/*
 * libSpiff - XSPF playlist handling library
 *
 * Copyright (C) 2007, Sebastian Pipping / Xiph.Org Foundation
 * All rights reserved.
 *
 * Redistribution  and use in source and binary forms, with or without
 * modification,  are permitted provided that the following conditions
 * are met:
 *
 *     * Redistributions   of  source  code  must  retain  the   above
 *       copyright  notice, this list of conditions and the  following
 *       disclaimer.
 *
 *     * Redistributions  in  binary  form must  reproduce  the  above
 *       copyright  notice, this list of conditions and the  following
 *       disclaimer   in  the  documentation  and/or  other  materials
 *       provided with the distribution.
 *
 *     * Neither  the name of the Xiph.Org Foundation nor the names of
 *       its  contributors may be used to endorse or promote  products
 *       derived  from  this software without specific  prior  written
 *       permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS  IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT  NOT
 * LIMITED  TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS
 * FOR  A  PARTICULAR  PURPOSE ARE DISCLAIMED. IN NO EVENT  SHALL  THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL,    SPECIAL,   EXEMPLARY,   OR   CONSEQUENTIAL   DAMAGES
 * (INCLUDING,  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES;  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT  LIABILITY,  OR  TORT (INCLUDING  NEGLIGENCE  OR  OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Sebastian Pipping, sping@xiph.org
 */

#include <spiff/Spiff.h>
#include <stdio.h>
#include <cstdlib>

using namespace Spiff;



class DemoPlaylist : public SpiffReaderCallback {

private:
	void addTrack(SpiffTrack * track) {
		open(_PT("Track"));
			printKeyValue(_PT("Album"), track->getAlbum());
			printKeyValue(_PT("Annotation"), track->getAnnotation());
			printKeyValue(_PT("Creator"), track->getCreator());
			printKeyValue(_PT("Duration"), track->getDuration());
			printIdentifiers(track);
			printKeyValue(_PT("Image"), track->getImage());
			printKeyValue(_PT("Info"), track->getInfo());
			printLinks<SpiffTrack>(track);
			printLocations(track);
			printMetas<SpiffTrack>(track);
			printKeyValue(_PT("Title"), track->getTitle());
			printKeyValue(_PT("TrackNum"), track->getTrackNum());
		close();
		delete track;
	}

	void setProps(SpiffProps * props) {
		open(_PT("Props"));
			printKeyValue(_PT("Annotation"), props->getAnnotation());
			printAttributions(props);
			printKeyValue(_PT("Creator"), props->getCreator());
			printKeyValue(_PT("Date"), props->getDate());
			printKeyValue(_PT("Identifier"), props->getIdentifier());
			printKeyValue(_PT("Image"), props->getImage());
			printKeyValue(_PT("Info"), props->getInfo());
			printKeyValue(_PT("License"), props->getLicense());
			printLinks<SpiffProps>(props);
			printKeyValue(_PT("Location"), props->getLocation());
			printMetas<SpiffProps>(props);
			printKeyValue(_PT("Title"), props->getTitle());
			printKeyValue(_PT("Version"), props->getVersion());
		close();
		delete props;
	}

	static void open(const XML_Char * type) {
		PORT_PRINTF(_PT("%s\n"), type);
	}

	static void close() {
		PORT_PRINTF(_PT("\n"));
	}

	static void printKeyValue(const XML_Char * key, const XML_Char * value) {
		if (value == NULL) {
			PORT_PRINTF(_PT("\t%s: -\n"), key);
		} else {
			PORT_PRINTF(_PT("\t%s: '%s'\n"), key, value);
		}
	}

	static void printKeyValue(const XML_Char * key, int value) {
		if (value == -1) {
			PORT_PRINTF(_PT("\t%s: -\n"), key);
		} else {
			PORT_PRINTF(_PT("\t%s: '%i'\n"), key, value);
		}
	}

	static void printKeyValue(const XML_Char * key, const SpiffDateTime * value) {
		if (value == NULL) {
			PORT_PRINTF(_PT("\t%s: -\n"), key);
		} else {
			PORT_PRINTF(_PT("\t%s: %04i-%02i-%02i %02i:%02i:%02i %s%02i:%02i\n"), key,
					value->getYear(), value->getMonth(), value->getDay(), value->getHour(), value->getMinutes(),
					value->getSeconds(), (value->getDistHours() < 0) ? _PT("-") : _PT("+"),
					std::abs(value->getDistHours()), std::abs(value->getDistMinutes()));
		}
	}

	static void printHelper(const XML_Char * key, int size) {
		PORT_PRINTF(_PT("\t%s: %i entr%s\n"), key, size, (size == 1) ? _PT("y") : _PT("ies"));
	}

	template <class T>
	static void printLinks(T * data) {
		const int size = (static_cast<SpiffData *>(data))->getLinkCount();
		printHelper(_PT("Links"), size);
	}

	template <class T>
	static void printMetas(T * data) {
		const int size = (static_cast<SpiffData *>(data))->getMetaCount();
		printHelper(_PT("Metas"), size);
	}

	static void printAttributions(SpiffProps * props) {
		const int size = props->getAttributionCount();
		printHelper(_PT("Attributions"), size);
	}

	static void printIdentifiers(SpiffTrack * track) {
		const int size = track->getIdentifierCount();
		printHelper(_PT("Identifier"), size);
	}

	static void printLocations(SpiffTrack * track) {
		const int size = track->getLocationCount();
		printHelper(_PT("Locations"), size);
	}

};



int PORT_MAIN() {
	SpiffReader reader;
	const int res = reader.parseFile(_PT("playlist.xspf"), new DemoPlaylist());
	if (res != SPIFF_READER_SUCCESS) {
		PORT_PRINTF(_PT("Error %i at line %i: '%s'\n"), res, reader.getErrorLine(), reader.getErrorText());
	} else {
		PORT_PRINTF(_PT("Everything fine.\n"));
	}
	return 0;
}
