/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include "libpolyxmass-monomerspec.h"



/* NEW'ING FUNCTIONS, DUPLICATING FUNCTIONS ...
 */
PxmMonomerSpec *
libpolyxmass_monomerspec_new (void)
{
  PxmMonomerSpec *ms = NULL;
  
  ms = g_malloc0 (sizeof (PxmMonomerSpec));

  return ms;
}


gboolean
libpolyxmass_monomerspec_set_raster (PxmMonomerSpec *ms, gchar *file)
{
  g_assert (ms != NULL && file != NULL);

  if (ms->raster != NULL)  
    g_free (ms->raster);
  
  ms->raster = g_strdup (file);
  
  return TRUE;
}  

gboolean
libpolyxmass_monomerspec_set_vector (PxmMonomerSpec *ms, gchar *file)
{
  g_assert (ms != NULL && file != NULL);

  if (ms->vector != NULL)  
    g_free (ms->vector);
  
  ms->vector = g_strdup (file);
  
  return TRUE;
}  


gboolean
libpolyxmass_monomerspec_set_name (PxmMonomerSpec *ms, gchar *name)
{
  g_assert (ms != NULL && name != NULL);

  if (ms->name != NULL)  
    g_free (ms->name);
  
  ms->name = g_strdup (name);
  
  return TRUE;
}


gboolean
libpolyxmass_monomerspec_set_code (PxmMonomerSpec *ms, gchar *code)
{
  g_assert (ms != NULL && code != NULL);

  if (ms->code != NULL)  
    g_free (ms->code);
  
  ms->code = g_strdup (code);
  
  return TRUE;
}


gboolean
libpolyxmass_monomerspec_set_name_sound (PxmMonomerSpec *ms, gchar *name)
{
  g_assert (ms != NULL && name != NULL);

  if (ms->name_sound != NULL)  
    g_free (ms->name_sound);
  
  ms->name_sound = g_strdup (name);
  
  return TRUE;
}


gboolean
libpolyxmass_monomerspec_set_code_sound (PxmMonomerSpec *ms, gchar *code)
{
  g_assert (ms != NULL && code != NULL);
  
  if (ms->code_sound != NULL)  
    g_free (ms->code_sound);
  
  ms->code_sound = g_strdup (code);
  
  return TRUE;
}





/*  LOCATING FUNCTIONS
 */
PxmMonomerSpec *
libpolyxmass_monomerspec_get_ptr_by_code (GPtrArray *GPA, gchar *code)
{
  return libpolyxmass_monomerspec_get_ptr_top_by_code (GPA, code);
}


PxmMonomerSpec *
libpolyxmass_monomerspec_get_ptr_top_by_code (GPtrArray *GPA, gchar *code)
{
  gint iter = 0;
  PxmMonomerSpec *monomerspec = NULL;
  

  g_assert (code != NULL);
  g_assert (GPA != NULL);
  
  for (iter = 0; iter < GPA->len; iter ++)
    {
      monomerspec = g_ptr_array_index (GPA, iter);
      g_assert (monomerspec != NULL);
	    
      if (0 == strcmp (monomerspec->code, code))
	return monomerspec;
    }
  
  return NULL;
}


PxmMonomerSpec *
libpolyxmass_monomerspec_get_ptr_bottom_by_code (GPtrArray *GPA, 
						   gchar *code)
{
  gint iter = 0;
  PxmMonomerSpec *monomerspec = NULL;
  

  g_assert (code != NULL);
  g_assert (GPA != NULL);
  
  if (GPA->len > 0)
    {
      for (iter = GPA->len -1 ; iter >= 0; iter--)
	{
	  monomerspec = g_ptr_array_index (GPA, iter);
	  g_assert (monomerspec != NULL);

	  if (0 == strcmp (monomerspec->code, code))
	    return monomerspec;
	}
    }
    
  return NULL;
}


gint
libpolyxmass_monomerspec_get_index_by_code (GPtrArray *GPA, gchar *code)
{
  return libpolyxmass_monomerspec_get_index_top_by_code (GPA, code);
}


gint
libpolyxmass_monomerspec_get_index_top_by_code (GPtrArray *GPA, gchar *code)
{
  gint iter = 0;
  PxmMonomerSpec *monomerspec = NULL;
  

  g_assert (GPA != NULL);
  g_assert (code != NULL);

  for (iter = 0; iter < GPA->len; iter ++)
    {
      monomerspec = g_ptr_array_index (GPA, iter);
      g_assert (monomerspec != NULL);
      
      if (0 == strcmp (monomerspec->code, code))
	return iter;
    }
  
  return -1;
}

gint
libpolyxmass_monomerspec_get_index_bottom_by_code (GPtrArray *GPA, 
						   gchar *code)
{
  gint iter = 0;
  PxmMonomerSpec *monomerspec = NULL;
  

  g_assert (GPA != NULL);
  g_assert (code != NULL);
    
  if (GPA->len > 0)
    {
      for (iter = GPA->len -1 ; iter >= 0; iter--)
	{
	  monomerspec = g_ptr_array_index (GPA, iter);
	  g_assert (monomerspec != NULL);
	  
	  if (0 == strcmp (monomerspec->code, code))
	    return iter;
	}
    }
    
  return -1;
}





/* UTILITY FUNCTIONS
 */
PxmMonomerSpec *
libpolyxmass_monomerspec_init_from_monicons_dic_line (gchar *line, 
						      gchar *dir) 
{
  gint iter = 0;

  GString * gs = NULL;

  gchar *help = NULL;
  
  gboolean code_done = FALSE;

  gboolean vector_done = FALSE;
  gboolean raster_done = FALSE;
  
  PxmMonomerSpec *ms = NULL;


  g_assert (line != NULL);
  g_assert (dir != NULL);
  
  gs = g_string_new ("");
  g_assert (gs != NULL);

  /* We have a line of the kind
     "monomer;code=pixmapfile.svg|pixmapfile.xpm" (note the newline
     character may be found if the line in question is not the last of
     the file!), we just need to deconstruct this into three pieces:
     code, pixmapfile.svg and pixmapfile.xpm.
   

     Note that one of the two graphics files may miss. Which is that
     we may have something like:
     
     monomer;R=arginine.svg|
     or
     monomer;R=|arginine.xpm
  */
  
  /* We should place the pointer to the first letter of the code, that
     is on the character right of the ';' of "monomer;".
  */
  g_assert (g_str_has_prefix (line, "monomer;"));

  help = strchr (line, ';');
  line = ++help;
  
  /* Allocate the monomerspec instance that we'll fill with data 
     parsed out of the currently analyzed line.
  */
  ms = libpolyxmass_monomerspec_new ();

  while (line [iter] != '\x0' && line [iter] != '\n')
    {
      if (line [iter] == '=')
	{
	  /* We should have something in gs, namely the code.
	   */
	  if (strlen (gs->str) <= 0)
	    {
	      g_string_free (gs, TRUE);
	      
	      libpolyxmass_monomerspec_free (ms);
	      
	      return NULL;
	    }

	  libpolyxmass_monomerspec_set_code (ms, gs->str);
	  
	  gs = g_string_truncate (gs, 0);
	  
	  code_done = TRUE;
	}

      else if (line [iter] == '|')
	{
	  /* We should have something in gs, namely the vector
	     filename. However, the string gs may be empty, as
	     explained above.
	   */
	  if (strlen (gs->str) > 0)
	    {
	      /* Make an absolute filename.
	       */
	      help = g_strdup_printf ("%s/%s", dir, gs->str);
	      
	      /* 
		 Make sure that the file actually exists. Also be
		 certain that the help string is an absolute filename.
	      */
	      g_assert (TRUE == g_file_test (help, G_FILE_TEST_EXISTS));
	      g_assert (TRUE == g_path_is_absolute (help));
	      
	      libpolyxmass_monomerspec_set_vector (ms, help);
	      
	      g_free (help);
	      
	      gs = g_string_truncate (gs, 0);
	      
	      vector_done = TRUE;
	    }
	}
      
      else
	{
	  /* We are iterating in some character, so we should simply
	   * put this into the gs string.
	   */
	  gs = g_string_append_c (gs, line[iter]);
	}

      iter++;		    
    }

  /* And now we are at the end of the line, so the raster file name
   * should have completed now, in gs as above. However, the gs string
   * may be empty, because the raster filename may be empty, if the
   * vector filename is not. So we have to check this now. Also, do
   * not forget to check that the code was done already.
   */
  if (strlen (gs->str) > 0)
    {
      /* Make an absolute filename.
       */
      help = g_strdup_printf ("%s/%s", dir, gs->str);
      
      /* 
	 Make sure that the file actually exists. Also be
	 certain that the help string is an absolute filename.
      */
      g_assert (TRUE == g_file_test (help, G_FILE_TEST_EXISTS));
      g_assert (TRUE == g_path_is_absolute (help));
	      
      libpolyxmass_monomerspec_set_raster (ms, help);
      
      g_free (help);

      raster_done = TRUE;
    }

  else 
    {
      if (vector_done == FALSE)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: both vector/raster file names cannot be emtpy\n"),
		 __FILE__, __LINE__);

	  libpolyxmass_monomerspec_free (ms);
	  
	  g_string_free (gs, TRUE);
	  
	  return NULL;
	}
    }
  
  /* Apparently, at least one of the two vector/raster file names 
   * is not empty, which is good. We still have to check that the
   * code was set properly in a previous parsing step.
   */
  if (code_done == FALSE)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: line not well formed: %s\n"),
	     __FILE__, __LINE__, line);
      
      libpolyxmass_monomerspec_free (ms);
      
      g_string_free (gs, TRUE);
      
      return NULL;
    }
  
  g_string_free (gs, TRUE);
  
  return ms;
}


gint
libpolyxmass_monomerspec_parse_monicons_dic_file (gchar *file,
						  GPtrArray *GPA)
{
  gint count = 0;
  
  gchar *line = NULL; 
  gchar *read = NULL;
  gchar * dir = NULL;

  FILE *cfgp = NULL;

  PxmMonomerSpec *ms = NULL;


  g_assert (GPA != NULL);
  g_assert (file != NULL);
  
  if (strlen (file) <= 0)
    return -1;
  
  /* Note that we NEED that the file name passed as parameter be an
     absolute file name because we are going to use the dirname to
     construct the PxmPolchemdefSpec data.
   */
  if (FALSE == g_path_is_absolute (file))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: file name is not absolute: '%s'\n"),
	     __FILE__, __LINE__, file);

      return -1;
    }

  /* Get the dir name from the abolute file name.
   */
  dir = g_path_get_dirname (file);

  cfgp = fopen (file, "r");
  
  if (NULL == cfgp)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: file not found: %s\n"),
	     __FILE__, __LINE__, file);

      g_free (dir);
      
      return -1;
    }

  /* The lines that we are interested in always start with the prefix
     "monomer;". So it is easy to find them...
  */

  /* Allocate the memory into which each line of the file will be
   * store for recursive parsing.
   */
  line = g_malloc0 (sizeof (gchar) * (MAX_LINE_LENGTH + 1));
  
  read = line;
  
  while (read != NULL)
    {
      read = fgets (read, MAX_LINE_LENGTH, cfgp);

      if (read != NULL && strlen (read) > 1)
	{
	  /* Apparently something interesting should be 
	   * in 'read' for us to parse.
	   */
	  if (FALSE == g_str_has_prefix (line, "monomer;"))
	    continue;
	  
	  ms = 
	    libpolyxmass_monomerspec_init_from_monicons_dic_line (read,
								  dir);
	  
	  if (NULL == ms)
	    {
	      g_free (line);
	      
	      fclose (cfgp);
	      
	      g_free (dir);
	      
	      return -1;
	    }
	  else
	    {
	      count++;
	      
	      g_ptr_array_add (GPA, ms);
	      
	      continue;
	    }
	}
      /* End of
	 if (read != NULL && strlen (read) > 1)
      */
    }
  /* End of   
     while (read != NULL)
  */

  /* Now that we do not need this string anymore, since the file is
   * finished parsing, we can free the memory for reuse of the pointer
   */
  g_free (line);
  
  fclose (cfgp);

  g_free (dir);
  
  return count;
}



PxmMonomerSpec *
libpolyxmass_monomerspec_init_from_sounds_dic_line (gchar *line, 
						    gchar *dir) 
{
  gint iter = 0;

  GString * gs = NULL;

  gchar *help = NULL;
  
  gboolean code_done = FALSE;

  gboolean name_sound_done = FALSE;
  gboolean code_sound_done = FALSE;
  
  PxmMonomerSpec *ms = NULL;


  g_assert (line != NULL);
  g_assert (dir != NULL);
  
  gs = g_string_new ("");
  g_assert (gs != NULL);

  /* We have a line of the kind "monomer;A=alanine.ogg|a.ogg" (note
     the newline character may be found if the line in question is not
     the last of the file!), we just need to deconstruct this into
     three pieces: code, monomer's name sound file and monomer(s code
     soudn file.
   */

  /* The lines that we have to parse here are in this format:

     monomer;A=alanine.ogg|a.ogg
     
     Note that any of the two file may miss. Which is that we may have
     something like:
     
     R=arginine.ogg|
     or
     R=|r.ogg
   
     Note that this file contains a line like the following

     silence-sound$silence.ogg
     
     which indicates which file should be used to insert silent spaces
     in the ogg files to be constructed (delays between monomer
     sounds, for example).
     
     That line is parsed by another function, so we do not care about it
     here.
  */

  /* We should place the pointer to the first letter of the code, that
     is on the character right of the ';' of "monomer;".
  */
  g_assert (g_str_has_prefix (line, "monomer;"));

  help = strchr (line, ';');
  line = ++help;

  /* Allocate the monomerspec instance that we'll fill with data 
   * parsed out of the currently analyzed line.
   */
  ms = libpolyxmass_monomerspec_new ();

  while (line [iter] != '\x0' && line [iter] != '\n' && line [iter] != '$')
    {
      if (line [iter] == '=')
	{
	  /* We should have something in gs, namely the code.
	   */
	  if (strlen (gs->str) <= 0)
	    {
	      g_string_free (gs, TRUE);
	      
	      libpolyxmass_monomerspec_free (ms);
	      
	      return NULL;
	    }

	  libpolyxmass_monomerspec_set_code (ms, gs->str);
	  
	  gs = g_string_truncate (gs, 0);
	  
	  code_done = TRUE;
	}
      else if (line [iter] == '|')
	{
	  /* We should have something in gs, namely the name_sound
	     filename. However, the string gs may be empty, as
	     explained above.
	   */
	  if (strlen (gs->str) > 0)
	    {
	      /* Make an absolute filename.
	       */
	      help = g_strdup_printf ("%s/%s", dir, gs->str);
	      
	      /* 
		 Make sure that the file actually exists. Also be
		 certain that the help string is an absolute filename.
	      */
	      g_assert (TRUE == g_file_test (help, G_FILE_TEST_EXISTS));
	      g_assert (TRUE == g_path_is_absolute (help));
	      
	      libpolyxmass_monomerspec_set_name_sound (ms, help);
	      
	      g_free (help);
	      
	      gs = g_string_truncate (gs, 0);
	      
	      name_sound_done = TRUE;
	    }
	}
      else
	{
	  /* We are iterating in some character, so we should simply
	   * put this into the gs string.
	   */
	  gs = g_string_append_c (gs, line[iter]);
	}

      iter++;		    
    }

  /* And now we are at the end of the line, so the code_sound filename
     should have completed now, in gs as above. However, the gs string
     may be empty, because the code_sound filename may be empty, if
     the name_sound filename is not. So we have to check this
     now. Also, do not forget to check that the code was done already.
   */
  if (strlen (gs->str) > 0)
    {
      /* Make an absolute filename.
       */
      help = g_strdup_printf ("%s/%s", dir, gs->str);
      
      /* 
	 Make sure that the file actually exists. Also be
	 certain that the help string is an absolute filename.
      */
      g_assert (TRUE == g_file_test (help, G_FILE_TEST_EXISTS));
      g_assert (TRUE == g_path_is_absolute (help));
	      
      libpolyxmass_monomerspec_set_code_sound (ms, help);
      
      g_free (help);

      code_sound_done = TRUE;
    }
  else 
    {
      if (name_sound_done == FALSE)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: both name/code sound file names "
		   "cannot be emtpy\n"),
		 __FILE__, __LINE__);

	  libpolyxmass_monomerspec_free (ms);
	  
	  g_string_free (gs, TRUE);
	  
	  return NULL;
	}
    }
  
  /* Apparently, at least one of the two name/code sound file names is
     not empty, which is good. We still have to check that the code
     was set properly defined in a previous parsing step.
   */
  if (code_done == FALSE)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: line not well formed: %s\n"),
	     __FILE__, __LINE__, line);
      
      libpolyxmass_monomerspec_free (ms);
      
      g_string_free (gs, TRUE);
      
      return NULL;
    }
  
  g_string_free (gs, TRUE);
  
  return ms;
}


gint
libpolyxmass_monomerspec_parse_sounds_dic_file (gchar *file,
						GPtrArray *GPA)
{
  gint count = 0;
  
  gchar *line = NULL; 
  gchar *read = NULL;
  gchar * dir = NULL;

  FILE *cfgp = NULL;

  PxmMonomerSpec *ms = NULL;
  PxmMonomerSpec *ms_local = NULL;

  g_assert (GPA != NULL);
  g_assert (file != NULL);
  
  if (strlen (file) <= 0)
    return -1;
  
  /* Note that we NEED that the file name passed as parameter be an
     absolute file name because we are going to use the dirname to
     construct the PxmPolchemdefSpec data.
   */
  if (FALSE == g_path_is_absolute (file))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: file name is not absolute: '%s'\n"),
	     __FILE__, __LINE__, file);

      return -1;
    }

  /* Get the dir name from the abolute file name.
   */
  dir = g_path_get_dirname (file);

  cfgp = fopen (file, "r");
  
  if (NULL == cfgp)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: file not found: %s\n"),
	     __FILE__, __LINE__, file);

      g_free (dir);
      
      return -1;
    }

  /* The lines that we have to parse here are in this format:
   
     monomer;A=alanine.ogg|a.ogg
   
     Note that any of the two file may miss. Which is that we may have
     something like:
   
     monomer;R=arginine.ogg|
     or
     monomer;R=|r.ogg
   */

  /* The lines that we are interested in always start with the prefix
     "monomer;". So it is easy to find them...
  */

  /* Allocate the memory into which each line of the file will be
   * store for recursive parsing.
   */
  line = g_malloc0 (sizeof (gchar) * (MAX_LINE_LENGTH + 1));

  read = line;

  while (read != NULL)
    {
      read = fgets (read, MAX_LINE_LENGTH, cfgp);

      if (read != NULL && strlen (read) > 1)
	{
	  /* Apparently something interesting should be 
	   * in 'read' for us to parse.
	   */
	  if (FALSE == g_str_has_prefix (line, "monomer;"))
	    continue;
	  
	  ms = libpolyxmass_monomerspec_init_from_sounds_dic_line (read, dir);
	  
	  if (NULL == ms)
	    {
	      g_free (line);
	      
	      fclose (cfgp);
	      
	      g_free (dir);
	      
	      return -1;
	    }
	  else
	    {
	      /* At this point, we add this new monomer spec instance
		 in the array of the monomer spec instances, only if
		 one such item does not exist already. If one such
		 item exists already, we have to update only the items
		 we are interested in: sound data exclusively, as
		 other monomer raster/vector data were setup already
		 in another function and we do not want to overwrite
		 these.
	      */
	      ms_local = libpolyxmass_monomerspec_get_ptr_by_code (GPA,
								   ms->code);
	      if (ms_local == NULL)
		{
		  /* No item in the array has the same code, so we 
		     just add the new one and that's done.
		  */
		  count++;
		  
		  g_ptr_array_add (GPA, ms);
		  
		  continue;
		}
	      else
		{
		  /* Apparently an item by the same code already
		     exists.  We only do update the data that we
		     are interested in here:
		  */
		  if (ms->name_sound != NULL)
		    libpolyxmass_monomerspec_set_name_sound 
		      (ms_local, ms->name_sound);
		  
		  if (ms->code_sound != NULL)
		    libpolyxmass_monomerspec_set_code_sound 
		      (ms_local, ms->code_sound);
		  
		  /* At this time we can free the ms that was returned
		     from the parsing code, as, this time, we do not
		     need it.
		  */
		  count++;
		  
		  continue;
		}
	    }
	}
      /* Enf of 
	 if (read != NULL && strlen (read) > 1)
      */
    }
  /* End of   
     while (read != NULL)
  */

  /* Now that we do not need this string anymore, since the file is
   * finished parsing, we can free the memory for reuse of the pointer
   */
  g_free (line);
  
  fclose (cfgp);

  g_free (dir);
  
  return count;
}



/* FREE'ING FUNCTIONS
 */
gint
libpolyxmass_monomerspec_free (PxmMonomerSpec *ms)
{
  g_assert (ms != NULL);
  
  if (ms->raster != NULL)
    g_free (ms->raster);
  
  if (ms->vector != NULL)
    g_free (ms->vector);
  
  if (ms->name != NULL)
    g_free (ms->name);
  
  if (ms->code != NULL)
    g_free (ms->code);
  
  if (ms->name_sound != NULL)
    g_free (ms->name_sound);
 
  if (ms->code_sound != NULL)
    g_free (ms->code_sound);
  
  g_free (ms);
  
  return 1;
}




/* GPtrArray-RELATED FUNCTIONS
 */
gint
libpolyxmass_monomerspec_GPA_empty (GPtrArray *GPA)
{
  gint count = 0;
  
  PxmMonomerSpec *ms = NULL;
  

  g_assert (GPA != NULL);
  
  while (GPA->len > 0)
    {
      ms = g_ptr_array_remove_index (GPA, 0);
      g_assert (ms != NULL);
      libpolyxmass_monomerspec_free (ms);
      count++;
    }
  
  return count;
}


gint
libpolyxmass_monomerspec_GPA_free (GPtrArray *GPA)
{
  gint count = 0;
  

  g_assert (GPA != NULL);

  count = libpolyxmass_monomerspec_GPA_empty (GPA);
    
  g_ptr_array_free (GPA, TRUE);

  return count;
}


