/***************************************************************************
  $RCSfile: keyfile.cpp,v $
  -------------------
  cvs         : $Id: keyfile.cpp,v 1.10 2003/05/18 21:46:36 aquamaniac Exp $
  begin       : Mon Apr 14 2003
  copyright   : (C) 2003 by Martin Preuss
  email       : martin@libchipcard.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef __declspec
# if BUILDING_DLL
#  define DLLIMPORT __declspec (dllexport)
# else /* Not BUILDING_DLL */
#  define DLLIMPORT __declspec (dllimport)
# endif /* Not BUILDING_DLL */
#else
# define DLLIMPORT
#endif


#include "openhbci.h"
#include "keyfile.h"
#include "mediumkeyfile.h"
#include "hbcistring.h"
#include <string.h>


namespace HBCI {



KeyFilePlugin::KeyFilePlugin(const API *api)
    :MediumPlugin()
,_api(api)
{

}


KeyFilePlugin::~KeyFilePlugin(){
}


Pointer<Medium> KeyFilePlugin::mediumFactory(const string &name){
  Pointer<Medium> m;

  m=new MediumKeyfile(_api, name);
  return m;
}



Error KeyFilePlugin::mediumCheck(string &name) const {
  string tmp;
  int size;
  Error err;
  Pointer<File> f;

  if (name.empty()) {
    return Error("KeyfilePlugin::mediumCheck",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_INVALID,
		 ERROR_ADVISE_ABORT,
		 "Empty name.");
  }

  f=new File(name);

  // check whether the file exists
  err=f.ref().accessFile(FILE_ACCESS_EXIST);
  if (!err.isOk()) {
    return Error("KeyfilePlugin::mediumCheck",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_INVALID,
		 ERROR_ADVISE_ABORT,
		 "File does not exist.");
  }

  // check whether the file is read- and writeable
  err=f.ref().accessFile(FILE_ACCESS_RW);
  if (!err.isOk()) {
    return Error("KeyfilePlugin::mediumCheck",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_INVALID,
		 ERROR_ADVISE_ABORT,
		 "File exists but I have no rights on it.");
  }

  // ok, now open the file
  err=f.ref().openFile(FILE_AM_OPEN_EXISTING);
  if (!err.isOk()) {
    return Error("KeyfilePlugin::mediumCheck",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_INVALID,
		 ERROR_ADVISE_ABORT,
		 "File exists, I have all rights but I could not open it.");
  }

  // read tag type and size
  err=f.ref().readData(tmp,3);
  f.ref().closeFile();
  if (!err.isOk()) {
    return Error("KeyfilePlugin::mediumCheck",
		 err);
  }

  if (tmp.length()!=3)
    return Error("KeyfilePlugin::mediumCheck",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_BAD_MEDIUM,
		 ERROR_ADVISE_ABORT,
		 "This seems not to be a key file.");
  if (String::typeTLV(tmp)!=MEDIUMKEYFILE_TAG_CRYPT)
    return Error("KeyfilePlugin::mediumCheck",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_BAD_MEDIUM,
		 ERROR_ADVISE_ABORT,
		 "This seems not to be a key file (bad type).");
  size=String::sizeTLV(tmp);
  if (size % 8)
    return Error("KeyfilePlugin::mediumCheck",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_BAD_MEDIUM,
		 ERROR_ADVISE_ABORT,
		 "This seems not to be a key file (bad size).");
  return Error();
}





Pointer<Medium> KeyFilePlugin::createNewMedium(bool readonly,
					       int country,
					       const string &bankId,
					       const string &userId,
					       const string &name){
  Pointer<Medium> medium;
  Error err;

  // name empty ?
  if (name.empty())
    throw Error("KeyFilePlugin::createNewMedium()",
		ERROR_LEVEL_NORMAL,
		0,
		ERROR_ADVISE_DONTKNOW,
		"need name"
		"for RDH mode the name contains the path for the keyfile");

  // check if file already exists
  File f(name);
  if (!f.accessFile(FILE_ACCESS_EXIST).isOk()) {
    // File does not exist

    // create medium; Create new MediumRDH object
    MediumKeyfile *mediumtmp = new MediumKeyfile(_api,name);
    // Store it in the upcast version
    medium = mediumtmp;
    // For convenience, use a reference in the following
    MediumKeyfile &mediumrdh = *mediumtmp;

    string pin;

    err=_api->authentificator().ref().getSecret(0,
						name,
						pin,
						true);
    if (!err.isOk())
      throw Error("KeyFilePlugin::createNewMedium",err);

    // create context
    err=mediumrdh.createMedium(country,
			       bankId,
			       userId,
			       pin);
    if (!err.isOk())
      throw Error("KeyFilePlugin::createNewMedium()", err);

    // mount medium without customer
    err=mediumrdh.mountMedium(pin);
    if (!err.isOk())
      throw Error("KeyFilePlugin::createNewMedium()", err);

    // select context
    err=mediumrdh.selectContext(country, bankId, userId);
    if (!err.isOk())
      throw Error("KeyFilePlugin::createNewMedium()", err);

    // create keys
    err=mediumrdh.createUserKeys();
    if (!err.isOk())
      throw Error("KeyFilePlugin::createNewMedium()", err);

    // unmount medium
    err=mediumrdh.unmountMedium(pin);
    if (!err.isOk())
      throw Error("KeyFilePlugin::createNewMedium()", err);
  } // if not exists
  else
    // File exists, so: create medium
    medium=new MediumKeyfile(_api,name);

  // finished

  return medium;
}


Pointer<Medium> KeyFilePlugin::mediumFromConfig(SimpleConfig &cfg,
						cfgPtr group){
  string typ;
  string name;
  Pointer<Medium> m;

  typ=cfg.getVariable("mediumtype","",group);
  if (strcasecmp(typ.c_str(), mediumTypeName().c_str())!=0) {
    throw Error("KeyFilePlugin::mediumFromConfig",
		ERROR_LEVEL_NORMAL,
		HBCI_ERROR_CODE_INVALID,
		ERROR_ADVISE_DONTKNOW,
		"bad medium type");
  }
  name=cfg.getVariable("mediumname",
		       cfg.getVariable("keyfile","",group),
		       group);
  m=new MediumKeyfile(_api, name);
  return m;
}


Error KeyFilePlugin::mediumToConfig(Pointer<Medium> m,
				    SimpleConfig &cfg,
				    cfgPtr group){
  Pointer<MediumKeyfile> mkf;

  try {
    mkf=m.cast<MediumKeyfile>();
  }
  catch (Error xerr) {
    return Error("KeyFilePlugin::mediumToConfig",xerr);
  }
  cfg.setVariable("mediumtype",m.ref().mediumTypeName(),group);
  cfg.setVariable("mediumname",mkf.ref().mediumName(),group);

  return Error();
}


Error _checkVersion(const API *api) {
  int vmajor, vminor, vpl, vbuild;

  /* do some version checking
   * Please note that the constants used below are defined upon
   * compilation of this plugin. If you compile this plugin using
   * OpenHBCI 0.9.10 and try to use it on a system where OpenHBCI 0.9.9
   * is installed, this module will refuse to work. This plugin only
   * works with versions of OpenHBCI which are the same or newer than
   * the version used to compile this plugin.
   * If the major version differs then this module believes that the
   * version of OpenHBCI used is totally incompatible with the version
   * used to build this plugin.
   */
  api->libraryVersion(vmajor, vminor, vpl, vbuild);
  if (vmajor!=OPENHBCI_VERSION_MAJOR) {
    /* this is the major compatibility check. The major version is
     * incremented if a newer version is incompatible with older ones */
    fprintf(stderr,
	    " Different major versions, "
	    "please recompile RDHFile plugin.\n");
    return HBCI::Error("Keyfile Plugin",
		       ERROR_LEVEL_NORMAL,
		       HBCI_ERROR_CODE_MEDIUM,
		       ERROR_ADVISE_DONTKNOW,
		       "Major version does not match");
  }
  if (!(
	((vminor==OPENHBCI_VERSION_MINOR)&&(vpl>OPENHBCI_VERSION_PATCHLEVEL))
	||
	((vminor==OPENHBCI_VERSION_MINOR) &&
	 (vpl==OPENHBCI_VERSION_PATCHLEVEL) &&
	 (vbuild>=OPENHBCI_VERSION_BUILD))
       )
     ) {
    fprintf(stderr,
	    "This plugin needs OpenHBCI v%d.%d.%d.%d or newer.\n",
	    OPENHBCI_VERSION_MAJOR,
	    OPENHBCI_VERSION_MINOR,
	    OPENHBCI_VERSION_PATCHLEVEL,
            OPENHBCI_VERSION_BUILD);
    return HBCI::Error("RDHFile Plugin",
		       ERROR_LEVEL_NORMAL,
		       HBCI_ERROR_CODE_MEDIUM,
		       ERROR_ADVISE_DONTKNOW,
		       "need OpenHBCI v"OPENHBCI_VERSION_FULL_STRING
		       " or newer");
  }

  return Error();
}



}; // namespace


extern "C" {
  // The version of openhbci's plugin interface system (i.e. the
  // convention that this plugin has to define a registerYourSelf()
  // function.)
  int rdhfile_openhbci_plugin_interface_version = OPENHBCI_PLUGIN_INTERFACE_VERSION;

  // Version numbers of this plugin
  int rdhfile_openhbci_module_current=KEYFILE_MODULE_CURRENT;
  int rdhfile_openhbci_module_revision=KEYFILE_MODULE_REVISION;
  int rdhfile_openhbci_module_age=KEYFILE_MODULE_AGE;


  // The function that registers this plugin at the HBCI::API. This
  // function will be looked up by dlsym() and called from the plugin
  // interface system.
  HBCI::Error rdhfile_registerYourSelf(HBCI::API *api) {
    HBCI::Pointer<HBCI::MediumPlugin> mp;
    HBCI::Error err;

    err=_checkVersion(api);
    if (!err.isOk())
      return HBCI::Error("Keyfile Plugin",err);

    mp=new HBCI::KeyFilePlugin(api);
    return api->registerMediumPlugin(mp);
  }

  /*
   * This function simply creates a medium plugin without registering it
   */
  HBCI::Pointer<HBCI::MediumPlugin> rdhfile_createPlugin(const HBCI::API *api){
    HBCI::Pointer<HBCI::MediumPlugin> mp;
    HBCI::Error err;

    err=_checkVersion(api);
    if (!err.isOk())
      throw HBCI::Error("Keyfile Plugin",err);

    mp=new HBCI::KeyFilePlugin(api);
    mp.setDescription("KeyFilePlugin");
    return mp;
  }


}; // extern "C"


