/* LogJam, a GTK LiveJournal client.
 * Copyright (C) 2000,2001 Evan Martin <evan@livejournal.com>
 * vim:ts=4:sw=4:
 *
 * $Id: network.c,v 1.11 2002/01/29 04:51:25 martine Exp $
 */

#include <gtk/gtk.h>
#include <gdk/gdk.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>

#include <curl/curl.h>
#include <pthread.h>

#include "protocol.h"
#include "util.h"
#include "dotconf.h"
#include "network.h"
#include "convert.h"

#include "../pixmaps/data1.xpm"
#include "../pixmaps/data2.xpm"
#include "../pixmaps/data3.xpm"
#include "../pixmaps/data4.xpm"

/* delay between rotation of the "throbber" */
#define PIXROTATE_TIMEOUT 200

/* if the content of the request is at least this many bytes, show
 * the progress bar. */
#define MINBYTES_FOR_PROGRESSBAR 4000

typedef struct {
	GtkWidget *win, *label, *button, *pix, *progress;

	CURL *curl;
	CURLcode res;
	char curlerrorbuf[CURL_ERROR_SIZE];
	
	GString *responsedata;

	int pixidx;
	int pixrotate_tag;

	int refcount;
	int cancelled;
} net_request;

static GdkPixmap *datapix[4]  = { 0 };
static GdkBitmap *datamask[4] = { 0 };

static void
stop_throbber(net_request *nr) {
	if (nr->pixrotate_tag) {
		gtk_timeout_remove(nr->pixrotate_tag);
		nr->pixrotate_tag = 0;
	}
}

static void 
set_status(net_request *nr, char *text) {
	gtk_label_set_text(GTK_LABEL(nr->label), text);
}

static void 
error(net_request *nr, char *text, int std) {
	char *msg;

	if (std) 
		msg = g_strdup_printf("Request failed:\n%s: %s.", 
				text, strerror(errno));
	else 
		msg = g_strdup_printf("Request failed:\n%s.", text);
	set_status(nr, msg);
	g_free(msg);

	stop_throbber(nr);
	gtk_main(); /* the cancel button will end this. */
}


static void 
create_pixs(GtkWidget *win) {
	datapix[0] = gdk_pixmap_create_from_xpm_d(win->window, &datamask[0], 
			NULL, data1_xpm);
	datapix[1] = gdk_pixmap_create_from_xpm_d(win->window, &datamask[1], 
			NULL, data2_xpm);
	datapix[2] = gdk_pixmap_create_from_xpm_d(win->window, &datamask[2], 
			NULL, data3_xpm);
	datapix[3] = gdk_pixmap_create_from_xpm_d(win->window, &datamask[3], 
			NULL, data4_xpm);
}

static gint 
pixrotate_cb(net_request *nr) {
	/* glib timeouts need to get the gdk lock. */
	gdk_threads_enter();
	nr->pixidx++;
	if (nr->pixidx >= 4) nr->pixidx = 0;
	gtk_pixmap_set(GTK_PIXMAP(nr->pix), 
		datapix[nr->pixidx], datamask[nr->pixidx]);
	gdk_threads_leave();
	return TRUE;
}

static void 
cancel_cb(GtkWidget *w, net_request *nr) {
	gtk_widget_destroy(nr->win);
	nr->cancelled = TRUE;

	gtk_main_quit(); 
}

static void 
win_destroy_cb(GtkWidget *w, net_request *nr) {
	stop_throbber(nr);
	nr->win = NULL;
}

static void 
create_win(net_request *nr, char *title, GtkWidget *parent) {
	GtkWidget *frame;
	GtkWidget *vbox;
	GtkWidget *hbox;

	nr->win = gtk_window_new(GTK_WINDOW_DIALOG);

	gtk_widget_realize(nr->win);
	gdk_window_set_decorations(nr->win->window, 0);
	gtk_window_set_title(GTK_WINDOW(nr->win), title);
	gtk_window_set_modal(GTK_WINDOW(nr->win), TRUE);
	gtk_window_set_default_size(GTK_WINDOW(nr->win), 200, -1);
	gtk_signal_connect(GTK_OBJECT(nr->win), "destroy",
		GTK_SIGNAL_FUNC(win_destroy_cb), nr);

	lj_win_set_icon(nr->win);

	if (datapix[0] == NULL)
		create_pixs(nr->win);

	if (parent)
		gtk_window_set_transient_for(GTK_WINDOW(nr->win), 
				GTK_WINDOW(parent));


	vbox = gtk_vbox_new(FALSE, 5); 
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);


	hbox = gtk_hbox_new(FALSE, 5); 

	nr->pix = gtk_pixmap_new(datapix[0], datamask[0]);
	nr->pixrotate_tag = gtk_timeout_add(PIXROTATE_TIMEOUT, 
			(GtkFunction)pixrotate_cb, nr);
	gtk_signal_connect_object(GTK_OBJECT(nr->pix), "destroy",
			GTK_SIGNAL_FUNC(gtk_timeout_remove), 
			GINT_TO_POINTER(nr->pixrotate_tag));
	gtk_box_pack_start(GTK_BOX(hbox), nr->pix, FALSE, FALSE, 0);

	nr->label = gtk_label_new("Network request processing...");
	gtk_label_set_line_wrap(GTK_LABEL(nr->label), TRUE);
	gtk_box_pack_start(GTK_BOX(hbox), nr->label, TRUE, TRUE, 0);

	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);

	hbox = gtk_hbox_new(FALSE, 5);
	nr->progress = gtk_progress_bar_new();
	gtk_box_pack_start(GTK_BOX(hbox), nr->progress, 
			TRUE, TRUE, 0);

	nr->button = gtk_button_new_with_label("  Cancel  ");
	gtk_signal_connect(GTK_OBJECT(nr->button), "clicked",
			GTK_SIGNAL_FUNC(cancel_cb), nr);
	gtk_box_pack_end(GTK_BOX(hbox), nr->button, 
			FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_OUT);
	gtk_container_add(GTK_CONTAINER(frame), vbox);
	gtk_widget_show_all(frame);

	gtk_container_add(GTK_CONTAINER(nr->win), frame);

	gtk_widget_hide(nr->progress);
	gtk_widget_show(nr->win);
}

static size_t 
curlwrite_nothread_cb(void *ptr, size_t size, size_t nmemb, void *data) {
	net_request *nr = data;

	g_string_append(nr->responsedata, ptr);

	if (nr->responsedata->len > MINBYTES_FOR_PROGRESSBAR) {
		double contentlength;
		curl_easy_getinfo(nr->curl, 
				CURLINFO_CONTENT_LENGTH_DOWNLOAD, &contentlength);

		gtk_progress_configure(GTK_PROGRESS(nr->progress),
				nr->responsedata->len, 0, contentlength);
		gtk_widget_show(nr->progress);
	}
	return size*nmemb;
}

static size_t 
curlwrite_cb(void *ptr, size_t size, size_t nmemb, void *data) {
	net_request *nr = data;

	if (nr->cancelled) /* don't do anything if they cancelled. */
		return -1;

	gdk_threads_enter();

	g_string_append(nr->responsedata, ptr);

	if (nr->responsedata->len > MINBYTES_FOR_PROGRESSBAR) {
		double contentlength;
		curl_easy_getinfo(nr->curl, 
				CURLINFO_CONTENT_LENGTH_DOWNLOAD, &contentlength);

		gtk_progress_configure(GTK_PROGRESS(nr->progress),
				nr->responsedata->len, 0, contentlength);
		gtk_widget_show(nr->progress);
	}
	gdk_threads_leave();
	return size*nmemb;
}

static void
hash_append_str(gpointer key, gpointer value, gpointer data) {
	GString *string = data;
	gchar *en_key, *en_value;

	if (key == NULL) return;
	if (conf.convertcharset == CONVERT_KOI2WIN) {
		koi2win(key);
		koi2win(value);
	}
	en_key = curl_escape(key, strlen(key));
	en_value = curl_escape(value, strlen(value));
	g_string_sprintfa(string, "%s=%s&", en_key, en_value);

	free(en_key);
	free(en_value);
}

static void
nr_decref(net_request *nr) {
	if (--nr->refcount <= 0) /* no locking needed because we use the gdk lock */
		g_free(nr);
}

static void *
curl_thread(void *data) {
	net_request *nr = data;

	nr->refcount++;
	nr->res = curl_easy_perform(nr->curl);
	curl_easy_cleanup(nr->curl);

	gdk_threads_enter();
	/* if the request wasn't cancelled already,
	 * we need to signal completion by ending the mainloop. */
	if (!nr->cancelled)
		gtk_main_quit();
	nr_decref(nr);
	gdk_threads_leave();
	return NULL;
}

char*
net_errmsg(GHashTable *response) {
	char *errmsg = NULL;
	
	if (response == NULL) {
		errmsg = "Malformed response from server";
	} else {
		errmsg = g_hash_table_lookup(response, "errmsg");
	}

	if (errmsg == NULL)
		errmsg = "Unable to parse server response";
	
	return errmsg;
}

static void 
handle_response_gui(net_request *nr, GHashTable *response) {
	if (lj_protocol_request_succeeded(response)) {
		gtk_widget_destroy(nr->win);
	} else {
		if (nr->res == CURLE_OK) {
			error(nr, net_errmsg(response), FALSE);
		} else {
			error(nr, nr->curlerrorbuf, FALSE);
		}
	}
}

static GHashTable* 
curl_request_run(net_request *nr, GHashTable *request) {
	GHashTable *response;
	GString *requestdata;
	char urlreq[1024], urlproxy[1024];

	nr->curl = curl_easy_init();
	if (nr->curl == NULL)
		return NULL;

	curl_easy_setopt(nr->curl, CURLOPT_VERBOSE, conf.netdump != 0);
	curl_easy_setopt(nr->curl, CURLOPT_NOPROGRESS, 1);
	curl_easy_setopt(nr->curl, CURLOPT_ERRORBUFFER, nr->curlerrorbuf);

	/*
	 * curl's progress function is mostly useless; we instead track writes.
	 *
	 * curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION, progress_cb);
	 * curl_easy_setopt(curl, CURLOPT_PROGRESSDATA, nr);
	 */
	
	snprintf(urlreq, sizeof(urlreq), "%s/interface/flat", conf.ljserver);
	curl_easy_setopt(nr->curl, CURLOPT_URL, urlreq);

	if (conf.fastserver) {
		curl_easy_setopt(nr->curl, CURLOPT_COOKIE, "ljfastserver=1;");
	}

	if (conf.useproxy) {
		snprintf(urlproxy, sizeof(urlproxy), "%s:%d", conf.proxyserver, 
				 conf.proxyport);
		curl_easy_setopt(nr->curl, CURLOPT_PROXY, urlproxy);
	}

	requestdata = g_string_sized_new(2048);
	g_hash_table_foreach(request, hash_append_str, requestdata);
	curl_easy_setopt(nr->curl, CURLOPT_POSTFIELDS, requestdata->str);
	curl_easy_setopt(nr->curl, CURLOPT_POSTFIELDSIZE, requestdata->len-1);
	curl_easy_setopt(nr->curl, CURLOPT_POST, 1);

	nr->responsedata = g_string_sized_new(2048);

	curl_easy_setopt(nr->curl, CURLOPT_FILE, nr);
	
	nr->res = !CURLE_OK; /* just set it to something other than ok. */

	if (conf.nothread || conf.postmode == POSTMODE_CMD) {
		curl_easy_setopt(nr->curl, CURLOPT_WRITEFUNCTION, curlwrite_nothread_cb);
		nr->res = curl_easy_perform(nr->curl);
		curl_easy_cleanup(nr->curl);
	} else {
		pthread_t threadid;
		curl_easy_setopt(nr->curl, CURLOPT_WRITEFUNCTION, curlwrite_cb);
		pthread_create(&threadid, NULL, curl_thread, nr);
		/* block here until the request completes. */
		gtk_main(); /* the thread or the user will quit this... */
	}

	if (conf.netdump) 
		g_print("Response: [%s]\n", nr->responsedata->str);

	g_string_free(requestdata, TRUE);

	response = lj_protocol_parse_response(nr->responsedata->str);
	g_string_free(nr->responsedata, TRUE);

	return response;
}

GHashTable*
net_request_run(GtkWidget *parent, char *title, GHashTable *request) {
	net_request *nr;

	GHashTable *response = NULL;

	nr = g_new0(net_request, 1);
	nr->refcount = 1; /* we use a refcount system because the thread can
						 live beyond this function, or die before this function
						 finishes. */

	create_win(nr, title, parent);

	response = curl_request_run(nr, request);
	if (response && !nr->cancelled) { 
		handle_response_gui(nr, response);
	}
	nr_decref(nr);

	return response;
}

GHashTable*
net_request_new(char *mode) {
	GHashTable *request = lj_protocol_request_new(mode, 
			conf.username, conf.password, conf.usejournal);

	return request;
}

GHashTable*
net_request_run_cli(GHashTable *request) {
	net_request actual_nr = {0}, *nr = &actual_nr;
	GHashTable *response = NULL;

	curl_request_run(nr, request);

	if (nr->res != CURLE_OK) {
		g_print("HTTP error: %s\n", nr->curlerrorbuf);
	} else {
		response = lj_protocol_parse_response(nr->responsedata->str);
	}

	g_string_free(nr->responsedata, TRUE);

	return response;
}

gboolean
net_request_succeeded(GHashTable *response) {
	return lj_protocol_request_succeeded(response);
}
