/* LogJam, a GTK LiveJournal client.
 * Copyright (C) 2000,2001 Evan Martin <evan@livejournal.com>
 * vim:ts=4:sw=4:
 *
 * $Id: friendgroups.c,v 1.2 2001/07/14 05:56:17 martine Exp $
 */

#include "config.h"

#include <gtk/gtk.h>
#include "dotconf.h"
#include "util.h"
#include "network.h"
#include "friends.h"
#include "friendgroupedit.h"

typedef struct {
	GtkWidget *win;
	GtkWidget *lgroups, *linc, *lexc;

	GList *friends;
} friendgroups_dlg;

static int 
findfreegroup() {
	int freegroup;
	friendgroup *fg;
	GList *l;

	for (freegroup = 1; freegroup <= 30; freegroup++) {
		for (l = conf.friendgroups; l != NULL; l = l->next) {
			fg = l->data;
			if (fg->id == freegroup) break;
		}
		if (l == NULL) {
			/* this id isn't in use! */
			return freegroup;
		}
	}
	g_warning("Couldn't find free friend group?");
	return -1; /* we didn't find one. */
}

static void
dlg_add_friend_group(friendgroups_dlg *fdlg, friendgroup *fg) {
	char *text[] = { fg->name };
	int row;

	row = gtk_clist_append(GTK_CLIST(fdlg->lgroups), text);
	gtk_clist_set_row_data(GTK_CLIST(fdlg->lgroups), row, fg);
}

static void
new_cb(GtkWidget *w, friendgroups_dlg *fdlg) {
	friendgroup *fg;
	int freegroup;

	freegroup = findfreegroup();
	if (freegroup == -1) return; /* FIXME */

	fg = friend_group_edit_dlg_run(fdlg->win, NULL, freegroup);
	if (fg == NULL) return; /* they cancelled. */

	/* new friend group! */
	conf.friendgroups = g_list_append(conf.friendgroups, fg);
	dlg_add_friend_group(fdlg, fg);
}

static void
edit_cb(GtkWidget *w, friendgroups_dlg *fdlg) {
	friendgroup *fg;
	gint row;
	
	if (GTK_CLIST(fdlg->lgroups)->selection == NULL)
		return;
	row = (gint) GTK_CLIST(fdlg->lgroups)->selection->data;
	fg = gtk_clist_get_row_data(GTK_CLIST(fdlg->lgroups), row);

	fg = friend_group_edit_dlg_run(fdlg->win, fg, -1);
	if (fg == NULL) return; /* they cancelled. */

	gtk_clist_set_text(GTK_CLIST(fdlg->lgroups), row, 0, fg->name);
}

static void
remove_cb(GtkWidget *w, friendgroups_dlg *fdlg) {
	GHashTable *request, *result;
	friendgroup *fg;
	gint row;
	
	if (GTK_CLIST(fdlg->lgroups)->selection == NULL)
		return;
	row = (gint) GTK_CLIST(fdlg->lgroups)->selection->data;
	fg = gtk_clist_get_row_data(GTK_CLIST(fdlg->lgroups), row);
	
	if (conf.confirmdelete && lj_confirm(fdlg->win, "Delete Friend Group", "Really delete this friend group?") == FALSE) 
		return;
	
	request = net_request_new("editfriendgroups");

	g_hash_table_insert(request, 
			g_strdup_printf("efg_delete_%d", fg->id),
			g_strdup("1"));

	result = net_request_run(fdlg->win, "Removing Friend Group...", request);
	hash_destroy(request);

	if (net_request_succeeded(result)) {
		gtk_clist_remove(GTK_CLIST(fdlg->lgroups), row);
		conf.friendgroups = g_list_remove(conf.friendgroups, fg);
		g_free(fg->name);
		g_free(fg);
	}
	hash_destroy(result);
}

static void
populate_grouplist(friendgroups_dlg *fdlg) {
	GList *lgroup;

	for (lgroup = conf.friendgroups; lgroup != NULL; lgroup = lgroup->next) {
		friendgroup *fg = lgroup->data;

		dlg_add_friend_group(fdlg, fg);
	}
	gtk_clist_set_column_width(GTK_CLIST(fdlg->lgroups), 0,
			gtk_clist_optimal_column_width(GTK_CLIST(fdlg->lgroups), 0));
}

static void
row_selected(GtkCList *list, 
		gint row, gint col, GdkEventButton *e, friendgroups_dlg *fdlg) {
	friendgroup *fg;
	GList *l;
	guint mask;

	fg = gtk_clist_get_row_data(list, row);
	mask = (1L << fg->id);
	for (l = fdlg->friends; l != NULL; l = l->next) {
		friend *f = l->data;
		char *text[1];
		gint row;

		if (!(f->conn & FRIEND_MY))
			continue;

		text[0] = f->username;
		if (f->groupmask & mask) {
			row = gtk_clist_append(GTK_CLIST(fdlg->linc), text);
			gtk_clist_set_row_data(GTK_CLIST(fdlg->linc), row, f);
		} else {
			row = gtk_clist_append(GTK_CLIST(fdlg->lexc), text);
			gtk_clist_set_row_data(GTK_CLIST(fdlg->lexc), row, f);
		}
	}
}

static void
row_unselected(GtkCList *list, 
		gint row, gint col, GdkEventButton *e, friendgroups_dlg *fdlg) {
	gtk_clist_clear(GTK_CLIST(fdlg->linc));
	gtk_clist_clear(GTK_CLIST(fdlg->lexc));
}

static GtkWidget*
fg_list_create(friendgroups_dlg *fdlg) {
	char *titles[] = { "Friend Groups" };

	fdlg->lgroups = gtk_clist_new_with_titles(1, titles);

	gtk_signal_connect(GTK_OBJECT(fdlg->lgroups), "select-row",
			GTK_SIGNAL_FUNC(row_selected), fdlg);
	gtk_signal_connect(GTK_OBJECT(fdlg->lgroups), "unselect-row",
			GTK_SIGNAL_FUNC(row_unselected), fdlg);

	populate_grouplist(fdlg);
	return scroll_wrap(fdlg->lgroups);
}

static void
rowsel_sensitive(GtkCList *list, 
		gint row, gint col, GdkEventButton *e, GtkWidget *w) {
	if (list->selection)
		gtk_widget_set_sensitive(w, TRUE);
	else
		gtk_widget_set_sensitive(w, FALSE);
}

static void
addrem_cb(friendgroups_dlg *fdlg, 
		GtkCList *source, GtkCList *dest, gboolean add) {
	gint row;
	friend *f;
	friendgroup *fg;
	GList *sel;
	GHashTable *request, *result;
	guint32 mask, gmask;
	
	if (source->selection == NULL) return;

	fg = clist_data_from_selection(GTK_CLIST(fdlg->lgroups));
	if (fg == NULL) return;

	gmask = (1L << fg->id);
	request = net_request_new("editfriendgroups");

	for (sel = source->selection; sel != NULL; sel = sel->next) {
		row = (gint) sel->data;
		f = gtk_clist_get_row_data(source, row);

		mask = f->groupmask;
		if (add) {
			mask |= gmask;
		} else {
			mask &= ~gmask;
		}

		g_hash_table_insert(request, 
				g_strdup_printf("editfriend_groupmask_%s", f->username),
				g_strdup_printf("%d", mask));
	}

	result = net_request_run(fdlg->win, "Modifying Friends...", request);
	hash_destroy(request);

	if (net_request_succeeded(result)) {
		char *text[1];

		sel = source->selection;
		while (sel != NULL) {
			row = (gint) sel->data;
			f = gtk_clist_get_row_data(source, row);

			if (add) {
				f->groupmask |= gmask;
			} else {
				f->groupmask &= ~gmask;
			}
			gtk_clist_remove(source, row);
			text[0] = f->username;
			row = gtk_clist_append(dest, text);
			gtk_clist_set_row_data(dest, row, f);

			if (sel != source->selection) 
				sel = source->selection;
			else
				sel = sel->next;
		}
	}
	hash_destroy(result);
}

static void
add_cb(GtkWidget *w, friendgroups_dlg *fdlg) {
	addrem_cb(fdlg, GTK_CLIST(fdlg->lexc), GTK_CLIST(fdlg->linc), TRUE);
}

static void
rem_cb(GtkWidget *w, friendgroups_dlg *fdlg) {
	addrem_cb(fdlg, GTK_CLIST(fdlg->linc), GTK_CLIST(fdlg->lexc), FALSE);
}

static GtkWidget*
fg_list_buttonbox(friendgroups_dlg *fdlg) {
	GtkWidget *bnew, *bedit, *brem;
	GtkWidget *box;

	bnew = gtk_button_new_with_label("New...");
	gtk_signal_connect(GTK_OBJECT(bnew), "clicked",
			GTK_SIGNAL_FUNC(new_cb), fdlg);

	bedit = gtk_button_new_with_label("Edit...");
	gtk_signal_connect(GTK_OBJECT(bedit), "clicked",
			GTK_SIGNAL_FUNC(edit_cb), fdlg);
	gtk_widget_set_sensitive(GTK_WIDGET(bedit), FALSE);

	brem = gtk_button_new_with_label("Remove");
	gtk_signal_connect(GTK_OBJECT(brem), "clicked",
			GTK_SIGNAL_FUNC(remove_cb), fdlg);
	gtk_widget_set_sensitive(GTK_WIDGET(brem), FALSE);

	gtk_signal_connect(GTK_OBJECT(fdlg->lgroups), "select-row",
			GTK_SIGNAL_FUNC(rowsel_sensitive), bedit);
	gtk_signal_connect(GTK_OBJECT(fdlg->lgroups), "unselect-row",
			GTK_SIGNAL_FUNC(rowsel_sensitive), bedit);
	gtk_signal_connect(GTK_OBJECT(fdlg->lgroups), "select-row",
			GTK_SIGNAL_FUNC(rowsel_sensitive), brem);
	gtk_signal_connect(GTK_OBJECT(fdlg->lgroups), "unselect-row",
			GTK_SIGNAL_FUNC(rowsel_sensitive), brem);

	box = lj_dialog_buttonbox_new();
	lj_dialog_buttonbox_add(box, bnew);
	lj_dialog_buttonbox_add(box, bedit);
	lj_dialog_buttonbox_add(box, brem);
	return box;
}

void
friendgroups_dialog_new(GtkWidget *parent, GList *friends) {
	friendgroups_dlg *fdlg;
	GtkWidget *paned;

	fdlg = g_new0(friendgroups_dlg, 1);
	fdlg->friends = friends;
	fdlg->win = lj_dialog_new(parent, "Friend Groups", 500, -1);
	gtk_signal_connect_object(GTK_OBJECT(fdlg->win), "destroy",
			GTK_SIGNAL_FUNC(g_free), (GtkObject*)fdlg);

	paned = gtk_hpaned_new(); {
		GtkWidget *vbox, *hbox;
		
		gtk_paned_set_gutter_size(GTK_PANED(paned),
				GTK_PANED(paned)->handle_size+2);

		vbox = gtk_vbox_new(FALSE, 5); {
			gtk_box_pack_start(GTK_BOX(vbox), 
					fg_list_create(fdlg), TRUE, TRUE, 0);
			hbox = gtk_hbox_new(FALSE, 0);
			gtk_box_pack_end(GTK_BOX(hbox), 
					fg_list_buttonbox(fdlg), FALSE, FALSE, 0);
			gtk_box_pack_start(GTK_BOX(vbox), 
					hbox, FALSE, FALSE, 0);
		}
		gtk_paned_pack1(GTK_PANED(paned), vbox, TRUE, TRUE);

		hbox = gtk_hbox_new(FALSE, 5); {
			GtkWidget *bbox, *badd, *brem;
			char *titles[1];

			titles[0] = "Included";
			fdlg->linc = gtk_clist_new_with_titles(1, titles);
			gtk_clist_set_auto_sort(GTK_CLIST(fdlg->linc), TRUE);
			gtk_clist_set_selection_mode(GTK_CLIST(fdlg->linc),
					GTK_SELECTION_MULTIPLE);
			gtk_box_pack_start(GTK_BOX(hbox), 
					scroll_wrap(fdlg->linc), TRUE, TRUE, 0);
			
			bbox = gtk_vbox_new(FALSE, 5); {
				badd = gtk_button_new_with_label("<-");
				gtk_widget_set_sensitive(badd, FALSE);
				gtk_signal_connect(GTK_OBJECT(badd), "clicked",
						GTK_SIGNAL_FUNC(add_cb), fdlg);
				gtk_box_pack_end(GTK_BOX(bbox), badd, FALSE, FALSE, 0);

				brem = gtk_button_new_with_label("->");
				gtk_widget_set_sensitive(brem, FALSE);
				gtk_signal_connect(GTK_OBJECT(brem), "clicked",
						GTK_SIGNAL_FUNC(rem_cb), fdlg);
				gtk_box_pack_end(GTK_BOX(bbox), brem, FALSE, FALSE, 0);
			}
			gtk_box_pack_start(GTK_BOX(hbox), bbox, FALSE, FALSE, 0);

			titles[0] = "Excluded";
			fdlg->lexc = gtk_clist_new_with_titles(1, titles);
			gtk_clist_set_auto_sort(GTK_CLIST(fdlg->lexc), TRUE);
			gtk_clist_set_selection_mode(GTK_CLIST(fdlg->lexc),
					GTK_SELECTION_MULTIPLE);
			gtk_box_pack_start(GTK_BOX(hbox), 
					scroll_wrap(fdlg->lexc), TRUE, TRUE, 0);

			gtk_signal_connect(GTK_OBJECT(fdlg->linc), "select-row",
					GTK_SIGNAL_FUNC(rowsel_sensitive), brem);
			gtk_signal_connect(GTK_OBJECT(fdlg->linc), "unselect-row",
					GTK_SIGNAL_FUNC(rowsel_sensitive), brem);
			gtk_signal_connect(GTK_OBJECT(fdlg->lexc), "select-row",
					GTK_SIGNAL_FUNC(rowsel_sensitive), badd);
			gtk_signal_connect(GTK_OBJECT(fdlg->lexc), "unselect-row",
					GTK_SIGNAL_FUNC(rowsel_sensitive), badd);
		}
		gtk_paned_pack2(GTK_PANED(paned), hbox, TRUE, TRUE);
	}

	gtk_paned_set_position(GTK_PANED(paned), 175);

	lj_dialog_set_contents(fdlg->win, paned);
	lj_dialog_add_close(fdlg->win);

	gtk_widget_show(fdlg->win);
}
