// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptLineEdit"

#include "kvi_error.h"
#include "kvi_lineedit.h"
#include "kvi_script_lineedit.h"
#include "kvi_script_objectclassdefinition.h"

/*
	@class: lineedit
	@short:
		A simple single line editor widget
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:

		!fn: $text()
		Returns the text currently displayed.<br>

		!fn: $setText(&lt;text&gt;)
		Sets the text to be edited.<br>

		!fn: $setReadOnly(&lt;bReadOnly&gt;)
		Sets this line edit to read only mode or read/write mode.<br>
		if &lt;bReadOnly&gt; is '1' then the read-only mode is set,
		if it is '0' then the read/write mode is set.<br>

		!fn: $readOnly()
		Returns '1' if this widget is in read only mode, '0' otherwise.<br>

	@description:
		This class represents a simple text line editor.<br>
		It can be easily used in configuration dialogs.<br>
		The constructor for this class accepts an additional (optional) parameter:
		the initial text that has to be edited.<br>
		In this way you can avoid to call the [classfnc:lineedit]$setText[/classfnc] function.<br>

	@examples:

	@seealso:
		class [class]object[/class], <br>
		class [class]widget[/class], <br>
		class [class]label[/class], <br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/

/**
 * LINEEDIT class
 */
void KviScriptLineEdit::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("setText",     (scriptObjectFunction) &KviScriptLineEdit::builtinFunction_SETTEXT);
	d->addBuiltinFunction("text",        (scriptObjectFunction) &KviScriptLineEdit::builtinFunction_TEXT);
	d->addBuiltinFunction("readOnly",    (scriptObjectFunction) &KviScriptLineEdit::builtinFunction_READONLY);
	d->addBuiltinFunction("setReadOnly", (scriptObjectFunction) &KviScriptLineEdit::builtinFunction_SETREADONLY);
	d->addBuiltinFunction("home",        (scriptObjectFunction) &KviScriptLineEdit::builtinFunction_HOME);
	d->addBuiltinFunction("end",         (scriptObjectFunction) &KviScriptLineEdit::builtinFunction_END);
	d->addBuiltinFunction("clear",       (scriptObjectFunction) &KviScriptLineEdit::builtinFunction_CLEAR);
}

KviScriptLineEdit::KviScriptLineEdit(
	KviScriptObjectController *cntrl, KviScriptObject *p, const char *name, KviScriptObjectClassDefinition *pDef)
	: KviScriptWidget(cntrl, p, name, pDef)
{
	// Nothing here
}

KviScriptLineEdit::~KviScriptLineEdit()
{
	// Nothing here
}

bool KviScriptLineEdit::init(QPtrList<KviStr> *params)
{
	if( parent() ) {
		if( parent()->inherits("KviScriptWidget") ) {
			m_pWidget = new KviLineEdit(((KviScriptWidget *) parent())->m_pWidget, name());
		}
	}
	if( !m_pWidget )
		m_pWidget = new KviLineEdit(0, name());
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget, SIGNAL(destroyed()), this, SLOT(widgetDestroyed()));
	if( params ) {
		KviStr *pS = params->first();
		if( pS )
			((KviLineEdit *) m_pWidget)->setText(pS->ptr());
	}
	return true;
}

int KviScriptLineEdit::builtinFunction_TEXT(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append(((KviLineEdit *) m_pWidget)->text());
	return KVI_ERROR_Success;
}

int KviScriptLineEdit::builtinFunction_SETTEXT(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			((KviLineEdit *) m_pWidget)->setText(pS->ptr());
			return KVI_ERROR_Success;
		}
	}
	((KviLineEdit *) m_pWidget)->setText("");
	return KVI_ERROR_Success;
}

int KviScriptLineEdit::builtinFunction_READONLY(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append(((KviLineEdit *) m_pWidget)->isReadOnly() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptLineEdit::builtinFunction_SETREADONLY(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			((KviLineEdit *) m_pWidget)->setReadOnly(kvi_strEqualCI(pS->ptr(), "1"));
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptLineEdit::builtinFunction_HOME(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			((KviLineEdit *) m_pWidget)->home(kvi_strEqualCI(pS->ptr(), "1"));
		}
	}
	return KVI_ERROR_Success;
}

int KviScriptLineEdit::builtinFunction_END(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			((KviLineEdit *) m_pWidget)->end(kvi_strEqualCI(pS->ptr(), "1"));
		}
	}
	return KVI_ERROR_Success;
}

int KviScriptLineEdit::builtinFunction_CLEAR(QPtrList<KviStr> *, KviStr &buffer)
{
	((KviLineEdit *) m_pWidget)->clear();
	return KVI_ERROR_Success;
}

#include "m_kvi_script_lineedit.moc"
