#ifndef _KVIRC_PLUGIN_H_INCLUDED_
#define _KVIRC_PLUGIN_H_INCLUDED_

//
//   File : kvirc_plugin.h (/usr/build/KVIrc/kvirc/src/plugins/kvirc_plugin.h)
//   Last major modification : Wed Oct 15 1999 04:53:14 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

//
//   KVIrc plugins interface file
//

#include "kvi_string.h"
#include <qlist.h>

#ifdef __KVIRC_PLUGIN__

// included from a plugin source

#include "kvi_app.h"
#include "kvi_frame.h"
#include "kvi_window.h"
#include "kvi_defines.h"
#include "kvi_error.h"
#include "kvi_ircsocket.h"
#include "kvi_console.h"
#include "kvi_statusbar.h"
#include "kvi_systray.h"

#else

// included from kvirc executable sources
// just forward declarations

class KviWindow;
class KviApp;
class KviFrame;
class KviIrcSocket;
class KviConsole;
class KviStatusBar;
class KviDockableWidget;

#endif

// structure passed to the routines (except for the cleanup_routine)

typedef struct _KviPluginCommandStruct
{
	void          * handle;     // handle of this plugin (required for many interface functions , always set)
	KviApp        * app;        // pointer to the application (it is the same as g_pApp , may be 0 in the init_routine)
	QList<KviStr> * params;     // list of parameters (may be 0...so check it if you mess with the list directly (kvirc_plugin_param() will always work)) , the first parameter (index 0) is always the command name
	KviWindow     * window;     // the window that this command is bound to (may be 0 in the init_routine of the plugin)
	KviFrame      * frame;      // parent frame of the window above (may be 0 in the init routine!)
	KviConsole    * console;    // console window belonging to this frame (may be 0 in the init_routine)
	KviIrcSocket  * sock;       // irc socket (is NOT connected in the init and cleanup routine and may be 0 there)
	int             error;      // last error code to be set when an error occures
	KviStr          errorstr;   // error description to be set when an error occures
} KviPluginCommandStruct;

//
// All plugins must export an instance of this struct
// named 'kvirc_plugin' and NOT declared static
//

typedef struct _KviPlugin
{
	const char * module_name;                         // name of the plugin module: "Hello"
	const char * service_name;                        // service name             : "A Sample plugin"
	const char * version;                             // version                  : "0.1"
	const char * author;                              // author's name and info   : "Szymon Stefanek <kvirc@tin.it>"
	const char * description;                         // detailed description of the exported commands & styles
	bool (*init_routine)(KviPluginCommandStruct *);   // init routine of the plugin (return false if you want the plugin loading to be aborted)
	void (*cleanup_routine)(void);                    // cleanup routine (may be 0)
	void (*config_routine)(void);                     // configuration routine (may be 0)
	void (*help_routine)(void);                       // help routine (may be 0)
} KviPlugin;

#ifdef __KVIRC_PLUGIN__

#include "kvi_plugin.h"
#include "kvi_event.h"
#include "kvi_regusersdb.h"
#include "kvi_options.h"
#include "kvi_config.h"
#include "kvi_uparser.h"
#include "kvi_mime.h"

#include <qstyle.h>
#include <qsplitter.h>

// INTERNAL STUFF

extern KviPluginManager *g_pPluginManager;

// ================================================================================================
// PLUGIN INTERFACE FUNCTIONS
// ================================================================================================
//
// with a decent compiler all this stuff goes inlined
// so it is as fast as a #define

// ------------------------------------------------------------------------------------------------
// Command registration routines
// ------------------------------------------------------------------------------------------------

//
// Registers a new user command (called <cmdname>)
// It can be accessed from the commandline and scripts
// Each time that this command is executed , the handler_routine callback is called.
// The callback routine should have the following prototype:
//
//    bool my_handler_routine(KviPluginCommandStruct *cmd);
//
// You should return true if the command has executed succesfully and the parsing
// can continue. If you return false the parsing of the commands will be stopped;
// if you set cmd->error to a meaningful error code (see kvi_error.h) the error
// will be printed in the current window. Since the error types
// are not enough for all kind of errors, if you feel that no error code declared in
// kvi_error.h is appropriate , you can set cmd->error to a generic error code
// such as KVI_ERROR_InvalidOperation and then put a more detailed
// error description in cmd->errorstr.
// If you return false and NOT set cmd->error kvirc will behave just like
// the command has been executed succesfully and immediately after halt was called.
// You will typically use this call in the init routine of the plugin
//
inline void kvirc_plugin_register_command(void * handle,const char * cmdname,bool (*handler_routine)(KviPluginCommandStruct *))
{
	g_pPluginManager->registerCommand(handle,cmdname,handler_routine);
}

//
// Unregisters an user command (called <cmdname>)
// It is not required to unregister commands in the cleanup_routine of the plugin
// KVIrc will unregister all the commands by itself.
//
inline void kvirc_plugin_unregister_command(void * handle,const char * cmdname)
{
	g_pPluginManager->unregisterCommand(handle,cmdname);
}

//
// Unregisters all the commands owned by this plugin
//
inline void kvirc_plugin_unregister_all_commands(void * handle)
{
	g_pPluginManager->unregisterCommandsFor(handle);
}


// ------------------------------------------------------------------------------------------------
// Function & identifier registration routines
// ------------------------------------------------------------------------------------------------

//
// Registers a new user function (called <fncname>)
// It can be accessed from the commandline and scripts
// Each time that this function is called the handler routine is called.
// The callback routine prototype is:
//
//     bool my_handler_routine(KviPluginCommandStruct *cmd,KviStr *buffer);
//
// If the function was accessed as identifier ($fncname form) the param list
// of the passed KviPluginCommandStruct will be 0.
// If the function was accessed as real function ($fncname(parameters))
// the param list will contain the parameters passed.
// The handler should set result of the call in the KviStr object passed.
// (buffer->append(your result) or *str = yourresult)
// The return value of the callback routine has exactly the same meaning
// as for the command handlers.
//
inline void kvirc_plugin_register_function(void * handle,const char * fncname,bool (*handler_routine)(KviPluginCommandStruct *,KviStr *))
{
	g_pPluginManager->registerFunction(handle,fncname,handler_routine);
}

//
// Unregisters an user function (called <fncname>)
// It is not required to unregister functions in the cleanup_routine of the plugin
// KVIrc will unregister all the functions by itself.
//
inline void kvirc_plugin_unregister_function(void * handle,const char * fncname)
{
	g_pPluginManager->unregisterFunction(handle,fncname);
}

//
// Unregisters all the functions owned by this plugin
//
inline void kvirc_plugin_unregister_all_functions(void * handle)
{
	g_pPluginManager->unregisterFunctionsFor(handle);
}

// ------------------------------------------------------------------------------------------------
// Hook registration routines
// ------------------------------------------------------------------------------------------------

//
// Registers a hook routine to be called when an event is fired in KVIrc
// The event identifiers (eventidx) are declared in kvi_event.h
// The prototype is exaclty the same as for the command handlers.
// The callback is called BEFORE the script handlers.
// Since the event handler code is not followed by any other code
// the handling of the return value is quite different in this case:
// you should return 'true' if you want to act like halt was called:
// you will usually stop the KVIrc output for the event.
// A 'false' return value will signal that you want the output to be
// printed normally.
// To signal an error you have to print it by yourself:
// there is no way to tell kvirc that the execution of the routine
// was unsuccessfull , and in most cases there is no point in doing it.
// (well...in most cases , if you have errors inside
// a hardcoded event handler you should think about rewriting your code).
//
inline void kvirc_plugin_add_hook(void * handle,int eventidx,bool (*handler_routine)(KviPluginCommandStruct *))
{
	g_pPluginManager->registerHook(handle,eventidx,handler_routine);
}

//
// Unegisters a hook routine
// It is not required to unregister hooks in the cleanup_routine
// KVIrc will cleanup all the hooks when this plugin is unloaded
//
inline void kvirc_plugin_remove_hook(void * handle,int eventidx)
{
	g_pPluginManager->unregisterHook(handle,eventidx);
}

//
// Unegisters all hooks owned by this plugin
//
inline void kvirc_plugin_remove_all_hooks(void * handle)
{
	g_pPluginManager->unregisterHooksFor(handle);
}

//
// Is a specified event_hook registered for this plugin ?
//
inline bool kvirc_plugin_is_hook_registered(void * handle,int eventidx)
{
	return g_pPluginManager->isHookRegistered(handle,eventidx);
}

// ------------------------------------------------------------------------------------------------
// Registered user database interface routines
// ------------------------------------------------------------------------------------------------

//
// Finds a registered user matching the given mask.
// Returns the KviRegisteredUser structure pointer or 0 if there is no match
//
// prototype of the KviRegisteredUser class
//
// typedef struct KviRegisteredUser
// {
//    KviIrcUser user;                // class declared in kvi_ircuser.h
//    KviStr allFlags;                // flags
//    KviStr passwd;                  // optional password
//    KviStr comment;                 // comment field
//    char   bNotify;                 // 1 if the user is in the notify list (the flag 'n' is also in allflags)
//    char   bIgnore;                 // 1 if the user is ignored (the flag 'i' is also in allFlags)
// };
//
inline KviRegisteredUser * kvirc_plugin_find_registered_user(const char *mask)
{
	return g_pOptions->m_pRegUsersDb->findUserByMask(mask);
}

//
// As above but returns the structure only if the user is registered
// and has the specified flag in the allFlags field
//
inline KviRegisteredUser * kvirc_plugin_find_registered_user_with_flag(const char *mask,char flag)
{
	return g_pOptions->m_pRegUsersDb->findUserWithFlagByMask(mask,flag);
}

// ------------------------------------------------------------------------------------------------
// Misc helpers
// ------------------------------------------------------------------------------------------------

//
// Places in <buffer> the path for the plugin's config file.
// You can be almost sure that the filename returned by this routine is
// readable & writeable. A good helper for handing config files is
// the KviConfig class declared in kvi_config.h
//
inline void kvirc_plugin_get_config_file_path(KviStr &buffer,const char *plugin_name)
{
	g_pApp->getPluginConfigFilePath(buffer,plugin_name);
}

//
// Places in <buffer> the path for the plugin's config file.
// This function should be used only if you install a default config file
// in the global KVIrc directory. (config/plugins)
// It will first look for a saved config file in the local user KVIrc directory
// and place its path in <buffer> if existing.
// Otherwise it will look for the default file in the application global
// KVIrc directory (config/plugins).
// If the file is found , its path will be placed in <buffer>.
// This function returns true if the file exists (and then the file path
// is in <buffer>), or false if there is no config file for the plugin.
// The filepath returned by this function is NOT writable:
// You should use it only to LOAD the configuration.
// To save it , get the filepath from kvirc_plugin_get_config_file_path
//
inline bool kvirc_plugin_get_read_only_config_file_path(KviStr &buffer,const char *plugin_name)
{
	return g_pApp->getReadOnlyPluginConfigFilePath(buffer,plugin_name);
}

//
// Finds an open KVIrc MDI window belonging to the specified KviFrame.
// Returns 0 if the window can not be found
//
inline KviWindow * kvirc_plugin_find_window(KviFrame * frame,const char *name)
{
	return frame->findWindow(name);
}

//
// Returns the parameter at index <paramindex> or 0 if there is no such parameter.
// (Note: KviStr tmp = kvirc_plugin_param(cmd,anynumber); will always work , even if the returned value is 0,
//  you will just get an empty tmp string.)
//
inline const char * kvirc_plugin_param(KviPluginCommandStruct * cmd,int paramindex)
{
	return (cmd->params) ? (cmd->params->at(paramindex) ? cmd->params->at(paramindex)->ptr() : 0) : 0;
}

//
// Execute a buffer of script commands , binding the execution to a specified window
// Returns false if the execution of the command fails (syntax or semantic errors ?)
// Hardcoded scripting inside plugins...just don't try to unload yourself. :)
//
inline bool kvirc_plugin_execute_command(KviWindow * window,const char *buffer)
{
	return window->m_pFrm->m_pUserParser->parseCommand(buffer,window);
}

//
// Simulate typing <text> into the commandline of a specified window
// and pressing return. :)
//
inline void kvirc_plugin_simulate_typing(KviWindow * window,KviStr &text)
{
	window->m_pFrm->m_pUserParser->parseUserCommand(text,window);
}

//
// Pops up the help window with the page <pagename> opened
//
inline void kvirc_plugin_show_help_page(const char * pagename)
{
	g_pApp->requestHelpOn(pagename);
}

//
// Puts a message in the statusbar with a specified timeout
//
inline void kvirc_plugin_statusbar_message(KviFrame * frame,const char *text,int msecs)
{
	frame->m_pStatusBar->tempText(text,msecs);
}

//
// This is a quick message box
//
inline void kvirc_plugin_warning_box(const char *string)
{
	g_pApp->warningBox(string);
}

//
// Lookups the mimetype for the specified filename
// Take a look at kvi_mime.h for the definition of the KviMimeType struct
//
inline KviMimeType * kvirc_plugin_get_mime_type(const char *filename)
{
	return g_pOptions->m_pMimeManager->findMatch(filename,true);
}

// ------------------------------------------------------------------------------------------------
// Style handling routines
// ------------------------------------------------------------------------------------------------

inline void kvirc_plugin_register_style(void * handle,const char *name,QStyle * (*style_provider_routine)())
{
	g_pApp->registerStyle(name,handle,style_provider_routine);
}

inline void kvirc_plugin_unregister_style(void * handle,const char *name)
{
	g_pApp->unregisterStyle(name,handle);
}

inline void kvirc_plugin_unregister_all_styles(void * handle)
{
	g_pApp->unregisterStylesFor(handle);
}

// ------------------------------------------------------------------------------------------------
// MDI window registration routines
// ------------------------------------------------------------------------------------------------

inline void kvirc_plugin_add_window(void * handle,KviFrame * frame,KviWindow * wnd,bool bShow = true)
{
	g_pPluginManager->addPluginWindow(handle,frame,wnd,bShow);
}

inline void kvirc_plugin_remove_all_windows(void * handle)
{
	g_pPluginManager->removeAllPluginWindows(handle);
}

// ------------------------------------------------------------------------------------------------
// Statusbar docked widgets registration routines
// ------------------------------------------------------------------------------------------------

inline void kvirc_plugin_add_docked_widget(void * handle,KviFrame * frame,KviDockableWidget * w)
{
	g_pPluginManager->addPluginDockedWidget(handle,frame,w);
}

inline void kvirc_plugin_remove_all_docked_widgets(void * handle)
{
	g_pPluginManager->removeAllPluginDockedWidgets(handle);
}

// ------------------------------------------------------------------------------------------------
// SysTray widgets registration routines
// ------------------------------------------------------------------------------------------------

// This is for the sys tray widget creation
inline KviSysTray * kvirc_plugin_get_systray(KviFrame * frame)
{
	return frame->m_pSysTrayBar->m_pSysTray;
}

inline void kvirc_plugin_add_systray_widget(void * handle,KviFrame * frame,KviSysTrayWidget * w, bool _show = true)
{
	g_pPluginManager->addPluginSysTrayWidget(handle,frame,w, _show);
}

inline void kvirc_plugin_remove_systray_widget(KviFrame * frame,KviSysTrayWidget *w,bool bDelete)
{
	frame->m_pSysTrayBar->m_pSysTray->removeWidget(w,bDelete);
}

inline void kvirc_plugin_remove_systray_widgets(void * handle,KviFrame * frame)
{
	frame->m_pSysTrayBar->m_pSysTray->killPluginWidgets(handle);
}

inline void kvirc_plugin_remove_all_systray_widgets(void * handle)
{
	g_pPluginManager->removeAllPluginSysTrayWidgets(handle);
}

inline KviSysTrayWidget * kvirc_plugin_lookup_systray_widget(KviFrame *frame,const char * className)
{
	return frame->m_pSysTrayBar->m_pSysTray->findSysTrayWidget(className);
}

inline KviSysTrayWidget * kvirc_plugin_current_systray_widget(KviFrame *frame)
{
	return frame->m_pSysTrayBar->m_pSysTray->currentWidget();
}

inline bool kvirc_plugin_set_current_systray_widget(KviFrame * frame,KviSysTrayWidget *w)
{
	return frame->m_pSysTrayBar->m_pSysTray->setCurrentWidget(w);
}

// ------------------------------------------------------------------------------------------------
// Routines for adding widgets to MDI windows
// ------------------------------------------------------------------------------------------------

//
// Is there a clean way to add a widget to this window ?
// Yes....if there is a splitter
//
inline bool kvirc_plugin_mdi_window_can_add_widget(KviWindow *wnd)
{
	return (wnd->m_pSplitter != 0);
}

//
// The clean way to get a splitter from a mdi window
// WARNING : MAY BE 0!
// so not all the mdi windows can accept a new widget
//
inline QSplitter * kvirc_plugin_mdi_window_splitter(KviWindow *wnd)
{
	return wnd->m_pSplitter;
}

//
// This will add a widget to a mdi window preserving the right focus processing
// The widget MUST have been created with the kvirc_plugin_mdi_window_splitter as parent!
// To remove the widget simply delete it
// Please note that you MUST remove your widgets in the cleanup routine
// if they have not been deleted yet.
// If you don't do it , kvirc will probably be killed by the
// system dynamic linker that will fail to resolve library symbols.
// If the user closes the window the widget will be automatically destroyed.
// 
inline void kvirc_plugin_mdi_window_add_widget(QWidget *w,KviWindow *wnd)
{
	wnd->addWidgetToSplitter(w);
}

//
// Hack for Qt.... this is ALWAYS NEEDED When a plugin is exiting...
//

inline void kvirc_plugin_unregister_meta_object(const char * metaObjectName)
{
	g_pPluginManager->unregisterMetaObject(metaObjectName);
}

#endif //__KVIRC_PLUGIN__

#endif
