#ifndef _KVI_USERLIST_H_INCLUDED_
#define _KVI_USERLIST_H_INCLUDED_

//
//   File : kvi_userlist.h (/usr/build/NEW_kvirc/kvirc/src/kvilib/kvi_userlist.h)
//   Last major modification : Sun Mar 14 1999 03:36:39 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_ircuser.h"

//
// Shared list of users with reference count
// Small memory usage and relatively fast access
// This is the GLOBAL LIST and should be UNIQUE (PER CONNECTION).
//

typedef struct KviIrcUserListNode
{
	KviIrcUserListNode * prev;
	KviIrcUser         * pUser;
	int                  nRefs;
	KviIrcUserListNode * next;
};

class KviIrcUserList{
	friend class KviIrcUserChanList;
public:
	KviIrcUserList();
	~KviIrcUserList();
private:
	KviIrcUserListNode *m_pHead;
	KviIrcUserListNode *m_pTail;
	KviIrcUserListNode *m_pCur; //This is defined ONLY after a call to first(),firstNode(),next(),...
protected:
	void insertNode(KviIrcUserListNode * node);
	void removeNode(KviIrcUserListNode * node); //deletes the node!
	KviIrcUserListNode * findNode(const KviIrcUser &user);
	KviIrcUserListNode * findNode(const char *nick);
	/**
	* Add an user to the list
	* Making a deep copy of data if the user is not already there
	* or increasing the reference count and updating the host/username data (if possible and necessary)
	* Returns the pointer to the internal KviIrcUser user class (that should never be changed!)
	* You can NOT add directly an user here , you must do it through the client class
	*/
	KviIrcUserListNode * addUser(const KviIrcUser &user);
//	void killUser(KviIrcUser &user);
	void killUserByNode(KviIrcUserListNode * node);
public:
	void clearList();
	//CAUTION
	//firstNode(),lastNode(),first() or next() MUST be called before
	//any iteration attempt with nextNode(),next(),prevNode() and prev().
	KviIrcUserListNode * firstNode(){ m_pCur = m_pHead; return m_pHead; };
	KviIrcUserListNode * lastNode(){ m_pCur = m_pTail; return m_pTail; };
	KviIrcUserListNode * nextNode(){ if(m_pCur)m_pCur = m_pCur->next; return m_pCur; };
	KviIrcUserListNode * prevNode(){ if(m_pCur)m_pCur = m_pCur->prev; return m_pCur; };
	KviIrcUser * first(){ m_pCur = m_pHead; return m_pCur ? m_pCur->pUser : 0; };
	KviIrcUser * last(){ m_pCur = m_pTail; return m_pCur ? m_pCur->pUser : 0; };
	KviIrcUser * next(){ if(m_pCur)m_pCur = m_pCur->next; return m_pCur ? m_pCur->pUser : 0; };
	KviIrcUser * prev(){ if(m_pCur)m_pCur = m_pCur->prev; return m_pCur ? m_pCur->pUser : 0; };
	void getFirstFiveNicksToUpdate(KviStr &buffer);
	// Updates the mask of the user (assuming he is in the list!)
	// Returns true if the user was in the list
	bool updateUser(const KviIrcUser &user);
	// I forgot what does this one...:)
	KviIrcUser * findUser(const KviIrcUser &user);
	KviIrcUser * findUser(const char *nick);
};

//
// Client of the global list.
// This keeps only pointers to data and flags
//
// NOTE: I am sure that one day I'll forget completly how it works :)
//

typedef struct KviIrcUserChanData
{
	// THIS IS (should be) A READ-ONLY STRUCTURE!
	// NEWER CHANGE DIRECTLY (except for extFlag that is used in the KviListBox to keep track of selections)
	KviIrcUserChanData * prev;
	KviIrcUserListNode * pNode;  //pointer to the global data structure
	char qFlag;		// UnrealIRCd owner -Trisk
	char oFlag;                  //is user +o on this channel ?
	char hFlag;
	char vFlag;                  //is user +v on this channel ?
	char uFlag;		// Userop... -Trisk
	char extFlag;                //The only R/W flag in this structure
	KviIrcUserChanData * next;
};

class KviIrcUserChanList{
public:
	KviIrcUserChanList(KviIrcUserList * pList);
	~KviIrcUserChanList(); // Makes everyone part from this channel....
private:
	KviIrcUserList * m_pGlobalList;
	KviIrcUserChanData * m_pHead;
	KviIrcUserChanData * m_pTail;
	KviIrcUserChanData * m_pCur; //Same commens as for the global class
	int m_iCount;
	int m_iOwnerCount;
	int m_iOpCount;
	int m_iHalfOpCount;
	int m_iVoiceCount;
	int m_iUserOpCount;
public:
	// Unsafe iterators
	// CAUTION
	// firstNode(),lastNode(),first() or next() MUST be called before
	// any iteration attempt with nextNode(),next(),prevNode() and prev().
	KviIrcUser * firstUser(){ m_pCur = m_pHead; return m_pCur ? m_pCur->pNode->pUser : 0; };
	KviIrcUser * lastUser(){ m_pCur = m_pTail; return m_pCur ? m_pCur->pNode->pUser : 0; };
	KviIrcUser * nextUser(){ if(m_pCur)m_pCur = m_pCur->next; return m_pCur ? m_pCur->pNode->pUser : 0; };
	KviIrcUser * prevUser(){ if(m_pCur)m_pCur = m_pCur->prev; return m_pCur ? m_pCur->pNode->pUser : 0; };
	KviIrcUserChanData * firstData(){ m_pCur = m_pHead; return m_pHead; };
	KviIrcUserChanData * lastData(){ m_pCur = m_pTail; return m_pTail; };
	KviIrcUserChanData * nextData(){ if(m_pCur)m_pCur = m_pCur->next; return m_pCur; };
	KviIrcUserChanData * prevData(){ if(m_pCur)m_pCur = m_pCur->prev; return m_pCur; };

	void clearList();
	// An user joins...adds an entry here and in the global list
	KviIrcUserChanData * join(const KviIrcUser &user,char bOp=0,char bVoice=0,char bHalfOp=0,char bUserOp=0,char bOwner=0);
	// An user parts...removes the entry here and in the global list
	bool part(const char *nick);
	bool part(const KviIrcUser &user); //returns false if the user was not here

	bool select(const char *nick);
	bool select(const KviIrcUser &user);
	
	bool deselect(const char *nick);
	bool deselect(const KviIrcUser &user);
	// Careful with that AXE Lilian!
	// Newer change the KviIrcUserChanData structure directly
	// (I mean from outside this class).
	// or all this stuff will coredump sooner or later.
	// There are surely safer ways to do the same thing
	// but are also SURELY slowlier (and a bit more complex inside ?)....
	KviIrcUserChanData * findData(const KviIrcUser &user);
	KviIrcUserChanData * findData(const char *nick);
	// Finds an user in this list
	KviIrcUser * findUser(const KviIrcUser &user);
	KviIrcUser * findUser(const char *nick);
	int findUserPosition(const char *nick);
	// Ops the user...this is the right way to do it
	// Returns false if the user was not in the list
	bool owner(const char *nick,char bOwner);
	bool owner(const KviIrcUser &user,char bOwner){ return owner(user.nick(),bOwner); };
	bool op(const char *nick,char bOp);
	bool op(const KviIrcUser &user,char bOp){ return op(user.nick(),bOp); };
	bool halfop(const char *nick,char bHalfOp);
	bool halfop(const KviIrcUser &user,char bHalfOp){ return halfop(user.nick(),bHalfOp); };
	bool voice(const char *nick,char bVoice);
	bool voice(const KviIrcUser &user,char bVoice){ return voice(user.nick(),bVoice); };
	bool userop(const char *nick,char bUserOp);
	bool userop(const KviIrcUser &user,char bUserOp){ return userop(user.nick(),bUserOp); };
	bool isOwner(const char *nick);
	bool isOwner(const KviIrcUser &user){ return isOwner(user.nick()); };
	bool isOp(const char *nick);
	bool isOp(const KviIrcUser &user){ return isOp(user.nick()); };
	bool isHalfOp(const char *nick);
	bool isHalfOp(const KviIrcUser &user){ return isHalfOp(user.nick()); };
	bool isVoice(const char *nick);
	bool isVoice(const KviIrcUser &user){ return isVoice(user.nick()); };
	bool isUserOp(const char *nick);
	bool isUserOp(const KviIrcUser &user){ return isUserOp(user.nick()); };
	bool nickChange(const KviIrcUser &nicker,const char *newNick);
	// Internal user count
	int count(){ return m_iCount; };
	int ownerCount(){ return m_iOwnerCount; };
	int opCount(){ return m_iOpCount; };
	int halfopCount(){ return m_iHalfOpCount; };
	int voiceCount(){ return m_iVoiceCount; };
	int useropCount(){ return m_iUserOpCount; };
	// Debug stuff : dumps the list to the shell (in the debug compilation mode)
//	void dump();
protected:
	// Insert data keeping the op-voice-alphabetic order
	void insertData(KviIrcUserChanData * data);
	// Insert an owner
	void insertOwnerData(KviIrcUserChanData * data);
	// Skips owners and inserts an op user
	void insertOpData(KviIrcUserChanData * data);
	// Skips ops and inserts a halfop user
	void insertHalfOpData(KviIrcUserChanData * data);
	// Skips ops and halfops and inserts a voiced user
	void insertVoiceData(KviIrcUserChanData * data);
	// Skips ops, halfops, and voiced users, and inserts a userop user
	void insertUserOpData(KviIrcUserChanData * data);
	// Skips ops, halfops, userops, and voiced users, and inserts a normal user
	void insertNormalData(KviIrcUserChanData * data);
	// Does not remove the user list entry from the global list (it must be done before)! (deletes the data structure)
	void removeData(KviIrcUserChanData * data);
	// Used for internal reorganisation (op-deop list order changes)
	void removeDataNoDelete(KviIrcUserChanData * data);
};

#endif //!_KVI_USERLIST_H_INCLUDED_
