#ifndef MODULEMANAGER_H
#define MODULEMANAGER_H

#include <kore/kore.h>
#include <kore/serviceprovider.h>
#include <set>

namespace kore
{
/**
 * class kore::ModuleManager - provides easier Kore Module management.
 * When creating a new Module instance, you have to register that instance
 * to the ModuleManager, as well as unregister it right before the instance
 * will get destroyed.
 * The ModuleManager can be queried (registeredModules()) for the existing
 * registered Modules (i.e. by a "module browser").
 */
class KORE_API ModuleManager : public ServiceProvider
{
public:
    /**
     * Default constructor. Creates a ModuleManager.
     */
    ModuleManager();
    /**
     * Destructor.
     */
    virtual ~ModuleManager();

    /**
     * Registers a Module to this ModuleManager.
     * @param module - the Module to be registered.
     */
    virtual void registerModule(Module* module);
    /**
     * Unregisters a Module from this ModuleManager.
     * @param module - the Module to be unregistered.
     */
    virtual void unregisterModule(Module* module);
    /**
     * Registers a list of Modules to this ModuleManager.
     * @param modules - the Modules to be registered.
     */
    virtual void registerModules(Module** modules);
    /**
     * Unregisters a list of Modules from this ModuleManager.
     * @param modules - the list of Modules to be unregistered.
     */
    virtual void unregisterModules();

    /**
     * Gets a NULL-terminated array of Modules registered to this ModuleManager.
     * IMPORTANT: the user has to delete[] the returned array when it's no
     * longer needed.
     * @return - the Modules registered to this ModuleManager.
     */
    virtual Module** registeredModules() const;

protected:
private:
    // current MM version
    const Version* _mmVersion;
    // Kernel API version required by MM
    const Version* _mmAPIVersion;
    // MM info
    const Info* _mmInfo;
    // basic MM service
    const Service* _mmService;

    // needed by _modules set.
    struct ltptr
    {
        bool operator()(Module* mod1, Module* mod2) const
        {
            // just compare the pointers (addresses).
            return mod1 < mod2;
        }
    };
    typedef set<Module* , ltptr> mod_set_type;
    // the list of modules.
    mod_set_type _modules;
};
};

#endif
