/***************************************************************************
 *
 * knetworkmanager-network.cpp - A NetworkManager frontend for KDE
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Timo Hoenig        <thoenig@suse.de>, <thoenig@nouse.net>
 *         Will Stephenson    <wstephenson@suse.de>, <wstephenson@kde.org>
 *         Valentine Sinitsyn <e_val@inbox.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#include "knetworkmanager-network.h"
#include "knetworkmanager-encryption.h"

#include <kconfigbase.h>
#include <kdebug.h>

void
Network::setObjectPath (const QString & obj_path)
{
	_obj_path = obj_path;
}

QString
Network::getObjectPath () const
{
	return _obj_path;
}

void
Network::setEssid (const QString & essid)
{
	_dirty |= (_essid != essid);
	_essid = essid;
}

QString
Network::getEssid () const
{
	return _essid;
}

void
Network::insertHardwareAddress (const QString & hw_address)
{
	if ( ( hw_address != "00:00:00:00:00:00" )  && ( _hw_addresses.find( hw_address ) == _hw_addresses.end() ) ) {
		_hw_addresses.append( hw_address );
		_dirty = true;
	}
}

void
Network::removeHardwareAddress (const QString & hw_address)
{
	QStringList::Iterator it = _hw_addresses.find( hw_address );
	if ( it != _hw_addresses.end() ) {
		_hw_addresses.remove( it );
		_dirty = true;
	}
}

QStringList
Network::getHardwareAddresses () const
{
	return _hw_addresses;
}

void
Network::setHardwareAddresses( const QStringList & addresses )
{
	_dirty |= (_hw_addresses != addresses);
	_hw_addresses = addresses;
}

void
Network::setStrength (int strength)
{
	_strength = strength;
}

int
Network::getStrength () const
{
	return _strength;
}

void
Network::setFrequency (double frequency)
{
	_frequency = frequency;
}

double
Network::getFrequency () const
{
	return _frequency;
}

void
Network::setRate (int rate)
{
	_rate = rate;
}

int
Network::getRate () const
{
	return _rate;
}

void
Network::setMode (int mode)
{
	_mode = mode;
}

int
Network::getMode () const
{
	return _mode;
}

void
Network::setCapabilities (int capabilties)
{
	_capabilities = capabilties;
}

int
Network::getCapabilities () const
{
	return _capabilities;
}

void
Network::setActive (bool active)
{
	_active = active;
}

bool
Network::isActive () const
{
	return _active;
}

Encryption*
Network::getEncryption (void) const
{
	return _encryption;
}

void
Network::setEncryption (Encryption* encryption)
{
	_encryption = encryption;
	_dirty = true;
	if ( _encryption )
		_encryption->setNetwork(this);
}

QStringList
Network::getEncryptionProtocol () const
{
	QStringList encryptionProtocol;

	/* string freeze
	if (_capabilities & NM_802_11_CAP_PROTO_NONE)
		encryptionProtocol.append (i18n ("None"));
	*/
	if (_capabilities & NM_802_11_CAP_PROTO_WEP)
		encryptionProtocol.append ("WEP");
	if (_capabilities & NM_802_11_CAP_PROTO_WPA)
		encryptionProtocol.append ("WPA");
	if (_capabilities & NM_802_11_CAP_PROTO_WPA2)
		encryptionProtocol.append ("WPA2");

	return encryptionProtocol;
}

bool
Network::isEncrypted () const
{
	if (_capabilities & NM_802_11_CAP_PROTO_NONE) {
		return false;
	} else {
		return true;
	}
}

bool Network::isTrusted() const
{
	return _trusted;
}

void Network::setTrusted( bool trusted )
{
	_dirty |= (_trusted != trusted);
	_trusted = trusted;
}

bool Network::isHidden () const
{
	return _hidden;
}

void Network::setHidden (bool hidden)
{
	_dirty |= (_hidden != hidden);
	_hidden = hidden;
}

bool
Network::operator == (Network net)
{
	return (this->getObjectPath () == net.getObjectPath ());
}

bool
Network::isModified (void) const
{
	/* 
	 * Since Encryption objects are persisted on par with Network, we need
	 * to mark Network dirty if it was modified itself or it contains modified Encryption
	 */
	return ( _dirty || _encryption->isModified() );
}

Network::Network (const QString & obj_path): _encryption (0), _obj_path (obj_path), _essid (""), _hw_addresses (),
		  _strength (0), _frequency (0), _rate (0), _passphrase (""), _active( false ), _mode (0),
		  _capabilities( NM_DEVICE_CAP_NONE ), _trusted( true )
{
	_encryption = new EncryptionNone ();
	_encryption->setNetwork(this);
	_dirty = true;
}

Network::Network () : _encryption (0), _obj_path (""), _essid (""), _hw_addresses (), _strength (0), _frequency (0),
		      _rate (0), _passphrase (""), _active( false ), _mode (0), _capabilities( NM_DEVICE_CAP_NONE ),
		      _trusted( true )
{
	_encryption = new EncryptionNone ();
	_encryption->setNetwork(this);
	_dirty = true;
}

Network::~Network ()
{
	delete _encryption;
}

QDateTime Network::getTimestamp( void ) const
{
	return _timeStamp;
}

void Network::setTimestamp( const QDateTime & dt )
{
	_timeStamp = dt;
}

void Network::persist( KConfigBase * cfg, bool updateTimestamp, bool withKey ) const
{
	// write network settings
	// ESSID
	cfg->writeEntry( "ESSID", getEssid() );
	// trusted
	cfg->writeEntry( "Trusted", _trusted );
	// hardware addresses
	//if ( !_hw_addresses.isEmpty() )
	cfg->writeEntry( "HardwareAddresses", _hw_addresses );

	if ( updateTimestamp )
		persistTimestamp( cfg );

	if ( _encryption && _encryption->isModified( ) )
		_encryption->persist( cfg, withKey );

	_dirty = false;	
}

void Network::restore( KConfigBase * cfg, double version, bool withKey )
{
	setEssid( cfg->readEntry( "ESSID", QString::null ) );
	//setEncryption( 0 ); /* Memory leak here */
	QDateTime defaultTime;
	defaultTime.setTime_t( 1 ); // invalid time
	setTimestamp( cfg->readDateTimeEntry( "Timestamp", &defaultTime) );
	kdDebug() << "restore read timestamp " << _timeStamp << endl;
	_hw_addresses = cfg->readListEntry( "HardwareAddresses" );
	QString encryption = cfg->readEntry( "Encryption", "none" );
	_trusted = cfg->readBoolEntry( "Trusted", false );
	if ( _encryption ) {
		delete _encryption;
		_encryption = 0;
	}
	if ( "WPA" == encryption )
		_encryption = new EncryptionWPAPersonal ( );
	else if ( "WPA-EAP" == encryption )
		_encryption = new EncryptionWPAEnterprise ( );
	else if ( "WEP" == encryption )
		_encryption = new EncryptionWEP( WEP_ASCII ); // actual type is not important, restore overwrites it
	else if ( "none" == encryption )
		_encryption = new EncryptionNone();
	_encryption->setNetwork(this);
	_encryption->restore( cfg, version, withKey);
	_dirty = false;
}

void Network::persistTimestamp( KConfigBase *cfg ) const
{
	QDateTime current = QDateTime::currentDateTime( );
	cfg->writeEntry( "Timestamp", current );
}
