// ForthCompiler.cpp
//
// FORTH compiler to generate FORTH Byte Code (FBC) from expressions
//   or programs
//
// Copyright (c) 1998--2004 Krishna Myneni and David P. Wallace, 
//   Creative Consulting for Research and Education
//
// This software is provided under the terms of the General Public License.
//
// Revisions:
// 	9-12-1998
//	9-15-1998 added SP@, RP@, +!
//      9-16-1998 added -ROT, PICK, ROLL, A@ 
//	9-18-1998 error checking for incomplete structures at end of definition
//	10-6-1998 added ?DUP
//	10-14-1998 fixed COUNT
//	10-19-1998 added 0<, 0=, 0>, TRUE, FALSE, INVERT
//      02-09-1999 added EXECUTE, ' (tick)
//      03-01-1999 added OPEN, LSEEK, CLOSE, READ, WRITE
//      03-02-1999 added IOCTL
//      03-03-1999 added USLEEP
//      03-07-1999 added FILL, CMOVE
//      03-27-1999 added +LOOP, UNLOOP
//      03-31-1999 added CMOVE>, KEY
//      05-06-1999 added FLOOR, FROUND
//      05-24-1999 added FATAN2, LSHIFT, RSHIFT
//      05-27-1999 added ACCEPT
//      05-29-1999 added QUIT, BASE, BINARY, DECIMAL, HEX, U<, U.
//      06-02-1999 added */, */MOD, NUMBER?
//      06-05-1999 added CHAR (ASCII)
//      06-09-1999 function IsInt now calls Forth's NUMBER? 
//      06-16-1999 fixed to allow multiple LEAVEs within single DO-LOOP
//      07-18-1999 added FIND
//      08-24-1999 compiler reports redefinition of words
//      09-06-1999 added use of global ptr pTIB to permit implemetation of TICK, WORD, etc.
//      09-12-1999 added SYSTEM
//      10-2-1999  used precedence byte to determine execution of non-deferred words
//      10-4-1999 added CREATE, VARIABLE, FVARIABLE as intrinsic words
//      10-6-1999 added CONSTANT, FCONSTANT as intrinsic words
//      10-7-1999 added CHDIR
//      10-8-1999 added ERASE, [']
//      10-9-1999 added TIME&DATE, MS, ?, 2@, 2!, BL
//      10-20-1999 moved global input and output stream pointers into
//                   this module from ForthVM.cpp; added >FILE, CONSOLE
//      10-28-1999 added KEY?
//      11-16-1999 added RECURSE
//      12-14-1999 fixed ExtractName for case of null string
//      12-24-1999 added U>, F0=, F0<, S>D, D>F, F>D
//      12-25-1999 added CELLS, CELL+, CHAR+, DFLOATS, DFLOAT+, SFLOATS, SFLOAT+
//      12-27-1999 added BYE
//      1-13-2000  added ?ALLOT
//      1-23-2000  added 0<>, .R, U.R, [CHAR]; removed ASCII
//      1-24-2000  added LITERAL, made '"' and '."' immediate words
//      2-14-2000  added M*, UM*, FM/MOD
//      2-26-2000  added SM/REM, UM/MOD
//      3-1-2000   display VM errors
//      3-5-2000   changed DO, LEAVE, BEGIN, WHILE, REPEAT, UNTIL, AGAIN,
//                   IF, ELSE, THEN, RECURSE, and '(' from compiler directives
//                   to actual words.
//      3-7-2000   ensure control stacks are cleared after VM error.
//      5-17-2000  added DOES>
//      6-11-2000  added CASE, ENDCASE, OF, ENDOF
//      6-15-2000  added ?DO, ABORT"
//      8-08-2000  added default directory search for include files  DPW
//      9-05-2000  added M/, M+, D.
//      11-29-2000 added DABS, DNEGATE, M*/
//      04-22-2001 added D+, D-
//      04-24-2001 added 2>R, 2R>
//      05-13-2001 added .(, D<, D=, D0=
//      05-20-2001 added <#, #, #S, #>, SIGN, HOLD
//      05-30-2001 modified loop code to handle ?DO
//      09-03-2001 added >BODY, IMMEDIATE, NONDEFERRED, POSTPONE; fixed
//                   immediate execution of defined words.
//      12-08-2001 added EVALUATE
//      02-10-2002 made .( a non-deferred word
//      08-01-2002 added \ as a word; added STATE; fixed behavior of 
//                   POSTPONE for non-immediate words; added MS@;
//                   code cleanup in ForthCompiler()
//      09-25-2002 updated include statements and added "using" directives
//                   to resolve std namespace definitions for gcc 3.2
//      09-29-2002 added IMMEDIATE as a regular word; cleaned up logic
//                   for INCLUDE
//      04-11-2003 changed F>S to FROUND>S
//      04-15-2003 added FTRUNC and FTRUNC>S
//      01-31-2004 extended intrinsic wordlist to include INCLUDE,
//                   SOURCE, REFILL, NONDEFERRED, STATE
//      02-09-2004 added ALLOCATE and FREE.
//      03-18-2004 added FSINCOS, FSINH, FCOSH, FTANH, FASINH, FACOSH, 
//                   FATANH, SP!, RP!
//      04-23-2004 changed variable "debug" to type "bool"
//      04-xx-2004 changed intrinsic wordlist specification to
//                   structure array following suggestion of BK.
//      06-19-2004 updated error handling method after call to ForthVM()
#include <iostream>
#include <fstream>
using std::cout;
using std::endl;
using std::istream;
using std::ostream;
using std::ifstream;
using std::ofstream;
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include "fbc.h"
#include <vector>
#include <stack>
using std::vector;
using std::stack;
#include "ForthCompiler.h"

const int IMMEDIATE   = PRECEDENCE_IMMEDIATE;
const int NONDEFERRED = PRECEDENCE_NON_DEFERRED;

extern bool debug;

// Provided by ForthVM.cpp

extern vector<DictionaryEntry> Dictionary;
extern vector<char*> StringTable;
void ClearControlStacks();
void OpsPushInt (int);
void PrintVM_Error (int);
int ForthVM (vector<byte>*, int**, byte**);
vector<DictionaryEntry>::iterator LocateWord (char*);
void RemoveLastWord();

extern "C" {

  // Provided by ForthVM.cpp

  int CPP_then();
  int CPP_immediate();
  int CPP_nondeferred();
  int CPP_source();
  int CPP_refill();

  // Provided by vm.s, vmc.c

  int C_numberquery();
  int L_abort();
}


extern "C" int* GlobalSp;
extern "C" int* GlobalRp;

extern "C" int* JumpTable;
extern "C" int Base;
extern "C" int State;  // TRUE = compile, FALSE = interpret
extern "C" char* pTIB; 
extern "C"  char TIB[];  // contains current line of input

void strupr (char*);

// stacks for keeping track of nested control structures

vector<int> ifstack;	// stack for if-then constructs
vector<int> beginstack;	// stack for begin ... constructs
vector<int> whilestack;	// stack for while jump holders
vector<int> dostack;    // stack for do loops
vector<int> querydostack; // stack for conditional do loops
vector<int> leavestack; // stack for leave jumps
vector<int> recursestack; // stack for recursion
vector<int> casestack;  // stack for case jumps
vector<int> ofstack;   // stack for of...endof constructs

int linecount;

// The global input and output streams

istream* pInStream ;
ostream* pOutStream ;

// Global ptr to current opcode vector

vector<byte>* pCurrentOps;

// The word currently being compiled (needs to be global)

DictionaryEntry NewWord;

WordTemplate IntrinsicWords[] =
{
    { "WORD",      OP_WORD,         NONDEFERRED },
    { "WORDS",     OP_WORDS,        0 },
    { "FIND",      OP_FIND,         0 },
    { "'",         OP_TICK,         NONDEFERRED },
    { "[']",       OP_BRACKETTICK,  IMMEDIATE },
    { "COMPILE,",  OP_COMPILECOMMA, 0 },
//    { "[COMPILE]", OP_BRACKETCOMPILE, IMMEDIATE },
    { "POSTPONE",  OP_POSTPONE,     IMMEDIATE },
    { "[",         OP_LBRACKET,     IMMEDIATE },
    { "]",         OP_RBRACKET,     IMMEDIATE },
    { "STATE",     OP_STATE,        0 },
    { "CREATE",    OP_CREATE,       NONDEFERRED },
    { "DOES>",     OP_DOES,         0 },
    { ">BODY",     OP_TOBODY,       0 },
    { "FORGET",    OP_FORGET,       NONDEFERRED },
    { "COLD",      OP_COLD,         NONDEFERRED },
    { ":",         OP_COLON,        NONDEFERRED },
    { ";",         OP_SEMICOLON,    IMMEDIATE },
    { "ALLOCATE",  OP_ALLOCATE,     0 },     
    { "FREE",      OP_FREE,         0 },
    { "ALLOT",     OP_ALLOT,        NONDEFERRED },
    { "?ALLOT",    OP_QUERYALLOT,   NONDEFERRED },
    { "LITERAL",   OP_LITERAL,      IMMEDIATE },
    { "EVALUATE",  OP_EVALUATE,     0 },
    { "INCLUDE",   OP_INCLUDE,      NONDEFERRED },
    { "SOURCE",    OP_SOURCE,       0 },
    { "REFILL",    OP_REFILL,       0 }, 
    { "IMMEDIATE", OP_IMMEDIATE,    0 },
    { "NONDEFERRED", OP_NONDEFERRED, 0 },
    { "CONSTANT",  OP_CONSTANT,     NONDEFERRED },
    { "FCONSTANT", OP_FCONSTANT,    NONDEFERRED },
    { "VARIABLE",  OP_VARIABLE,     NONDEFERRED },
    { "FVARIABLE", OP_FVARIABLE,    NONDEFERRED },
    { "CELLS",     OP_CELLS,        0 },
    { "CELL+",     OP_CELLPLUS,     0 },
    { "CHAR+",     OP_INC,          0 },
    { "DFLOATS",   OP_DFLOATS,      0 },
    { "DFLOAT+",   OP_DFLOATPLUS,   0 },
    { "SFLOATS",   OP_CELLS,        0 },
    { "SFLOAT+",   OP_CELLPLUS,     0 },
    { "?",         OP_QUESTION,     0 },
    { "@",         OP_FETCH,        0 },
    { "!",         OP_STORE,        0 },
    { "2@",        OP_DFFETCH,      0 },
    { "2!",        OP_DFSTORE,      0 },
    { "A@",        OP_AFETCH,       0 },
    { "C@",        OP_CFETCH,       0 },
    { "C!",        OP_CSTORE,       0 },
    { "W@",        OP_WFETCH,       0 },
    { "W!",        OP_WSTORE,       0 },
    { "F@",        OP_DFFETCH,      0 },
    { "F!",        OP_DFSTORE,      0 },
    { "DF@",       OP_DFFETCH,      0 },
    { "DF!",       OP_DFSTORE,      0 },
    { "SF@",       OP_SFFETCH,      0 },
    { "SF!",       OP_SFSTORE,      0 },
    { "SP@",       OP_SPFETCH,      0 },
    { "SP!",       OP_SPSTORE,      0 },
    { "RP@",       OP_RPFETCH,      0 },
    { "RP!",       OP_RPSTORE,      0 },
    { ">R",        OP_PUSH,         0 },
    { "R>",        OP_POP,          0 },
    { "R@",        OP_RFETCH,       0 },
    { "2>R",       OP_TWOPUSH,      0 }, 
    { "2R>",       OP_TWOPOP,       0 },
    { "2R@",       OP_TWORFETCH,    0 }, 
    { "?DUP",      OP_QUERYDUP,     0 },
    { "DUP",       OP_DUP,          0 },
    { "DROP",      OP_DROP,         0 },
    { "SWAP",      OP_SWAP,         0 },
    { "OVER",      OP_OVER,         0 },
    { "ROT",       OP_ROT,          0 }, 
    { "-ROT",      OP_MINUSROT,     0 }, 
    { "NIP",       OP_NIP,          0 },
    { "TUCK",      OP_TUCK,         0 }, 
    { "PICK",      OP_PICK,         0 },
    { "ROLL",      OP_ROLL,         0 },
    { "2DUP",      OP_2DUP,         0 },
    { "2DROP",     OP_2DROP,        0 },
    { "2SWAP",     OP_2SWAP,        0 },
    { "2OVER",     OP_2OVER,        0 },
    { "2ROT",      OP_2ROT,         0 },
    { "DEPTH",     OP_DEPTH,        0 },
    { "BASE",      OP_BASE,         0 },
    { "BINARY",    OP_BINARY,       NONDEFERRED },
    { "DECIMAL",   OP_DECIMAL,      NONDEFERRED },
    { "HEX",       OP_HEX,          NONDEFERRED },
    { "1+",        OP_INC,          0 },
    { "1-",        OP_DEC,          0 },
    { "2+",        OP_TWOPLUS,      0 },
    { "2-",        OP_TWOMINUS,     0 },
    { "2*",        OP_TWOSTAR,      0 },
    { "2/",        OP_TWODIV,       0 },
    { "DO",        OP_DO,           IMMEDIATE },
    { "?DO",       OP_QUERYDO,      IMMEDIATE },
    { "LOOP",      OP_LOOP,         0 },
    { "+LOOP",     OP_PLUSLOOP,     0 },
    { "LEAVE",     OP_LEAVE,        IMMEDIATE },
    { "UNLOOP",    OP_UNLOOP,       0 },
    { "I",         OP_I,            0 },
    { "J",         OP_J,            0 },
    { "BEGIN",     OP_BEGIN,        IMMEDIATE },
    { "WHILE",     OP_WHILE,        IMMEDIATE },
    { "REPEAT",    OP_REPEAT,       IMMEDIATE },
    { "UNTIL",     OP_UNTIL,        IMMEDIATE },
    { "AGAIN",     OP_AGAIN,        IMMEDIATE },
    { "IF",        OP_IF,           IMMEDIATE },
    { "ELSE",      OP_ELSE,         IMMEDIATE },
    { "THEN",      OP_THEN,         IMMEDIATE },
    { "CASE",      OP_CASE,         IMMEDIATE },
    { "ENDCASE",   OP_ENDCASE,      IMMEDIATE },
    { "OF",        OP_OF,           IMMEDIATE },
    { "ENDOF",     OP_ENDOF,        IMMEDIATE },
    { "RECURSE",   OP_RECURSE,      IMMEDIATE },
    { "BYE",       OP_BYE,          0 },
    { "EXIT",      OP_RET,          0 },
    { "QUIT",      OP_QUIT,         0 },
    { "ABORT",     OP_ABORT,        0 },
    { "ABORT\x22", OP_ABORTQUOTE,   IMMEDIATE },
    { "USLEEP",    OP_USLEEP,       0 },
    { "EXECUTE",   OP_EXECUTE,      0 },
    { "CALL",      OP_CALL,         0 },
    { "SYSTEM",    OP_SYSTEM,       0 },
    { "TIME&DATE", OP_TIMEANDDATE,  0 },
    { "MS",        OP_MS,           0 },
    { "MS@",       OP_MSFETCH,      0 },
    { "CHDIR",     OP_CHDIR,        0 },
    { ">FILE",     OP_TOFILE,       NONDEFERRED },
    { "CONSOLE",   OP_CONSOLE,      NONDEFERRED },
    { "\\",        OP_BACKSLASH,    IMMEDIATE | NONDEFERRED },
    { "(",         OP_LPAREN,       IMMEDIATE },
    { ".(",        OP_DOTPAREN,     IMMEDIATE | NONDEFERRED },
    { "\x22",      OP_CQUOTE,       IMMEDIATE },
    { "C\x22",     OP_CQUOTE,       IMMEDIATE },
    { "S\x22",     OP_SQUOTE,       IMMEDIATE },
    { "COUNT",     OP_COUNT,        0 },
    { "NUMBER?",   OP_NUMBERQUERY,  0 },
    { "<#",        OP_BRACKETSHARP, 0 },
    { "#",         OP_SHARP,        0 },
    { "#S",        OP_SHARPS,       0 },
    { "#>",        OP_SHARPBRACKET, 0 },
    { "SIGN",      OP_SIGN,         0 },
    { "HOLD",      OP_HOLD,         0 },
    { ".",         OP_DOT,          0 },
    { ".R",        OP_DOTR,         0 },
    { "D.",        OP_DDOT,         0 },
    { "U.",        OP_UDOT,         0 },
    { "U.R",       OP_UDOTR,        0 },
    { "F.",        OP_FDOT,         0 },
    { ".\x22",     OP_DOTQUOTE,     IMMEDIATE },
    { ".S",        OP_DOTS,         0 },
    { "CR",        OP_CR,           0 },
    { "SPACES",    OP_SPACES,       0 },
    { "EMIT",      OP_EMIT,         0 },
    { "TYPE",      OP_TYPE,         0 },
    { "BL",        OP_BL,           0 },
    { "[CHAR]",    OP_BRACKETCHAR,  IMMEDIATE },
    { "CHAR",      OP_CHAR,         NONDEFERRED },
    { "KEY",       OP_KEY,          0 },
    { "KEY?",      OP_KEYQUERY,     0 },
    { "ACCEPT",    OP_ACCEPT,       0 },
    { "SEARCH",    OP_SEARCH,       0 },
    { "COMPARE",   OP_COMPARE,      0 },
    { "=",         OP_EQ,           0 },
    { "<>",        OP_NE,           0 },
    { "<",         OP_LT,           0 },
    { ">",         OP_GT,           0 },
    { "<=",        OP_LE,           0 },
    { ">=",        OP_GE,           0 },
    { "U<",        OP_ULT,          0 },
    { "U>",        OP_UGT,          0 },
    { "0<",        OP_ZEROLT,       0 },
    { "0=",        OP_ZEROEQ,       0 },
    { "0<>",       OP_ZERONE,       0 },
    { "0>",        OP_ZEROGT,       0 },
    { "D<",        OP_DLT,          0 },
    { "D=",        OP_DEQ,          0 },
    { "D0=",       OP_DZEROEQ,      0 },
    { "FALSE",     OP_FALSE,        0 },
    { "TRUE",      OP_TRUE,         0 },
    { "AND",       OP_AND,          0 },
    { "OR",        OP_OR,           0 },
    { "XOR",       OP_XOR,          0 },
    { "NOT",       OP_NOT,          0 },
    { "INVERT",    OP_NOT,          0 },
    { "LSHIFT",    OP_LSHIFT,       0 },
    { "RSHIFT",    OP_RSHIFT,       0 },
    { "+",         OP_ADD,          0 },
    { "-",         OP_SUB,          0 },
    { "*",         OP_MUL,          0 },
    { "/",         OP_DIV,          0 },
    { "MOD",       OP_MOD,          0 },
    { "/MOD",      OP_SLASHMOD,     0 },
    { "*/",        OP_STARSLASH,    0 },
    { "*/MOD",     OP_STARSLASHMOD, 0 },
    { "+!",        OP_PLUSSTORE,    0 },
    { "D+",        OP_DPLUS,        0 },
    { "D-",        OP_DMINUS,       0 },
    { "M+",        OP_MPLUS,        0 },
    { "M*",        OP_MSTAR,        0 },
    { "M/",        OP_MSLASH,       0 },
    { "M*/",       OP_MSTARSLASH,   0 },
    { "UM*",       OP_UMSTAR,       0 },
    { "UM/MOD",    OP_UMSLASHMOD,   0 },
    { "FM/MOD",    OP_FMSLASHMOD,   0 },
    { "SM/REM",    OP_SMSLASHREM,   0 },
    { "ABS",       OP_ABS,          0 },
    { "NEGATE",    OP_NEG,          0 },
    { "MIN",       OP_MIN,          0 },
    { "MAX",       OP_MAX,          0 },
    { "DABS",      OP_DABS,         0 },
    { "DNEGATE",   OP_DNEGATE,      0 },
    { "OPEN",      OP_OPEN,         0 },
    { "LSEEK",     OP_LSEEK,        0 },
    { "CLOSE",     OP_CLOSE,        0 },
    { "READ",      OP_READ,         0 },
    { "WRITE",     OP_WRITE,        0 },
    { "IOCTL",     OP_IOCTL,        0 },
    { "FILL",      OP_FILL,         0 },
    { "ERASE",     OP_ERASE,        0 },
    { "CMOVE",     OP_CMOVE,        0 },
    { "CMOVE>",    OP_CMOVEFROM,    0 },
    { "FDUP",      OP_2DUP,         0 },
    { "FDROP",     OP_2DROP,        0 },
    { "FSWAP",     OP_2SWAP,        0 },
    { "FOVER",     OP_2OVER,        0 },
    { "FROT",      OP_2ROT,         0 },
    { "F=",        OP_FEQ,          0 },
    { "F<>",       OP_FNE,          0 },
    { "F<",        OP_FLT,          0 },
    { "F>",        OP_FGT,          0 },
    { "F<=",       OP_FLE,          0 },
    { "F>=",       OP_FGE,          0 },
    { "F0=",       OP_FZEROEQ,      0 },
    { "F0<",       OP_FZEROLT,      0 },
    { "F+",        OP_FADD,         0 },
    { "F-",        OP_FSUB,         0 },
    { "F*",        OP_FMUL,         0 },
    { "F/",        OP_FDIV,         0 },
    { "F**",       OP_FPOW,         0 },
    { "FSQRT",     OP_FSQRT,        0 },
    { "FABS",      OP_FABS,         0 },
    { "FNEGATE",   OP_FNEG,         0 },
    { "FLOOR",     OP_FLOOR,        0 },
    { "FROUND",    OP_FROUND,       0 },
    { "FTRUNC",    OP_FTRUNC,       0 },
    { "FMIN",      OP_FMIN,         0 },
    { "FMAX",      OP_FMAX,         0 },
    { "FSIN",      OP_FSIN,         0 },
    { "FCOS",      OP_FCOS,         0 },
    { "FSINCOS",   OP_FSINCOS,      0 },
    { "FTAN",      OP_FTAN,         0 },
    { "FACOS",     OP_FACOS,        0 },
    { "FASIN",     OP_FASIN,        0 },
    { "FATAN",     OP_FATAN,        0 },
    { "FATAN2",    OP_FATAN2,       0 },
    { "FSINH",     OP_FSINH,        0 },
    { "FCOSH",     OP_FCOSH,        0 },
    { "FTANH",     OP_FTANH,        0 },
    { "FASINH",    OP_FASINH,       0 },
    { "FACOSH",    OP_FACOSH,       0 },
    { "FATANH",    OP_FATANH,       0 },
    { "FLOG",      OP_FLOG,         0 },
    { "FLN",       OP_FLN,          0 },
    { "FEXP",      OP_FEXP,         0 },
    { "DEG>RAD",   OP_DEGTORAD,     0 },
    { "RAD>DEG",   OP_RADTODEG,     0 },
    { "S>D",       OP_STOD,         0 },
    { "S>F",       OP_STOF,         0 },
    { "D>F",       OP_DTOF,         0 },
    { "F>D",       OP_FTOD,         0 },
    { "FROUND>S",  OP_FROUNDTOS,    0 },
    { "FTRUNC>S",  OP_FTRUNCTOS,    0 }
};
//---------------------------------------------------------------


char* C_ErrorMessages[] =
{
	"",
	"",
	"End of definition with no beginning",
	"End of string",	 
        "Not allowed inside colon definition",
	"Error opening file",
	"Incomplete IF...THEN structure",
	"Incomplete BEGIN structure",
	"Unknown word",
	"No matching DO",
	"Incomplete DO loop",
	"Incomplete CASE structure",
	"VM returned error"
};
//---------------------------------------------------------------

char* ExtractName (char* str, char* name)
{
// Starting at ptr str, extract the non delimiter text into
//   a buffer starting at name with null terminator appended
//   at the end. Return a pointer to the next position in
//   str.

    char* delim = "\n\r\t ";
    char *pStr = str, *pName = name;

    if (*pStr)
      {
	while (strchr(delim, *pStr)) ++pStr;
	while (*pStr && (strchr(delim, *pStr) == NULL))
	  {
	    *pName = *pStr;
	    ++pName;
	    ++pStr;
	  }
      }
    *pName = 0;
    return pStr;
}
//---------------------------------------------------------------
	
int IsForthWord (char* name, DictionaryEntry* pE)
{
// Locate and Return a copy of the dictionary entry
//   with the specified name.  Return True if found,
//   False otherwise. A copy of the entry is returned
//   in *pE.

    vector<DictionaryEntry>::iterator i = LocateWord (name);

    if (i != (vector<DictionaryEntry>::iterator) NULL)
    {
        *pE = *i;
        return TRUE;
    }
    else
        return FALSE;
}
//---------------------------------------------------------------

int IsFloat (char* token, double* p)
{
// Check the string token to see if it is an LMI style floating point
//   number; if so set the value of *p and return True, otherwise
//   return False.

    char *pStr = token;

//    cout << "\nIsFloat: token = " << token;

    if (strchr(pStr, 'E'))
    {
        while ((isdigit(*pStr)) || (*pStr == '-')
          || (*pStr == 'E') || (*pStr == '+') || (*pStr == '.'))
        {
            ++pStr;
//            cout << ' ' << ((int) *pStr);
        }
        if (*pStr == 0)
        {
            // LMI Forth style

            --pStr;
            if (*pStr == 'E') *pStr = '\0';
            *p = atof(token);
            return TRUE;
        }
    }

    return FALSE;
}
//----------------------------------------------------------------

int IsInt (char* token, int* p)
{
// Check the string token to see if it is an integer number;
//   if so set the value of *p and return True, otherwise return False.

  char s[256];
  *s = (unsigned char) strlen(token);
  strcpy (s+1, token);
  *GlobalSp-- = (int) s;

  
  int err = C_numberquery();
  if (err)
    {
      // stack has probably become corrupted -- call abort

      cout << "Stack error during compilation.\n";
      L_abort();
      return FALSE;
    }

  ++GlobalSp;
  int b = *GlobalSp++;
  ++GlobalSp;
  *p = *GlobalSp;

  return b;
}
//---------------------------------------------------------------

void OutputForthByteCode (vector<byte>* pFBC)
{
// Output opcode vector to an output stream for use in
//   debugging the compiler.

    int i, n = pFBC->size();
    byte* bp = (byte*) &(*pFBC)[0]; // ->begin();

    *pOutStream << "\nOpcodes:\n";
    for (i = 0; i < n; i++)
    {
        *pOutStream << ((int) *bp) << ' ';
        if (((i + 1) % 8) == 0) *pOutStream << '\n';
        ++bp;
    }
    *pOutStream << '\n';
    return;
}
//---------------------------------------------------------------

void SetForthInputStream (istream& SourceStream)
{
  // Set the input stream for the Forth Compiler and Virtual Machine

  pInStream = &SourceStream;
}
//--------------------------------------------------------------

void SetForthOutputStream (ostream& OutStream)
{
  // Set the output stream for the Forth Compiler and Virtual Machine

  pOutStream = &OutStream;
}
//---------------------------------------------------------------

int ForthCompiler (vector<byte>* pOpCodes, int* pLc)
{
// The FORTH Compiler
//
// Reads and compile the source statements from the input stream
//   into a vector of FORTH Byte Codes.
//
// Return value:
//
//  0   no error
//  other --- see ForthCompiler.h

  int ecode = 0, opcount = 0;
  char s[256], WordToken[256], filename[256]; 
  char *begin_string, *end_string, *str;
  double fval;
  int i, j, ival, *sp;
  vector<byte>::iterator ib1, ib2;
  vector<int>::iterator iI;
  DictionaryEntry d;
  vector<DictionaryEntry>::iterator id;
  byte opval, *fp, *ip, *bp, *tp;

  static bool postpone = FALSE;

  if (debug) cout << ">Compiler Sp: " << GlobalSp << " Rp: " << GlobalRp << endl;

  fp = (byte *) &fval;
  ip = (byte *) &ival;

  // if (! State) linecount = 0;
  linecount = *pLc;
  pCurrentOps = pOpCodes;

  while (TRUE)
    {
      // Read each line and parse

      pInStream->getline(TIB, 255);
      if (debug) (*pOutStream) << TIB << endl;

      if (pInStream->fail())
	{
	  if (State)
	    {
	      ecode = E_C_ENDOFSTREAM;  // reached end of stream before end of definition
	      break;
	    }
	  // pOpCodes->push_back(OP_RET);
	  break;    // end of stream reached
	}
      ++linecount;
      pTIB = TIB;
      while (*pTIB && (pTIB < (TIB + 255)))
	{
	  if (*pTIB == ' ' || *pTIB == '\t')
	    ++pTIB;

	  else
	    {
	      pTIB = ExtractName (pTIB, WordToken);
	      strupr(WordToken);

	      if (IsForthWord(WordToken, &d))
		{
		    if (d.WordCode >> 8) d.WordCode = OP_DEFINITION;
 		    pOpCodes->push_back(d.WordCode);
		  
		  if (d.WordCode == OP_DEFINITION)
		    {
		      OpsPushInt((int) d.Cfa);
		    }
		  else if (d.WordCode == OP_ADDR)
		    {
		      // push address into the byte code vector

		      OpsPushInt((int) d.Pfa);
		    }
		  else if (d.WordCode == OP_IVAL)
		    {
		      // push value into the byte code vector

		      OpsPushInt(*((int*)d.Pfa));			
		    }
		  else if (d.WordCode == OP_FVAL)
		    {
		      // push float value into the vector

		      bp = (byte*) d.Pfa;
		      for (i = 0; i < sizeof(double); i++)
			pOpCodes->push_back(*(bp + i));
		    }
		  else if (d.WordCode == OP_UNLOOP)
		    {
		      if (dostack.empty())
			{
			  ecode = E_C_NODO;
			  goto endcompile;
			}
		    }
		  else if (d.WordCode == OP_LOOP || d.WordCode == OP_PLUSLOOP)
		    {
		      if (dostack.empty())
			{
			  ecode = E_C_NODO;
			  goto endcompile;
			}
		      i = dostack[dostack.size() - 1];
		      if (leavestack.size())
			{
			  do
			    {
			      j = leavestack[leavestack.size() - 1];
			      if (j > i)
				{
				  ival = pOpCodes->size() - j + 1;
				  ib1 = pOpCodes->begin() + j;
				  *ib1++ = *ip;       // write the relative jump count
				  *ib1++ = *(ip + 1);
				  *ib1++ = *(ip + 2);
				  *ib1 = *(ip + 3);
				  leavestack.pop_back();
				}
			    } while ((j > i) && (leavestack.size())) ;
			}
		      dostack.pop_back();
		      if (querydostack.size())
			{
			  j = querydostack[querydostack.size() - 1];
			  if (j >= i)
			    {
			      CPP_then();
			      querydostack.pop_back();
			    }
			}
		    }
		  else
		    {
		      ;
		    }

		  int execution_method = EXECUTE_NONE;

		  switch (d.Precedence)
		    {
		      case IMMEDIATE:
			execution_method = EXECUTE_CURRENT_ONLY;
			break;
		      case NONDEFERRED:
			if (State)
			  NewWord.Precedence |= NONDEFERRED ;
			else
			  execution_method = EXECUTE_UP_TO;
			break;
		      case (NONDEFERRED + IMMEDIATE):
			execution_method = State ? EXECUTE_CURRENT_ONLY :
			  EXECUTE_UP_TO;
			break;
		      default:
			;
		    }

		  vector<byte> SingleOp;
		  
		  switch (execution_method)
		    {
		    case EXECUTE_UP_TO:
		      // Execute the opcode vector immediately up to and
		      //   including the current opcode

		      pOpCodes->push_back(OP_RET);
		      if (debug) OutputForthByteCode (pOpCodes);
		      ecode = ForthVM (pOpCodes, &sp, &tp);
		      pOpCodes->erase(pOpCodes->begin(), pOpCodes->end());
		      if (ecode) goto endcompile; 
		      break;

		    case EXECUTE_CURRENT_ONLY:
		      i = (d.WordCode == OP_DEFINITION) ? 5 : 1;
		      ib1 = pOpCodes->end() - i;
		      for (j = 0; j < i; j++) SingleOp.push_back(*(ib1+j));
		      SingleOp.push_back(OP_RET);
		      pOpCodes->erase(ib1, pOpCodes->end());
		      ecode = ForthVM (&SingleOp, &sp, &tp);
		      SingleOp.erase(SingleOp.begin(), SingleOp.end());
		      if (ecode) goto endcompile; 
		      pOpCodes = pCurrentOps; // may have been redirected
		      break;

		    default:
		      ;
		    }

		}  // end if (IsForthWord())

	      else if (IsInt(WordToken, &ival))
		{
		  pOpCodes->push_back(OP_IVAL);
		  OpsPushInt(ival);
		}
	      else if (IsFloat(WordToken, &fval))
		{
		  pOpCodes->push_back(OP_FVAL);
		  for (i = 0; i < sizeof(double); i++)
		    pOpCodes->push_back(*(fp + i)); // store in proper order
		}
	      else
		{
		  *pOutStream << endl << WordToken << endl;
		  ecode = E_C_UNKNOWNWORD;  // unknown keyword
		  goto endcompile;
		}
	    }
	} // end while (*pTIB ...)
	
      if ((State == 0) && pOpCodes->size())
	{
	  // Execute the current line in interpretation state
	  pOpCodes->push_back(OP_RET);
	  if (debug) OutputForthByteCode (pOpCodes);
	  ecode = ForthVM (pOpCodes, &sp, &tp);
	  pOpCodes->erase(pOpCodes->begin(), pOpCodes->end());
	  if (ecode) goto endcompile; 
	}

    } // end while (TRUE)

endcompile:
    
  if ((ecode != E_C_NOERROR) && (ecode != E_C_ENDOFSTREAM))
    {
      // A compiler error occurred; reset to interpreter mode and
      //   clear all flow control stacks.

      State = FALSE;
      ClearControlStacks();
    }
  if (debug) 
    {
      *pOutStream << "Error: " << ecode << " State: " << State << endl;
      *pOutStream << "<Compiler Sp: " << GlobalSp << " Rp: " << GlobalRp << endl;
    }
  *pLc = linecount;
  return ecode;
}


void strupr (char* p)
{
// convert string to upper case

  while (*p) {*p = toupper(*p); ++p;}
}

