/***************************************************************************
                          keybled.cpp  -  description
                             -------------------
    begin                : Fri Jul 13 2001
    copyright            : (C) 2001 by Aurelien Jarno
    email                : aurelien@aurel32.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// includes files for Qt
#include <qcombobox.h>
#include <qtooltip.h>
#include <qcheckbox.h>
#include <qradiobutton.h>
#include <qslider.h>
#include <qgroupbox.h>
#include <qtabwidget.h>
#include <qfont.h>
#include <qcursor.h>

// includes files for KDE
#include <kglobal.h>
#include <kglobalsettings.h>
#include <klocale.h>
#include <kconfig.h>
#include <kapp.h>
#include <kwin.h>
#include <kmessagebox.h>
#include <kaboutdata.h>
#include <kaboutapplication.h>
#include <kcolorbtn.h>
#include <kpopupmenu.h>
#include <kiconloader.h>
#include <kcolordialog.h>

// application specific includes
#include "keybled.h"
#include "keybled.moc"
#include "popup.h"
#include "confdialog.h"
#include "kledtoggle.h"

// X11 includes
extern "C"
{
  #include <X11/Xlib.h>
  #include <X11/keysym.h>
  #include <X11/extensions/XTest.h>
}

extern "C"
{
  KPanelApplet* init(QWidget *parent, const QString& configFile)
  {
    KGlobal::locale()->insertCatalogue("keybled");
    return new KeybLED(configFile, KPanelApplet::Normal,
                      KPanelApplet::About | KPanelApplet::Help | KPanelApplet::Preferences,
                      parent, "keybled");
  }
}

KeybLED::KeybLED(const QString& configFile, Type type, int actions, QWidget *parent, const char *name)
       : KPanelApplet(configFile, type, actions, parent, name),
       confDialog(0), ledColor("green")
{
  // Get the current application configuration handle
  ksConfig = config();
  readSettings();

  // Xtest stuff
  xtest_init();
  SetLEDStartup();

  // NUM LED creation
  numLed = new KLedToggle(this, "numLed");
  numLed->installEventFilter(this);
  QToolTip::add(numLed, i18n("NUM.LOCK"));
  if (numlock_mask != 0) connect(numLed, SIGNAL(toggled()), this, SLOT(slotToggleNumLED()));

  numLabel = new QLabel(i18n("N"), this, "numLabel");
  numLabel->resize(20, 10);
  numLabel->setAlignment(AlignHCenter | AlignVCenter);
  numLabel->installEventFilter(this);
  QToolTip::add(numLabel, i18n("NUM.LOCK"));

  // CAPS LED creation
  capsLed = new KLedToggle(this, "capsLed");
  capsLed->installEventFilter(this);
  QToolTip::add(capsLed, i18n("CAPS.LOCK"));
  if (capslock_mask != 0) connect(capsLed, SIGNAL(toggled()), this, SLOT(slotToggleCapsLED()));

  capsLabel = new QLabel(i18n("C"), this, "capsLabel");
  capsLabel->resize(20, 10);
  capsLabel->setAlignment(AlignHCenter | AlignVCenter);
  capsLabel->installEventFilter(this);
  QToolTip::add(capsLabel, i18n("CAPS.LOCK"));

  // SCROLL LED creation
  scrollLed = new KLedToggle(this, "scrollLed");
   scrollLed->installEventFilter(this);
  QToolTip::add(scrollLed, "SCROLL.LOCK");
	if (scrolllock_mask != 0) connect(scrollLed, SIGNAL(toggled()), this, SLOT(slotToggleScrollLED()));

  scrollLabel = new QLabel(i18n("S"), this, "scrollLabel");
  scrollLabel->resize(20, 10);
  scrollLabel->setAlignment(AlignHCenter | AlignVCenter);
  scrollLabel->installEventFilter(this);
  QToolTip::add(scrollLabel, i18n("SCROLL.LOCK"));

  popup = new PopUp();
  connect(popup, SIGNAL(clicked()), this, SLOT(hidePopUp()));
  connect(popup, SIGNAL(savePos()), this, SLOT(savePopUpPos()));

  timer = new QTimer(this);
  connect(timer, SIGNAL(timeout()), this, SLOT(timerEvent()));
  timer->start(refreshTime, False);

  timerEvent();
  applySettings();
}

KeybLED::~KeybLED()
{
  if (numlock_mask != 0) disconnect(numLed, SIGNAL(toggled()), this, SLOT(slotToggleNumLED()));
  delete numLed;
  delete numLabel;

  if (scrolllock_mask != 0) disconnect(scrollLed, SIGNAL(toggled()), this, SLOT(slotToggleScrollLED()));
  delete capsLed;
  delete capsLabel;

  if (capslock_mask != 0) disconnect(capsLed, SIGNAL(toggled()), this, SLOT(slotToggleCapsLED()));
  delete scrollLed;
  delete scrollLabel;

  timer->stop();
  delete timer;

  delete popup;

  if (confDialog)
  {
    confDialog->close();
    delete confDialog;
  }
}

void KeybLED::xtest_init()
{
  display = this->x11Display();
  numlock_mask = 0 ; capslock_mask = 0 ; scrolllock_mask = 0;

  XModifierKeymap* map = XGetModifierMapping(display);
  KeyCode numlock_keycode = XKeysymToKeycode(display, XK_Num_Lock);
  KeyCode capslock_keycode = XKeysymToKeycode(display, XK_Caps_Lock);
  KeyCode scrolllock_keycode = XKeysymToKeycode(display, XK_Scroll_Lock);

  for(int i = 0 ; i < 8; ++i)
  {
    if (map->modifiermap[map->max_keypermod * i] == numlock_keycode) numlock_mask = 1 << i;
    if (map->modifiermap[map->max_keypermod * i] == capslock_keycode) capslock_mask = 1 << i;
    if (map->modifiermap[map->max_keypermod * i] == scrolllock_keycode) scrolllock_mask = 1 << i;
  }

  if (numlock_keycode == NoSymbol) numlock_mask = 0;
  if (capslock_keycode == NoSymbol) capslock_mask = 0;
  if (scrolllock_keycode == NoSymbol) scrolllock_mask = 0;

  XFreeModifiermap(map);
}

void KeybLED::toggleNumLED()
{
  if (numlock_mask != 0)
  {
    XTestFakeKeyEvent(display, XKeysymToKeycode(display, XK_Num_Lock), true, CurrentTime);
    XTestFakeKeyEvent(display, XKeysymToKeycode(display, XK_Num_Lock), false, CurrentTime);
  }
}

void KeybLED::toggleCapsLED()
{
  if (capslock_mask != 0)
  {
    XTestFakeKeyEvent(display, XKeysymToKeycode(display, XK_Caps_Lock), true, CurrentTime);
    XTestFakeKeyEvent(display, XKeysymToKeycode(display, XK_Caps_Lock), false, CurrentTime);
  }
}

void KeybLED::toggleScrollLED()
{
  if (scrolllock_mask != 0)
  {
    XTestFakeKeyEvent(display, XKeysymToKeycode(display, XK_Scroll_Lock), true, CurrentTime);
    XTestFakeKeyEvent(display, XKeysymToKeycode(display, XK_Scroll_Lock), false, CurrentTime);
  }
}

// Read the current configuration entries from the data base.
void KeybLED::readSettings()
{
  ksConfig->setGroup("LEDs");
  ledColor = ksConfig->readColorEntry("ledColor", &ledColor);
  ledLook = (KLed::Look) ksConfig->readNumEntry("ledLook", KLed::Raised);
  ledShape = (KLed::Shape) ksConfig->readNumEntry("ledShape", KLed::Circular);
  ledSize = ksConfig->readNumEntry("ledSize", 5);
  ledOrientation = ksConfig->readNumEntry("ledOrientation", Horizontal);

  ksConfig->setGroup("Label");
  labelEnabled = ksConfig->readBoolEntry("labelEnabled", true);
  labelSize = ksConfig->readNumEntry("labelSize", 4);
  labelPositionVertical = ksConfig->readNumEntry("labelPositionVertical", Left);
  labelPositionHorizontal = ksConfig->readNumEntry("labelPositionHorizontal", Bottom);

  ksConfig->setGroup("Display");
  numLEDEnabled = ksConfig->readBoolEntry("numLEDEnabled", true);
  capsLEDEnabled = ksConfig->readBoolEntry("capsLEDEnabled", true);
  scrollLEDEnabled = ksConfig->readBoolEntry("scrollLEDEnabled", true);
  popupEnabled = ksConfig->readBoolEntry("popupEnabled", true);
  refreshTime = ksConfig->readNumEntry("refreshTime", 100);
  mouseToggle = ksConfig->readBoolEntry("mouseToggle", true);
  tooltipsEnabled = ksConfig->readBoolEntry("tooltipsEnabled", true);
  QToolTip::setEnabled(tooltipsEnabled);

  ksConfig->setGroup("Popup");
  popupX = ksConfig->readNumEntry("PopupX", 0);
  popupY = ksConfig->readNumEntry("PopupY", 0);

  ksConfig->setGroup("Startup");
  numLEDStartup = ksConfig->readNumEntry("numLEDStartup", 0);
  capsLEDStartup = ksConfig->readNumEntry("capsLEDStartup", 0);
  scrollLEDStartup = ksConfig->readNumEntry("scrollLEDStartup", 0);

}

// Dump the current configuration entries to the data base.
void KeybLED::writeSettings()
{
  ksConfig->setGroup("LEDs");
  ksConfig->writeEntry("ledColor", ledColor);
  ksConfig->writeEntry("ledLook", ledLook);
  ksConfig->writeEntry("ledShape", ledShape);
  ksConfig->writeEntry("ledSize", ledSize);
  ksConfig->writeEntry("ledOrientation", ledOrientation);

  ksConfig->setGroup("Label");
  ksConfig->writeEntry("labelEnabled", labelEnabled);
  ksConfig->writeEntry("labelSize", labelSize);
  ksConfig->writeEntry("labelPositionVertical", labelPositionVertical);
  ksConfig->writeEntry("labelPositionHorizontal", labelPositionHorizontal);

  ksConfig->setGroup("Display");
  ksConfig->writeEntry("tooltipsEnabled", tooltipsEnabled);
  ksConfig->writeEntry("popupEnabled", popupEnabled);
  ksConfig->writeEntry("refreshTime", refreshTime);
  ksConfig->writeEntry("mouseToggle", mouseToggle);
  ksConfig->writeEntry("numLEDEnabled", numLEDEnabled);
  ksConfig->writeEntry("capsLEDEnabled", capsLEDEnabled);
  ksConfig->writeEntry("scrollLEDEnabled", scrollLEDEnabled);

  ksConfig->setGroup("Startup");
  ksConfig->writeEntry("numLEDStartup", numLEDStartup);
  ksConfig->writeEntry("capsLEDStartup", capsLEDStartup);
  ksConfig->writeEntry("scrollLEDStartup", scrollLEDStartup);

  ksConfig->setGroup("Popup");
  ksConfig->writeEntry("PopupX", popupX);
  ksConfig->writeEntry("PopupY", popupY);

  ksConfig->sync();
}

void KeybLED::SetLEDStartup()
{
  Window dummy1, dummy2;
  int dummy3, dummy4, dummy5, dummy6;
  unsigned int indicatorStates;

  XQueryPointer(display, DefaultRootWindow(display), &dummy1, &dummy2,&dummy3, &dummy4, &dummy5, &dummy6, &indicatorStates);

  if ((numlock_mask != 0 && numLEDStartup != unchanged)
     && (((indicatorStates & numlock_mask) == numlock_mask) != (numLEDStartup == on)))
     toggleNumLED();

  if ((capslock_mask != 0 && capsLEDStartup != unchanged)
     && (((indicatorStates & capslock_mask) == capslock_mask) != (capsLEDStartup == on)))
     toggleCapsLED();

  if ((scrolllock_mask != 0 && scrollLEDStartup != unchanged)
     && (((indicatorStates & scrolllock_mask) == scrolllock_mask) != (scrollLEDStartup == on)))
     toggleScrollLED();
}

void KeybLED::preferences()
{
  if (confDialog)
  {
    KWin::setActiveWindow(confDialog->winId());
    return;
  }

  confDialog = new ConfDialog(this, 0, FALSE, WDestructiveClose);

  connect(confDialog->buttonOk, SIGNAL(clicked()), this, SLOT(dlgOkClicked()));
  connect(confDialog->buttonApply, SIGNAL(clicked()), this, SLOT(dlgApplyClicked()));
  connect(confDialog->buttonCancel, SIGNAL(clicked()), this, SLOT(dlgCancelClicked()));
  connect(confDialog->buttonHelp, SIGNAL(clicked()), this, SLOT(dlgHelpClicked()));
  connect(confDialog, SIGNAL(destroyed()), SLOT(dlgDeleted()));

  // Display tab
  confDialog->numLEDEnabled->setChecked(numLEDEnabled);
  confDialog->capsLEDEnabled->setChecked(capsLEDEnabled);
  confDialog->scrollLEDEnabled->setChecked(scrollLEDEnabled);
  confDialog->tooltipsEnabled->setChecked(tooltipsEnabled);
  confDialog->popupEnabled->setChecked(popupEnabled);
  confDialog->refreshTimeSlider->setValue(refreshTime);
  confDialog->mouseToggle->setChecked(mouseToggle);
  confDialog->mouseToggle->setEnabled(numlock_mask != 0 || capslock_mask != 0 || scrolllock_mask != 0);

  // LEDs tab
  confDialog->ledSize->setValue(ledSize);

  switch(ledOrientation)
  {
    case Vertical : confDialog->ledVertical->setChecked(true);
      break;
    case Horizontal : confDialog->ledHorizontal->setChecked(true);
      break;
  }

  switch(ledLook)
  {
    case KLed::Flat : confDialog->ledFlat->setChecked(true);
      break;
    case KLed::Raised : confDialog->ledRaised->setChecked(true);
      break;
    case KLed::Sunken : confDialog->ledSunken->setChecked(true);
      break;
  }

  switch(ledShape)
  {
    case KLed::Rectangular : confDialog->ledRectangular->setChecked(true);
      break;
    case KLed::Circular : confDialog->ledCircular->setChecked(true);
      break;
  }

  confDialog->ledColor->setColor(ledColor);

  // Label tab
  confDialog->labelEnabled->setChecked(labelEnabled);
  confDialog->labelSize->setValue(labelSize);
  confDialog->labelSizeBox->setEnabled(labelEnabled);

  confDialog->labelPositionBox->setEnabled(labelEnabled);

  switch(labelPositionVertical)
  {
    case Left : confDialog->labelLeft->setChecked(true);
      break;
    case Right : confDialog->labelRight->setChecked(true);
      break;
  }

  switch(labelPositionHorizontal)
  {
    case Top : confDialog->labelTop->setChecked(true);
      break;
    case Bottom : confDialog->labelBottom->setChecked(true);
      break;
  }

  // Startup tab
  confDialog->numLEDStartup->setCurrentItem(numLEDStartup);
  confDialog->numLEDStartup->setEnabled(numlock_mask != 0);
  confDialog->numLEDStartupLabel->setEnabled(numlock_mask != 0);

  confDialog->capsLEDStartup->setCurrentItem(capsLEDStartup);
  confDialog->capsLEDStartup->setEnabled(capslock_mask != 0);
  confDialog->capsLEDStartupLabel->setEnabled(capslock_mask != 0);

  confDialog->scrollLEDStartup->setCurrentItem(scrollLEDStartup);
  confDialog->scrollLEDStartup->setEnabled(scrolllock_mask != 0);
  confDialog->scrollLEDStartupLabel->setEnabled(scrolllock_mask != 0);

  confDialog->show();
}

void KeybLED::dlgOkClicked()
{
    dlgApplyClicked();
    delete confDialog;
}

void KeybLED::dlgApplyClicked()
{
  // Display tab
  refreshTime = confDialog->refreshTimeSlider->value();
  tooltipsEnabled = confDialog->tooltipsEnabled->isChecked();
  popupEnabled = confDialog->popupEnabled->isChecked();
  mouseToggle = confDialog->mouseToggle->isChecked();
  numLEDEnabled = confDialog->numLEDEnabled->isChecked();
  capsLEDEnabled = confDialog->capsLEDEnabled->isChecked();
  scrollLEDEnabled = confDialog->scrollLEDEnabled->isChecked();

  // LEDs tab
  if (confDialog->ledRectangular->isChecked()) ledShape = KLed::Rectangular;
  if (confDialog->ledCircular->isChecked()) ledShape = KLed::Circular;
  if (confDialog->ledFlat->isChecked()) ledLook = KLed::Flat;
  if (confDialog->ledRaised->isChecked()) ledLook = KLed::Raised;
  if (confDialog->ledSunken->isChecked()) ledLook = KLed::Sunken;
  ledColor = confDialog->ledColor->color();
  ledSize = confDialog->ledSize->value();
  if (confDialog->ledVertical->isChecked()) ledOrientation = Vertical;
  if (confDialog->ledHorizontal->isChecked()) ledOrientation = Horizontal;

  // Labels tab
  labelEnabled = confDialog->labelEnabled->isChecked();
  labelSize = confDialog->labelSize->value();

  if (confDialog->labelLeft->isChecked()) labelPositionVertical = Left;
  if (confDialog->labelRight->isChecked()) labelPositionVertical = Right;

  if (confDialog->labelTop->isChecked()) labelPositionHorizontal = Top;
  if (confDialog->labelBottom->isChecked()) labelPositionHorizontal = Bottom;

  // Startup tab
  numLEDStartup = confDialog->numLEDStartup->currentItem();
  capsLEDStartup = confDialog->capsLEDStartup->currentItem();
  scrollLEDStartup = confDialog->scrollLEDStartup->currentItem();

  writeSettings();
  applySettings();
  emit updateLayout();
}

void KeybLED::dlgHelpClicked()
{
  KApplication *app = (KApplication*) KGlobal::instance();
  switch (confDialog->tabWidget->currentPageIndex())
  {
    case 0: app->invokeHelp("preferences-display", "keybled"); break;
    case 1: app->invokeHelp("preferences-leds", "keybled"); break;
    case 2: app->invokeHelp("preferences-labels", "keybled"); break;
    case 3: app->invokeHelp("preferences-startup", "keybled"); break;
  }
}

void KeybLED::dlgCancelClicked()
{
  delete confDialog;
}

void KeybLED::dlgDeleted()
{
  confDialog = 0;
}

void KeybLED::applySettings()
{
  numLed->setShape(ledShape);
  numLed->setLook(ledLook);
  numLed->setColor(ledColor);

  capsLed->setShape(ledShape);
  capsLed->setLook(ledLook);
  capsLed->setColor(ledColor);

  scrollLed->setShape(ledShape);
  scrollLed->setLook(ledLook);
  scrollLed->setColor(ledColor);

  QToolTip::setEnabled(tooltipsEnabled);
  timer->changeInterval(refreshTime);

  updateGeometry();
}

void KeybLED::about()
{
  KAboutData aboutData ("keybled", I18N_NOOP("KeybLED"), VERSION,
                        I18N_NOOP("This application displays a keyboard's LED status. It is particulary\n"
                                  "useful for keyboards which lack their own LED status indicators (e.g.\n" 
                                  "cordless keyboards), or just for the fun!"),
                        KAboutData::License_GPL, "(c) 2001, Aurelien Jarno", 0, 0, "aurelien@aurel32.net");
  aboutData.addAuthor("Aurelien Jarno", 0, "aurelien@aurel32.net", "http://www.aurel32.net");
  KAboutApplication dialog(&aboutData);
  dialog.show();
}

void KeybLED::help()
{
  KApplication *app = (KApplication*) KGlobal::instance();
  app->invokeHelp("", "keybled");
}

void KeybLED::openContextMenu()
{
  KPopupMenu *menu = new KPopupMenu();
  menu->insertTitle(SmallIcon("keybled"), i18n("KeybLED"));

  KPopupMenu *ledEnableMenu = new KPopupMenu(menu);
  ledEnableMenu->insertItem(i18n("&NUM.LOCK"), 11);
  ledEnableMenu->insertItem(i18n("&CAPS.LOCK"), 12);
  ledEnableMenu->insertItem(i18n("&SCROLL.LOCK"), 13);
  if (numLEDEnabled) ledEnableMenu->setItemChecked(11, true);
  if (capsLEDEnabled) ledEnableMenu->setItemChecked(12, true);
  if (scrollLEDEnabled) ledEnableMenu->setItemChecked(13, true);
  menu->insertItem(i18n("LE&Ds"), ledEnableMenu, 10);

  KPopupMenu *ledShapeMenu = new KPopupMenu(menu);
  ledShapeMenu->insertItem(i18n("&Rectangular"), 21);
  ledShapeMenu->insertItem(i18n("&Circular"), 22);
  if (ledShape == KLed::Rectangular) ledShapeMenu->setItemChecked(21, true);
  if (ledShape == KLed::Circular) ledShapeMenu->setItemChecked(22, true);
  menu->insertItem(i18n("LEDs &shape"), ledShapeMenu, 20);

  KPopupMenu *ledLookMenu = new KPopupMenu(menu);
  ledLookMenu->insertItem(i18n("&Flat"), 31);
  ledLookMenu->insertItem(i18n("&Raised"), 32);
  ledLookMenu->insertItem(i18n("&Sunken"), 33);
  if (ledLook == KLed::Flat) ledLookMenu->setItemChecked(31, true);
  if (ledLook == KLed::Raised) ledLookMenu->setItemChecked(32, true);
  if (ledLook == KLed::Sunken) ledLookMenu->setItemChecked(33, true);
  menu->insertItem(i18n("LEDs loo&k"), ledLookMenu, 30);

  KPopupMenu *ledSizeMenu = new KPopupMenu(menu);
  ledSizeMenu->insertItem(i18n("&Tiny"), 41);
  ledSizeMenu->insertItem(i18n("&Small"), 42);
  ledSizeMenu->insertItem(i18n("&Medium"), 43);
  ledSizeMenu->insertItem(i18n("&Big"), 44);
  ledSizeMenu->insertItem(i18n("&Very big"), 45);
  if (ledSize == 2) ledSizeMenu->setItemChecked(41, true);
  if (ledSize == 3) ledSizeMenu->setItemChecked(42, true);
  if (ledSize == 4) ledSizeMenu->setItemChecked(43, true);
  if (ledSize == 5) ledSizeMenu->setItemChecked(44, true);
  if (ledSize == 6) ledSizeMenu->setItemChecked(45, true);
  menu->insertItem(i18n("LEDs si&ze"), ledSizeMenu, 40);

  KPopupMenu *ledOrientationMenu = new KPopupMenu(menu);
  ledOrientationMenu->insertItem(i18n("&Vertical"), 51);
  ledOrientationMenu->insertItem(i18n("&Horizontal"), 52);
  if (ledOrientation == Vertical) ledOrientationMenu->setItemChecked(51, true);
  if (ledOrientation == Horizontal) ledOrientationMenu->setItemChecked(52, true);
  menu->insertItem(i18n("LEDs o&rientation"), ledOrientationMenu, 50);

  menu->insertItem(i18n("LEDs &color"), 60);

  menu->insertSeparator();

  menu->insertItem(i18n("Show &labels"), 70);
  if (labelEnabled) menu->setItemChecked(70, true);

  KPopupMenu *labelSizeMenu = new KPopupMenu(menu);
  labelSizeMenu->insertItem(i18n("&Tiny"), 81);
  labelSizeMenu->insertItem(i18n("&Small"), 82);
  labelSizeMenu->insertItem(i18n("&Medium"), 83);
  labelSizeMenu->insertItem(i18n("&Big"), 84);
  labelSizeMenu->insertItem(i18n("&Very big"), 85);
  if (labelSize == 2) labelSizeMenu->setItemChecked(81, true);
  if (labelSize == 3) labelSizeMenu->setItemChecked(82, true);
  if (labelSize == 4) labelSizeMenu->setItemChecked(83, true);
  if (labelSize == 5) labelSizeMenu->setItemChecked(84, true);
  if (labelSize == 6) labelSizeMenu->setItemChecked(85, true);
  menu->insertItem(i18n("Labels s&ize"), labelSizeMenu, 80);

  KPopupMenu *labelPositionHorizontalMenu = new KPopupMenu(menu);
  labelPositionHorizontalMenu->insertItem(i18n("&Top"), 93);
  labelPositionHorizontalMenu->insertItem(i18n("&Bottom"), 94);
  if (labelPositionHorizontal == Top) labelPositionHorizontalMenu->setItemChecked(93, true);
  if (labelPositionHorizontal == Bottom) labelPositionHorizontalMenu->setItemChecked(94, true);
  KPopupMenu *labelPositionVerticalMenu = new KPopupMenu(menu);
  labelPositionVerticalMenu->insertItem(i18n("&Left"), 95);
  labelPositionVerticalMenu->insertItem(i18n("&Right"), 96);
  if (labelPositionVertical == Left) labelPositionVerticalMenu->setItemChecked(95, true);
  if (labelPositionVertical == Right) labelPositionVerticalMenu->setItemChecked(96, true);
  KPopupMenu *labelPositionMenu = new KPopupMenu(menu);
  labelPositionMenu->insertItem(i18n("When LEDs are &horizontal"), labelPositionHorizontalMenu, 91);
  labelPositionMenu->insertItem(i18n("When LEDs are &vertical"), labelPositionVerticalMenu, 92);
  menu->insertItem(i18n("Labels p&osition"), labelPositionMenu, 90);

  menu->insertSeparator();

  KPopupMenu *refreshMenu = new KPopupMenu(menu);
  refreshMenu->insertItem(i18n("50 ms"), 101);
  refreshMenu->insertItem(i18n("100 ms"), 102);
  refreshMenu->insertItem(i18n("150 ms"), 103);
  refreshMenu->insertItem(i18n("200 ms"), 104);
  refreshMenu->insertItem(i18n("250 ms"), 105);
  refreshMenu->insertItem(i18n("300 ms"), 106);
  refreshMenu->insertItem(i18n("350 ms"), 107);
  refreshMenu->insertItem(i18n("400 ms"), 108);
  refreshMenu->insertItem(i18n("450 ms"), 109);
  refreshMenu->insertItem(i18n("500 ms"), 110);
  menu->insertItem(i18n("Re&fresh time"), refreshMenu, 100);
  if (refreshTime < 75) refreshMenu->setItemChecked(101, true);
  if (refreshTime >= 75 && refreshTime < 125) refreshMenu->setItemChecked(102, true);
  if (refreshTime >= 125 && refreshTime < 175) refreshMenu->setItemChecked(103, true);
  if (refreshTime >= 175 && refreshTime < 225) refreshMenu->setItemChecked(104, true);
  if (refreshTime >= 225 && refreshTime < 275) refreshMenu->setItemChecked(105, true);
  if (refreshTime >= 275 && refreshTime < 325) refreshMenu->setItemChecked(106, true);
  if (refreshTime >= 325 && refreshTime < 375) refreshMenu->setItemChecked(107, true);
  if (refreshTime >= 375 && refreshTime < 425) refreshMenu->setItemChecked(108, true);
  if (refreshTime >= 425 && refreshTime < 475) refreshMenu->setItemChecked(109, true);
  if (refreshTime >= 475) refreshMenu->setItemChecked(110, true);

  menu->insertItem(i18n("E&nable tool tips"), 120);
  if (tooltipsEnabled) menu->setItemChecked(120, true);

  menu->insertItem(i18n("Enable pop&up"), 130);
  if (popupEnabled) menu->setItemChecked(130, true);

  menu->insertItem(i18n("LEDs can be toggled with &mouse"), 140);
  if (mouseToggle) menu->setItemChecked(140, true);

  menu->insertSeparator();

  menu->insertItem(i18n("&About..."), 150);
  menu->insertItem(SmallIcon("help"), i18n("&Help..."), 160);
  menu->insertSeparator();
  menu->insertItem(SmallIcon("configure"), i18n("&Preferences..."), 170);

  int result = menu->exec(QCursor::pos());

  if ((result >= 0) && (result <= 149))
  {
    if (result == 11)
    {
      numLEDEnabled = !numLEDEnabled;
      if (confDialog) confDialog->numLEDEnabled->setChecked(numLEDEnabled);
    }
    else if (result == 12)
    {
      capsLEDEnabled = !capsLEDEnabled;
      if (confDialog) confDialog->capsLEDEnabled->setChecked(capsLEDEnabled);
    }
    else if (result == 13)
    {
      scrollLEDEnabled = !scrollLEDEnabled;
      if (confDialog) confDialog->scrollLEDEnabled->setChecked(scrollLEDEnabled);
    }
    else if (result == 21)
    {
      ledShape = KLed::Rectangular;
      if (confDialog) confDialog->ledRectangular->setChecked(true);
    }
    else if (result == 22)
    {
      ledShape = KLed::Circular;
      if (confDialog) confDialog->ledCircular->setChecked(true);
    }
    else if (result == 31)
    {
      ledLook = KLed::Flat;
      if (confDialog) confDialog->ledFlat->setChecked(true);
    }
    else if (result == 32)
    {
      ledLook = KLed::Raised;
      if (confDialog) confDialog->ledRaised->setChecked(true);
    }
    else if (result == 33)
    {
      ledLook = KLed::Sunken;
      if (confDialog) confDialog->ledSunken->setChecked(true);
    }
    else if (result >= 41 && result <= 45)
    {
      switch (result)
      {
        case 41 : ledSize = 2; break;
        case 42 : ledSize = 3; break;
        case 43 : ledSize = 4; break;
        case 44 : ledSize = 5; break;
        case 45 : ledSize = 6; break;
      }
      if (confDialog) confDialog->ledSize->setValue(ledSize);
    }
    else if (result == 51)
    {
      ledOrientation = Vertical;
      if (confDialog) confDialog->ledVertical->setChecked(true);
    }
    else if (result == 52)
    {
      ledOrientation = Horizontal;
      if (confDialog) confDialog->ledHorizontal->setChecked(true);
    }
    else if (result == 60)
    {
       QColor color = ledColor;
       if (KColorDialog::getColor(color) == KColorDialog::Accepted)
       {
         ledColor = color;
         if (confDialog) confDialog->ledColor->setColor(ledColor);
       }
    }
    else if (result == 70)
    {
      labelEnabled = !labelEnabled;
      if (confDialog) confDialog->labelEnabled->setChecked(labelEnabled);
    }
    else if (result >= 81 && result <= 85)
    {
      switch (result)
      {
        case 81 : labelSize = 2; break;
        case 82 : labelSize = 3; break;
        case 83 : labelSize = 4; break;
        case 84 : labelSize = 5; break;
        case 85 : labelSize = 6; break;
      }
      if (confDialog) confDialog->labelSize->setValue(labelSize);
    }
    else if (result == 93)
    {
      labelPositionHorizontal = Top;
      if (confDialog) confDialog->labelTop->setChecked(true);
    }
    else if (result == 94)
    {
      labelPositionHorizontal = Bottom;
      if (confDialog) confDialog->labelBottom->setChecked(true);
    }
    else if (result == 95)
    {
      labelPositionVertical = Left;
      if (confDialog) confDialog->labelLeft->setChecked(true);
    }
    else if (result == 96)
    {
      labelPositionVertical = Right;
      if (confDialog) confDialog->labelRight->setChecked(true);
    }
    else if (result >= 101 && result <= 110)
    {
      switch (result)
      {
        case 101 : refreshTime = 50; break;
        case 102 : refreshTime = 100; break;
        case 103 : refreshTime = 150; break;
        case 104 : refreshTime = 200; break;
        case 105 : refreshTime = 250; break;
        case 106 : refreshTime = 300; break;
        case 107 : refreshTime = 350; break;
        case 108 : refreshTime = 400; break;
        case 109 : refreshTime = 450; break;
        case 110 : refreshTime = 500; break;
      }
      if (confDialog) confDialog->refreshTimeSlider->setValue(refreshTime);
    }
    else if (result == 120)
    {
      tooltipsEnabled = !tooltipsEnabled;
      if (confDialog) confDialog->tooltipsEnabled->setChecked(tooltipsEnabled);
    }
    else if (result == 130)
    {
      popupEnabled = !popupEnabled;
      if (confDialog) confDialog->popupEnabled->setChecked(popupEnabled);
    }
    else if (result == 140)
    {
      mouseToggle = !mouseToggle;
      if (confDialog) confDialog->mouseToggle->setChecked(mouseToggle);
    }

    writeSettings();
    applySettings();
    emit updateLayout();
  }
  else if (result == 150) about();
  else if (result == 160) help();
  else if (result == 170) preferences();
  delete menu;
}

void KeybLED::updateGeometry()
{
  int led = 0; int label = 0; int intled = 0; int xled = 0; int xlabel = 0; int y = 0;
  int nbLED = numLEDEnabled + capsLEDEnabled + scrollLEDEnabled;

  QFont f(KGlobalSettings::generalFont());

  if (nbLED > 0)
  {
    if (orientation() == KPanelApplet::Horizontal)
    {
       if (ledOrientation == Vertical)
       {
        led = ((height() / nbLED) * ledSize) / 6;
        label = ((height() / nbLED) * labelSize) / 6;
        intled = (height() - (nbLED * led)) / (nbLED + 1);

        if (labelPositionVertical == Right || !labelEnabled)
        {
          xled = 1;
          xlabel = 2 + led;
        }
        else
        {
          xled = 2 + label;
          xlabel = 1;
        }
      }
      else
      {
        led = (height() * ledSize) / 12;
        if (labelEnabled)
        {
          label = (height() * labelSize) / 12;
          if (label > led) intled = 1 + (label - led); else intled = 1;
          if (labelPositionHorizontal == Top)
          {
            xlabel = ((height() - (led + label)) / 4);
            xled = ((height() - (led + label)) / 2) + label;
          }
          else
          {
            xled = ((height() - (led + label)) / 4);
            xlabel = ((height() - (led + label)) / 2) + led;
          }
        }
        else
        {
          label = 0;
          xled = (height() - led) / 2;
          xlabel = 0;
          intled = 1;
        }
      }
    }
    else
    {
      if (ledOrientation == Vertical)
      {
        led = (width() * ledSize) / 12;
        if (labelEnabled)
        {
          label = (width() * labelSize) / 12;
          if (label > led) intled = 1 + (label - led); else intled = 1;
          if (labelPositionVertical == Left)
          {
            xlabel = ((width() - (led + label)) / 4);
            xled = ((width() - (led + label)) / 2) + label;
          }
          else
          {
            xled = ((width() - (led + label)) / 4);
            xlabel = ((width() - (led + label)) / 2) + led;
          }
        }
        else
        {
          label = 0;
          xled = (width() - led) / 2;
          xlabel = 0;
          intled = 1;
        }
      }
      else
      {
        led = ((width() / nbLED) * ledSize) / 6;
        label = ((width() / nbLED) * labelSize) / 6;
        intled = (width() - (nbLED * led)) / (nbLED + 1);

        if (labelPositionHorizontal == Bottom || !labelEnabled)
        {
          xled = 1;
          xlabel = 2 + led;
        }
        else
        {
          xled = 2 + label;
          xlabel = 1;
        }
      }
    }

    f.setPointSize(label);
    y = intled;
  }

  if (numLEDEnabled)
  {
    numLed->resize(led, led);
    if (ledOrientation == Vertical) numLed->move(xled, y);
    else numLed->move(y, xled);
    numLed->show();

    if (labelEnabled)
    {
      numLabel->resize(label, label);
      if (ledOrientation == Vertical) numLabel->move(xlabel, y + ((led - label) / 2));
      else numLabel->move(y + ((led - label) / 2), xlabel);
      numLabel->show();
      numLabel->setFont(f);
    }
    else numLabel->hide();

    y += intled + led;
  }
  else
  {
    numLed->hide();
    numLabel->hide();
  }

  if (capsLEDEnabled)
  {
    capsLed->resize(led, led);
    if (ledOrientation == Vertical) capsLed->move(xled, y);
    else capsLed->move(y, xled);
    capsLed->show();

    if (labelEnabled)
    {
      capsLabel->resize(label, label);
      if (ledOrientation == Vertical) capsLabel->move(xlabel, y + ((led - label) / 2));
      else capsLabel->move(y + ((led - label) / 2), xlabel);
      capsLabel->show();
      capsLabel->setFont(f);
    }
    else capsLabel->hide();

    y += intled + led;
  }
  else
  {
    capsLed->hide();
    capsLabel->hide();
  }

  if (scrollLEDEnabled)
  {
    scrollLed->resize(led, led);
    if (ledOrientation == Vertical) scrollLed->move(xled, y);
    else scrollLed->move(y, xled);
    scrollLed->show();

    if (labelEnabled)
    {
      scrollLabel->resize(label, label);
      if (ledOrientation == Vertical) scrollLabel->move(xlabel, y + ((led - label) / 2));
      else scrollLabel->move(y + ((led - label) / 2), xlabel);
      scrollLabel->show();
      scrollLabel->setFont(f);
    }
    else scrollLabel->hide();
  }
  else
  {
    scrollLed->hide();
    scrollLabel->hide();
  }
}

int KeybLED::widthForHeight(int height) const
{
  int nbLED = numLEDEnabled + capsLEDEnabled + scrollLEDEnabled;

  if (nbLED > 0)
  {
    if (ledOrientation == Vertical)
    {
      if (labelEnabled) return (3 + (height * (ledSize + labelSize)) / (6 * nbLED));
      else return (2 + (height * ledSize) / (6 * nbLED));
    }
    else
    {
      if ((labelSize > ledSize) && labelEnabled) return 1 + ((height * labelSize) / 12 + 1) * nbLED;
      else return (1 + ((height * ledSize) / 12 + 1) * nbLED);
    }
  }
  else return 0;
}

int KeybLED::heightForWidth(int width) const
{
  int nbLED = numLEDEnabled + capsLEDEnabled + scrollLEDEnabled;

  if (nbLED > 0)
  {
    if (ledOrientation == Horizontal)
    {
      if (labelEnabled) return (3 + (width * (ledSize + labelSize)) / (6 * nbLED));
      else return (2 + (width * ledSize) / (6 * nbLED));
    }
    else
    {
      if ((labelSize > ledSize) && labelEnabled) return 1 + ((width * labelSize) / 12 + 1) * nbLED;
      else return (1 + ((width * ledSize) / 12 + 1) * nbLED);
    }
  }
  else return 0;
}

void KeybLED::resizeEvent(QResizeEvent *event)
{
  updateGeometry();
}

void KeybLED::timerEvent()
{
  Window dummy1, dummy2;
  int dummy3, dummy4, dummy5, dummy6;
  unsigned int indicatorStates;

  XQueryPointer(display, DefaultRootWindow(display), &dummy1, &dummy2,&dummy3, &dummy4, &dummy5, &dummy6, &indicatorStates);

  if (indicatorStates & numlock_mask) numLed->setState(KLed::On); else numLed->setState(KLed::Off);
  if (indicatorStates & capslock_mask) capsLed->setState(KLed::On); else capsLed->setState(KLed::Off);
  if (indicatorStates & scrolllock_mask) scrollLed->setState(KLed::On); else scrollLed->setState(KLed::Off);

  if (popupOn)
  {
    QString text;

    text = QString(i18n("NUM. LOCK      "));
    if (indicatorStates & numlock_mask) text.append(i18n("Enabled")); else text.append(i18n("Disabled"));

    text.append("\n");
    text.append(i18n("CAPS. LOCK     "));
    if (indicatorStates & capslock_mask) text.append(i18n("Enabled")); else text.append(i18n("Disabled"));

    text.append("\n");
    text.append(i18n("SCROLL. LOCK   "));
    if (indicatorStates & scrolllock_mask) text.append(i18n("Enabled")); else text.append(i18n("Disabled"));

    popup->setText(text);
  }
}

// catch the mouse clicks of our child widgets
bool KeybLED::eventFilter(QObject *object, QEvent *event)
{
  if ((object == numLabel || object == capsLabel || object == scrollLabel)
     && (event->type() == QEvent::MouseButtonPress))
  {
    mousePressEvent(static_cast<QMouseEvent*>(event) );
    return true;
  }

  if ((event->type() == QEvent::MouseButtonPress)
     && (object == numLed || object == capsLed || object == scrollLed)
     && (!mouseToggle
          || (static_cast<QMouseEvent*>(event))->button() != QMouseEvent::LeftButton
          || (object == numLed && numlock_mask == 0)
          || (object == capsLed && capslock_mask == 0)
          || (object == scrollLed && scrolllock_mask == 0)))
  {
    mousePressEvent(static_cast<QMouseEvent*>(event) );
    return true;
  }

  return KPanelApplet::eventFilter(object, event);
}

void KeybLED::mousePressEvent(QMouseEvent *event)
{
 if (event->button() == QMouseEvent::LeftButton)
 {
   if (!popupOn) showPopUp(event->globalPos());
   else hidePopUp();
 }
 if (event->button() == QMouseEvent::RightButton) openContextMenu();
}


void KeybLED::slotToggleNumLED()
{
  if (mouseToggle)
  {
    toggleNumLED();
    timerEvent();
  }
}

void KeybLED::slotToggleCapsLED()
{
  if (mouseToggle)
  {
    toggleCapsLED();
    timerEvent();
  }
}

void KeybLED::slotToggleScrollLED()
{
  if (mouseToggle)
  {
    toggleScrollLED();
    timerEvent();
  }
}

void KeybLED::showPopUp(QPoint Point)
{
  popupOn = TRUE;

  if(popupX == 0 && popupY == 0)
  {
    popupX = Point.x() - 220;
    popupY = Point.y() - 100;
  }

  popup->move(popupX, popupY);
  popup->show();
}

void KeybLED::hidePopUp()
{
  popupOn = FALSE;
  savePopUpPos();
  popup->hide();
}

void KeybLED::savePopUpPos()
{
  popupX = popup->x();
  popupY = popup->y();
  writeSettings();
}
