// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Anders Dahnielson (anders@dahnielson.com)
*/

#include "simple_bitmap_modifier.h"

#include <k3dsdk/i18n.h>
#include <k3dsdk/module.h>

namespace libk3dbitmap
{

/////////////////////////////////////////////////////////////////////////////
// bitmap_color_monochrome

class bitmap_color_monochrome :
	public simple_bitmap_modifier
{
	typedef simple_bitmap_modifier base;

public:
	bitmap_color_monochrome(k3d::idocument& Document) :
		base(Document),
		m_red_weight(init_owner(*this) + init_name("red_weight") + init_label(_("Red weight")) + init_description(_("Scale Red componnt value")) + init_value(0.3)),
		m_green_weight(init_owner(*this) + init_name("green_weight") + init_label(_("Green weight")) + init_description(_("Scale Green component value")) + init_value(0.59)),
		m_blue_weight(init_owner(*this) + init_name("blue_weight") + init_label(_("Blue weight")) + init_description(_("Scale Blue component value")) + init_value(0.11))
	{
		m_red_weight.changed_signal().connect(make_update_bitmap_slot());
		m_green_weight.changed_signal().connect(make_update_bitmap_slot());
		m_blue_weight.changed_signal().connect(make_update_bitmap_slot());
	}

	void on_update_bitmap(const k3d::bitmap& Input, k3d::bitmap& Output)
	{
		const double red_weight = m_red_weight.value();
		const double green_weight = m_green_weight.value();
		const double blue_weight = m_blue_weight.value();

		k3d::bitmap::const_iterator input = Input.begin();
		const k3d::bitmap::const_iterator end = Input.end();

		k3d::bitmap::iterator output = Output.begin();

		for(; input != end; ++input, ++output)
		{
			output->red = output->green = output->blue = static_cast<double>(input->red) * red_weight + static_cast<double>(input->green) * green_weight +  static_cast<double>(input->blue) * blue_weight;
			output->alpha = input->alpha;
		}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<bitmap_color_monochrome>,
			k3d::interface_list<k3d::ibitmap_source,
			k3d::interface_list<k3d::ibitmap_sink> > > factory(
				k3d::uuid(0x73cd5356, 0x4d3d4eaf, 0xacda0d76, 0x8070a016),
				"BitmapColorMonochrome",
				_("Make any image black and white"),
				"Bitmap",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_red_weight;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_green_weight;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_blue_weight;
};

/////////////////////////////////////////////////////////////////////////////
// bitmap_color_monochrome_factory

k3d::iplugin_factory& bitmap_color_monochrome_factory()
{
	return bitmap_color_monochrome::get_factory();
}

} // namespace libk3dbitmap


