// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/classes.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/mesh.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// select_face_by_number_implementation

class select_face_by_number_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	select_face_by_number_implementation(k3d::idocument& Document) :
		base(Document),
		m_index(k3d::init_name("index") + k3d::init_description("Face index [number]") + k3d::init_value(0UL) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_document(Document))
	{
		register_property(m_index);

		m_input_mesh.changed_signal().connect(SigC::slot(*this, &select_face_by_number_implementation::on_reset_geometry));
		m_index.changed_signal().connect(SigC::slot(*this, &select_face_by_number_implementation::on_reset_geometry));
		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &select_face_by_number_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Create output geometry ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);

		const unsigned long index = m_index.property_value();

		for(k3d::mesh::polyhedra_t::const_iterator polyhedron = output->polyhedra.begin(); polyhedron != output->polyhedra.end(); ++polyhedron)
			{
				k3d::deselect(document(), k3d::deep_selection(document().dag(), k3d::make_selection(**polyhedron)));

				if(index < (*polyhedron)->faces.size())
					k3d::select(document(), k3d::deep_selection(document().dag(), k3d::make_selection(*(*polyhedron)->faces[index])));
			}

		return output;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<select_face_by_number_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0xe6925525, 0xe5aa4bb2, 0xb6e0d3d0, 0xb01d7e91),
				"SelectFaceByNumber",
				"Selects a face from the input mesh by its index number",
				"Objects",
				k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}

private:
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_index;
};

/////////////////////////////////////////////////////////////////////////////
// select_face_by_number_factory

k3d::iplugin_factory& select_face_by_number_factory()
{
	return select_face_by_number_implementation::get_factory();
}

} // namespace libk3dmesh


