// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/imaterial.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_collection.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/mouse_event_observer.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/transform.h>
#include <k3dsdk/viewport.h>

#include <sdpgl/sdpgl.h>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// freehand_polygon_implementation

class freehand_polygon_implementation :
	public k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > >,
	public k3d::mouse_event_observer,
	public k3d::viewport::idrawable
{
	typedef k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > > base;

public:
	freehand_polygon_implementation(k3d::idocument& Document) :
		base(Document),
		k3d::mouse_event_observer("LMB to draw polygon"),
		m_size(k3d::init_name("size") + k3d::init_description("Size [number]") + k3d::init_value(5.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_document(Document)),
		m_no_intersection(k3d::init_name("no_intersection") + k3d::init_description("No self-intersection [boolean]") + k3d::init_value(true) + k3d::init_document(Document))
	{
		enable_serialization(k3d::persistence::proxy(m_size));
		enable_serialization(k3d::persistence::proxy(m_no_intersection));

		register_property(m_size);
		register_property(m_no_intersection);

		m_material.changed_signal().connect(SigC::slot(*this, &freehand_polygon_implementation::on_reset_geometry));

		m_size.changed_signal().connect(SigC::slot(*this, &freehand_polygon_implementation::on_reset_geometry));
		m_no_intersection.changed_signal().connect(SigC::slot(*this, &freehand_polygon_implementation::on_reset_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &freehand_polygon_implementation::on_create_geometry));
	}

	void draw_polygon_2d()
	{
		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		glOrtho(0, 1, 0, 1, 0, 1);

		glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		glLoadIdentity();

		glDisable(GL_LIGHTING);
		glDisable(GL_DEPTH_TEST);
		glDisable(GL_BLEND);
		glDisable(GL_TEXTURE_1D);
		glDisable(GL_TEXTURE_2D);

		// Draw points
		glColor3d(1.0, 0.0, 0.0);
		glPointSize(6.0f);

		glBegin(GL_POINTS);
		for(unsigned long j = 0; j < m_points.size(); ++j)
			{
				glVertex3d(m_points[j][0], 1 - m_points[j][1], 0);
			}
		glEnd();

		// Draw line
		glLineWidth(1.0f);
		glBegin(GL_LINES);
		for(unsigned long j = 0; j + 1 < m_points.size(); ++j)
			{
				glVertex3d(m_points[j][0], 1 - m_points[j][1], 0);
				glVertex3d(m_points[j + 1][0], 1 - m_points[j + 1][1], 0);
			}
		glEnd();

		glPopMatrix();
	}

	bool OnLButtonClick(const k3d::imouse_event_observer::event_state& State, const k3d::vector2& Current)
	{
		const double mousex = Current[0];
		const double mousey = Current[1];

		m_points.push_back(k3d::vector3(mousex, mousey, 0));
		on_reset_geometry();

		return true;
	}

	bool OnLButtonDrag(const k3d::imouse_event_observer::event_state& State, const k3d::vector2& Current, const k3d::vector2& Last, const k3d::vector2& Start, const drag_type_t DragType)
	{
		const double mousex = Current[0];
		const double mousey = Current[1];

		m_points.push_back(k3d::vector3(mousex, mousey, 0));
		on_reset_geometry();

		return true;
	}

	void viewport_draw(const k3d::viewport::render_state& State)
	{
		sdpgl::store_attributes attributes();

		on_viewport_draw(State);
	}

	void on_viewport_draw(const k3d::viewport::render_state& State)
	{
		draw_polygon_2d();
	}

	void viewport_select(const k3d::viewport::render_state& State)
	{
		on_viewport_select(State);
	}

	void on_viewport_select(const k3d::viewport::render_state& State)
	{
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		// Check for empty polygon ...
		if(!m_points.size())
			return 0;

		std::auto_ptr<k3d::mesh> mesh(new k3d::mesh());

		k3d::polyhedron* polyhedron = new k3d::polyhedron();
		return_val_if_fail(polyhedron, 0);
		mesh->polyhedra.push_back(polyhedron);
		//polyhedron->material = m_material.interface();

		const double size = m_size.property_value();
		const bool no_intersection = m_no_intersection.property_value();

		// Create polygon
		k3d::polyhedron::edges_t edges;
		for(unsigned long n = 0; n < m_points.size(); ++n)
			{
				k3d::vector3 coords(m_points[n]);

				// Center around origin and scale
				coords -= k3d::vector3(0.5, 0.5, 0);
				coords *= size;

				// Create mesh point
				k3d::point* const point = new k3d::point(coords);
				return_val_if_fail(point, 0);
				mesh->points.push_back(point);

				// Create split edge
				k3d::split_edge* edge = new k3d::split_edge(point);
				return_val_if_fail(edge, 0);
				edges.push_back(edge);
			}

		// Create face
		k3d::loop_edges(edges.begin(), edges.end());
		polyhedron->edges.insert(polyhedron->edges.end(), edges.begin(), edges.end());

		k3d::face* const face = new k3d::face(*edges.begin());
		return_val_if_fail(face, 0);
		polyhedron->faces.push_back(face);

		return_val_if_fail(is_valid(*polyhedron), 0);

		return mesh.release();
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<freehand_polygon_implementation>, k3d::interface_list<k3d::imesh_source > > factory(
			k3d::uuid(0x36157b8e, 0xdf0f46cb, 0x886dc893, 0xe7b09afc),
			"FreehandPolygon",
			"Creates a freehand polygon",
			"Objects",
			k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}

private:
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_size;
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_no_intersection;

	std::vector<k3d::vector3> m_points;
};

/////////////////////////////////////////////////////////////////////////////
// freehand_polygon_factory

k3d::iplugin_factory& freehand_polygon_factory()
{
	return freehand_polygon_implementation::get_factory();
}

} // namespace libk3dmesh


