/***************************************
  $Header: /cvsroot/petscgraphics/utility.c,v 1.1 2004/08/17 15:05:42 hazelsct Exp $

  This file contains small utility functions for various aspects of
  visualization and storage.
  ***************************************/


#include "config.h" /* esp. for inline */
#include "illuminator.h" /* Just to make sure the interface is "right" */


#undef __FUNCT__
#define __FUNCT__ "minmax_scale"

/*++++++++++++++++++++++++++++++++++++++
  Determine a sensible scale for plotting, returned in *minmax.  If a scalar
  field, returns the minimum and maximum; if a vector field, returns the
  minimum and maximum magnitudes (in 1-D, just plain minimum and maximum); if a
  ternary, returns the corners of the smallest equilateral triangle in ternary
  space in which all of the data fit.

  int minmax_scale Returns zero or an error code.

  PetscScalar *global_array Array with values to scan for scale.

  int points Number of points in array to scan.

  int num_fields Number of fields in array.

  int display_field This display field (at least the start).

  field_plot_type fieldtype Type of field.

  int dimensions Number of dimensions.

  PetscScalar *minmax Array in which to return the minimum/maximum values.
  ++++++++++++++++++++++++++++++++++++++*/

int minmax_scale
(PetscScalar *global_array, int points, int num_fields, int display_field,
 field_plot_type fieldtype, int dimensions, PetscScalar *minmax)
{
  int i;

  if (minmax == NULL)
    SETERRQ (PETSC_ERR_ARG_BADPTR, "Invalid null pointer");

  if ((fieldtype == FIELD_VECTOR || fieldtype == FIELD_VECTOR+1) &&
      dimensions == 1)
    fieldtype = FIELD_SCALAR;

  switch (fieldtype)
    {
    case FIELD_SCALAR:
    case FIELD_SCALAR+1:
      {
	minmax [0] = minmax [1] = global_array [display_field];
	for (i=1; i<points; i++)
	  {
	    minmax [0] = PetscMin
	      (minmax [0], global_array [i*num_fields + display_field]);
	    minmax [1] = PetscMax
	      (minmax [1], global_array [i*num_fields + display_field]);
	  }
	return 0;
      }
    case FIELD_VECTOR:
    case FIELD_VECTOR+1:
      {
	/* Find the minimum and maximum square magnitudes, then sqrt them. */
	minmax[0] = minmax[1] =
	  global_array [display_field] * global_array [display_field] +
	  global_array [display_field+1] * global_array [display_field+1] +
	  ((dimensions < 3) ? 0. :
	   global_array [display_field+2] * global_array [display_field+2]);
	for (i=1; i<points; i++)
	  {
	    PetscScalar localmag =
	      global_array [i*num_fields + display_field] *
	      global_array [i*num_fields + display_field] +
	      global_array [i*num_fields + display_field+1] *
	      global_array [i*num_fields + display_field+1] +
	      ((dimensions < 3) ? 0. :
	       global_array [i*num_fields + display_field+2] *
	       global_array [i*num_fields + display_field+2]);
	    minmax[0] = PetscMin (minmax [0], localmag);
	    minmax[1] = PetscMax (minmax [1], localmag);
	  }
	minmax [0] = sqrt (minmax [0]);
	minmax [1] = sqrt (minmax [1]);
	return 0;
      }
    case FIELD_TERNARY:
      {
	/* Find the minimum x and y, and maximum sum, then fill in corners. */
	PetscScalar maxxpy =
	  global_array [display_field] + global_array [display_field+1];
	minmax[0] = global_array [display_field];
	minmax[1] = global_array [display_field+1];
	for (i=1; i<points; i++)
	  {
	    minmax [0] = PetscMin (minmax[0], global_array [display_field]);
	    minmax [1] = PetscMin (minmax[1], global_array [display_field+1]);
	    maxxpy = PetscMax (maxxpy, global_array [display_field] +
			       global_array [display_field+1]);
	  }
	minmax [2] = minmax [0];
	minmax [3] = maxxpy - minmax [0];
	minmax [4] = maxxpy - minmax [1];
	minmax [5] = minmax [1];
	return 0;
      }
    default:
      SETERRQ (PETSC_ERR_ARG_OUTOFRANGE, "Field type not yet supported");
    }
}


#undef __FUNCT__
#define __FUNCT__ "field_indices"

/*++++++++++++++++++++++++++++++++++++++
  Given an array of
  +latex+{\tt field\_plot\_type} enums, fill (part of) the {\tt indices}
  +html+ <tt>field_plot_type</tt> enums, fill (part of) the <tt>indices</tt>
  array with integers pointing to the true variable starts.  For example, in
  2-D with a vector field (two fields), a scalar field (one field), a symmetric
  tensor field (three fields) and a ternary composition field (two fields) for
  a total of 8 fields, this will fill the indices array with the values 0, 2,
  3, 6 and pad the rest of indices with -1, indicating when those true field
  variables start in the overall set of field variables.

  int nfields Total number of fields.

  int ds Dimensionality of the space (used to determine the number of fields
  used for a vector or tensor field).

  field_plot_type *plottypes Array of
  +latex+{\tt field\_plot\_type} enums with length {\tt nfields}.
  +html+ <tt>field_plot_type</tt> enums with length <tt>nfields</tt>.

  int *indices Array to hold the return values.
  ++++++++++++++++++++++++++++++++++++++*/

void field_indices (int nfields, int ds, field_plot_type *plottypes,
		    int *indices)
{
  int i, j;
  for (i=0, j=0; i<nfields; i++, j++)
    {
      indices [j] = i;
      if (plottypes [i] == FIELD_VECTOR ||
	  plottypes [i] == FIELD_VECTOR+1)
	i += ds-1;
      else if (plottypes [i] == FIELD_TERNARY)
	i += 1;
      else if (plottypes [i] == FIELD_TENSOR_FULL)
	i += ds*ds-1;
      else if (plottypes [i] == FIELD_TENSOR_SYMMETRIC)
	i += ds*(ds+1)/2 -1;
      else if (plottypes [i] == FIELD_TENSOR_SYMMETRIC_ZERODIAG)
	i += ds*(ds+1)/2 -2;
    }
  while (j<i)
    indices [j++] = -1;
}
