  /* 
   ikeyd - reacts on iBook2/Powerbooks hotkeys.

   Copyright (C) 2002 Stefan Pfetzing

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

/* Don't wonder about these triple "{" and numbers, these are markers for vim's
   folding function */

/* some includes {{{1 */
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <getopt.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <sys/mount.h>
#include <linux/version.h>
#include <linux/input.h>
#include <linux/cdrom.h>
#include <linux/soundcard.h>
#include <linux/types.h>
#include <linux/cdrom.h>
#include <linux/fd.h>
#include "system.h"
/* }}}1 */

/* define default settings {{{1 */
#define DEFAULT_MIXER "/dev/mixer"
#define DEFAULT_CDROM "/dev/cdrom"
#define DEFAULT_PIDFILE "/var/run/ikeyd.pid"
#define EVENT_PATH "/dev/input/event"
/* }}}1 */


static void usage (int status);

/* The name the program was run with, stripped of any leading path. */
char *program_name;

/* getopt_long return codes {{{1 */
enum
{ DUMMY_CODE = 129
};
/* }}}1 */

/* Option flags and variables {{{1 */

int want_quiet;			/* --quiet, --silent */
int want_verbose;		/* --verbose */
int want_nodaemon;		/* --verbose */
char *mixer;			/* --mixer */
char *cdrom;			/* --cdrom */
char *pidfile;			/* --pidfile */
/* }}}1 */

/* name of the event device */
char *event_device;

/* the structure of all long options for getopt {{{1 */
static struct option const long_options[] = {
  {"quiet", no_argument, 0, 'q'},
  {"silent", no_argument, 0, 'q'},
  {"verbose", no_argument, 0, 'v'},
  {"help", no_argument, 0, 'h'},
  {"version", no_argument, 0, 'V'},
  {"mixer", required_argument, 0, 'm'},
  {"cdrom", required_argument, 0, 'c'},
  {"pidfile", required_argument, 0, 'p'},
  {"nodaemon", no_argument, 0, 'n'},
  {NULL, 0, NULL, 0}
};
/* }}}1 */

/* prototypes of all functions {{{1 */
static int decode_switches (int argc, char **argv);

static void check_mixer ();

static void check_cdrom ();

static void check_pidfile ();

static int find_button_device ();

static void eject_cdrom ();

static void volume_change (int level);

static void remove_pidfile (int signum);

static void wait_child (int signum);
/* }}}1 */

/* the structure for stat(...) */
struct stat stat_buf;

/* int main (int argc, char **argv) {{{1 */
int
main (int argc, char **argv)
{
  /* declare some variables and structures {{{2 */
  /* the filedescriptor of the event interface */
  int eventfd = -1;
  /* the filedescriptor of the pidfile */
  FILE *pidstream;
  /* the pid of the child */
  pid_t childpid = 0;
  /* the structure for the input event, see input.h */
  struct input_event inp;
  /* the count of events caught */
  int count = 0;
  /* the keycode of the last key presssed */
  int last_keycode = 0;
  int i;
  /* the returnvalue from read */
  ssize_t ret;
  /* the sigaction handler for SIGTERM */
  struct sigaction sigterm = { {remove_pidfile}, {{0}}, SA_RESTART, 0 };
  /* the sigaction handler for SIGCHLD */
  struct sigaction sigchld = { {wait_child}, {{0}}, SA_RESTART, 0 };
  /* }}}2 */

  /* install the signal handlers {{{2 */
  if (sigaction (SIGINT, &sigterm, 0))
    {
      fprintf (stderr, "%s: could not install SIGINT handler\n", PACKAGE);
      exit (EXIT_FAILURE);
    }

  if (sigaction (SIGTERM, &sigterm, 0))
    {
      fprintf (stderr, "%s: could not install SIGTERM handler\n", PACKAGE);
      exit (EXIT_FAILURE);
    }

  if (sigaction (SIGCHLD, &sigchld, 0))
    {
      fprintf (stderr, "%s: could not install SIGCHLD handler\n", PACKAGE);
      exit (EXIT_FAILURE);
    }
  /* }}}2 */

  program_name = argv[0];

  i = decode_switches (argc, argv);

  /* check if the given mixer is correct */
  check_mixer ();

  /* check if the given cdrom is correct */
  check_cdrom ();

  /* now search the button device */
  eventfd = find_button_device ();

  /* check the pidfile */
  check_pidfile ();

  if (!want_nodaemon)		/* {{{2 */
    {
      /* fork and write the PID of the child into
       * the pidfile {{{3 */
      childpid = fork ();
      if (childpid)
	{
	  pidstream = fopen (pidfile, "w+a");
	  if (pidstream != NULL)
	    {
	      fprintf (pidstream, "%i\n", childpid);
	    }
	  else
	    {
	      fprintf (stderr, "%s: warning, couldn't save the pid in %s!\n",
		       PACKAGE, pidfile);
	    }
	  fclose (pidstream);
	  exit (0);
	}			/* }}}3 */

      /* make sure the daemon process doesn't do
       * anything weired. {{{3 */
      fclose (stdin);
      fclose (stderr);
      fclose (stdout);

      chdir ("/");

      setsid ();		/* }}}3 */
    }				/* }}}2 */

  /* do the work {{{2 */
  while ((ret = read (eventfd, &inp, sizeof (inp))) > 0)
    {
      /* on keypress inp.value is set, on release it is not set
       * if (inp.value) {{{3 */
      if (inp.value)
	{
	  /* on keypress */
	  if (want_verbose && want_nodaemon)
	    printf ("%s: keycode %i\n", PACKAGE, inp.code);

	  /* if the last key pressed is not the same as the
	   * current key, reset the count, else increment it. */
	  if (last_keycode != inp.code)
	    count = 1;
	  else
	    count++;
	  last_keycode = inp.code;

	  /* choose the action for the keycode */
	  switch (inp.code)
	    {
	    case KEY_MUTE:
	      /* mute the mixer */
	      volume_change (0);
	      break;
	    case KEY_VOLUMEDOWN:
	      /* decrement the volume */
	      volume_change (-8);
	      break;
	    case KEY_VOLUMEUP:
	      /* increment the volume */
	      volume_change (+8);
	      break;
	    case KEY_EJECTCD:
	      /* eject the cdrom */
	      if (count == 15)
		{
		  eject_cdrom ();
		}
	      break;
	    }
	}			/* }}}3 */
      else
	{			/* {{{3 */
	  /* on keyrelease */
	  if (last_keycode == KEY_EJECTCD && count == 15)
	    {
	      eject_cdrom ();
	    }
	  last_keycode = 0;
	  count = 0;
	}			/* }}}3 */
    }				/* }}}2 */

  /* check the return value of read(...) {{{2 */
  if (ret < 0)
    {
      fprintf (stderr, "%s: Couldn't read from %s: %s", PACKAGE, event_device,
	       strerror (errno));
    }				/* }}}2 */

  return 0;
}				/* }}}1 */

/* Set all the option flags according to the switches specified.
   Return the index of the first non-option argument.  */

/* static int decode_switches (int argc, char **argv) {{{1 */
static int
decode_switches (int argc, char **argv)
{
  int c;


  while ((c = getopt_long (argc, argv, "q"	/* quiet or silent */
			   "v"	/* verbose */
			   "h"	/* help */
			   "V"	/* version */
			   "m:"	/* mixer */
			   "c:"	/* cdrom */
			   "p:"	/* pidfile */
			   "n",	/* nodaemon */
			   long_options, (int *) 0)) != EOF)
    {
      switch (c)
	{
	case 'q':		/* --quiet, --silent */
	  want_quiet = 1;
	  break;
	case 'v':		/* --verbose */
	  want_verbose = 1;
	  break;
	case 'm':		/* --mixer */
	  mixer = optarg;
	  break;
	case 'c':		/* --cdrom */
	  cdrom = optarg;
	  break;
	case 'p':		/* --pidfile */
	  pidfile = optarg;
	  break;
	case 'n':		/* --nodaemon */
	  want_nodaemon = 1;
	  break;
	case 'V':
	  printf ("%s %s\n", PACKAGE, VERSION);
	  exit (0);

	case 'h':
	  usage (0);

	default:
	  usage (EXIT_FAILURE);
	}
    }

  return optind;
}				/* }}}1 */

/* static void usage (int status) {{{1 */
static void
usage (int status)
{
  printf (_("%s - \
reacts on iBook2/Powerbooks hotkeys.\n"), program_name);
  printf (_("Usage: %s [OPTION]... \n"), program_name);
  printf (_("\
Options:\n\
  -m, --mixer=MIXER          use alternative mixer-device,\n\
                             default is %s\n\
  -c, --cdrom=CDROM          use alternative cdrom-device,\n\
                             default is %s\n\
  -p, --pidfile=PIDFILE      use alternative pidfile,\n\
                             default is %s\n\
  -n, --nodaemon             do not run as a daemon\n\
  -q, --quiet, --silent      inhibit usual output\n\
  --verbose                  print more information\n\
  -h, --help                 display this help and exit\n\
  -V, --version              output version information and exit\n\
"), DEFAULT_MIXER, DEFAULT_CDROM, DEFAULT_PIDFILE);
  exit (status);
}				/* }}}1 */

/* static void check_mixer () {{{1 */
static void
check_mixer ()
{
  /* set the default mixer */
  if (mixer == NULL)
    {
      mixer = DEFAULT_MIXER;
    }

  /* check if the given mixer device is correct */
  if (stat (mixer, &stat_buf))
    {
      /* an error while stating occured */
      fprintf (stderr, "%s: ", PACKAGE);
      perror (mixer);
      exit (EXIT_FAILURE);
    }
  else
    {
      /* the mixer does exist, now check the type of the file. */
      if (!S_ISCHR (stat_buf.st_mode))
	{
	  printf ("%s: mixer %s is not a character device!\n", PACKAGE,
		  mixer);
	  exit (EXIT_FAILURE);
	}
    }
  if (want_verbose)
    {
      printf ("%s: Mixer is: %s\n", PACKAGE, mixer);
    }
}				/* }}}1 */

/* static void check_cdrom () {{{1 */
static void
check_cdrom ()
{
  /* set the default cdrom */
  if (cdrom == NULL)
    cdrom = DEFAULT_CDROM;

  /* check if the given cdrom device is correct */
  if (stat (cdrom, &stat_buf))
    {
      /* an error while stating occured */
      if (strcmp(cdrom,DEFAULT_CDROM) == 0)
	{
	  fprintf (stderr, "Something went wrong with %s:\n", DEFAULT_CDROM);
	  fprintf (stderr, "%s: ", PACKAGE);
	  perror (cdrom);
	  exit (0);
	}
      else
	{
	  fprintf (stderr, "%s: ", PACKAGE);
	  perror (cdrom);
	  exit (EXIT_FAILURE);
	}
    }
  else
    {
      /* the mixer does exist, now check the type of the file. */
      if (!S_ISBLK (stat_buf.st_mode))
	{
	  printf ("%s: cdrom %s is not a block device!\n", PACKAGE, cdrom);
	  exit (EXIT_FAILURE);
	}
    }
  if (want_verbose)
    printf ("%s: Cdrom is: %s\n", PACKAGE, cdrom);
}				/* }}}1 */

/* static void check_pidfile () {{{1 */
static void
check_pidfile ()
{
  /* set the default pidfile */
  if (pidfile == NULL)
    pidfile = DEFAULT_PIDFILE;

  /* check if the given pidfile is correct */
  if (stat (pidfile, &stat_buf))
    {
      if (errno != ENOENT)
	{
	  /* an error while stating occured */
	  fprintf (stderr, "%s: ", PACKAGE);
	  perror (pidfile);
	  exit (EXIT_FAILURE);
	}
    }
  else
    {
      fprintf (stderr, "%s: %s exists. %s seems to be running!\n", PACKAGE,
	       pidfile, PACKAGE);
      exit (EXIT_FAILURE);
    }

  if (want_verbose)
    printf ("%s: pidfile is: %s\n", PACKAGE, pidfile);
}				/* }}}1 */

/* static int find_button_device () {{{1 */
static int
find_button_device ()
{
  /* the filename of the event interface */
  char *filename;
  /* the filedescriptor of the event interface */
  int eventfd = -1;
  short ids[4];
  int i;

  for (i = 0; i < 32; i++)
    {
      filename = malloc (strlen (EVENT_PATH) + 3);
      sprintf (filename, "%s%i", EVENT_PATH, i);
      /* try to open the device */
      eventfd = open (filename, O_RDONLY);
      /* if opening suceeded */
      if (eventfd >= 0)
	{
	  /* check the id of the event interface */
	  ioctl (eventfd, EVIOCGID, ids);
	  if ((ids[2] & 0xfff) == 0x71f)
	    {
	      if (!want_quiet)
		printf ("%s: Button device found at %s\n", PACKAGE, filename);
	      event_device = filename;
	      break;
	    }
	  else
	    eventfd = -1;
	}
    }

  /* when no event filedesctiptor is set e.g. the event device could not be found */
  if (!(eventfd >= 0))
    {
      free (filename);

      /* check if we are root */
      if(geteuid() != 0)
	{
	  printf("%s: Couldn't access /dev/input/event* perhaps you are not root?\n",PACKAGE);
	  exit(0);
	}

      /* check if the kernel module is loaded */
      if(!system("lsmod | grep evdev"))
	{
	  printf("%s: The evdev kernel module is already loaded, so it seems\n",PACKAGE);
	  printf("%s: something else is going wrong.\n",PACKAGE);
	  exit(0);
	}
      
      /* when the kernel module is not loaded and we are not root
       * try to load the kernel module... */
      printf("%s: I' will try to load some modules...\n",PACKAGE);
      printf("%s: loading evdev...\n",PACKAGE);
      if(!system("modprobe evdev"))
	{
	  printf("%s: loading succeeded.\n",PACKAGE);
	  return find_button_device();
	}
      else
	printf("%s: Something wrent wong loading the evdev module.\n",PACKAGE);

      exit (0);
    }

  return eventfd;
}				/* }}}1 */

/* static void eject_cdrom () {{{1 */
static void
eject_cdrom ()
{
  pid_t cdromchild = fork ();
  char *command;
  if (cdromchild == 0)
    {
      /* the filedescriptor of the cdrom */
      int cdromfd = -1;

      /* the command to be executed */
      command =
	malloc (strlen ("/bin/umount ") + strlen (cdrom) + 1);
      sprintf(command, "/bin/umount %s", cdrom);

      system(command);

      cdromfd = open (cdrom, O_RDONLY | O_NDELAY);
      if (cdromfd < 0)
	{
	  /* an error while opening occured */
	  fprintf (stderr, "%s: ", PACKAGE);
	  perror (cdrom);
	}
      if (want_verbose && want_nodaemon)
	printf ("%s: ejecting the cdrom\n", PACKAGE);
#ifdef CDROMEJECT
      ioctl (cdromfd, CDROMEJECT);
#endif
#ifndef CDROMEJECT
      if (want_verbose && want_nodaemon)
	printf ("%s: ejecting not possible.\n", PACKAGE);
#endif
      close (cdromfd);
      _exit (0);
    }
}				/* }}}1 */

/* changes the volume level, if level is 0 then it will mute the volume *
 * static void volume_change (int level) {{{1 */
static void
volume_change (int level)
{
  static int old_volume = 0;
  int mixerfd = -1;
  int volume;
  int temp;

  mixerfd = open (mixer, O_RDONLY);

  ioctl (mixerfd, MIXER_READ (SOUND_MIXER_VOLUME), &volume);

  if (level == 0)
    {
      temp = volume;
      volume = old_volume;
      old_volume = temp;
    }
  else
    {
      volume = (volume & 0xff) + level;
      if (volume < 0)
	volume = 0;
      if (volume > 100)
	volume = 100;
      volume |= volume << 8;
      old_volume = 0;
    }

  ioctl (mixerfd, MIXER_WRITE (SOUND_MIXER_VOLUME), &volume);

  close (mixerfd);
}				/* }}}1 */

/* signal handler for SIGKILL *
 * static void remove_pidfile (int signum) {{{1 */
static void
remove_pidfile (int signum)
{
  /* remove the pidfile */
  unlink (pidfile);
  exit (0);
}				/* }}}1 */

/* signal handler for SIGCHLD
 * SIGCHLD is sent to the parent by a child when the child exits *
 * static void wait_child (int signum) {{{1 */
static void
wait_child (int signum)
{
  /* make sure wait is called when a client exits, to prevent defunct
   * processes */
  pid_t wpid;
  int stat_loc;
  wpid = waitpid (-1, &stat_loc, WNOHANG);
}				/* }}}1 */

/* vim600:set foldmarker={{{,}}} foldmethod=marker nu: */
