/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iscript.h"


#include "icalculator.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "imath.h"
#include "iscriptkit.h"

//
//  Templates
//
#include "iarraytemplate.h"
//#include "icalculatortemplate.h"


iScript::iScript(iScript *parent) : mErrorStatus("Script")
{
	mParent = parent;

	mCalculator = 0;

	mAllowChildAccess = true;  //  children can only access my vars if this is true

	mNumStatementPrototypes = mNumAssignmentPrototypes = 0;

	mBlock = mIsCompiled = false;
	
	//mText = "";
	mThisEntry = mNextEntry = -1;

	//
	//  Special characters - can be overwritten in a child class
	//
	mCommentString = "#";

	//
	// Continuation symbols - can be overwritten in a child class 
	// (may be useful for creatting REALLY complex multi-line syntax!)
	//
	mHeadOfLineContinuationString = "";
	mTailOfLineContinuationString = "\\";  // standard C-type continuation line
	mAppendSeparator = "";  // this separates strings when they are appended - useful for complex syntax
}


iScript::~iScript()
{
	int i;

	this->Reset();
	this->RemoveVariables();

	for(i=0; i<mConstants.Size(); i++) delete mConstants[i];
	for(i=0; i<mPrototypes.Size(); i++) delete mPrototypes[i];

	while(mObservers.Size() > 0) mObservers.RemoveLast()->SetScript(0);

	if(mCalculator != 0) delete mCalculator;
}


//
//  Observers
//
void iScript::AddObserver(iScriptObserver *obs)
{
	if(obs != 0) mObservers.AddUnique(obs);
}


void iScript::RemoveObserver(iScriptObserver *obs)
{
	if(obs != 0) mObservers.Remove(obs);
}


//
//  Script Creation
//
void iScript::CreateAliasWord(const iString &alias, const iString &word)
{
	AliasWord tmp;
	tmp.Alias = alias;
	tmp.Word = word;
	mAliasWords.Add(tmp);
}


void iScript::CreateDummyWord(const iString &v)
{
	if(!v.IsEmpty()) mDummyWords.Add(v);
}


void iScript::AddConstant(iScriptKit::Value *value)
{
	IERROR_ASSERT(value);

	int i;
	for(i=0; i<mConstants.Size(); i++) if(value->Name() == mConstants[i]->Name()) return;
	mConstants.Add(value);
}


iScriptKit::Calculator* iScript::CreateCalculator()
{
	return new iScriptKit::Calculator(this);
}


//
//  Public interface
//
void iScript::SetText(const iString &s)
{
	static const char d = 'a' - 'A';

	if(s != mText)
	{
		mText = s;

		if(!mText.IsEmpty())
		{
			if(mText[mText.Length()-1] != '\n') mText += "\n";
			if(!mCaseSensitive) 
			{
				//
				//  Turn the script into lower case, but not the quoted strings or comments
				//
				const char *cs = mCommentString.ToCharPointer();
				char *ptr = mText.GetWritePointer(mText.Length());
				bool doit = true;
				int i, n = mText.Length(), l = mCommentString.Length();
				for(i=0; i<n; i++)
				{
					if(ptr[i] == '"') doit = !doit;
					if(memcmp(cs,ptr+i,l) == 0) doit = false;
					if(ptr[i] == '\n') doit = true;
					if(doit && ptr[i]>='A' && ptr[i]<='Z') ptr[i] += d;
				}
			}
		}

		//
		//  Do partial reset
		//
		this->Reset();
	}
}


bool iScript::IsDummyWord(const iString &s) const
{
	return (mDummyWords.Find(s) != -1);
}


bool iScript::IsCommandWord(const iString &s) const
{
	int i;
	for(i=0; i<mNumStatementPrototypes; i++) if(mPrototypes[i]->Command() == s) return true;
	return false;
}


bool iScript::IsVariableName(const iString &s) const
{
	int i;
	for(i=0; i<mVariables.Size(); i++) if(mVariables[i]->Name() == s) return true;
	return false;
}


bool iScript::IsConstantName(const iString &s) const
{
	int i;
	for(i=0; i<mConstants.Size(); i++) if(mConstants[i]->Name() == s) return true;
	return false;
}


bool iScript::IsReservedWord(const iString &s) const
{
	return this->IsDummyWord(s) || this->IsCommandWord(s) || this->IsConstantName(s);
}


const iString &iScript::GetReservedWord(int i) const
{
	static const iString none;

	if(i < 0) return none;

	if(i < mDummyWords.Size()) return mDummyWords[i];
	i -= mDummyWords.Size();
	
	if(i < mNumStatementPrototypes) return mPrototypes[i]->Command();
	i -= mNumStatementPrototypes;
	
	if(i < mConstants.Size()) return mConstants[i]->Name();
	
	return none;
}


//
//  Get the line of code
//
iString iScript::GetText(int i)const
{
	static const iString none;

	if(i >= 0) return mText.Section("\n",i,i); else return none;
}


//
//  Remove the pseudocode
//
void iScript::Reset()
{
	//
	//  Delete the code and the stack
	//
	while(mCode.Size() > 0) delete mCode.RemoveLast();
	while(mStack.Size() > 0) mStack.RemoveLast();

	while(mPrototypes.Size() > mNumAssignmentPrototypes)  //  remove unkept variables
	{
		this->RemoveVariable(mPrototypes.Last()->Command());
		delete mPrototypes.RemoveLast();
	}

	mLoopCur = -1;
	mLoopNum = 0;

	mThisEntry = mNextEntry = -1;
	mBlock = mIsCompiled = false;
}


//
//  Remove the declared variables
//
void iScript::RemoveVariables()
{
	//
	//  Delete all variables and their associated commands
	//
	while(mPrototypes.Size() > mNumStatementPrototypes)
	{
		this->RemoveVariable(mPrototypes.Last()->Command());
		delete mPrototypes.RemoveLast();
	}
	//
	//  Remove the rest of variables
	//
	while(mVariables.Size() > 0)
	{
		delete mVariables.RemoveLast();
	}
	mIsCompiled = false;
}


void iScript::ReportError(const iString &message, int pos, int line)
{
	this->GetErrorStatus()->Set(message);
	mErrorPosition = pos;
	if(line == -1) line = this->GetThisLine();
}


void iScript::Execute(bool keepVariables)
{
	if(this->StartExecute())
	{
		while(this->ExecuteOneLine() && this->GetErrorStatus()->NoError());
		this->StopExecute(keepVariables);
	}
}


bool iScript::StartExecute()
{
	if(!this->Compile()) return false;

	int i;
	for(i=0; i<mObservers.Size(); i++) mObservers[i]->OnScriptStart();

	mNextEntry = 0;

	return true;
}


void iScript::StopExecute(bool keepVariables)
{
	int i;
	for(i=0; i<mObservers.Size(); i++) mObservers[i]->OnScriptStop(this->GetErrorStatus()->Message());

	mCalculator->Reset();
	if(keepVariables)
	{
		if(this->GetErrorStatus()->NoError()) mNumAssignmentPrototypes = mPrototypes.Size();
	}
	else this->RemoveVariables();

	mThisEntry = mNextEntry = -1;
}


//
//  Interrupts
//
bool iScript::CheckAbort() const
{
	if(mThisEntry>=0 && mThisEntry<mCode.Size() && mStack.Last()==mCode[mThisEntry])
	{
		return this->CheckAbort(mLoopCur,mLoopNum,0);
	}
	else
	{
		return this->CheckAbort(-1,0,0);
	}
}


bool iScript::CheckAbort(int cur, int num, int l) const
{
	int i, level = mStack.Size() + l;
	
	bool abort = false;
	for(i=0; !abort && i<mObservers.Size(); i++) abort = abort || mObservers[i]->OnScriptCheckAbort(cur,num,level);
	return abort;
}


//
// Execute one line of script. Returns true if the script is not finished, and false
// if it is done.
//
bool iScript::ExecuteOneLine()
{
	//
	//  Reset error status
	//
	this->GetErrorStatus()->Clear();
	mErrorPosition = -1;

	if(mBlock)
	{
		this->GetErrorStatus()->Set("Attempting to call script recursively.");
		return false; //  do not multi-thread me!!!
	}

	if(mNextEntry < 0)
	{
		this->GetErrorStatus()->Set("Script has not been started.");
		return false;
	}

	mBlock = true;

	mThisEntry = mNextEntry;
	if(mThisEntry == mCode.Size()) // Script is done
	{
		mBlock = false;
		return false;
	}

	int i, line;
	iString text;
	if(mObservers.Size() > 0)
	{
		line = this->GetThisLine();
		text = this->GetText(line);
		for(i=0; i<mObservers.Size(); i++) mObservers[i]->OnScriptBeginLine(line,text);
	}

	mNextEntry++; // set it here, so that a flow control statement can change it inside Execute

	if(mCode[mThisEntry]->Execute())
	{
		if(this->CheckAbort())
		{
			this->GetErrorStatus()->SetAbort(-1);
			mBlock = false;
			return false;
		}

		for(i=0; i<mObservers.Size(); i++) mObservers[i]->OnScriptEndLine(line,text);

		mBlock = false;
		return true;
	}
	else
	{
		mBlock = false;
		return false;
	}
}


//
//  Compile the script.
//
bool iScript::Compile()
{
	int i, j, k, n = this->GetNumberOfLines();
	iString line, text, s;

	if(mIsCompiled) return true; // we can be compiled many times before the actual execution

	if(mCalculator == 0)
	{
		mCalculator = this->CreateCalculator(); IERROR_ASSERT(mCalculator);
		if(mParent!=0 && mParent->mAllowChildAccess) mCalculator->CopyVariables(mParent->mCalculator); // this ensures correct name resolution

		int i;
		for(i=0; i<mConstants.Size(); i++) mCalculator->AddVariable(mConstants[i]->CalculatorValue());
	}

	this->Reset();

	int headLen = mHeadOfLineContinuationString.Length();
	int tailLen = mTailOfLineContinuationString.Length();

	for(k=0; k<n; k++)
	{
		line = this->GetText(k);

		//
		//  If the following lines are just continuations of this one, append them including mAppendSeparator
		//
		while(k+1 < n)
		{
			s = this->GetText(k+1);
			//
			//  The head string preceedes the tail one
			//
			i = 0;
			while(i<s.Length() && s.IsWhiteSpace(i)) i++;
			if(headLen>0 && s.Part(i,headLen)==mHeadOfLineContinuationString)
			{
				line += " " + mAppendSeparator + s.Part(headLen);
				k++;
				continue;
			}
			i = s.Length() - 1;
			while(i>=0 && s.IsWhiteSpace(i)) i--;
			if(tailLen>0 && line.Part(i+1-tailLen,tailLen)==mTailOfLineContinuationString)
			{
				line = line.Part(0,line.Length()-tailLen) + mAppendSeparator + s;
				k++;
				continue;
			}
			break;
		}

		//
		//  Remove comments if they are present (but not inside a string)
		//
		i = line.Find(mCommentString);
		if(i>=0 && line.Part(0,i).Contains('"')%2==0) line = line.Part(0,i);

		//
		//  Replace alias words - but not when they are parts of other words.
		//
		for(i=0; i<mAliasWords.Size(); i++) 
		{
			while((j=this->FindCommandWord(line,mAliasWords[i].Alias))>-1) line.Replace(j,mAliasWords[i].Alias.Length(),mAliasWords[i].Word);
		}

		//
		//  No command on this line
		//
		s = line;
		s.ReduceWhiteSpace();
		if(s.IsEmpty() || (s.Length()==1 && s[0]==' '))
		{
			continue;
		}

		//
		//  Remove dummy words - but not when they are parts of other words.
		//
		for(i=0; i<mDummyWords.Size(); i++) 
		{
			while((j=this->FindCommandWord(line,mDummyWords[i]))>-1) line.Replace(j,mDummyWords[i].Length()," ");
		}

		//
		//  Search the list of command words.
		//  Be carefult to find the exact complete match beginning at the start of the line
		//
		s = line;
		s.ReduceWhiteSpace();
		for(i=0; i<mPrototypes.Size(); i++)
		{
			if(this->FindCommandWord(s,mPrototypes[i]->Command()) == 0)
			{
				break;
			}
		}

		if(i == mPrototypes.Size()) //  The word is not a valid command
		{
			this->ReportError("Syntax error - invalid command or undefined variable.",0,k);
			return false;
		}

		j = -1;
		line.FindIgnoringWhiteSpace(mPrototypes[i]->Command(),0,&j);
		if(!mPrototypes[i]->Compile(line.Part(j),k,j))
		{
			return false;
		}
	}

	//
	//  Check that all closing flow control statements are present
	//
	if(mStack.Size() > 0)
	{
		this->ReportError("Missing loop or branch closing statement(s).",-1,k);
		return false;
	}

	mIsCompiled = true;
	return true;
}


int iScript::FindCommandWord(const iString &context, const iString &word, int index) const
{
	int i = index;
	int l = word.Length();

	while((i=context.Find(word,i))>-1)
	{
		//
		// Is this a full word?
		//
		if(i>0 && this->IsCommandWordLetter(context[i-1])) { i+=l; continue; }
		if(i+l<context.Length() && this->IsCommandWordLetter(context[i+l])) { i+=l; continue; }
		//
		//  Is it behind a quote - meaning, it is a string?
		//
		if(context.Part(0,i).Contains('"')%2 == 1) { i+=l; continue; }
		break;
	}
	return i;
}


bool iScript::IsCommandWordLetter(char c) const
{
	//
	//  Script words may contain letters, numbers, and '_'.
	//
	return ((c>='0'&&c<='9') || (c>='A'&&c<='Z') || (c>='a'&&c<='z') || c=='_');
}


bool iScript::IsCommandWordFirstLetter(char c) const
{
	//
	//  Script words must start with a letter or '_'.
	//
	return ((c>='A'&&c<='Z') || (c>='a'&&c<='z') || c=='_');
}


bool iScript::IsValidVariableName(const iString &name) const
{
	return (mCalculator!=0 && mCalculator->IsValidVariableName(name));
}


void iScript::AddVariable(iScriptKit::Value *value)
{
	if(value != 0)
	{
#ifdef I_DEBUG
		int i;
		for(i=0; i<mVariables.Size(); i++) if(value->Name() == mVariables[i]->Name())
		{
			int ooo = 0;
		}
#endif
		mVariables.Add(value);
	}
}


void iScript::RemoveVariable(const iString &name)
{
	if(!name.IsEmpty())
	{
		int i;
		for(i=mVariables.MaxIndex(); i>=0; i--)  // variables are always looped over in the reverse order to insure proper name resolution
		{
			if(mVariables[i]->Name() == name)
			{
				delete mVariables[i];
				mVariables.Remove(i);
				return;  //  remove one at a time
			}
		}
	}
}


iScriptKit::Value* iScript::FindVariable(const iString &name) const
{
	int i;
	for(i=mVariables.MaxIndex(); i>=0; i--) if(mVariables[i]->Name() == name) return mVariables[i];

	//
	// This ensures correct name resolution.
	//
	if(mParent!=0 && mParent->mAllowChildAccess) return mParent->FindVariable(name); else return 0;
}


//
//  Only return released variables
//
void iScript::QueryVariables(iArray<const iScriptKit::Value*> &list) const
{
	int i;
	list.Clear();
	for(i=0; i<mVariables.Size(); i++) if(mVariables[i]->IsReleased())
	{
		list.Add(mVariables[i]);
	}
}


//
//  Other misc functions
//
void iScript::JumpToEntry(int entry)
{
	if(entry>=0 && entry<mCode.Size()) mNextEntry = entry;
}


int iScript::GetThisLine() const
{
	if(mThisEntry>=0 && mThisEntry<mCode.Size() && mCode[mThisEntry]!=0) return mCode[mThisEntry]->LineInScript(); else return -1;
}


int iScript::GetNextLine() const
{
	if(mNextEntry>=0 && mNextEntry<mCode.Size() && mCode[mNextEntry]!=0) return mCode[mNextEntry]->LineInScript(); else return -1;
}


int iScript::GetNumberOfLines() const
{
	return mText.Contains('\n');
}


void iScript::SetLoopParameters(int cur, int num)
{
	mLoopCur = cur;
	mLoopNum = num;
}


bool iScript::SatisfyRequestForVariables(const iString &)
{
	return true;  //  by default, there are no special, non-algebraic variables
}


void iScript::RegisterPrototype(iScriptKit::Prototype *prototype)
{
	IERROR_ASSERT(prototype);

	if(mCalculator == 0)
	{
		int i;
		for(i=0; i<this->mPrototypes.Size(); i++) if(prototype->Command() == this->mPrototypes[i]->Command()) return;
		this->mPrototypes.Add(prototype);
		this->mNumStatementPrototypes++;
		mNumAssignmentPrototypes = mNumStatementPrototypes;
	}
	else
	{
		IERROR_FATAL("Script creation must be completed before the first compilation.");
	}
}


//
//  Observer functions
//
iScriptObserver::iScriptObserver(iScript *s)
{
	mScript = s;
	if(mScript != 0) mScript->AddObserver(this);
}


iScriptObserver::~iScriptObserver()
{
	if(mScript != 0) mScript->RemoveObserver(this);
}


bool iScriptObserver::OnScriptCheckAbort(int cur, int num, int level)
{
	if(mScript != 0) return this->OnScriptCheckAbortBody(cur,num,level); else return false;
}


void iScriptObserver::OnScriptStart()
{
	if(mScript != 0) this->OnScriptStartBody();
}


void iScriptObserver::OnScriptStop(const iString &error)
{
	if(mScript != 0) this->OnScriptStopBody(error);
}


void iScriptObserver::OnScriptBeginLine(int line, const iString &text)
{
	if(mScript != 0) this->OnScriptBeginLineBody(line,text);
}


void iScriptObserver::OnScriptEndLine(int line, const iString &text)
{
	if(mScript != 0) this->OnScriptEndLineBody(line,text);
}


void iScriptObserver::SetScript(iScript *s)
{
	if(mScript != s)
	{
		if(mScript != 0) mScript->RemoveObserver(this);
		mScript = s;
		if(mScript != 0) mScript->AddObserver(this);
	}
}

