/* icetypes.h
 * - General Type Declarations
 *
 * Copyright (c) 1999 Jack Moffitt, Barath Raghavan, and Alexander Havng
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#ifndef __ICECAST_TYPES_H
#define __ICECAST_TYPES_H

typedef enum {listener_e = 0, pusher_e = 1, pulling_client_e = 2, unknown_client_e = -1 } client_type_t;
typedef enum {icy_e = 0, xaudiocast_e = 1 } protocol_t;
typedef enum {encoder_e = 0, puller_e = 1, on_demand_pull_e = 2, file_e = 3, unknown_source_e = -1} source_type_t;
typedef enum contype_e {client_e = 0, source_e = 1, admin_e = 2, unknown_connection_e = 3} contype_t;
typedef enum { deny = 0, allow = 1, all = 2 } acltype_t;
typedef enum scheme_e {html_scheme_e = 0, default_scheme_e = 1, tagged_scheme_e = 2} scheme_t;
typedef enum { conf_file_e = 1, log_file_e = 2, template_file_e = 3, static_file_e = 4} filetype_t;
typedef enum { linux_gethostbyname_r_e = 1, solaris_gethostbyname_r_e = 2, standard_gethostbyname_e = 3 } resolv_type_t;
typedef enum { relay_pull_e = 1, relay_push_e = 2 } relay_type_t;

typedef int icecast_function();
typedef const char *htf;
typedef htf HttpFunction();
typedef avl_tree vartree_t;
typedef int wid_t;

typedef enum type_e {integer_e, real_e, string_e, function_e} type_t;

#define BUFSIZE 8192
#define CHUNKLEN 64
#define MAXMETADATALENGTH (16 * 256)
#define SOURCE_BUFFSIZE 8192
#define SOURCE_READSIZE (SOURCE_BUFFSIZE - MAXMETADATALENGTH) /* Please be careful here... */
#define MAXLISTEN 5		/* max number of listening ports */

#ifndef HAVE_SOCKLEN_T
typedef int socklen_t;
#endif

#ifndef _WIN32
typedef int SOCKET;
#define DIR_DELIMITER '/'
#else
#define DIR_DELIMITER '\\'
#define W_OK 2
#define R_OK 3
#endif

#ifndef HAVE_SOCK_T
typedef int sock_t;
#else
typedef sock_t SOCKET;
#endif

typedef struct http_parsable_commandsSt
{
  char *name;         /* What will be substituted */
  HttpFunction *func; /* Function to call for the substitution
		       * This function should return a pointer
		       * to where the parser should jump */
} http_parsable_t;

typedef struct http_variableSt
{
  char *name;
  type_t type;
  void *valueptr;
}http_variable_t;

typedef struct varpair_St
{
	char *name;
	char *value;
} varpair_t;

typedef struct acl_controlSt
{
	avl_tree *all;  /* tree of restrict_t's */
} acl_control_t;

typedef struct request_St
{
	char path[BUFSIZE];
	char host[BUFSIZE];
	int port;
} request_t;

typedef struct alias_St
{
	request_t *name;
	request_t *real;
} alias_t;

typedef struct restrict_St
{
	unsigned long int id;
	char *mask;
	acltype_t type;	/* is this restriction an allow or a deny? */
} restrict_t;

typedef struct chunkSt
{
	char data[SOURCE_BUFFSIZE + MAXMETADATALENGTH];
	int len;
	int metalen;
	int clients_left;
} chunk_t;

typedef struct statistics_St
{
	unsigned long int read_bytes;   /* Bytes read from encoder(s) */
	unsigned long int read_megs;    /* Megs read from encoder(s) */
	unsigned long int write_bytes;  /* Bytes written to client(s) */
	unsigned long int write_megs;   /* Megs written to clients */
	unsigned long int client_connections; /* Number of connects from clients */
	unsigned long int source_connections; /* Number of connects from sources */
	unsigned long int client_connect_time; /* Total sum of the time each client has been connected (minutes) */
	unsigned long int source_connect_time; /* Total sum of the time each source has been connected (minutes) */
} statistics_t;

/* audiocast stuff */
typedef struct audiocast_St {
	char *name;		/* Name of Server */
	char *genre;		/* Genre of music */
	int bitrate;		/* Bitrate of server */
	char *url;		/* Url for server */
	char *mount;		/* Name of this particular channel */
	char *description;	/* Description of this stream */
	char *streammimetype;   /* MIMEtype of stream (defaults to something smart) */
	char *contentid;
	int public;		/* List on the directory? */
} audiocast_t;

typedef struct relay_idSt
{
	char *host;
	int id;
} relay_id_t;

typedef struct songinfo_St
{
	char *streamurl;
	char *streamtitle;
	char *streammsg;
	long int streamlength;
	unsigned long int udpseqnr;
} songinfo_t;

typedef struct source_St {
	int connected;                 /* Is connected? */
	source_type_t type;            /* Encoder, or pulling redirect */
	protocol_t protocol;
	mutex_t mutex;
	audiocast_t audiocast;
	avl_tree *clients;             /* Tree of clients */
	icethread_t thread;              /* Pointer to running thread */
	statistics_t stats;
	unsigned long int num_clients; /* Take a wild guess */
	chunk_t chunk[CHUNKLEN];
	int cid;
	char *dumpfile;
	int dumpfd;
	avl_tree *relay_tree;
	int priority;                  /* order for getting the default mount in the sourcetree */
	songinfo_t info;

} source_t;

typedef struct client_St {
	unsigned int use_udp:1;
	unsigned int use_icy:1;

	int errors;             /* Used at first to mark position in buf, 
			             later to mark error */
	int offset;
	int cid;
	int alive;
	client_type_t type;
	unsigned long int write_bytes;	/* Number of bytes written to client */
	int virgin;			/* Need sync? */
	source_t *source;        /* Pointer back to the source (to avoid having to find it) */
	int use_icy_metadata;
	unsigned long int metadataoffset;
	unsigned long int metadatalen;
	unsigned long int metadatawritten;
	unsigned long int udpseqnr;
} client_t;

typedef struct admin_St {
	unsigned int status:1; /* Show status information ? */ 
	unsigned int oper:1;
	unsigned int tailing:1;
	unsigned int alive:1;
	int commands;          /* Number of issued commands */
	icethread_t thread;       /* Pointer to running thread */
	int debuglevel;
	scheme_t scheme;
} admin_t;

typedef struct dirSt {
	int counter;     /* Number of touches (or number of errors if id < 0)  */
	int touches;
	int id;          /* our id on this server */
	char *host;      /* Hostname of the directory server */
	int port;        /* What port for the directory server? */
	protocol_t type; /* Standard xaudiocast or icy? */
	char *path;      /* Path to script */
	time_t touchtime;
} directory_server_t;
  
typedef struct connectionSt {
	contype_t type;
	union {
		client_t *client;
		source_t *source;
		admin_t *admin;
	} food;
	unsigned long int id;  /* Session unique connection id */
	struct sockaddr_in *sin;
	socklen_t sinlen;
	SOCKET sock;
	time_t connect_time;
	char *host;
	char *hostname;
	vartree_t *headervars;
} connection_t;
  
typedef struct relay_St
{
	request_t req;                  /* Host, port and path of remote machine */
	relay_type_t type;              /* Type of relay (pushing or pulling) */
	char localmount[BUFSIZE];       /* Name of the local mount (both for push and pull) */
	connection_t *con;              /* Connection struct of a connected relay (or NULL if not connected) */
	int reconnects;                 /* How many times have we tried reconnecting? */
	time_t last_reconnect;          /* When was the last reconnect? */
	char *password;                 /* Only for push, encoder password */
	protocol_t protocol;            /* Protocol in/to use */
	audiocast_t audiocast;          /* Audiocast headers (only push) */
	int reconnect_now;              /* Tell reconnector to reconnect this now */
} relay_t;

typedef struct {
  
	/* Global stuff */
	char *runpath;			/* the argv[0] */
	int port[MAXLISTEN];		/* Listen to what port(s)? */
	SOCKET listen_sock[MAXLISTEN];	/* Socket to listen to */

	/* Where icecast lives */
	char *etcdir;		/* Name of config file directory */
	char *logdir;
	char *templatedir;

	/* Encoder stuff */
	avl_tree *sources;	/* Source array */
	unsigned long int num_sources;	/* Encoders connected */
	unsigned long int max_sources;	/* Maximal number of encoders */
	char *encoder_pass;	/* Password to verify encoder */
	char *default_sourceopts; /* Default output for 'sources' */
  
	/* Admin stuff */
	char *oper_pass;	/* Operator password (this one can do it all) */
	avl_tree *admins;	/* Admin array */
	unsigned long int num_admins;		/* Number of connected admins */
	unsigned long int max_admins;		/* Maximal number of admins */
	char *remote_admin_pass;	/* Password for remote administration */

	/* Directory server stuff */
	avl_tree *d_servers;	/* Directory servers to rat on */
	int touch_freq;	        /* Frequency to update directory 
				   (same touch_freq for all directories) */
	unsigned long int max_directories;	/* Maximal number of directory servers */
	time_t directorylasttime;		/* Time of last touch() */
  
	/* Udp info stuff */
	time_t udpupdatelasttime;
	int udpupdatetime;

	/* Misc stuff */
	char *staticdir;	/* Name of static content directory */
	char *configfile;	/* Name of configuration file */
	char *userfile;
	char *groupfile;
	char *mountfile;
	char *prompt;

	/* Logfiles */
	char *logfilename;	/* Name of default log file */
        int logfile;		/* File descriptor for that */
	char *usagefilename;    /* Name of usage log file   */
	int usagefile;          /* File descriptor for that */
	char *accessfilename;   /* Access file name */
	int accessfile;         /* File descriptor for that */
	char *statsfilename;	/* Name of stats logfile */
	char *statshtmlfilename; /* Like previous, but in html format */
	int statsfile;	        /* File to dump stats to */
	int statstime;		/* How long between dumps to the stats file */
	time_t statslasttime;	/* Time since last dump */
	long server_start_time; /* The time the server started */
	time_t statuslasttime;
	int statustime;
	char *myhostname;	/* NULL unless we want to bind to specific ip */
	char *server_name;	/* Server name */
	int console_mode;	/* CONSOLE_BACKGROUND, CONSOLE_LOG or CONSOLE_ADMIN */

	int detach;
	int throttle_on;
	double throttle;
	double bandwidth_usage;
	double sleep_ratio;    /* 0.0 - 1.0, how much of the time between reads should a source sleep  */
	
	int reverse_lookups;
	int force_servername;
	int mount_fallback;

	icethread_t main_thread;

#ifndef _WIN32
	pthread_attr_t defaultattr;
#endif
	mutex_t source_mutex;
	mutex_t admin_mutex;
	mutex_t directory_mutex;
	mutex_t alias_mutex;
	mutex_t misc_mutex;
	mutex_t mount_mutex;
	mutex_t hostname_mutex;
	mutex_t acl_mutex;
	mutex_t double_mutex;
	mutex_t thread_mutex;
	mutex_t mutex_mutex;
	mutex_t relay_mutex;
#ifdef DEBUG_MEMORY
	mutex_t memory_mutex;
	avl_tree *mem;
#endif

	mutex_t resolvmutex;
	resolv_type_t resolv_type;

	/* How to deal with clients when no encoder is connected
	 * Negative value means keep them forever.
	 * 0 means kick them out instantly
	 * Other values are keep them X seconds */
	int client_timeout; 

	char *client_pass;
	unsigned long int num_clients;
	unsigned long int max_clients;
	unsigned long int max_clients_per_source;

	avl_tree *threads;
	avl_tree *mutexes;
	avl_tree *relays; /* Connected and not connected relays */

	long int threadid;
	long int mutexid;
	unsigned long int id;

	avl_tree *aliases;
	int transparent_proxy;

	int kick_relays;   /* Kick relays when they are not relaying to any client (recommended) */
	int relay_reconnect_time; /* Seconds to wait before reconnecting dead relay */
	int relay_reconnect_tries; /* Number of tries before giving up reconnect (-1 means go on forever) */
	int kick_clients;  /* Kick clients when their source dies, instead of moving them to the default */

	avl_tree *my_hostnames;

	/* ACL controll (yeah, primitive) */
	avl_tree *all_acl;
	avl_tree *admin_acl;
	avl_tree *source_acl;
	avl_tree *client_acl;
	int policy;
	int allow_http_admin;

	/* Statistics */
	statistics_t hourly_stats;
	statistics_t daily_stats;
	statistics_t total_stats;

	/* Server meta info */
	char *location;
	char *rp_email;
	char *server_url;

	int streamurllock;

	char *streamtitle;
	char *streamurl;

	/* Templates man */
	char *streamtitletemplate;
	char *nametemplate;
	char *descriptiontemplate;

	int use_meta_data;

	unsigned long int metainterval;

	int consoledebuglevel;
	int logfiledebuglevel;

} server_info_t;

typedef struct com_requestSt
{
	wid_t wid;           /* Request includes a window id, or -1 for none */
	connection_t *con;   /* Pointer to requester connection */
	char *arg;           /* Pointer to requester argument, might be NULL */
}com_request_t;

#endif




