/* Groundhog -- a simple logic game
 * Copyright (C) 1998-2001 Maurits Rijk
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <algorithm>
#include <functional>

#include "ball.h"
#include "pocket.h"

#include "pixmaps/left_blue.xpm"
#include "pixmaps/left_green.xpm"
#include "pixmaps/left_red.xpm"
#include "pixmaps/left_yellow.xpm"
#include "pixmaps/lower_blue.xpm"
#include "pixmaps/lower_green.xpm"
#include "pixmaps/lower_red.xpm"
#include "pixmaps/lower_yellow.xpm"
#include "pixmaps/right_blue.xpm"
#include "pixmaps/right_green.xpm"
#include "pixmaps/right_red.xpm"
#include "pixmaps/right_yellow.xpm"
#include "pixmaps/upper_blue.xpm"
#include "pixmaps/upper_green.xpm"
#include "pixmaps/upper_red.xpm"
#include "pixmaps/upper_yellow.xpm"


Pocket::Pocket(GtkWidget* table) : _table(table), _element(0), _ball(0)
{
}

void
Pocket::InsertInTable(int row, int col, GdkPixmap* pixmap)
{
   /* a pixmap widget to contain the pixmap */
   _pixmapwid = gtk_pixmap_new(_red_pixmap, _mask);
   gtk_widget_show(_pixmapwid);

   gtk_table_attach_defaults(GTK_TABLE(_table), _pixmapwid, col, col + 1, 
			     row, row + 1);
}

GdkPixmap*
Pocket::CreateFromXpm(char** data)
{
   GtkStyle* style = gtk_widget_get_style(_table);   
   return gdk_pixmap_colormap_create_from_xpm_d(
      _table->window, gtk_widget_get_colormap(_table), &_mask,
      &style->bg[GTK_STATE_NORMAL], data);
}

void 
Pocket::PutBall(Ball* ball)
{
   _ball = ball;
   switch (ball->GetColor()) {
   case Ball::Red:
      gtk_pixmap_set(GTK_PIXMAP(_pixmapwid), _red_pixmap, 0);
      break;
   case Ball::Green:
      gtk_pixmap_set(GTK_PIXMAP(_pixmapwid), _green_pixmap, 0);
      break;
   case Ball::Blue:
      gtk_pixmap_set(GTK_PIXMAP(_pixmapwid), _blue_pixmap, 0);
      break;
   case Ball::Yellow:
      gtk_pixmap_set(GTK_PIXMAP(_pixmapwid), _yellow_pixmap, 0);
      break;
   }
   Element::PutBall(ball);
}

bool
Pocket::Contains(Ball::Color color)
{
   return _ball->GetColor() == color;
}

void
SetOffPockets::Eject()
{
   for_each(_set.begin(), _set.end(), std::mem_fun(&Pocket::Eject));
}

bool 
SetOffPockets::HasRightBalls()
{
   return 
      find_if(_set.begin(), _set.end(), 
	      std::not1(std::mem_fun(&Pocket::HasRightBall))) == _set.end();
}

LeftPocket::LeftPocket(GtkWidget* table, int row, int col) : Pocket(table)
{
   _red_pixmap = CreateFromXpm(left_red_xpm);
   _green_pixmap = CreateFromXpm(left_green_xpm);
   _blue_pixmap = CreateFromXpm(left_blue_xpm);
   _yellow_pixmap = CreateFromXpm(left_yellow_xpm);
   InsertInTable(row, col, _red_pixmap);
}

void 
LeftPocket::MoveBall(Ball* ball)
{
   if (ball->GetDirection() == Ball::MovingLeft) {
      ball->SetDirection(Ball::NotMoving);
      _ball = ball;
   } else { 
      _element->PutBall(ball);
      _ball = 0;
   }
}

void 
LeftPocket::Eject()
{
   _ball->SetDirection(Ball::MovingRight);
}

UpperPocket::UpperPocket(GtkWidget* table, int row, int col) : Pocket(table)
{
   _red_pixmap = CreateFromXpm(upper_red_xpm);
   _green_pixmap = CreateFromXpm(upper_green_xpm);
   _blue_pixmap = CreateFromXpm(upper_blue_xpm);
   _yellow_pixmap = CreateFromXpm(upper_yellow_xpm);
   InsertInTable(row, col, _red_pixmap);
}

void 
UpperPocket::MoveBall(Ball* ball)
{
   if (ball->GetDirection() == Ball::MovingUp) {
      ball->SetDirection(Ball::NotMoving);
      _ball = ball;
   } else {
      _element->PutBall(ball);
      _ball = 0;
   }
}

void
UpperPocket::Eject()
{
   _ball->SetDirection(Ball::MovingDown);
}

RightPocket::RightPocket(GtkWidget* table, int row, int col) : Pocket(table)
{
   _red_pixmap = CreateFromXpm(right_red_xpm);
   _green_pixmap = CreateFromXpm(right_green_xpm);
   _blue_pixmap = CreateFromXpm(right_blue_xpm);
   _yellow_pixmap = CreateFromXpm(right_yellow_xpm);
   InsertInTable(row, col, _red_pixmap);
}

void 
RightPocket::MoveBall(Ball* ball)
{
   if (ball->GetDirection() == Ball::MovingRight) {
      ball->SetDirection(Ball::NotMoving);
      _ball = ball;
   } else {
      _element->PutBall(ball);
      _ball = 0;
   }
}

void
RightPocket::Eject()
{
   _ball->SetDirection(Ball::MovingLeft);
}

LowerPocket::LowerPocket(GtkWidget* table, int row, int col) : Pocket(table)
{
   _red_pixmap = CreateFromXpm(lower_red_xpm);
   _green_pixmap = CreateFromXpm(lower_green_xpm);
   _blue_pixmap = CreateFromXpm(lower_blue_xpm);
   _yellow_pixmap = CreateFromXpm(lower_yellow_xpm);
   InsertInTable(row, col, _red_pixmap);
}

void 
LowerPocket::MoveBall(Ball* ball)
{
   if (ball->GetDirection() == Ball::MovingDown) {
      ball->SetDirection(Ball::NotMoving);
      _ball = ball;
   } else {
      _element->PutBall(ball);
      _ball = 0;
   }
}

void
LowerPocket::Eject()
{
   _ball->SetDirection(Ball::MovingUp);
}
