/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*------------------------------------------------------------------------------
 
  rr - rr calculates the mean particle displacement for (Digital) Particle
       Image Velocimetry (DPIV) by means of FFT. 

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

 
 -------------------------------------------------------------------- */

#include <stdlib.h>
#include <gpiv.h>

#define PARFILE "gpivrc"	/* Parameter file name */

/* static float sqrarg; */
/* #define SQR(a) ((sqrarg=(a)) == 0.0 ? 0.0 : sqrarg*sqrarg) */
/* #define SWAP(a,b) tempr=(a);(a)=(b);(b)=tempr */

#define USAGE "\
Usage: rr [-c int] [--c_dif] [--no_cdif] [--cf int] [--cl int] [--cp int] \n\
          [-f filename] [-g] [-h | --help] [--iasize1 int] [--iasize2 int] \n\
          [--iashift int] [--ischeme int] [--ifit 0/1/2/3] [--linec int int int] \n\
          [--liner int int int] [-p | --print] [--peak 1/2/3] \n\
          [--p_piv] [--point int int] [-r int] [--rf int] [--rl int] [--rp int] \n\
          [-s float] [-v | --version] [--val_r int] [--val_s] [--val_t float] \n\
          [-x | --cross_corr] [--no_x | --auto_corr] < stdin > stdout \n\
\n\
keys: \n\
-c ncols:              number of columns \n\
--cf col:              first column of area to interrogate \n\
--cl col:              last column of area to interrogate \n\
--cp columns:          pre-shift in x-direction (columns) \n\
-f filename:           overrides stdin and stdout  \n\
-g:                    graphical visualization with gnuplot (needs -f) \n\
-h | --help:           this on-line help \n\
--iasize1 size:        size of first interrogation area \n\
--iasize2 size:        size of second interrogation area (>= isi1) \n\
--iashift shift:       shift of adjacent interrogation area \n\
--ischeme 0/1/2/3/4:   interrogation scheme: no correction (0), linear \n\
                       weighting (1), zero offset (2), zero offset with central \n\
                       differential (3), image deformation (4) \n\
--ifit 0/1/2/3:        interpolation type: none (0), Gauss (1), Parabolic (2) \n\
                       or Centre of Gravity (3). \n\
--linec c rf rl:       selects a vertical line at column r to interrogate \n\
                       from row rf to row rl \n\
--liner r cf cl:       selects an horizontal line at row r to interrogate \n\
                       from column cf to column cl \n\
-p | --print:          print parameters and other info to stdout \n\
--peak #:              find maximum of #-th covariance peak \n\
--p_piv:               prints piv results to stdout, even if -f has been used \n\
--point x y:           select one single point in the image to interrogate \n\
-r nrows:              number of rows \n\
--rf row:              first row of area to interrogate \n\
--rl row:              last row of area to interrogate \n\
--rp rows:             pre-shift in y-direction (rows) \n\
-s S:                  scale factor for graphic output with gnuplot \n\
-v | --version:        version number \n\
--val_r N:             validation; residu type calculated from: snr (0), median \n\
                       (1) or normalized median (2) \n\
--val_s N:             validation; substitution of erroneous vector by: nothing \n\
                       (0), local mean from the surroundings (1), \n\
                       the median of the surroundings (2), next highest \n\
                       correlation peak (3) (needs -f) \n\
--val_t F:             validation; threshhold of residus to be accepted \n\
-x | --cross_corr:     cross-correlation \n\
--no_x | --auto_corr:  auto-correlation \n\
"

#ifdef DEBUG
#define USAGE_DEBUG "\
Developers version also contains: \n\
rr:	  [-p_main int] \n\
\n\
keys: \n\
-p_'function' int:     prints data to be generated in the function; the \n\
                       higher the number, the more detailed the output.  \n\
                       For int = 10, rr will exit at the end of the function \n\
"
#endif

#define HELP "\
rr interrogates an image (pair) in order to obtain displacements of \n\
particles for (Digital) Particle Image Velocimetry (PIV). \n\
"

#define RCSID "$Id: rr.c,v 3.17 2006/01/31 14:18:03 gerber Exp $"

/*
 * Global parameters and variables
 */
gboolean fname_logic = FALSE;
gboolean print_par = FALSE;


#define GNUPLOT_DISPLAY_SIZE 500
#define GNUPLOT_DISPLAY_COLOR "LightBlue"

int gnuplot = 0;
float gnuplot_scale = 1.0;
gboolean gnuplot_logic = FALSE, gnuplot_scale_logic = FALSE;



#ifdef DEBUG
/*
 * Variables for developer version
 */
int print_main = 0;
#endif



static void 
command_args(int argc, 
             char *argv[], 
             char fname[GPIV_MAX_CHARS],
             GpivImagePar * image_par, 
             GpivEvalPar * piv_eval_par,
             GpivValidPar * piv_valid_par
             )
/*-----------------------------------------------------------------------------
 * Command line argument handling
 */
{
    char c;
    int argc_next;


/*
 * Processing of command line arguments
 */
    while (--argc > 0 && (*++argv)[0] == '-') {
        argc_next = 0;		/* argc_next is set to 1 if the next cmd 
                                 * line argument has to be searched for; 
                                 * in case that the command line argument
                                 * concerns more than one char or cmd 
                                 * line argument needs a parameter
                                 */
        while (argc_next == 0 && (c = *++argv[0]))
            switch (c) {

/*
 * number of columns
 */
            case 'c':
                image_par->ncolumns = atoi(*++argv);
                image_par->ncolumns_logic = TRUE;
                argc_next = 1;
/*
 * an integer value belongs to this argument; argc a has to skip one
 */
                --argc;
                break;

/*
 * use -f instead of stdin stdout
 */
            case 'f':
                strcpy(fname, *++argv);
                fname_logic = TRUE;
                argc_next = 1;
                --argc;
                break;

/*
 * graphic output with gnuplot
 */
            case 'g':	       
                gnuplot = 1;
                gnuplot_logic = TRUE;
                /* 		 argc_next = 1; */
                break;
                
            case 'h':
                printf("%s\n", RCSID);
                printf("%s\n", HELP);
                printf("%s", USAGE);
#ifdef DEBUG
                printf("\n%s", USAGE_DEBUG);
#endif
                exit(0);

/*
 * print paramaters to stdout
 */
            case 'p':
                print_par = TRUE;
                break;
                
/*
 * number of rows
 */
            case 'r':
                image_par->nrows = atoi(*++argv);
                image_par->nrows_logic = TRUE;
                --argc;
                argc_next = 1;
                break;

/*
 * scaling for graphic output with gnuplot
 */
            case 's':	       
                gnuplot_scale = atof(*++argv);
                gnuplot_scale_logic = TRUE;
                --argc;
                argc_next = 1;
                break;
                                
/*
 * Print version and exits
 * Use Revision Control System (RCS) for version
 */
            case 'v':
                printf("%s\n", RCSID);
                exit(0);
                
/*
 * forces Rr.x_corr = 1
 */
            case 'x':
                image_par->x_corr = 1;
                image_par->x_corr_logic = TRUE;
                break;
                
/*
 * long option keys
 */
            case '-':
                if (strcmp("-help", *argv) == 0) {
                    printf("\n%s", RCSID);
                    printf("\n%s", HELP);
                    printf("\n%s", USAGE);
                    exit(0);
                } else if (strcmp("-cross_corr", *argv) == 0) {
                    image_par->x_corr = 1;
                    image_par->x_corr_logic = TRUE;
                } else if (strcmp("-auto_corr", *argv) == 0) {
                    image_par->x_corr = 0;
                    image_par->x_corr_logic = TRUE;
                } else if (strcmp("-print", *argv) == 0) {
                    print_par = TRUE;
                } else if (strcmp("-version", *argv) == 0) {
                    printf("%s\n", RCSID);
                    exit(0);

/*
 * adaptive interr area
 */
                } else if (strcmp("-ad_int", *argv) == 0) {
                    piv_eval_par->ad_int = 1;
                    piv_eval_par->ad_int_logic = TRUE;
                    argc_next = 1;
                } else if (strcmp("-autokey", *argv) == 0) {
                    piv_eval_par->autokey = 1;
                    piv_eval_par->autokey_logic = TRUE;

/*
 * Rr.Col_start
 */
                } else if (strcmp("-cf", *argv) == 0) {
                    piv_eval_par->col_start = atoi(*++argv);
                    piv_eval_par->col_start_logic = TRUE;
                    piv_eval_par->int_geo = GPIV_AOI;
                    piv_eval_par->int_geo_logic = TRUE;
                    --argc;
                    argc_next = 1;

/*
 * overrides Rr.Col_end
 */
                } else if (strcmp("-cl", *argv) == 0) {
                    piv_eval_par->col_end = atoi(*++argv);
                    piv_eval_par->col_end_logic = TRUE;
                    piv_eval_par->int_geo = GPIV_AOI;
                    piv_eval_par->int_geo_logic = TRUE;
                    --argc;
                    argc_next = 1;

/*
 * pre-shift c (col)
 */
                } else if (strcmp("-cp", *argv) == 0) {
                    piv_eval_par->pre_shift_col = atoi(*++argv);
                    piv_eval_par->pre_shift_col_logic = TRUE;
                    --argc;
                    argc_next = 1;

/* 		} else if (strcmp("cmpr", *argv) == 0) { */
/* 		    cmpr = 1; */
/* 		    cmpr_logic = TRUE; */
/* 		    --argc; */
/* 		    argc_next = 1; */

/*
 * overrides Rr.Int_size_1
 */
                } else if (strcmp("-iasize1", *argv) == 0) {
                    piv_eval_par->int_size_1 = atoi(*++argv);
                    piv_eval_par->int_size_1_logic = TRUE;
                    --argc;
                    argc_next = 1;
                                        
/*
 * overrides Rr.Int_size_2
 */
                } else if (strcmp("-iasize2", *argv) == 0) {
                    piv_eval_par->int_size_2 = atoi(*++argv);
                    piv_eval_par->int_size_2_logic = TRUE;
                    --argc;
                    argc_next = 1;

/*
 * overrides Rr.int_shift
 */
                } else if (strcmp("-iashift", *argv) == 0) {
                    piv_eval_par->int_shift = atoi(*++argv);
                    piv_eval_par->int_shift_logic = TRUE;
                    --argc;
                    argc_next = 1;
                    
/*
 * overrides Rr.Int_scheme
 */
                } else if (strcmp("-ischeme", *argv) == 0) {
                    piv_eval_par->int_scheme = atoi(*++argv);
                    if (piv_eval_par->int_scheme != 0 
                        && piv_eval_par->int_scheme != 1 
                        && piv_eval_par->int_scheme != 2
                        && piv_eval_par->int_scheme != 3 
                        && piv_eval_par->int_scheme != 4) {
                        gpiv_error("%s: invalid value of Int_scheme (1, 2, 3 or 4)", 
                                   RCSID);
                    }
                    piv_eval_par->int_scheme_logic = TRUE;
                    --argc;
                    argc_next = 1;

/*
 * overrides Rr.Ifit
 */
                } else if (strcmp("-ifit", *argv) == 0) {
                    piv_eval_par->ifit = atoi(*++argv);
                    if (piv_eval_par->ifit != -1 && piv_eval_par->ifit != 0 && 
                        piv_eval_par->ifit != 1 && piv_eval_par->ifit != 2
                        && piv_eval_par->ifit != 3) {
                        gpiv_error("%s: invalid value of Ifit (1, 2 or 3)", 
                                   RCSID);
                    }
                    piv_eval_par->ifit_logic = TRUE;
                    --argc;
                    argc_next = 1;

/*
 * define interrogate at a line
 */
                } else if ((strcmp("-linec", *argv) == 0)) {
                    piv_eval_par->int_line_col = atoi(*++argv);
                    piv_eval_par->int_line_row_start = atoi(*++argv);
                    piv_eval_par->int_line_row_end =  atoi(*++argv);
                    piv_eval_par->int_geo = GPIV_LINE_C;
                    piv_eval_par->int_line_col_logic = TRUE;
                    piv_eval_par->int_line_row_start_logic = TRUE;
                    piv_eval_par->int_line_row_end_logic = TRUE;
                    if (piv_eval_par->int_geo_logic == TRUE) {
                        gpiv_error("%s: only a horizontal line"
                                   " may be defined\n"
                                   " not a vertical, a point or an area of"
                                   " interest as well\n",
                                   RCSID);
                    } else {
                        piv_eval_par->int_geo_logic = TRUE;
                    }
                    --argc;
                    --argc;
                    --argc;
                    argc_next = 1;
                } else if ((strcmp("-liner", *argv) == 0)) {
                    piv_eval_par->int_line_row = atoi(*++argv);
                    piv_eval_par->int_line_col_start = atoi(*++argv);
                    piv_eval_par->int_line_col_end =  atoi(*++argv);
                    piv_eval_par->int_geo = GPIV_LINE_R;
                    piv_eval_par->int_line_row_logic = TRUE;
                    piv_eval_par->int_line_col_start_logic = TRUE;
                    piv_eval_par->int_line_col_end_logic = TRUE;
                    if (piv_eval_par->int_geo_logic == TRUE) {
                        gpiv_error("%s: a vertical horizontal line"
                                   " may be defined\n"
                                   " not an horizontal, a point or an area of"
                                   " interest as well\n",
                                   RCSID);
                    } else {
                        piv_eval_par->int_geo_logic = TRUE;
                    }
                    --argc;
                    --argc;
                    --argc;
                    argc_next = 1;
                    
/*
 * print cov. function
 */
                } else if (strcmp("-p_cov", *argv) == 0) {
                    piv_eval_par->print_cov = 1;
                    piv_eval_par->print_cov_logic = TRUE;
                    argc_next = 1;
                    
/*
 * cov peak # to fit
 */
                } else if (strcmp("-peak", *argv) == 0) {
                    piv_eval_par->peak = atoi(*++argv);
                    --argc;
                    piv_eval_par->peak_logic = TRUE;
                    argc_next = 1;
                    
/*
 * # piv values to stdout
 */
                } else if (strcmp("-p_piv", *argv) == 0) {
                    piv_eval_par->print_piv = 1;
                    piv_eval_par->print_piv_logic = TRUE;
                    argc_next = 1;
                    
/*
 * interrogation at point(row, col)
 */
                } else if (strcmp("-point", *argv) == 0) {
                    piv_eval_par->int_point_col = atoi(*++argv);
                    piv_eval_par->int_point_row = atoi(*++argv);
                    piv_eval_par->int_point_col_logic = TRUE;
                    piv_eval_par->int_point_row_logic = TRUE;
                    piv_eval_par->int_geo = GPIV_POINT;
                    if (piv_eval_par->int_geo_logic == TRUE) {
                        gpiv_error("%s: only a point may be defined\n"
                                   " not an horizontal, a vertical line"
                                   " or an area of interest as well\n",
                                   RCSID);
                    } else {
                        piv_eval_par->int_geo_logic = TRUE;
                    }
                    --argc;
                    --argc;
                    argc_next = 1;
#ifdef DEBUG
                } else if (strcmp("-p_main", *argv) == 0) {
                    print_main = atoi(*++argv);
                    --argc;
                    argc_next = 1;
                    
#endif /* DEBUG */
                    
/*
 * overrides Rr.Row_start
 */
                } else if (strcmp("-rf", *argv) == 0) {
                    piv_eval_par->row_start = atoi(*++argv);
                    piv_eval_par->row_start_logic = TRUE;
                    piv_eval_par->int_geo = GPIV_AOI;
                    piv_eval_par->int_geo_logic = TRUE;
                    --argc;
                    argc_next = 1;

/*
 * overrides Rr.Row_end
 */
		} else if (strcmp("-rl", *argv) == 0) {
		    piv_eval_par->row_end = atoi(*++argv);
		    piv_eval_par->row_end_logic = TRUE;
                    piv_eval_par->int_geo = GPIV_AOI;
                    piv_eval_par->int_geo_logic = TRUE;
		    --argc;
		    argc_next = 1;
                    
/*
 * pre-shift r (row)
 */
		} else if (strcmp("-rp", *argv) == 0) {
		    piv_eval_par->pre_shift_row = atoi(*++argv);
		    piv_eval_par->pre_shift_row_logic = TRUE;
		    --argc;
		    argc_next = 1;



/*
 * validation parameter: residu type
 */
                } else if (strcmp("-val_r", *argv) == 0) {
                    piv_valid_par->residu_type = atoi(*++argv);
                    piv_valid_par->residu_type_logic = TRUE;
                    --argc;
                    argc_next = 1;
                    if (piv_valid_par->residu_type != 0 
                        && piv_valid_par->residu_type != 1 
                        && piv_valid_par->residu_type != 2) {
                        gpiv_error("%s: invalid value of VALID.Residu_type (0, 1, or 2)", 
                                   RCSID);
                    }
                    
/*
 * validation parameter: substitution type
 */
                } else if (strcmp("-val_s", *argv) == 0) {
                    piv_valid_par->subst_type = atoi(*++argv);
                    piv_valid_par->subst_type_logic = TRUE;
                    --argc;
                    argc_next = 1;
                    if (piv_valid_par->subst_type != 0 
                        && piv_valid_par->subst_type != 1 
                        && piv_valid_par->subst_type != 2
                        && piv_valid_par->subst_type != 3) {
                        gpiv_error("%s: invalid value of VALID.Subst_type (0, 1, 2 or 3)", 
                                   RCSID);
                    }
/*
 * validation parameter: threshold value
 */
                } else if (strcmp("-val_t", *argv) == 0) {
                    piv_valid_par->residu_max = atof(*++argv);
                    piv_valid_par->residu_max_logic = TRUE;
                    --argc;
                    argc_next = 1;
                    


/*
 * excluding of settings
 * forces Rr.x_corr = 0
 */
                } else if (strcmp("-no_x", *argv) == 0) {
		    image_par->x_corr = 0;
		    image_par->x_corr_logic = TRUE;

/*
 * adaptive interr area
 */
		} else if (strcmp("-no_ad_int", *argv) == 0) {
		    piv_eval_par->ad_int = 0;
		    piv_eval_par->ad_int_logic = TRUE;
		    argc_next = 1;

                } else {
		    gpiv_error("%s: unknown option: %s", RCSID, *argv);
		}
		argc_next = 1;
		break;

	    default:
		gpiv_error("%s: unknown option: %s", RCSID, *argv);
		break;

	    }
    }


    if (argc != 0) {
	gpiv_error("%s: unknown argument: %s", RCSID, *argv);
    }


}



static int 
make_fname(FILE * fp_par_out, 
           char *fname, 
           char *fname_in, 
           char *fname_int_first, 
           char *fname_int_second,
           char *fname_cov, 
           char *fname_out,
           GpivEvalPar piv_eval_par
           )
/*-----------------------------------------------------------------------------
 * generates filenames
 */
{

/*
 * Image filenames
 */
    strcpy(fname_in, fname);
    strcat(fname_in, GPIV_EXT_RAW_IMAGE);
    if (print_par)
        printf("\n# image file is: %s", fname_in);
    fprintf(fp_par_out, "\n# image file is: %s", fname_in);
        
/*
 * filenames for output PIV data
 */
    gpiv_io_make_fname(fname, GPIV_EXT_PIV, fname_out);
    if (piv_eval_par.print_piv == 0) {
        if (print_par)
            printf("\n# outputfile is: %s\n", fname_out);
        fprintf(fp_par_out, "\n# outputfile is: %s\n", fname_out);
    }

/*
 * filenames for output files
 */
    if (piv_eval_par.print_cov == 1) {
        strcpy(fname_cov, fname);
        strcat(fname_cov, GPIV_EXT_COV);
        if (print_par)
            printf("\n# covariance file is: %s", fname_cov);
        fprintf(fp_par_out, "\n# covariance file is: %s", fname_cov);
    }
    
    return 0;
}



static char*
alloc_pivdata_gridgen(GpivPivData *piv_data, 
                           GpivImagePar image_par, 
                           GpivEvalPar piv_eval_par
                           )
/*-----------------------------------------------------------------------------
 * Determines the number of grid points, allocating memory for output 
 * data and generates the grid
 */
{
    gchar *err_msg = NULL;
    GpivEvalPar piv_eval_par_ACT;
    
    gpiv_piv_cp_parameters(piv_eval_par, 
                           &piv_eval_par_ACT, 
                           TRUE, FALSE);
    if (piv_eval_par.ad_int 
        && piv_eval_par.int_shift <
        piv_eval_par.int_size_2 / GPIV_SHIFT_FACTOR) { 
        piv_eval_par_ACT.int_shift = 
            piv_eval_par_ACT.int_size_2 / GPIV_SHIFT_FACTOR;
    }
    if ((err_msg = 
         gpiv_piv_count_pivdata_fromimage(piv_data, 
                                          image_par, 
                                          piv_eval_par_ACT))
        != NULL) gpiv_error ("alloc_pivdata_gridgen: %s", err_msg);
    gpiv_null_pivdata(piv_data);
    gpiv_alloc_pivdata(piv_data);
    
    if ((err_msg = 
         gpiv_piv_gridgen(piv_data, 
                          image_par, 
                          piv_eval_par_ACT))
        != NULL) gpiv_error ("alloc_pivdata_gridgen: %s", err_msg);
    
    return err_msg;
}



static void 
report_progress(gboolean print_par,
                int index_y,
                int index_x,
                GpivEvalPar piv_eval_par,
                int sweep,
                gfloat cum_residu,
                PivData *piv_data,
                gint *progress_prev
                )
/*-----------------------------------------------------------------------------
 * Printing the progress of processing if its quantity has been changed
 */
{
    gint progress = 100 * (index_y * piv_data->nx + index_x + 1) / 
        (piv_data->nx * piv_data->ny);
    
    if (print_par && progress != *progress_prev) {
        *progress_prev = progress;
        if (index_y > 0 || index_x > 0)
            printf("\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b");
        if (piv_eval_par.int_scheme == GPIV_ZERO_OFF_FORWARD
            || piv_eval_par.int_scheme == GPIV_ZERO_OFF_CENTRAL
            || piv_eval_par.int_scheme == GPIV_ZERO_OFF_CENTRAL
            || piv_eval_par.int_scheme == GPIV_IMG_DEFORM
            || piv_eval_par.ad_int == 1) {
            printf
                ("\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b"
                 "\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b"
                 "\b\b\b\b\b\b\b\b\b\b\b"
                 "\b\b\b\b\b\b\b\b\b\b\b\b\b\b"
                 );
            printf("sweep #%2d, int_size = %d int_shift = %d residu = %.3f: ", 
                   sweep, piv_eval_par.int_size_1, piv_eval_par.int_shift, cum_residu);
        }
        printf("%3d %%", progress);
        fflush(stdout);
    }
}



static void
interrogate(GpivPivData * piv_data,
	    guint16 **img1, 
            guint16 **img2, 
	    GpivImagePar image_par, 
            GpivEvalPar piv_eval_par,
            GpivValidPar piv_valid_par
            )
/* ----------------------------------------------------------------------------
 * PIV interrogation of an image pair at an entire grid or single point
 */
{
    char *err_msg = NULL;
    int index_x = 0, index_y = 0;

    guint16 **img1_ACT, **img2_ACT;
    GpivPivData lgpd, *piv_data_ACT;
    GpivEvalPar piv_eval_par_ACT;
    
    float **intreg1;
    float **intreg2;
    int int_size_0;

    Covariance cov, w_k;
    int sweep = 1;
    gboolean sweep_last = FALSE, sweep_stop = FALSE; 
    gboolean grid_last = FALSE, isi_last = FALSE, cum_residu_reached = FALSE;
    gfloat sum_dxdy = 0.0, sum_dxdy_old = 0.0, cum_residu = 914;
    gint progress = 0;


    if (print_par) printf("\n");
/*
 * Testing parameters on consistency and initializing derived 
 * parameters/variables
 */
    if ((err_msg = 
         gpiv_piv_test_parameter(&image_par, &piv_eval_par))
        != NULL) gpiv_error ("%s: %s", RCSID, err_msg);

/*
 * Local (actualized) parameters
 * Setting initial parameters and variables for adaptive grid and 
 * Interrogation Area dimensions
 */
    gpiv_piv_cp_parameters(piv_eval_par, &piv_eval_par_ACT, TRUE, FALSE);
    if (piv_eval_par.int_scheme == GPIV_ZERO_OFF_FORWARD
        || piv_eval_par.int_scheme == GPIV_ZERO_OFF_CENTRAL
	|| piv_eval_par.int_scheme == GPIV_IMG_DEFORM
        || piv_eval_par.ad_int) {
        piv_eval_par_ACT.int_size_1 = piv_eval_par_ACT.int_size_2;
        sweep_last = FALSE;
    } else {
        sweep_last = TRUE;
    }
    
    if (piv_eval_par.int_shift < piv_eval_par.int_size_2 / 
        GPIV_SHIFT_FACTOR) {
        piv_eval_par_ACT.int_shift = 
            piv_eval_par_ACT.int_size_2 / GPIV_SHIFT_FACTOR;
    }
    
    if (piv_eval_par.int_scheme == GPIV_IMG_DEFORM) {
        img1_ACT = gpiv_alloc_img(image_par);
        img2_ACT = gpiv_alloc_img(image_par);
        gpiv_cp_img(img1, img1_ACT, image_par);
        gpiv_cp_img(img2, img2_ACT, image_par);
        piv_data_ACT = &lgpd;
        gpiv_null_pivdata(piv_data_ACT);
        piv_data_ACT->nx = piv_data->nx;
        piv_data_ACT->ny = piv_data->ny;
        gpiv_alloc_pivdata(piv_data_ACT);
        gpiv_cp_pivdata(piv_data, piv_data_ACT);
        gpiv_0_pivdata(piv_data_ACT);
    } else {
        img1_ACT = img1;
        img2_ACT = img2;
        piv_data_ACT = piv_data;
    }

/*
 * Reads eventually existing fftw wisdom
 */
    gpiv_fread_fftw_wisdom(1);
    gpiv_fread_fftw_wisdom(-1);

    while (sweep <= GPIV_MAX_EVAL_SWEEP 
           && !sweep_stop) {

/*
 * Memory allocation of interrogation area's and packed interrogation area 
 * arrays. Define weight kernel values
 */
        int_size_0 = GPIV_ZEROPAD_FACT * piv_eval_par_ACT.int_size_2;
	intreg1 = gpiv_matrix (int_size_0, int_size_0);
	intreg2 = gpiv_matrix(int_size_0, int_size_0);
        
	gpiv_piv_bounds_cov(&cov, int_size_0, image_par);
	gpiv_piv_bounds_cov(&w_k, int_size_0, image_par);
	cov.z = gpiv_matrix_index(cov.z_rl, cov.z_rh, cov.z_cl, cov.z_ch);
	w_k.z = gpiv_matrix_index(w_k.z_rl, w_k.z_rh, w_k.z_cl, w_k.z_ch);
        
        if (piv_eval_par_ACT.int_scheme == GPIV_LK_WEIGHT) {
            gpiv_piv_weight_kernel_lin(&w_k, int_size_0);
        } else {
            gpiv_piv_weight_kernel_1(&w_k);
        }
        
/*
 * Interrogates a single point
 */
	if (piv_eval_par_ACT.int_geo == GPIV_POINT) {
            if ((err_msg = 
                 gpiv_piv_interr_reg(0, 
				     0, 
				     img1_ACT, 
				     img2_ACT, 
				     intreg1, 
				     intreg2, 
				     &cov, 
				     &w_k, 
				     piv_data_ACT,
				     sweep, 
				     sweep_last, 
				     image_par,
				     piv_eval_par_ACT))
                != NULL) gpiv_error ("%s: %s", RCSID, err_msg);
            
	} else {

/*
 * Interrogates at a grid of points within the Area Of Interest of the image
 */
            for (index_y = 0; index_y < piv_data_ACT->ny; index_y++) {
                for (index_x = 0; index_x < piv_data_ACT->nx; index_x++) {

/*
 * Evaluates an interrogation area.
 */
                    if ((err_msg = 
                         gpiv_piv_interr_reg(index_y, 
					     index_x, 
					     img1_ACT,
					     img2_ACT,
					     intreg1, 
					     intreg2, 
					     &cov, 
					     &w_k,
					     piv_data_ACT, 
					     sweep, 
					     sweep_last,
					     image_par,
					     piv_eval_par_ACT))
                        != NULL) gpiv_error ("%s: %s", RCSID, err_msg);

/*
 * Printing the progress of processing 
 */
                    report_progress(print_par, 
                                    index_y,
                                    index_x, 
                                    piv_eval_par_ACT,
                                    sweep, 
                                    cum_residu,
                                    piv_data_ACT, 
                                    &progress);

                }
            }
        }

/*
 * Freeing memory: smaller sizes are eventually needed for a next sweep
 */
	gpiv_free_matrix(intreg1);
	gpiv_free_matrix(intreg2);
	gpiv_free_matrix_index(cov.z, cov.z_rl, cov.z_rh, cov.z_cl, cov.z_ch);
	gpiv_free_matrix_index(w_k.z, w_k.z_rl, w_k.z_rh, w_k.z_cl, w_k.z_ch);

	if (sweep_last) {
            sweep_stop = TRUE;
/*             g_message("gpiv_rr:: sweep_stop=TRUE"); */
        }

        if (piv_eval_par.int_scheme == GPIV_IMG_DEFORM
            || piv_eval_par.int_scheme == GPIV_ZERO_OFF_FORWARD
            || piv_eval_par.int_scheme == GPIV_ZERO_OFF_CENTRAL) {
/*
 * Test on outliers
 */
            GpivBinData klass;
            GpivLinRegData linreg;
            GpivPivData gpd;
            
            gpiv_null_pivdata(&gpd);
            gpd.nx = piv_data_ACT->nx;
            gpd.ny = piv_data_ACT->ny;
            gpiv_alloc_pivdata(&gpd);
            if ((err_msg = 
                 gpiv_valid_errvec(image_par, 
                                   piv_eval_par,
                                   piv_valid_par, 
                                   *piv_data_ACT, 
                                   &gpd, 
                                   &klass, 
                                   &linreg, 
                                   img1, 
                                   img2,
                                   TRUE))
                != NULL) gpiv_error ("%s: %s", RCSID, err_msg);
            gpiv_cp_pivdata(&gpd, piv_data_ACT);
            gpiv_free_pivdata(&gpd);


            if (piv_eval_par.int_scheme == GPIV_IMG_DEFORM) {
/*
 * Deformation of original image with updated estimators.
 */
                gpiv_cp_img(img1, img1_ACT, image_par);
                gpiv_cp_img(img2, img2_ACT, image_par);
                gpiv_add_dxdy_pivdata(piv_data_ACT, piv_data);
                gpiv_0_pivdata(piv_data_ACT);
                gpiv_imgproc_deform(piv_data,
                                    &image_par,
                                    img1, 
                                    img2,
                                    img1_ACT,
                                    img2_ACT);

                if (sweep_last && print_par) {
                    printf("\n");
                    if ((err_msg = 
                         gpiv_eval_write_deformed_image(img1_ACT, img2_ACT, 
                                                        image_par))
                        != NULL) {
                        g_error ("%s", err_msg);
                    }
                }
            }

/*
 * Check the cumulative residu for convergence
 * if final grid has been reached
 */
            if (isi_last && grid_last) {
                sum_dxdy_old = sum_dxdy;
                sum_dxdy = 0.0;
                gpiv_sum_dxdy_pivdata(*piv_data, &sum_dxdy);
                cum_residu = fabsf((sum_dxdy - sum_dxdy_old) / 
                             ((gfloat)piv_data->nx * (gfloat)piv_data->ny/*  * 2.0 */));
                if (cum_residu < GPIV_CUM_RESIDU_MIN) {
/*                     g_message("gpiv_rr:: cum_residu < GPIV_CUM_RESIDU_MIN"); */
                    cum_residu_reached = TRUE;
                }
            }
	} else {
            cum_residu_reached = TRUE;
        }

/*
 * Adaptive grid and interrogation size
 */
        if (/* piv_eval_par.ad_int && */
            piv_eval_par_ACT.int_shift > piv_eval_par.int_shift
            && !sweep_stop) {
            gpiv_piv_gridadapt(&image_par,
                               piv_eval_par, 
                               &piv_eval_par_ACT, 
                               piv_data, 
                               sweep, 
                               &grid_last);
            if (piv_eval_par.int_scheme == GPIV_IMG_DEFORM) {
                gpiv_free_pivdata(piv_data_ACT);
                gpiv_null_pivdata(piv_data_ACT);
                piv_data_ACT->nx = piv_data->nx;
                piv_data_ACT->ny = piv_data->ny;
                gpiv_alloc_pivdata(piv_data_ACT);
                gpiv_cp_pivdata(piv_data, piv_data_ACT);
                gpiv_0_pivdata(piv_data_ACT);
            }
        } else {
            grid_last = TRUE;
        }

        gpiv_piv_isizadapt(piv_eval_par, 
                           &piv_eval_par_ACT, 
                           &isi_last);
        
        if (cum_residu_reached && isi_last && grid_last) {
            sweep_last = TRUE;
            if (piv_eval_par.int_scheme == GPIV_ZERO_OFF_FORWARD
                || piv_eval_par.int_scheme == GPIV_ZERO_OFF_CENTRAL) {
                piv_eval_par_ACT.ifit = piv_eval_par.ifit;
            }
        }
        sweep++;
    }

/*
 * Writes existing fftw wisdom
 */
    gpiv_fwrite_fftw_wisdom(1);
    gpiv_fwrite_fftw_wisdom(-1);

    if (piv_eval_par.int_scheme == GPIV_IMG_DEFORM) {
        gpiv_free_img(img1_ACT, image_par);
        gpiv_free_img(img2_ACT, image_par);
        gpiv_free_pivdata(piv_data_ACT);
    }
    
    if (print_par)
	printf("\n");
}



int 
main(int argc, 
     char *argv[]
     )
/*-----------------------------------------------------------------------------
 * Main routine of rr to calculate estimators of particle image displacements
 * from PIV images 
 */
{
    char *err_msg = NULL;
    FILE *fp_par_out = NULL;
    char fname[GPIV_MAX_CHARS],
        fname_in[GPIV_MAX_CHARS],
        fname_header[GPIV_MAX_CHARS],
        fname_out[GPIV_MAX_CHARS],
        fname_int_first[GPIV_MAX_CHARS],
        fname_int_second[GPIV_MAX_CHARS],
        fname_cov[GPIV_MAX_CHARS],
        fname_parameter[GPIV_MAX_CHARS];
    char c_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS];
    guint16 **img1 = NULL, **img2 = NULL;
    int return_val;
    
    GpivPivData piv_data;
    GpivImagePar image_par;
    GpivEvalPar piv_eval_par, piv_eval_par_default;
    GpivValidPar piv_valid_par, piv_valid_par_default;
    
    gpiv_null_pivdata(&piv_data);
        
/* 
 * Initializing parameters
 */
    gpiv_img_parameters_logic(&image_par, FALSE);
    gpiv_piv_parameters_logic(&piv_eval_par, FALSE);
    gpiv_piv_default_parameters(&piv_eval_par_default, TRUE);
    gpiv_valid_parameters_logic(&piv_valid_par, FALSE);
    gpiv_valid_default_parameters(&piv_valid_par_default, TRUE);

    command_args(argc, 
                 argv, 
                 fname, 
                 &image_par, 
                 &piv_eval_par,
                 &piv_valid_par);
    if (print_par) {
        printf("\n# %s\n# Command line options:\n", RCSID);
        gpiv_img_print_parameters(image_par);
        gpiv_piv_print_parameters(piv_eval_par);
        gpiv_valid_print_parameters(piv_valid_par);
    }
    
    if (fname_logic) {
        gpiv_io_make_fname(fname, 
                           GPIV_EXT_PAR, 
                           fname_parameter);
        if (print_par)
            printf("\n# Parameters written to: %s", 
                   fname_parameter);
        if ((fp_par_out = fopen(fname_parameter, "w")) == NULL) {
            gpiv_error("%s error: failure opening %s for input",
                       RCSID, fname_parameter);
        }
        
/*
 * Write command-line args to parameterfile
 */
        fprintf(fp_par_out, "\n# %s\n# Command line options:\n", RCSID);
        gpiv_img_fprint_parameters(fp_par_out, 
                                   image_par);
        gpiv_piv_fprint_parameters(fp_par_out, 
                                   piv_eval_par);
/*
 * Creates image header filename (cannot be included in make_fname()!!)
 */
        gpiv_io_make_fname(fname, 
                           GPIV_EXT_HEADER, 
                           fname_header);
        fprintf(fp_par_out, "\n# image header is: %s\n", fname_header);
        
/*
 * Reads image header data from file.h and/or resource files 
 * if not overridden by the commandline options
 */
        gpiv_scan_parameter("", fname_header, &image_par, print_par);
        gpiv_scan_parameter(GPIV_IMAGE_PAR_KEY, PARFILE, &image_par, 
                            print_par);
        gpiv_scan_resourcefiles(GPIV_IMAGE_PAR_KEY, &image_par, print_par);
        gpiv_img_fprint_parameters(fp_par_out, image_par);

/*
 * Reads piv and valid parameters from PARFILE and/or resource files 
 * if not overridden by the commandline options
 * Prints tp parameter file
 */
        gpiv_scan_parameter(GPIV_EVAL_PAR_KEY, PARFILE, &piv_eval_par, 
                            print_par);
        gpiv_scan_resourcefiles(GPIV_EVAL_PAR_KEY, &piv_eval_par, print_par);
        gpiv_piv_fprint_parameters(fp_par_out, piv_eval_par);

        gpiv_scan_parameter(GPIV_VALID_PAR_KEY, PARFILE, &piv_valid_par, 
                            print_par);
        gpiv_scan_resourcefiles(GPIV_VALID_PAR_KEY, &piv_valid_par, print_par);
        gpiv_valid_fprint_parameters(fp_par_out, piv_valid_par);

    } else {
/*
 * use stdin, stdout
 * Reads image header data from resource files 
 * Reads piv and valid parameters from PARFILE and/or resource files 
 * if not overridden by the commandline options
 */
        gpiv_scan_parameter(GPIV_IMAGE_PAR_KEY, PARFILE, &image_par, 
                            print_par);
        gpiv_scan_resourcefiles(GPIV_IMAGE_PAR_KEY, &image_par, print_par);

        gpiv_scan_parameter(GPIV_EVAL_PAR_KEY, PARFILE, &piv_eval_par, 
                            print_par);
        gpiv_scan_resourcefiles(GPIV_EVAL_PAR_KEY, &piv_eval_par, print_par);

        gpiv_scan_parameter(GPIV_VALID_PAR_KEY, PARFILE, &piv_valid_par, 
                            print_par);
        gpiv_scan_resourcefiles(GPIV_VALID_PAR_KEY, &piv_valid_par, print_par);
    }

/*
 * Check if all parameters have been read
 * Else, set as defaults
 */
    if ((err_msg = 
         gpiv_piv_check_parameters_read(&piv_eval_par, piv_eval_par_default))
        != NULL) gpiv_warning ("%s: %s", RCSID, err_msg);
    
    if ((err_msg = 
         gpiv_valid_check_parameters_read(&piv_valid_par, 
                                          piv_valid_par_default))
        != NULL) gpiv_warning ("%s: %s", RCSID, err_msg);
    
    if (fname_logic == FALSE && gnuplot == 1) 
        gpiv_error("%s: -f filename has to be used in combination "
                   "with 'gnuplot'", 
                   RCSID);
    
/*
 * memory allocation of images
 */
    img1 = gpiv_alloc_img(image_par);
    if (image_par.x_corr) img2 = gpiv_alloc_img(image_par);

/*
 * reads image in binary format
 * interrogates and writes and displaying PIV data
 * Free memory of images and data
 */
    if (fname_logic) {
	if ((return_val = make_fname(fp_par_out, fname, fname_in,
                                     fname_int_first, fname_int_second,
                                     fname_cov, fname_out, piv_eval_par))
	    != 0) {
	    gpiv_error("%s: Failure calling make_fname", RCSID);
	}
        
        if ((err_msg = 
             gpiv_fread_image(fname_in, img1, img2, image_par))
            != NULL) gpiv_error ("%s: %s", RCSID, err_msg);
        
    } else {
	gpiv_read_image(img1, img2, image_par);
    }
    
    alloc_pivdata_gridgen(&piv_data, image_par, piv_eval_par);
    interrogate(&piv_data, img1, img2, image_par, piv_eval_par, piv_valid_par);

    if (piv_eval_par.print_piv == 1 && fname_logic == TRUE) {
        gpiv_write_pivdata(&piv_data, c_line, 0, 0, RCSID);
    } else if (fname_logic == TRUE) {
        if ((err_msg = 
             gpiv_fwrite_pivdata(fname_out, &piv_data, c_line, 0, 0, RCSID))
            != NULL) gpiv_error ("%s: %s", RCSID, err_msg);
        
    } else {
	gpiv_write_pivdata(&piv_data, c_line, 0, 0, RCSID);
    }
    
    
   if (gnuplot) gpiv_piv_gnuplot(fname, fname, gnuplot_scale, 
                                 GNUPLOT_DISPLAY_COLOR, GNUPLOT_DISPLAY_SIZE,
                                 image_par, piv_eval_par, piv_data, RCSID);
    

    gpiv_free_pivdata(&piv_data);
    gpiv_free_img(img1, image_par);
    if (image_par.x_corr == 1) {
        gpiv_free_img(img2, image_par);
    }
    if (fname_logic) fclose(fp_par_out);
    

    exit(0);
}
