#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# DESCRIPTION:
# A wxPython based user interface driver for GNUe forms.
#
# NOTES:
#
# $Id: UIdriver.py 7064 2005-02-25 13:55:58Z kilo $

import sys, os
import string

from gnue.forms.uidrivers._base import Exceptions

try:
  from wxPython import __version__ as WXVERSION
except ImportError:
  raise Exceptions.DriverNotSupported, _("The GNUe-Forms wx driver requires wxPython.")

from wxPython.wx import *

from gnue.common import events
from gnue.common.apps import GDebug
from gnue.common.apps import GConfig
from gnue.common.utils.TextUtils import lineWrap

from gnue.forms.GFForm import *

from gnue.forms.uidrivers._commonGuiToolkit import UIdriver as commonToolkit

from gnue.forms.uidrivers.wx.GFwxApp import *
from gnue.forms.uidrivers.wx.UIWXSplashScreen import *
from gnue.forms.uidrivers.wx.widgets._base  import *
from gnue.forms.uidrivers.wx.common import _eventObjTowxWindow
from PrintForm import printForm

from common import wxEncode


#
# GFUserInterface
#
# The public interface to the User Interface
# All UIs must provide this class
#
class GFUserInterface(commonToolkit.GFUserInterface):

  _MBOX_KIND = {'Info'    : {'style': wxICON_INFORMATION | wxOK,
                             'title': _("Information")},
                'Warning' : {'style': wxICON_EXCLAMATION | wxOK,
                             'title': _("Warning")},
                'Question': {'style': wxICON_QUESTION    | wxYES_NO,
                             'title': _("Question")},
                'Error'   : {'style': wxICON_ERROR       | wxOK,
                             'title': _("Error")}}

  _RESPONSE = {wxID_OK    : True,
               wxID_YES   : True,
               wxID_NO    : False,
               wxID_CANCEL: None}

  def _initialize(self):

    ################################################################
    # WX Specifics
    ################################################################

    self._disabledColour = wxLIGHT_GREY

    self._wxapp = getWxApp()

    #
    # SplashScreen
    #
    if not self._disableSplash:
      self.splash = UIWXSplashScreen()

    #
    # Create a dummy window used to compute sizes
    #
    dummyWindow = wxFrame(NULL, -1, "", wxDefaultPosition)
    initFont(dummyWindow,1)

    panel = wxPanel(dummyWindow, -1)
    self._disabledColour = panel.GetBackgroundColour()

    if gConfigForms('focusColor'):
      colorStr = gConfigForms('focusColor')
      rgbR = int(colorStr[:2], 16)
      rgbG = int(colorStr[2:4], 16)
      rgbB = int(colorStr[4:6], 16)
      self._focusColour = wxColour(rgbR, rgbG, rgbB)
    else:
      self._focusColour = None

    #
    # Find the max size of printable characters
    #  used to setup the grid
    #
    maxWidth, maxHeight, maxDescent, maxLeading = [0,0,0,0]
    checkchars = string.letters
    for letter in checkchars:
      width,height,descent,leading = dummyWindow.GetFullTextExtent(letter)
      maxWidth = max(maxWidth,width)
      maxHeight = max(maxHeight,height)
      maxDescent = max(maxDescent,descent)
      maxLeading = max(maxLeading,leading)

    self.textWidth    = int(maxWidth+maxLeading)  # The pixel width of text inside a widget
    self.textHeight   = int(maxHeight+maxDescent) # The pixel height of text inside a widget
    self.widgetWidth  = self.textWidth            # The pixel width of a 1 char widget (for things like buttons)
    self.widgetHeight = self.textHeight + 3       # The pixel height of a 1 char widget (for things like buttons)

    #
    # Close dummy window so app doesn't hang when all other windows closed
    #
    dummyWindow.Close()


  #############################################################################
  #
  # Private UIBase support functions
  #
  # Called in UIbase functions to perform UI interaction with this specific
  # widget set.
  #

  #
  # _exit
  #
  # Tells the application to close it's main window
  #
  def _exit(self,formName):
    exitApp = 1
    for child in self._children:
      if child._form.name == formName:
        child.mainWindow.Hide()

      exitApp = exitApp and not child.mainWindow.IsShown()

    if exitApp:
      for child in self._children:
        child.mainWindow.Close(true)

  def _beep(self):
    wxBell()

  #############################################################################
  #
  # Incoming Event Processors
  #
  # Processes the incoming events from other objects
  # From here down should be nothing but eventListeners listed

  #
  # mainLoop
  #
  # The primary loop of the user interface.  Called once the UI is
  # fully activated
  #
  def mainLoop(self):
    self._wxapp.MainLoop() # simply call the wxApp's MainLoop method

  #
  # formAlert
  #
  # Rings a bell and alters the statusbar to display
  # a line of text
  #
  def formAlert(self, event):
    wxBell()
    ui = self._gfObjToUIWidget[event._form]
    if ui.statusBar is not None:
      ui.statusBar.SetStatusText (wxEncode (event.data), 0)

  #
  # Called whenever forms goes into a "wait" state in which user cannot
  # interact with interface (e.g., while waiting for a query or a commit)
  #
  def beginWait (self, event):
    wxBeginBusyCursor()

  #
  # Called whenever forms leaves a "wait" state
  #
  def endWait (self, event):
    wxEndBusyCursor()


  #
  # Clipboard routines
  #
  # If a particular UI has a system-wide clipboard,
  # these methods should be overridden to use that
  # clipboard.
  #
  def getClipboardContents(self, event):

    if wxTheClipboard.Open():
      data = wxTextDataObject()
      success = wxTheClipboard.GetData(data)
      wxTheClipboard.Close()
    else:
      success = 0
      GDebug.printMesg(1,'Unable to open clipboard for read')

    if success:
      value = data.GetText()
    else:
      GDebug.printMesg(1,'Unable to obtain clipboard contents. Defaulting to Empty.')
      value = None

    GDebug.printMesg(6, "Getting clipboard value '%s'" % value)
    event.__result__ = value


  def setClipboardContents(self, event):

    GDebug.printMesg(6,"Setting clipboard '%s'" % event.text)

    if wxTheClipboard.Open():
      value = wxTheClipboard.SetData(wxTextDataObject(event.text))
      GDebug.printMesg(6,"Set Clipboard Status: %s" % value)
      wxTheClipboard.Close()

    else:
      GDebug.printMesg(6,'Unable to open clipboard for write')


  def printout(self, event):
    printForm(self._form, self)

  def setTitle(self, event):
    ui = self._gfObjToUIWidget[event._form]
    try:
      ui.mainWindow.SetTitle(event.title)
    except AttributeError:
      pass

  #############################################################################
  #
  # Internal Event Processors
  #
  # Processes the events from the widget set
  #

  # ---------------------------------------------------------------------------
  # Show a message dialog
  # ---------------------------------------------------------------------------

  def _showMessage (self, message, kind = 'Info', title = None, cancel = False):

    style = self._MBOX_KIND [kind]['style']
    if cancel:
      style |= wxCANCEL

    if title is None:
      title = self._MBOX_KIND [kind]['title']
    dialog = wxMessageDialog (None, wxEncode (lineWrap(message,80)), wxEncode (title), style)
    try:
      result = dialog.ShowModal ()
      return self._RESPONSE [result]

    finally:
      dialog.Destroy ()


  # ---------------------------------------------------------------------------
  # Show an exception
  # TODO: please implement a better dialog box, i.e. add a button for
  #       detail-display and so on
  # ---------------------------------------------------------------------------

  def _showException (self, group, name, message, detail):

    self._showMessage (wxEncode (detail), 'Error')
