------------------------------------------------------------------------------
--  Ada95 Interface to Oracle RDBMS                                         --
--  Copyright (C) 2000-2006 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-servers.adb,v 1.14 2008/07/03 06:22:48 vagul Exp $

with Interfaces.C;
with OCI.Thread;

package body OCI.Thick.Servers is

   use type Interfaces.C.int;

   ------------
   -- Attach --
   ------------

   function Attach (Name : String) return Server is
      H  : OCIHandle := Alloc_Handle (Thread.Environment, OCI_HTYPE_SERVER);
   begin
      Check_Error (OCIServerAttach
                     (OCIServer (H),
                      Thread.Error,
                      Interfaces.C.To_C (Name),
                      Name'Length));

      return (RF.Controlled_Reference with
              Handle      => H,
              Environment => Thread.Synch.Environment);
   exception
      when Lib_Error =>
         Free (H, OCI_HTYPE_SERVER);
         raise;
   end Attach;

   -----------
   -- Break --
   -----------

   procedure Break (It : Server) is
   begin
      Check_Error (OCIBreak (Handle (It), Thread.Error));
   end Break;

   -------------
   -- Destroy --
   -------------

   procedure Destroy (It : in out Server) is
      use type OCI.Lib.OCIHandle;
   begin
      if It.Handle /= Empty_Handle then
         if OCIServerDetach (OCIServer (It.Handle), Thread.Error)
            /= OCI_SUCCESS
         then
            null; -- ???
         end if;

         Free (It.Handle, OCI_HTYPE_SERVER);
      end if;
   end Destroy;

   -----------------
   -- Is_Blocking --
   -----------------

   function Is_Blocking (Item : in Server) return Boolean is
      function Get_Attrib is new Get_Attr_G (Ub1);
      use type Ub1;
   begin
      return Get_Attrib
               (Item.Handle, OCI_HTYPE_SERVER, OCI_ATTR_NONBLOCKING_MODE)
                = 0;
   end Is_Blocking;

   -----------
   -- Reset --
   -----------

   procedure Reset (It : Server) is
   begin
      Check_Error (OCIReset (Handle (It), Thread.Error));
   end Reset;

   --------------------
   -- Server_Version --
   --------------------

   function Server_Version (Item : in Server) return String is
      use type Ub4;
      Buff : aliased Text := (0 .. 511 => C.nul);
   begin
      Check_Error
        (OCIServerVersion
           (Hndlp    => Item.Handle,
            Errhp    => Thread.Error,
            Bufp     => CStr.To_Chars_Ptr (Buff'Unchecked_Access),
            Bufsz    => Buff'Length - 1,
            Hndltype => OCI_HTYPE_SERVER));
      return C.To_Ada (Buff);
   end Server_Version;

   ------------------
   -- Set_Blocking --
   ------------------

   procedure Set_Blocking (Item : in out Server; Mode : in Boolean) is
      B1 : aliased constant Ub1 := Boolean'Pos (not Mode);
   begin
      if Is_Blocking (Item) = Mode then
         return;
      end if;

      Set_Attr
        (Item.Handle,
         OCI_HTYPE_SERVER,
         OCI_ATTR_NONBLOCKING_MODE,
         Value => B1'Address);

      if Is_Blocking (Item) /= Mode then
         raise Program_Error with "Set blocking mode error.";
      end if;
   end Set_Blocking;

end OCI.Thick.Servers;
