/*--------------------------------------------------------------------
 *	$Id: grdedit.c,v 1.27 2006/04/13 08:34:41 pwessel Exp $
 *
 *	Copyright (c) 1991-2006 by P. Wessel and W. H. F. Smith
 *	See COPYING file for copying and redistribution conditions.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; version 2 of the License.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	Contact info: gmt.soest.hawaii.edu
 *--------------------------------------------------------------------*/
/*
 * grdedit reads an existing grdfile and takes command line
 * arguments to redefine some of the grdheader parameters:
 *
 *	x_min/x_max OR x_inc (the other is recomputed)
 *	y_min/y_max OR y_inc (the other is recomputed)
 *	z_scale_factor/z_add_offset
 *	x_units/y_units/z_units
 *	title/command/remark
 *
 * Author:	Paul Wessel
 * Date:	9-SEP-1998
 * Version:	4
 * Modified:	19-FEB-2000 PW: Added -V option
 *		24-FEB-2006 PW: Added -T option
 */
 
#include "gmt.h"

struct GRDEDIT_CTRL {
	struct A {	/* -A */
		BOOLEAN active;
	} A;
	struct D {	/* -D<xname>/<yname>/<zname>/<scale>/<offset>/<title>/<remark> */
		BOOLEAN active;
		char *information;
	} D;
	struct N {	/* N<xyzfile> */
		BOOLEAN active;
		char *file;
	} N;
	struct S {	/* -S */
		BOOLEAN active;
	} S;
	struct T {	/* -T */
		BOOLEAN active;
	} T;
};

int main (int argc, char **argv)
{
	BOOLEAN error = FALSE, got_region = FALSE;

	int i, j, k, n_expected_fields, n_fields, n_data;

	float *a;

	double w, e, s, n, shift_amount = 0.0, f, *in;

	char *grdfile, buffer[BUFSIZ];
	char *registration[2] = {"gridline", "pixel"};

	FILE *fp;

	struct GRD_HEADER grd;
	struct GRDEDIT_CTRL *Ctrl;

	void *New_Grdedit_Ctrl (), Free_Grdedit_Ctrl (struct GRDEDIT_CTRL *C);
	
	argc = GMT_begin (argc, argv);

	Ctrl = (struct GRDEDIT_CTRL *)New_Grdedit_Ctrl ();	/* Allocate and initialize a new control structure */
	
	grdfile = CNULL;
	w = e = s = n = 0.0;

	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
				/* Common parameters */

				case 'R':
				case 'H':
				case 'V':
				case ':':
  				case 'b':
  				case 'f':
				case '\0':
					error += GMT_parse_common_options (argv[i], &w, &e, &s, &n);
					if (argv[i][1] == 'R') got_region = TRUE;
					break;

				/* Supplemental parameters */

				case 'D':
					Ctrl->D.active = TRUE;
					Ctrl->D.information = strdup (&argv[i][2]);
					break;
				case 'A':
					Ctrl->A.active = TRUE;
					break;
				case 'N':
					Ctrl->N.active = TRUE;
					Ctrl->N.file = strdup (&argv[i][2]);
					break;
				case 'S':
					Ctrl->S.active = TRUE;
					break;
				case 'T':
					Ctrl->T.active = TRUE;
					break;
				default:
					error = TRUE;
					GMT_default_error (argv[i][1]);
					break;
			}
		}
		else
			grdfile = argv[i];
	}

	if (argc == 1 || GMT_give_synopsis_and_exit) {
		fprintf (stderr, "grdedit %s - Modifying the header in a netCDF grdfile\n\n", GMT_VERSION);
		fprintf (stderr, "usage: grdedit grdfile [-A] [%s]\n", GMT_GRDEDIT);
		fprintf (stderr, "\t[%s] [N<xyzfile>] [%s] [-S] [-T] [-V]\n", GMT_H_OPT, GMT_Rgeo_OPT);
		fprintf (stderr, "\t[%s] [%s] [%s]\n", GMT_t_OPT, GMT_bi_OPT, GMT_f_OPT);

		if (GMT_give_synopsis_and_exit) exit (EXIT_FAILURE);

		fprintf (stderr, "\tgrdfile is file to be modified\n");
		fprintf (stderr, "\n\tOPTIONS:\n");
		fprintf (stderr, "\t-A will adjust dx/dy to be compatible with the files domain or -R\n");
		fprintf (stderr, "\t-D to enter information.  Specify '=' to get default value\n");
		GMT_explain_option ('H');
		fprintf (stderr, "\t-N <file> has new xyz values to replace existing grid nodes\n");
		GMT_explain_option ('R');
		fprintf (stderr, "\t-S For global grids of 360 degree longitude range.\n");
		fprintf (stderr, "\t   Will rotate entire grid to coincide with new borders in -R\n");
		fprintf (stderr, "\t-T Toggle header from grid-line to pixel-registered grid or vice versa.\n");
		fprintf (stderr, "\t   This shrinks -R by 0.5*{dx,dy} going from pixel to grid-line registration\n");
		fprintf (stderr, "\t   and expands -R by 0.5*{dx,dy} going from grid-line to pixel registration.\n");
		GMT_explain_option ('V');
		GMT_explain_option (':');
		GMT_explain_option ('i');
		GMT_explain_option ('n');
		GMT_explain_option ('f');
		GMT_explain_option ('.');
		exit (EXIT_FAILURE);
	}

	if (Ctrl->S.active && Ctrl->A.active) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -S option:  Incompatible with -A\n", GMT_program);
		error++;
	}
	if (Ctrl->S.active && Ctrl->T.active) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -S option:  Incompatible with -T\n", GMT_program);
		error++;
	}
	if (Ctrl->S.active && Ctrl->N.active) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -S option:  Incompatible with -N\n", GMT_program);
		error++;
	}
	if (Ctrl->S.active && !got_region) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -S option:  Must also specify -R\n", GMT_program);
		error++;
	}
	if (Ctrl->S.active && fabs (w-e-360.0) > GMT_CONV_LIMIT) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -S option:  -R longitudes must span exactly 360 degrees\n", GMT_program);
		error++;
	}
	if (!grdfile) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Must specify input file\n", GMT_program);
		error++;
	}
	if (Ctrl->N.active) {
		if (!Ctrl->N.file) {
			fprintf (stderr, "%s: GMT SYNTAX ERROR -N option:  Must specify xyz ASCII file\n", GMT_program);
			error++;
		}
		if (GMT_io.binary[GMT_IN] && GMT_io.io_header[GMT_IN]) {
			fprintf (stderr, "%s: GMT SYNTAX ERROR.  Binary input data cannot have header -H\n", GMT_program);
			error++;
		}
		if (GMT_io.binary[GMT_IN] && GMT_io.ncol[GMT_IN] == 0) GMT_io.ncol[GMT_IN] = 3;
		if (GMT_io.binary[GMT_IN] && GMT_io.ncol[GMT_IN] < 2) {
			fprintf (stderr, "%s: GMT SYNTAX ERROR.  Binary input data (-bi) must have at least 2 columns\n", GMT_program);
			error++;
		}
	}
	if (error) exit (EXIT_FAILURE);

	GMT_put_history (argc, argv);	/* Update .gmtcommands4 */

	if (!strcmp (grdfile,  "=")) {
		fprintf (stderr, "%s: Piping of grdfile not supported!\n", GMT_program);
		exit (EXIT_FAILURE);
	}

	if (GMT_read_grd_info (grdfile, &grd)) {
		fprintf (stderr, "%s: Error opening file %s\n", GMT_program, grdfile);
		exit (EXIT_FAILURE);
	}

	if (Ctrl->S.active && fabs (grd.x_max - grd.x_min) != 360.0) {
		fprintf (stderr, "%s: Shift only allowed for global grids\n", GMT_program);
		exit (EXIT_FAILURE);
	}

	if (gmtdefs.verbose) fprintf (stderr, "%s: Editing parameters for file %s\n", GMT_program, grdfile);

	/* Decode grd information given, if any */

	if (Ctrl->D.active) {
		if (gmtdefs.verbose) fprintf (stderr, "%s: Decode and change attributes in file %s\n", GMT_program, grdfile);
		GMT_decode_grd_h_info (Ctrl->D.information, &grd);

	}

	if (Ctrl->S.active) {
		shift_amount = w - grd.x_min;
		if (gmtdefs.verbose) fprintf (stderr, "%s: Shifting longitudes in file %s by %g degrees\n", GMT_program, grdfile, shift_amount);
		a = (float *) GMT_memory (VNULL, (size_t)(grd.nx * grd.ny), sizeof (float), GMT_program);
		if (GMT_read_grd (grdfile, &grd, a, 0.0, 0.0, 0.0, 0.0, GMT_pad, FALSE)) {
			fprintf (stderr, "%s: Error reading file %s\n", GMT_program, grdfile);
			exit (EXIT_FAILURE);
		}
		GMT_grd_shift (&grd, a, shift_amount);
		if (GMT_write_grd (grdfile, &grd, a, 0.0, 0.0, 0.0, 0.0, GMT_pad, FALSE)) {
			fprintf (stderr, "%s: Error writing file %s\n", GMT_program, grdfile);
			exit (EXIT_FAILURE);
		}
		GMT_free ((void *)a);
	}
	else if (Ctrl->N.active) {
		if (gmtdefs.verbose) fprintf (stderr, "%s: Replacing nodes using xyz values from file %s\n", GMT_program, Ctrl->N.file);
		if (GMT_io.binary[GMT_IN] && gmtdefs.verbose) {
			char *type[2] = {"double", "single"};
			fprintf (stderr, "%s: Expects %d-column %s-precision binary data\n", GMT_program, GMT_io.ncol[GMT_IN], type[GMT_io.single_precision[GMT_IN]]);
		}
		a = (float *) GMT_memory (VNULL, (size_t)(grd.nx * grd.ny), sizeof (float), GMT_program);
		if (GMT_read_grd (grdfile, &grd, a, 0.0, 0.0, 0.0, 0.0, GMT_pad, FALSE)) {
			fprintf (stderr, "%s: Error reading file %s\n", GMT_program, grdfile);
			exit (EXIT_FAILURE);
		}
		if ((fp = GMT_fopen (Ctrl->N.file, GMT_io.r_mode)) == NULL) {
			fprintf (stderr, "%s: Could not open file %s\n", GMT_program, Ctrl->N.file);
			exit (EXIT_FAILURE);
		}
		if (GMT_io.io_header[GMT_IN]) for (i = 0; i < GMT_io.n_header_recs; i++) GMT_fgets (buffer, BUFSIZ, fp);

		n_expected_fields = (GMT_io.ncol[GMT_IN]) ? GMT_io.ncol[GMT_IN] : 3;

		n_data = 0;
		while ((n_fields = GMT_input (fp, &n_expected_fields, &in)) >= 0 && !(GMT_io.status & GMT_IO_EOF)) {	/* Not yet EOF */
			n_data++;
			if (GMT_io.status & GMT_IO_MISMATCH) {
				fprintf (stderr, "%s: Mismatch between actual (%d) and expected (%d) fields near line %d\n", GMT_program, n_fields,  n_expected_fields, n_data);
				exit (EXIT_FAILURE);
			}
			j = GMT_y_to_j(in[1], grd.y_min, grd.y_inc, grd.xy_off, grd.ny);
			if (j < 0 || j >= grd.ny) continue;
			i = GMT_x_to_i(in[0], grd.x_min, grd.x_inc, grd.xy_off, grd.nx);
			if (i < 0 || i >= grd.nx) continue;
			k = j * grd.nx + i;
			a[k] = (float)in[2];
		}
		if (fp != GMT_stdin) GMT_fclose (fp);

		if (GMT_write_grd (grdfile, &grd, a, 0.0, 0.0, 0.0, 0.0, GMT_pad, FALSE)) {
			fprintf (stderr, "%s: Error writing file %s\n", GMT_program, grdfile);
			exit (EXIT_FAILURE);
		}
		GMT_free ((void *)a);
	}
	else {
		if (got_region) {
			if (gmtdefs.verbose) fprintf (stderr, "%s: Reset region in file %s to %g/%g/%g/%g\n",
				GMT_program, grdfile, w, e, s, n);
			grd.x_min = w;	grd.x_max = e;
			grd.y_min = s;	grd.y_max = n;
			Ctrl->A.active = TRUE;	/* Must ensure -R -I compatibility */
		}
		if (Ctrl->A.active) {
			grd.x_inc = GMT_get_inc (grd.x_min, grd.x_max, grd.nx, grd.node_offset);
			grd.y_inc = GMT_get_inc (grd.y_min, grd.y_max, grd.ny, grd.node_offset);
			if (gmtdefs.verbose) fprintf (stderr, "%s: Reset grid-spacing in file %s to %g/%g\n",
				GMT_program, grdfile, grd.x_inc, grd.y_inc);
		}
		if (Ctrl->T.active) {	/* Grid-line <---> Pixel toggling of the header */
			f = (grd.node_offset) ? -0.5 : +0.5;
			grd.x_min -= f * grd.x_inc;	grd.x_max += f * grd.x_inc;
			grd.y_min -= f * grd.y_inc;	grd.y_max += f * grd.y_inc;
			grd.node_offset = 1 - grd.node_offset;
			if (gmtdefs.verbose) fprintf (stderr, "%s: Toggled registration mode in file %s from %s to %s\n", GMT_program, grdfile, registration[1-grd.node_offset], registration[grd.node_offset]);
			if (gmtdefs.verbose) fprintf (stderr, "%s: Reset region in file %s to %g/%g/%g/%g\n", GMT_program, grdfile, grd.x_min, grd.x_max, grd.y_min, grd.y_max);
		}
		if (GMT_update_grd_info (grdfile, &grd)) {
			fprintf (stderr, "%s: Error updating file %s\n", GMT_program, grdfile);
			exit (EXIT_FAILURE);
		}
	}

	if (gmtdefs.verbose) fprintf (stderr, "%s: File %s updated.\n", GMT_program, grdfile);

	Free_Grdedit_Ctrl (Ctrl);	/* Deallocate control structure */

	GMT_end (argc, argv);

	exit (EXIT_SUCCESS);
}

void *New_Grdedit_Ctrl () {	/* Allocate and initialize a new control structure */
	struct GRDEDIT_CTRL *C;
	
	C = (struct GRDEDIT_CTRL *) GMT_memory (VNULL, 1, sizeof (struct GRDEDIT_CTRL), "New_Grdedit_Ctrl");
	
	/* Initialize values whose defaults are not 0/FALSE/NULL */

	return ((void *)C);
}

void Free_Grdedit_Ctrl (struct GRDEDIT_CTRL *C) {	/* Deallocate control structure */
	if (C->D.information) GMT_free ((void *)C->D.information);	
	if (C->N.file) GMT_free ((void *)C->N.file);	
	GMT_free ((void *)C);	
}
