/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#include "gimageview.h"

#include "gimv_image.h"
#include "fileutil.h"
#include "image_info.h"


GHashTable *ImageInfoTable = NULL;


static ImageInfo *
image_info_new (const gchar *filename)
{
   ImageInfo *info;

   info = g_new0 (ImageInfo, 1);
   g_return_val_if_fail (info, NULL);

   if (!file_exists (filename)) return NULL;

   info->filename = g_strdup (filename);
   info->format = gimv_image_detect_type_by_ext (filename);
   stat (filename, &info->st);
   info->width  = -1;
   info->height = -1;
   info->sync_to_orig_image = FALSE;
   info->ref_count = 1;

   return info;
}


ImageInfo *
image_info_get (const gchar *filename)
{
   ImageInfo *info;
   struct stat st;

   g_return_val_if_fail (filename, NULL);

   if (!ImageInfoTable) {
      ImageInfoTable = g_hash_table_new (g_str_hash, g_str_equal);
   }

   info = g_hash_table_lookup (ImageInfoTable, filename);
   if (!info) {
      info = image_info_new (filename);
      if (!info) return NULL;
      g_hash_table_insert (ImageInfoTable, info->filename, info);
   } else {
      if (stat (filename, &st)) return NULL;
      if (info->st.st_size != st.st_size
	  || info->st.st_mtime != st.st_mtime
	  || info->st.st_ctime != st.st_ctime)
      {
	 memcpy (&info->st, &st, sizeof (struct stat));
	 info->width = -1;
	 info->height = -1;
	 info->sync_to_orig_image = FALSE;
      }
      image_info_ref (info);
   }

   return info;
}


void
image_info_finalize (ImageInfo *info)
{
   guint num;

   g_return_if_fail (info);

   /* remove data from hash table */
   g_hash_table_remove (ImageInfoTable, info->filename);
   if (info->filename)
      g_free (info->filename);
   g_free (info);

   /* destroy hash table if needed */
   num = g_hash_table_size (ImageInfoTable);
   if (num < 1) {
      g_hash_table_destroy (ImageInfoTable);
      ImageInfoTable = NULL;
   }
}


ImageInfo *
image_info_ref (ImageInfo *info)
{
   g_return_val_if_fail (info, NULL);

   info->ref_count++;

   return info;
}


void
image_info_unref (ImageInfo *info)
{
   g_return_if_fail (info);

   info->ref_count--;
   if (info->ref_count < 1)
      image_info_finalize (info);
}


gboolean
image_info_set_data_from_image (ImageInfo *info, GimvImage *image)
{
   gboolean is_exist;

   g_return_val_if_fail (info, FALSE);

   if (!image) return FALSE;

   is_exist = !stat (info->filename, &info->st);
   if (!is_exist) return FALSE;

   gimv_image_get_size (image, &info->width, &info->height);
   info->sync_to_orig_image = TRUE;

   return TRUE;
}
