/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 *  These codes are mostly taken from GTK See.
 *  Bitfields compression, 32bit, and OS2 BMP support was added.
 *  GTK See Copyright (C) 1998 Hotaru Lee <jkhotaru@mail.sti.com.cn> <hotaru@163.net>
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "bmp.h"

#define BI_RGB 0
#define BI_RLE8 1
#define BI_RLE4 2
#define BI_BITFIELDS 3

#define BitSet(byte, bit)  (((byte) & (bit)) == (bit))
#define ReadOK(file,buffer,len)  (fread(buffer, len, 1, file) != 0)
#define ToL(buffer, off)  (buffer[(off)] | buffer[(off)+1]<<8 | buffer[(off)+2]<<16 | buffer[(off)+3]<<24)
#define ToS(buffer, off)  (buffer[(off)] | buffer[(off)+1]<<8);


typedef struct BmpBitFields_Tag
{
   gint r_mask, r_shift, r_bits;
   gint g_mask, g_shift, g_bits;
   gint b_mask, b_shift, b_bits;
} BmpBitFields;


gboolean BmpReadColorMap (FILE *fd,
			  guchar buffer[256][3],
			  gint number,
			  gint size,
			  gint *grey);
gboolean BmpReadBitmasks (FILE *fd,
			  BmpBitFields *bf);
guchar  *BmpReadImage    (FILE *fd,
			  gint len,
			  gint height,
			  guchar cmap[256][3],
			  BmpBitFields *bf,
			  gint ncols,
			  gint bpp,
			  gint compression,
			  gint spzeile,
			  gint grey);

gboolean
bmp_get_header(const gchar *filename,
	       bmp_info *info)
{
   FILE *fd;
   guchar buffer[50];
   gulong bfSize, reserverd, bfOffs, biSize;
	
   fd = fopen (filename, "rb");
   if (!fd) return FALSE;

   /*
    *   ID (2byte)
    *      BM - Windows
    *      BA - OS2
    */
   if (!ReadOK(fd,buffer, 2)
       || !(!strncmp(buffer, "BM", 2) || !strncmp(buffer, "BA", 2)))
   {
      fclose (fd);
      return FALSE;
   }
	
   if (!ReadOK (fd, buffer, 16)) {
      fclose(fd);
      return FALSE;
   }

   bfSize    = ToL(buffer, 0);           /* File Size          (1dword) */
                                         /*    Windows      bfSize = 40 */
                                         /*    OS2          bfSize = 12 */
   reserverd = ToL(buffer, 4);           /* Reserved           (1dword) */
   bfOffs    = ToL(buffer, 8);           /* Bitmap Data Offset (1dword) */
   biSize    = ToL(buffer, 12);          /* Bitmap Header Size (1dword) */

   if (!(biSize == 40 || biSize == 12)) {
      fclose(fd);
      return FALSE;
   }
	
   if (!ReadOK (fd, buffer, 36)) {
      fclose(fd);
      return FALSE;
   }

   /* both Windows and OS2 */
   if (biSize == 40) {
      info -> width   = ToL(buffer, 0x00);  /* width    (Windows - 1dword) */
      info -> height  = ToL(buffer, 0x04);  /* height   (Windows - 1dword) */
   } else if (biSize == 12) {
       info -> width  = ToS(buffer, 0x00);  /*          (OS2     - 1 word) */
       info -> height = ToS(buffer, 0x04);  /*          (OS2     - 1 word) */
   }
   info -> planes  = ToS(buffer, 0x08);     /* Planes             (1 word) */
   info -> bitCnt  = ToS(buffer, 0x0A);     /* Bits Per Pixels    (1 word) */

   /* Windows only */
   if (biSize == 40) {
      info -> compr   = ToL(buffer, 0x0C);  /* Compression        (1dword) */
                                            /*    0 - RGB
					     *    1 - RLE8bit
					     *    2 - RLE4bit
					     *    3 - bitfields
					     */
      info -> sizeIm  = ToL(buffer, 0x10);  /* Bitmap Data Size   (1dword) */
      info -> xPels   = ToL(buffer, 0x14);  /* HResolution        (1dword) */
      info -> yPels   = ToL(buffer, 0x18);  /* VResolution        (1dword) */
      info -> clrUsed = ToL(buffer, 0x1C);  /* Colors             (1dword) */
      info -> clrImp  = ToL(buffer, 0x20);  /* Important Colors   (1dword) */
                                            /* Palette Table      (Palette num * 4byte) */
                                            /* Bitmap Data ... */
   }else {
      info->compr = 0;
   }

   fclose(fd);

   if (info -> bitCnt > 24) return FALSE;

   return TRUE;
}

guchar *
bmp_load(const gchar *filename,
	 gint *widthp,
	 gint *heightp)
{
   FILE *fd;
   guchar buffer[50];
   gulong bfSize, reserverd, bfOffs, biSize;
   gint ColormapSize, SpeicherZeile, Maps, Grey;
   guchar ColorMap[256][3];
   guchar *dest;
   bmp_info cinfo;
   BmpBitFields bf;

   g_return_val_if_fail (filename && widthp && heightp, NULL);

   fd = fopen (filename, "rb");
   if (!fd) return NULL;

   if (!ReadOK(fd, buffer, 2)
       || !(!strncmp (buffer, "BM", 2) || !strncmp (buffer, "BA", 2)))
   {
      fclose(fd);
      return NULL;
   }

   if (!ReadOK (fd, buffer, 16)) {
      fclose(fd);
      return NULL;
   }

   bfSize    = ToL (buffer, 0);
   reserverd = ToL (buffer, 4);
   bfOffs    = ToL (buffer, 8);
   biSize    = ToL (buffer, 12);

   if (!(biSize == 40 || biSize == 12)) {
      fclose (fd);
      return NULL;
   }
	
   if (!ReadOK (fd, buffer, 36)) {
      fclose (fd);
      return NULL;
   }
	
   if (biSize == 40) {          /* Windows */
      cinfo.width   = ToL (buffer, 0x00);
      cinfo.height  = ToL (buffer, 0x04);
   } else if (biSize == 12) {   /* OS2 */
      cinfo.width   = ToS (buffer, 0x00);
      cinfo.height  = ToS (buffer, 0x02);
   }
   cinfo.planes  = ToS (buffer, 0x08);
   cinfo.bitCnt  = ToS (buffer, 0x0A);
   if (biSize == 40) {
      cinfo.compr   = ToL (buffer, 0x0C);
      cinfo.sizeIm  = ToL (buffer, 0x10);
      cinfo.xPels   = ToL (buffer, 0x14);
      cinfo.yPels   = ToL (buffer, 0x18);
      cinfo.clrUsed = ToL (buffer, 0x1C);
      cinfo.clrImp  = ToL (buffer, 0x20);
   } else {
      cinfo.compr   = 0;
   }

   /*
   if (cinfo.bitCnt > 24) {
      fclose(fd);
      return NULL;
   }
   */

   Maps = 4;
   ColormapSize = (bfOffs - biSize - 14) / Maps;
   if ((cinfo.clrUsed == 0) && (cinfo.bitCnt < 24))
      cinfo.clrUsed = ColormapSize;
   if (cinfo.bitCnt == 32 || cinfo.bitCnt == 24 || cinfo.bitCnt == 16) {
      SpeicherZeile = ((bfSize - bfOffs) / cinfo.height);
   } else {
      SpeicherZeile = ((bfSize - bfOffs) / cinfo.height) * (8 / cinfo.bitCnt);
   }

   if (cinfo.compr == BI_BITFIELDS) {   /* BitFields compression */
      if (!BmpReadBitmasks (fd, &bf)) {
	 fclose (fd);
	 return NULL;
      }
   } else {                            /* Get Colormap (1, 4, 8 bpp) */
      if (!BmpReadColorMap (fd, ColorMap, ColormapSize, Maps, &Grey)) {
	 fclose (fd);
	 return NULL;
      }
   }

   /* read image!! */
   dest = BmpReadImage(fd, cinfo.width, cinfo.height, ColorMap, &bf,
		       cinfo.clrUsed, cinfo.bitCnt, cinfo.compr,
		       SpeicherZeile, Grey);
   if (!dest) {
      fclose(fd);
      return NULL;
   }

   fclose(fd);

   *widthp  = cinfo.width;
   *heightp = cinfo.height;

   return dest;
}


gboolean
BmpReadColorMap (FILE *fd,
		 guchar buffer[256][3],
		 gint number,
		 gint size,
		 gint *grey)
{
   gint i;
   guchar rgb[4];
	
   *grey = (number > 2);
   for (i = 0; i < number ; i++) {
      if (!ReadOK (fd, rgb, size)) {
	 return FALSE;
      }
      if (size == 4) {      
	 buffer[i][0] = rgb[2];
	 buffer[i][1] = rgb[1];
	 buffer[i][2] = rgb[0];
      } else {
	 buffer[i][0] = rgb[1];
	 buffer[i][1] = rgb[0];
	 buffer[i][2] = rgb[2];
      }
      *grey = ((*grey) && (rgb[0] == rgb[1]) && (rgb[1] == rgb[2]));
   }
   return TRUE;
}


static void
find_bits (gint n, gint *lowest, gint *n_set)
{
   gint i;

   *n_set = 0;

   for (i = 31; i >= 0; i--) {
      if (n & (1 << i)) {
	 *lowest = i;
	 (*n_set)++;
      }
   }
}


gboolean
BmpReadBitmasks (FILE *fd,
		 BmpBitFields *bf)
{
   guchar buffer[12];

   if (!ReadOK (fd, &buffer, 12)) {
      return FALSE;
   }

   bf->r_mask = ToL (buffer, 0);
   bf->g_mask = ToL (buffer, 4);
   bf->b_mask = ToL (buffer, 8);

   find_bits (bf->r_mask, &bf->r_shift, &bf->r_bits);
   find_bits (bf->g_mask, &bf->g_shift, &bf->g_bits);
   find_bits (bf->b_mask, &bf->b_shift, &bf->b_bits);

   if (bf->r_bits == 0 || bf->g_bits == 0 || bf->b_bits == 0) {
      bf->r_mask = 0x7c00;
      bf->r_shift = 10;
      bf->g_mask = 0x03e0;
      bf->g_shift = 5;
      bf->b_mask = 0x001f;
      bf->b_shift = 0;

      bf->r_bits = bf->g_bits = bf->b_bits = 5;
   }

   return TRUE;
}


/* FIXME!! Too long ;-P */
guchar *
BmpReadImage (FILE *fd,
	      gint len,
	      gint height,
	      guchar cmap[256][3],
	      BmpBitFields *bf,
	      gint ncols,
	      gint bpp,
	      gint compression,
	      gint spzeile,
	      gint grey)
{
   guchar v, wieviel;
   guchar buf[16];
   gint xpos = 0, ypos = 0;
   guchar *dest, *temp;
   gint i, j, pix, egal;

   ypos = height - 1;	/* Bitmaps begin in the lower left corner */
                        /* FIXME!! If height is negative value, it will not work correctly. */

   dest = g_malloc(sizeof(guchar) * len * height * 3);

   if (bpp == 32 && compression == BI_BITFIELDS) {
      gint r_lshift, r_rshift;
      gint g_lshift, g_rshift;
      gint b_lshift, b_rshift;

      r_lshift = 8 - bf->r_bits;
      g_lshift = 8 - bf->g_bits;
      b_lshift = 8 - bf->b_bits;

      r_rshift = bf->r_bits - r_lshift;
      g_rshift = bf->g_bits - g_lshift;
      b_rshift = bf->b_bits - b_lshift;

      while (ReadOK(fd, buf, 4)) {
	 gint v, r, g, b;

	 temp = dest + (xpos++ * 4) + (len * ypos * 4);

	 v = buf[0] | (buf[1] << 8) | (buf[2] << 16);

	 r = (v & bf->r_mask) >> bf->r_shift;
	 g = (v & bf->g_mask) >> bf->g_shift;
	 b = (v & bf->b_mask) >> bf->b_shift;

	 *temp++ = (r << r_lshift) | (r >> r_rshift);
	 *temp++ = (g << g_lshift) | (g >> g_rshift);
	 *temp++ = (b << b_lshift) | (b >> b_rshift);
	 /* *temp++ = buf[3]; */

	 if (xpos == len) {
	    if (spzeile > len * 4) {
	       fseek(fd, spzeile - (len * 4), SEEK_CUR);
	    }
	    ypos--;
	    xpos = 0;
	 }
	 if (ypos < 0) break;
      }

   } else if (bpp == 32) {
      while (ReadOK(fd, buf, 4)) {
	 temp = dest + (xpos++ * 4) + (len * ypos * 4);

	 *temp++ = buf[2];
	 *temp++ = buf[1];
	 *temp++ = buf[0];
	 /* *temp++ = buf[3]; */

	 if (xpos == len) {
	    if (spzeile > len * 4) {
	       fseek(fd, spzeile - (len * 4), SEEK_CUR);
	    }
	    ypos--;
	    xpos = 0;
	 }
	 if (ypos < 0) break;
      }

   } else if (bpp == 24) {
      while (ReadOK(fd, buf, 3)) {
	 temp = dest + (xpos++ * 3) + (len * ypos * 3);
	 *temp++ = buf[2];
	 *temp++ = buf[1];
	 *temp   = buf[0];
	 if (xpos == len) {
	    if (spzeile > len * 3) {
	       fseek(fd, spzeile - (len * 3), SEEK_CUR);
	    }
	    ypos--;
	    xpos = 0;
	 }
	 if (ypos < 0) break;
      }

   } else if (bpp == 16 && compression == BI_BITFIELDS) {
      gint r_lshift, r_rshift;
      gint g_lshift, g_rshift;
      gint b_lshift, b_rshift;

      r_lshift = 8 - bf->r_bits;
      g_lshift = 8 - bf->g_bits;
      b_lshift = 8 - bf->b_bits;

      r_rshift = bf->r_bits - r_lshift;
      g_rshift = bf->g_bits - g_lshift;
      b_rshift = bf->b_bits - b_lshift;

      while (ReadOK(fd, buf, 2)) {
	 gint v, r, g, b;

	 temp = dest + (xpos++ * 3) + (len * ypos * 3);

	 v = (int) buf[0] | ((int) buf[1] << 8);

	 r = (v & bf->r_mask) >> bf->r_shift;
	 g = (v & bf->g_mask) >> bf->g_shift;
	 b = (v & bf->b_mask) >> bf->b_shift;

	 *temp++ = (r << r_lshift) | (r >> r_rshift);
	 *temp++ = (g << g_lshift) | (g >> g_rshift);
	 *temp   = (b << b_lshift) | (b >> b_rshift);

	 if (xpos == len) {   /* egal = ReadOK(fd, buf, spzeile - (len * 2)); */
	    ypos--;
	    xpos = 0;
	 }
	 if (ypos < 0) break;
      }

   } else if (bpp == 16) {
      while (ReadOK(fd, buf, 2)) {
	 temp = dest + (xpos++ * 3) + (len * ypos * 3);

	 *temp++ = (buf[1] << 1) & 0xf8;
	 *temp++ = ((buf[1] >> 5) & 0x04) | ((buf[1] << 6) & 0xc0) | ((buf[0] >> 2) & 0x38);
	 *temp   = (buf[0] << 3) & 0xf8;

	 if (xpos == len) {   /* egal = ReadOK(fd, buf, spzeile - (len * 2)); */
	    ypos--;
	    xpos = 0;
	 }
	 if (ypos < 0) break;
      }

   } else {
      switch(compression) {
      case BI_RGB:  	/* uncompressed */
	 while (ReadOK(fd, &v, 1)) {
	    for (i = 1; (i <= (8 / bpp)) && (xpos < len); i++, xpos++) {
	       temp = dest + (xpos * 3) + (len * ypos * 3);
	       /* look at my bitmask !! */
	       pix = (v & (((1 << bpp) - 1) << (8 - (i * bpp)))) >> (8 - (i * bpp));
	       *temp++ = cmap[pix][0];
	       *temp++ = cmap[pix][1];
	       *temp   = cmap[pix][2];
	    }
	    if (xpos == len) {
	       if (spzeile > len) {
		  fseek(fd, (spzeile-len)/(8 / bpp), SEEK_CUR);
	       }
	       ypos--;
	       xpos = 0;
	    }
	    if (ypos < 0) break;
	 }
	 break;

      default:	/* Compressed images */
	 while (TRUE) {
	    egal = ReadOK(fd, buf, 2);
	    if ((guchar) buf[0] != 0) 
	       /* Count + Color - record */
	    {
	       for (j = 0; ((guchar) j < (guchar) buf[0]) && (xpos < len);) {
		  for (i = 1;
		       ((i <= (8 / bpp)) && (xpos < len) && ((guchar) j < (guchar) buf[0]));
		       i++, xpos++, j++)
		  {
		     temp = dest + (xpos * 3) + (len * ypos * 3);
		     pix = (buf[1] & (((1 << bpp) - 1) << (8 - (i * bpp)))) >> (8 - (i * bpp));
		     *temp++ = cmap[pix][0];
		     *temp++ = cmap[pix][1];
		     *temp   = cmap[pix][2];
		  }
	       }
	    }
	    if (((guchar) buf[0] == 0) && ((guchar) buf[1] > 2))
	       /* uncompressed record */
	    {
	       wieviel = buf[1];
	       for (j = 0; j < wieviel; j += (8 / bpp)) {
		  egal = ReadOK(fd, &v, 1);
		  i=1;
		  while ((i <= (8 / bpp)) && (xpos < len)) {
		     temp = dest + (xpos * 3) + (len * ypos * 3);
		     pix = (v & (((1 << bpp) - 1) << (8 - (i * bpp)))) >> (8 - (i * bpp));
			*temp++ = cmap[pix][0];
			*temp++ = cmap[pix][1];
			*temp   = cmap[pix][2];
			i++;
			xpos++;
		  }
	       }
	       if ((wieviel / (8 / bpp)) % 2) egal = ReadOK(fd, &v, 1);
	       /*if odd(x div (8 div bpp )) then blockread(f,z^,1);*/
	    }
	    if (((guchar) buf[0] == 0) && ((guchar) buf[1] == 0)) {   /* Zeilenende */
	       ypos--;
	       xpos = 0;
	    }
	    if (((guchar) buf[0] == 0) && ((guchar) buf[1] == 1)) {   /* Bitmapende */
	       break;
	    }
	    if (((guchar) buf[0] == 0) && ((guchar) buf[1] == 2)) {   /* Deltarecord */
	       xpos += (guchar) buf[2];
	       ypos += (guchar) buf[3];
	    }
	 }
	 break;
      }
   }
   
   /* g_free(dest); */
   return dest;
}
