static char rcsid[] = "$Id: splinmin.c,v 1.3 1997/07/18 03:02:36 dhb Exp $";

/*
** $Log: splinmin.c,v $
** Revision 1.3  1997/07/18 03:02:36  dhb
** Fix for getopt problem; getopt(), optopt and optind are now
** G_getopt(), G_optopt and G_optind.
**
** Revision 1.2  1993/02/24 21:15:47  dhb
** 1.4 to 2.0 command argument changes.
**
** Revision 1.1  1992/12/11  19:06:12  dhb
** Initial revision
**
*/

/* By : Upinder S. Bhalla, 1992, Caltech. */
#include <stdio.h>
#include <math.h>
#include "olf_ext.h"

float *vector();
void free_vector();
int find_nmin();
void splint();
float spbrent();
float bsplint();
float bsplinebrent();

#define NDIMS 20
#define TOLERANCE 0.0002
#define DX 0.002
#define RATIO 5.0
#define EPS 1.0e-8

#define ITMAX 200

#define NONE 0
#define FRESH 1
#define GRAD1LO 2
#define GRAD1HI 3
#define GRAD2LO 4
#define GRAD2HI 5
#define LINMIN 6
#define SPLINMIN 7
#define RANGE_DEFAULT 2.5

/*
** The splinmin routine does some severe approximations. It uses a
** fixed number of sample points along the minimization vector,
** and uses a spline fit for estimating the minimum. This method
** is justfiable only in some special situations.
** 
** 1 : if the function has a lot of small local minima, it is very
** easy for the standard linmin to get stuck. Splinmin can avoid
** this in two ways : using bezier (smoothing splines) and by using
** an asymmetric parabola estimation of slope for the gradient, so
** we are sampling at two length scales.
**
** 2 : If we are using parallel machines to do the minimization,
** the spline fit approach allows parallelization of an otherwise
** serial algorithm. All the derivatives are evaluated in one swell
** foop by calculating the 2N function values on separate nodes.
** Then all the spline sample points are evaluated in parallel, and
** if N is large enough they could be done on the same 2N nodes.
** This means that the entire minimization takes only 2*ITER cycles,
** where the number of iterations is typically 5 to 10.
** 
** Note : the number of sample points for the spline should be at
** least 16 in order to cover the space sufficiently.
*/

int do_splinmin(argc,argv)
	int argc;
	char	**argv;
{
	static float *p;
	float match,retmatch;
	int iter,i,ndim;
	struct table_type	*table;
	static int ret = 1;
	int	exp_flag=0,init_flag=0,cspline_flag=0;
	float tolerance;
	float dx=DX,ratio=RATIO;
	int niter = 0;
	int	status;

	tolerance = TOLERANCE;

/* Handle io options */
	initopt(argc, argv, "ndim table-element match -exponential -initialize -tolerance t -cspline -dx dx -ratio r -niterations n");
	while ((status = G_getopt(argc, argv)) == 1)
	  {
	    if (strcmp(G_optopt, "-exponential") == 0)
		exp_flag=1;
	    else if (strcmp(G_optopt, "-initialize") == 0)
		init_flag=1;
	    else if (strcmp(G_optopt, "-cspline") == 0)
		cspline_flag=1;
	    else if (strcmp(G_optopt, "-tolerance") == 0)
		tolerance=Atof(optargv[1]);
	    else if (strcmp(G_optopt, "-niterations") == 0)
		niter=atoi(optargv[1]);
	    else if (strcmp(G_optopt, "-dx") == 0)
		dx=Atof(optargv[1]);
	    else if (strcmp(G_optopt, "-ratio") == 0)
		ratio=Atof(optargv[1]);
	  }

	if (status < 0) {
		printoptusage(argc, argv);
		return(-1);
	}

	ndim=atoi(optargv[1]);
	if (ndim <=0) {
		printf("Error : ndim = %d must be > 0\n",ndim);
		return(-1);
	}
	table=(struct table_type *) GetElement(optargv[2]);
	if (!table) {
		printf("Error : table element %s not found\n",optargv[2]);
		return(-1);
	}
	if (strcmp(table->object->name,"table") !=0) {
		printf("Error : element %s is not of type table\n",optargv[2]);
		return(-1);
	}
	if (!table->alloced) {
		printf("Error : table %s is not allocated\n",optargv[2]);
		return(-1);
	}
	if (table->table->xdivs < ndim) {
		printf("Error : table %s is size %d, less than ndim=%d\n",
			optargv[2],table->table->xdivs,ndim);
		return(-1);
	}
	match=Atof(optargv[3]);

	/* Moved out of if statement because of possible pointer 
	** errors in sim, leaking over to here */
	if (ret) {
		p=vector(1,ndim); /* Stupid num rec convention */
		if (init_flag) {
			if (exp_flag) {
				for(i=1;i<=ndim;i++) p[i]=log(table->table->table[i-1]);
			} else  {
				for(i=1;i<=ndim;i++) p[i]=table->table->table[i-1];
			}
		} else {
			for(i=1;i<=ndim;i++) p[i]=0.0;
		}
	}
/* Do minimization */
	ret=find_splinmin(p,ndim,match,tolerance,dx,ratio,cspline_flag,
		&retmatch,niter);
	if (exp_flag)
		for(i=1;i<=ndim;i++) table->table->table[i-1]=exp(p[i]);
	else 
		for(i=1;i<=ndim;i++) table->table->table[i-1]=p[i];
	table->output=retmatch;
	if (ret)
		free_vector(p,1,ndim);
	return(ret);
}


int find_splinmin(p,ndim,match,tolerance,dx,ratio,cspline_flag,retmatch,
	niter)
	float *p;
	int ndim;
	float match;
	float tolerance;
	float dx,ratio;
	int cspline_flag;
	float *retmatch;
	int niter;
{
static float *ptemp,*xi,fret;
static int state=FRESH,laststate=NONE;
static float *g,*h;
static int i,j,k;
static float fp;
static float temp;
static int its;
static float gg,gam,dgg;
static float range,center;
static int nloops=0;

	/* finish up what was left over to do from last time */
	switch(laststate) {
		case FRESH :
			fp=match;
			break;
		case GRAD1LO :
		case GRAD2LO :
			p[i]=temp; /* restore previous value */
			xi[i]= (fp-match)*ratio*ratio -fp;
			break;
		case GRAD1HI :
			p[i]=temp;
			xi[i]+=match;
			xi[i]/=ratio*(ratio+1.0)*dx; /* finish evaln of gradient */
			i++;
			if (state==SPLINMIN) { /* initializing for minimization */
				for(j=1;j<=ndim;j++){
					g[j]= -xi[j];
					xi[j]=h[j]=g[j];
				}
				its=1;
				/* since p is being used for reporting params to the
				** simulation func */
				for(k=1;k<=ndim;k++) ptemp[k]=p[k];
				match=fp;
				range=RANGE_DEFAULT;
				center=0.0;
			}
			break;
		case GRAD2HI :
			p[i]=temp;
			xi[i]+=match;
			xi[i]/=ratio*(ratio+1.0)*dx; /* finish evaln of gradient */
			i++;
			if (state==SPLINMIN) { /* initializing for minimization */
		        dgg=gg=0.0;
        		for (j=1;j<=ndim;j++) {
            		gg += g[j]*g[j];
/* 			        dgg += xi[j]*xi[j];   */
            		dgg += (xi[j]+g[j])*xi[j];
        		}
        		if (gg == 0.0) {
            		/* do cleanup stuff here */
					free_vector(ptemp,1,ndim);
					free_vector(g,1,ndim);
					free_vector(h,1,ndim);
					free_vector(xi,1,ndim);
            		return(1);
        		}
        		gam=dgg/gg;
        		for (j=1;j<=ndim;j++) {
            		g[j] = -xi[j];
            		xi[j]=h[j]=g[j]+gam*h[j];
        		}
				its++;
				if (its>ITMAX) {
					printf("Error : too many iterations in GRAD2HI\n");
					return(-1);
				}
				/* since p is being used for reporting params to the
				** simulation func */
				for(k=1;k<=ndim;k++) ptemp[k]=p[k];
				match=fp;
				range=RANGE_DEFAULT;
				center=0.0;
			}
			break;
		case SPLINMIN :
			if (state==GRAD2LO) {
				/* Set the value of the returned function call at p */
				fp=match;
			}
			break;
		default :
			break;
	}
	laststate=state;

	switch(state) {
		case FRESH :
			/* Initialization */
			ptemp=vector(1,ndim); /* Stupid num rec convention */
			for(i=1;i<=ndim;i++) 
				ptemp[i]=0.0;
			g=vector(1,ndim);
			h=vector(1,ndim);
			xi=vector(1,ndim);
			i=1;
			state=GRAD1LO;
			break;
		case GRAD1LO :
			/* requests the function value at small basis vector
			** offsets in order to calculate gradients */
			temp=p[i];
			p[i]=temp-dx;
			state=GRAD1HI; /* evaluate lo end of gradient */
			break;
		case GRAD1HI :
			p[i]=temp+ratio*dx;
			if (i>=ndim) { /* This is the last eval in GRAD1HI */
				state=SPLINMIN;
			} else
				state=GRAD1LO; /* evaluate hi end of gradient */
			break;
		case SPLINMIN :
			temp=match; /* this is a horrible compiler error */
			j=splinmin(ptemp,xi,ndim,&temp,p,&range,&center,tolerance,
				cspline_flag);
			*retmatch=fret=temp;
			if (j>=1) { /* finished minimization */
				/* The pt to be evaluated is now ptemp */
				/* Since we could not predict what the last eval
				** in MIN would be, we have to set off the next
				** state from here itself. It is just a function
				** evaluation. The following state will be gradient
				** calculation */
				for(k=1;k<=ndim;k++) p[k]=ptemp[k];
				if (niter>0) {
					if (nloops>=niter) {
						free_vector(ptemp,1,ndim);
						free_vector(g,1,ndim);
						free_vector(h,1,ndim);
						free_vector(xi,1,ndim);
						nloops=0;
						return(1); /* finally done */
					} else {
						nloops++;
					}
				} else {
					if (j==2 && 2.0*fabs(fret-fp) <=
						tolerance*(fabs(fret)+fabs(fp)+EPS)) {
						/* free stuff here */
						free_vector(ptemp,1,ndim);
						free_vector(g,1,ndim);
						free_vector(h,1,ndim);
						free_vector(xi,1,ndim);
						return(1); /* finally done */
					}
				}
				/* Since we could not predict what the last eval
				** in MIN would be, we have to set off the next
				** state from here itself. It is just a function
				** evaluation. The following state will be gradient
				** calculation */
				/* for(i=1;i<ndim;i++) p[i]=ptemp[i]; */

				/* if it got here, then it did find a line min, but
				** not the final answer. So we go and do another 
				** grad evaluation */
				i=1;
				state=GRAD2LO;
			}
			break;
		case GRAD2LO :
			/* requests the function value at small basis vector
			** offsets in order to calculate gradients */
			temp=p[i];
			p[i]=temp-dx;
			state=GRAD2HI; /* evaluate lo end of gradient */
			break;
		case GRAD2HI :
			p[i]=temp+ratio*dx;
			if (i>=ndim) { /* This is the last eval in GRAD1HI */
				state=SPLINMIN;
			} else
				state=GRAD2LO; /* evaluate hi end of gradient */
			break;
		default :
			printf("Error : state should never be here \n");
			break;
	}
	return(0);
}

#undef NDIMS 
#undef TOLERANCE
#undef DX
#undef EPS

#undef ITMAX

#undef NONE
#undef FRESH
#undef GRAD1LO
#undef GRAD1HI
#undef GRAD2LO
#undef GRAD2HI
#undef LINMIN
#undef SPLINMIN
#undef RANGE_DEFAULT

#define MIN_INIT 1
#define MIN_BRAK 2
#define MIN_BRENT 3
#define MIN_SCAN 4
#define MIN_FIND 5

/* Should always be an odd number */
#define MAX_X_SAMPLE_PTS 17

static int ncom=0; /* defining declarations */
static float *pcom=0,*xicom=0;
float Spbrent();

static void setp(p,x)
	float *p;
	float x;
{
	int i;

	for(i=1;i<=ncom;i++) p[i]=pcom[i]+x*xicom[i];
}


int splinmin(p,xi,n,match,peval,range,center,tolerance,cspline_flag)
	float *p,*xi;
	int n;
	float *match,*peval,*range,*center;
	float tolerance;
	int cspline_flag;
{
	static int state=MIN_INIT;
	static float xx,xmin,fx,fb,fa,bx,ax;
	static float *x,*y,*y2;
	int		i,j;
	static int scan_count;
	static float  *x_sample_pts;
	int converge_flag=0;
	/*
	static float  x_sample_pts[]={-2.0,
		-1.0,-0.75,-0.5,-0.3,-0.15,
		-0.1,-0.05,-0.02,0.0,0.02,0.05,0.1,
		0.15,0.3,0.5,0.75,1.0
	};
	*/
	int besti;
	float ymin;
	float xiscale;
	FILE	*fp,*fopen();
	float temp;

	switch (state) {
		case MIN_INIT :
			x=vector(1,MAX_X_SAMPLE_PTS);
			y=vector(1,MAX_X_SAMPLE_PTS);
			y2=vector(1,MAX_X_SAMPLE_PTS);
			x_sample_pts=vector(1,MAX_X_SAMPLE_PTS+2);
			pcom=vector(1,n);
			xicom=vector(1,n);
			ncom=n;

			if (cspline_flag) {
				temp=0.01;
				xiscale=exp(log(100.0)/(float)(MAX_X_SAMPLE_PTS/2));
				for(i=1;i<=MAX_X_SAMPLE_PTS/2;i++) {
					x_sample_pts[MAX_X_SAMPLE_PTS/2 - i + 1]= -temp;
					x_sample_pts[MAX_X_SAMPLE_PTS/2 + i + 1]= temp;
					temp*=xiscale;
					x_sample_pts[MAX_X_SAMPLE_PTS/2+1]=0.0;
				}
			} else {
				/* range of -1 to +1 */
				temp=2.0/(float)(MAX_X_SAMPLE_PTS -1);
				for(i=1;i<=MAX_X_SAMPLE_PTS;i++) 
					x_sample_pts[i]=(float)(i-1) * temp - 1.0;
			}

				

			/* Enforce xi to be a unit vector, since we want to
			** avoid local minima */

			xiscale=0.0;
			for (j=1;j<=n;j++) {
				xiscale += xi[j] * xi[j];
			}
			if (xiscale > 0.0)
				xiscale=sqrt(xiscale);
			else 
				xiscale=1.0;

			for (j=1;j<=n;j++) {
				pcom[j]=p[j];
				xicom[j]=xi[j]/xiscale;
			}
			scan_count=1;
			/* Set it to the value at the initial p vector */
			y[1+MAX_X_SAMPLE_PTS/2]= *match;
			/* Set up the x array */
			for (i=1;i<=MAX_X_SAMPLE_PTS;i++)
				x[i]= *center + *range * x_sample_pts[i];
			setp(peval,x[scan_count]);

			state=MIN_SCAN;
			break;
		case MIN_SCAN :	
			y[scan_count]= *match;
			scan_count++;
			/* Skip if this is the middle pt */
			if (scan_count==1+MAX_X_SAMPLE_PTS/2)
				scan_count++;
			setp(peval,x[scan_count]);
			if (scan_count>=MAX_X_SAMPLE_PTS)
				state=MIN_FIND;
			break;
		case MIN_FIND :
			y[scan_count]= *match;
			fp=fopen("foo","a");
			fprintf(fp,"\n/newplot\n");
			for(i=1;i<=MAX_X_SAMPLE_PTS;i++) {
				fprintf(fp,"%f	%f\n",x[i],y[i]);
			}
			fclose(fp);
			ymin=y[1];
			besti=1;
			for(i=2;i<=MAX_X_SAMPLE_PTS;i++) {
				if (ymin>y[i]) {
					ymin=y[i];
					besti=i;
				}
			}
			/* Avoid problems with completely flat lines */
			if (besti==1 && y[1]==y[2]) besti=2;
			if (besti==MAX_X_SAMPLE_PTS &&
				y[MAX_X_SAMPLE_PTS]==y[MAX_X_SAMPLE_PTS-1])
				besti=MAX_X_SAMPLE_PTS-1;
			if (besti==1) { /* min was lower than the sampled range*/
			/* shift the sample range left by half, and double it */
				*center-= *range;
				*range *= 2;
			/* Set the new middle pt to the current low point */
				y[1+MAX_X_SAMPLE_PTS/2]=y[1];
			/* set the x array to the new scale */
				for (i=1;i<=MAX_X_SAMPLE_PTS;i++)
					x[i]= *center + *range * x_sample_pts[i];
			/* do another set of samples, alas */
				scan_count=1;
				setp(peval,x[scan_count]);
				state=MIN_SCAN;
			} else if (besti>=MAX_X_SAMPLE_PTS) {
			/* max was larger than the sampled range*/
			/* shift the sample range right by half, and double it */
				*center+= *range;
				*range *= 2;
			/* Set the new middle pt to the current high point */
				y[1+MAX_X_SAMPLE_PTS/2]=y[MAX_X_SAMPLE_PTS];
			/* set the x array to the new scale */
				for (i=1;i<=MAX_X_SAMPLE_PTS;i++)
					x[i]= *center + *range * x_sample_pts[i];
			/* do another set of samples, alas */
				scan_count=1;
				setp(peval,x[scan_count]);
				state=MIN_SCAN;
			} else {
			/* The best point is somewhere in here */
				if (cspline_flag) {
					spline(x,y,MAX_X_SAMPLE_PTS,1e30,1e30,y2);
					ax= x[besti-1];
					xx= x[besti];
					bx= x[besti+1];
					/* find the minimum */
					*match=
						spbrent(ax,xx,bx,tolerance,&xmin,x,y,y2,
						MAX_X_SAMPLE_PTS);
					/* set the p vector to return */
					setp(p,xmin);
				} else { /* the default b-spline */
					ax= x[besti-1];
					xx= x[besti];
					bx= x[besti+1];
					*match=
					bsplinebrent(ax,xx,bx,tolerance,&xmin,x,y,
						MAX_X_SAMPLE_PTS);
					if (xmin < tolerance)
						converge_flag=1;
					if (*match > y[besti]) 
						*match=y[besti];
					/* set the p vector to return */
					setp(p,xmin);
				}

				/* clean up */
				free_vector(x,1,MAX_X_SAMPLE_PTS);
				free_vector(y,1,MAX_X_SAMPLE_PTS);
				free_vector(y2,1,MAX_X_SAMPLE_PTS);
				free_vector(x_sample_pts,1,MAX_X_SAMPLE_PTS+2);
				free_vector(pcom,1,n);
				free_vector(xicom,1,n);

				state=MIN_INIT;
				if (converge_flag)
					return(2);
				else
					return(1);
			}
		break;
	}
	return(0);
}

#undef MIN_INIT
#undef MIN_BRAK
#undef MIN_BRENT
#undef MIN_SCAN 
#undef MIN_FIND
#undef MAX_X_SAMPLE_PTS 


/*
float *vector(nl,nh)
int nl,nh;
{
	float *v;

	v=(float *)malloc((unsigned) (nh-nl+1)*sizeof(float));
	if (!v) {
		printf("Failed to allocate vector %d %d\n",nl,nh);
		exit(1);
	}
	return v-nl;
}

void free_vector(v,nl,nh)
float *v;
int nl,nh;
{
	free((char*) (v+nl));
}
*/




#define ITMAX 100
#define CGOLD 0.3819660
#define ZEPS 1.0e-10
#define SIGN(a,b) ((b) > 0.0 ? fabs(a) : -fabs(a))
#define SHFT(a,b,c,d) (a)=(b);(b)=(c);(c)=(d);

#define NONE 0
#define INIT 1
#define LOOP 2

float bsplinebrent(ax,bx,cx,tol,xmin,xa,ya,n)
float ax,bx,cx,tol,*xmin;
float *xa,*ya;
int n;
{
	int iter;
	float a,b,d,etemp,fu,fv,fw,fx,p,q,r,tol1,tol2,u,v,w,x,xm;
	float e=0.0;
	void nrerror();
	float ret;

	a=((ax < cx) ? ax : cx);
	b=((ax > cx) ? ax : cx);
	x=w=v=bx;
	ret=bsplint(x,xa[2]-xa[1],xa[1],ya+1,n);
	fw=fv=fx=ret;
	for (iter=1;iter<=ITMAX;iter++) {
		xm=0.5*(a+b);
		tol2=2.0*(tol1=tol*fabs(x)+ZEPS);
		if (fabs(x-xm) <= (tol2-0.5*(b-a))) {
			*xmin=x;
			return fx;
		}
		if (fabs(e) > tol1) {
			r=(x-w)*(fx-fv);
			q=(x-v)*(fx-fw);
			p=(x-v)*q-(x-w)*r;
			q=2.0*(q-r);
			if (q > 0.0) p = -p;
			q=fabs(q);
			etemp=e;
			e=d;
			if (fabs(p) >= fabs(0.5*q*etemp) || p <= q*(a-x) || p >= q*(b-x))
				d=CGOLD*(e=(x >= xm ? a-x : b-x));
			else {
				d=p/q;
				u=x+d;
				if (u-a < tol2 || b-u < tol2)
					d=SIGN(tol1,xm-x);
			}
		} else {
			d=CGOLD*(e=(x >= xm ? a-x : b-x));
		}
		u=(fabs(d) >= tol1 ? x+d : x+SIGN(tol1,d));
		ret=bsplint(u,xa[2]-xa[1],xa[1],ya+1,n);
		fu=ret;
		if (fu <= fx) {
			if (u >= x) a=x; else b=x;
			SHFT(v,w,x,u)
			SHFT(fv,fw,fx,fu)
		} else {
			if (u < x) a=u; else b=u;
			if (fu <= fw || w == x) {
				v=w;
				w=u;
				fv=fw;
				fw=fu;
			} else if (fu <= fv || v == x || v == w) {
				v=u;
				fv=fu;
			}
		}
	}
	nrerror("Too many iterations in BRENT");
	*xmin=x;
	return fx;
}


float spbrent(ax,bx,cx,tol,xmin,xa,ya,y2,n)
float ax,bx,cx,tol,*xmin;
float *xa,*ya,*y2;
int n;
{
	int iter;
	float a,b,d,etemp,fu,fv,fw,fx,p,q,r,tol1,tol2,u,v,w,x,xm;
	float e=0.0;
	void nrerror();
	float ret;

	a=((ax < cx) ? ax : cx);
	b=((ax > cx) ? ax : cx);
	x=w=v=bx;
	splint(xa,ya,y2,n,x,&ret);
	fw=fv=fx=ret;
	for (iter=1;iter<=ITMAX;iter++) {
		xm=0.5*(a+b);
		tol2=2.0*(tol1=tol*fabs(x)+ZEPS);
		if (fabs(x-xm) <= (tol2-0.5*(b-a))) {
			*xmin=x;
			return fx;
		}
		if (fabs(e) > tol1) {
			r=(x-w)*(fx-fv);
			q=(x-v)*(fx-fw);
			p=(x-v)*q-(x-w)*r;
			q=2.0*(q-r);
			if (q > 0.0) p = -p;
			q=fabs(q);
			etemp=e;
			e=d;
			if (fabs(p) >= fabs(0.5*q*etemp) || p <= q*(a-x) || p >= q*(b-x))
				d=CGOLD*(e=(x >= xm ? a-x : b-x));
			else {
				d=p/q;
				u=x+d;
				if (u-a < tol2 || b-u < tol2)
					d=SIGN(tol1,xm-x);
			}
		} else {
			d=CGOLD*(e=(x >= xm ? a-x : b-x));
		}
		u=(fabs(d) >= tol1 ? x+d : x+SIGN(tol1,d));
		splint(xa,ya,y2,n,u,&ret);
		fu=ret;
		if (fu <= fx) {
			if (u >= x) a=x; else b=x;
			SHFT(v,w,x,u)
			SHFT(fv,fw,fx,fu)
		} else {
			if (u < x) a=u; else b=u;
			if (fu <= fw || w == x) {
				v=w;
				w=u;
				fv=fw;
				fw=fu;
			} else if (fu <= fv || v == x || v == w) {
				v=u;
				fv=fu;
			}
		}
	}
	nrerror("Too many iterations in BRENT");
	*xmin=x;
	return fx;
}

#undef ITMAX
#undef CGOLD
#undef ZEPS
#undef SIGN
#undef NONE
#undef INIT
#undef LOOP

void splint(xa,ya,y2a,n,x,y)
float xa[],ya[],y2a[],x,*y;
int n;
{
	int klo,khi,k;
	float h,b,a;
	void nrerror();

	klo=1;
	khi=n;
	while (khi-klo > 1) {
		k=(khi+klo) >> 1;
		if (xa[k] > x) khi=k;
		else klo=k;
	}
	h=xa[khi]-xa[klo];
	if (h == 0.0) {
		printf("Bad XA input to routine SPLINT");
		return;
	}
	a=(xa[khi]-x)/h;
	b=(x-xa[klo])/h;
	*y=a*ya[klo]+b*ya[khi]+((a*a*a-a)*y2a[klo]+(b*b*b-b)*y2a[khi])*(h*h)/6.0;
}

#define NSA 0.1666666666666666667
#define NSB 0.6666666666666666667

float bsplint(x,dx,lo,yc,n)
    float x,dx,lo;
    float *yc;
	int	n;
{
    float t,t1,t2,t3,t2a,t3a,nc1,nc2,nc3,nc4;
    int i;
    float y;

    if(x<(lo+dx)) /* lin interp between lo and dx */
        return(yc[0]+(yc[1]-yc[0])*(x-lo)/dx);
    i=(x-lo)/dx;
    /* t=(x-(dx*(float)i)-lo)/dx; */
    t=(x-lo)/dx -(float)i;
    if(i>n-3) { /* lin interp till end of data */
        if(i==n-1)
            return(yc[i]);
        return(yc[n-2]+(yc[n-1]-yc[n-2])*t);
    }

    t1=t/2.0;
    t2=t*t;
    t2a=t2/2.0;
    t3=t2*t;
    t3a=t3/2.0;
    nc1=t2a-t1+NSA-NSA*t3;
    nc2=t3a-t2+NSB;
    nc3= -t3a+t2a+t1+NSA;
    nc4=NSA*t3;
    /* y=nc1*yc[i-1]+nc2*yc[i]+nc3*yc[i+1]+nc4*yc[i+2]; */
    y=nc1*yc[i-1]+nc2*yc[i]+nc3*yc[i+1]+nc4*yc[i+2];
    /*
    printf("i=%d,t=%f,t1=%f,t2=%f,t3=%f,nc1=%f,nc2=%f,nc3=%f,nc4=%f\n",
        i,t,t1,t2,t3,nc1,nc2,nc3,nc4); */
    return(y);
}

#undef NSA
#undef NSB

