#
# Rhythmlet - gDesklets sensor for Rhythmbox
# Copyright (c) 2004 Alex Revo
# ----------------------------------------------------------------------
#  __init__.py - Rhythmlet sensor for gDesklets
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
########################################################################


VERSION="0.3i"

# Import the required (system) modules.
import Image
import md5
import os
import re
import string
import threading
import time
import types
import urllib
from   shutil				import copy
from   xml.sax.saxutils	    import escape,unescape

# Import gDesklets-specific modules.
from   sensor.Sensor	    import Sensor
from   utils				import i18n
from   utils.datatypes	    import *

# Import Rhythmlet-specific modules.
import RLBase
import Rhythmbox
import RLCovers
import AudioScrobbler

# Import optional modules.
#  * (SOAPpy is needed for Amazon)
try:
    import SOAPpy
    HAVE_SOAP = 1
except:
    HAVE_SOAP = 0
#  * Python Imaging Library is needed for resizing
try:
	import Image, ImageOps
	HAVE_PIL = 1
except:
	HAVE_PIL = 0


class Rhythmlet(Sensor, RLBase.RLBase):
#-----------------------------------------------------------------------
	"""Rhythmlet class for gDesklets."""

	# Initialise variables:
	RB = ""



	def __init__(self, coverSize=50, seekLength=112, dispString="%a%N%t", scrollChars=0, *args):
	#-------------------------------------------------------------------
		"""Constructor for Rhythmlet class."""

		Sensor.__init__(self)
		RLBase.RLBase.__init__(self)

		self.dPrint("Include everything below when filing bug reports.")
		self.dPrint("--- BEGIN DEBUG OUTPUT ---")
		self.dPrint("Rhythmlet " + VERSION + " is starting.")

		global _; _ = i18n.Translator("rhythmlet")

		# Set the default configurator settings.
		self._set_config_type("dispString", TYPE_STRING, dispString)
		self._set_config_type("coverSize", TYPE_INT, coverSize)
		self._set_config_type("showOutline", TYPE_BOOL, 1)
		self._set_config_type("showCover", TYPE_BOOL, 1)
		self._set_config_type("tryLocal", TYPE_BOOL, 1)
		self._set_config_type("asUsername", TYPE_STRING, "")
		self._set_config_type("asPassword", TYPE_SECRET_STRING, "")
		self._set_config_type("enableAudioScrobbler", TYPE_BOOL, 0)
		if (HAVE_SOAP):
			self._set_config_type("tryAmazon", TYPE_BOOL, 1)
			self._set_config_type("saveAmazon", TYPE_BOOL, 1)
		else:
			self._set_config_type("tryAmazon", TYPE_BOOL, 0)
			self._set_config_type("saveAmazon", TYPE_BOOL, 0)
		self._set_config_type("scrollSecs", TYPE_INT, 10)
		self._set_config_type("scrollChars", TYPE_INT, int(scrollChars))

		# Initialise variables for Rhythmlet class.
		self._RB				= Rhythmbox.Rhythmbox()
		self._RLCover			= RLCovers.RLCovers()
		self._Scrobbler			= AudioScrobbler.AudioScrobbler()
		self._needCover			= 0
		self._seekLength		= int(seekLength)
		self._dictVars			= {
			"album":			"",
			"artist":			"",
			"backgroundColour":	"",
			"bitrate":			"",
			"cover":			"",
			"coverAlbum":		"",
			"coverArtist":		"",
			"coverSizeX":		"",
			"coverSizeY":		"",
			"dispString":		"",
			"duration":			"",
			"durationSecs":		"",
			"elapsedTime":		"",
			"filePath":			"",
			"filesize":			"",
			"filesizeMB":		"",
			"genre":			"",
			"imgPath":			"",
			"imgPlayPause":		"",
			"imgRepeat":		"",
			"imgShuffle":		"",
			"launchBrowser":	"",
			"link":				"",
			"playCount":		"",
			"playing":			"",
			"playingTime":		0,
			"progress":			"",
			"rating":			"",
			"rating1":			"",
			"rating2":			"",
			"rating3":			"",
			"rating4":			"",
			"rating5":			"",
			"scrolledDispString":"",
			"showCover":		"",
			"title":			"",
			"trackNumber":		""
		}
		self._dictVarsOld		= self._dictVars.copy()
		self._failAlert			= 0

		# AudioScrobbler vars:
		self._lastRefresh		= 0
		self._oldArtist			= ""
		self._oldTime			= ""
		self._oldTitle			= ""
		self._trackQueued		= 0


		# Add threads and timers.
		self.dPrint("Adding threads and timers.")
		# Some things (self._get_config...) aren't available to __init__
		# so add a thread to do further init stuff.
		self._add_thread(self.postInitInit)

		self._add_timer(1000, self.refresh)
		self._add_timer(10000, self.unfuckBonobo)
		self._add_thread(self.getCover_thread)
		self._add_thread(self.doScrobbler_thread)
		self._watch_config(self.configWatcher)
	#-------------------------------------------------------------------

	def postInitInit(self):
	#-------------------------------------------------------------------
		# Run once then die.

		if (self._get_config("showCover") == 0):
			self.setVar("coverSizeX", 0)
			self.setVar("coverSizeY", 0)
		# end if

		# Set Scrobbler username/password.
		if (self._get_config("asUsername")):
			self._Scrobbler.setUsername(self._get_config("asUsername"))
		if (self._get_config("asPassword")):
			self._Scrobbler.setPassword(self._get_config("asPassword"))
	#-------------------------------------------------------------------


	def refresh(self):
	#-------------------------------------------------------------------
		"""Refresh data and update gDesklets watchers."""

		# Initialise default values.
		myDispString	= "Not Playing?"
		myDuration		= 0
		myFilesize		= 0
		myImgPlayPause	= "gfx/rhythmbox-play.png"
		myImgRepeat		= "gfx/repeat-off.png"
		myImgShuffle	= "gfx/shuffle-off.png"
		myProgress		= 0
		myRating1		= "gfx/rhythmbox-unset-star.png"
		myRating2		= "gfx/rhythmbox-unset-star.png"
		myRating3		= "gfx/rhythmbox-unset-star.png"
		myRating4		= "gfx/rhythmbox-unset-star.png"
		myRating5		= "gfx/rhythmbox-unset-star.png"
		myBackgroundColour= self._get_config("backgroundColour")
		myShowCover		= self._get_config("showCover")
		myShowOutline	= self._get_config("showOutline")

		# Tell the Rhythmbox class to refresh its data.
		self._RB.refresh()

		# Check if Rhythmbox is active; if not, report errors.
		if (not self._failAlert):
			if (not self._RB.isActive()):
				print("***")
				print("Bonobo reports that Rhythmbox is not running.")
				print("Running rhythmbox with the --no-register argument will cause this error.")
				print("***")
			if (not self._RB.hasSongInfo()):
				self.dPrint("(Bonobo) Rhythmbox is missing SongInfo struct (not playing?)")
			# end if

			self._failAlert = 1
		# end if

		try:
			if (self._RB.hasSongInfo()):
				self._failAlert = 0

				# Grab song info from Rhythmbox.
				if (int(self._RB["rating"]) >= 1):	myRating1		= "gfx/rhythmbox-set-star.png"
				if (int(self._RB["rating"]) >= 2):	myRating2		= "gfx/rhythmbox-set-star.png"
				if (int(self._RB["rating"]) >= 3):	myRating3		= "gfx/rhythmbox-set-star.png"
				if (int(self._RB["rating"]) >= 4):	myRating4		= "gfx/rhythmbox-set-star.png"
				if (int(self._RB["rating"]) >= 5):	myRating5		= "gfx/rhythmbox-set-star.png"
				if (self._RB["playing"]):			myImgPlayPause	= "gfx/rhythmbox-pause.png"
				if (self._RB["shuffle"]):			myImgShuffle	= "gfx/shuffle-on.png"
				if (self._RB["duration"] > 0):		myProgress		= (100 * int(self._RB["playingTime"]) / int(self._RB["duration"]))
				if (self._RB["duration"]):			myDuration		= int(self._RB["duration"])

				# Don't choke on Rhythmbox 0.8.4...
				try:
					if (self._RB["repeat"]):		myImgRepeat		= "gfx/repeat-on.png"
				except:
					pass
				# end try

				# Parse user-configurable display string.
				myDispString = self._get_config("dispString")
				myDispString = myDispString.replace("%N", "\n")
				myDispString = myDispString.replace("%a", str(self._RB["artist"]))
				myDispString = myDispString.replace("%A", str(self._RB["album"]))
				myDispString = myDispString.replace("%c", str(self._RB["play_count"]))
				myDispString = myDispString.replace("%d", "%d:%02d" % divmod(int(myDuration), 60))
				myDispString = myDispString.replace("%e", "%d:%02d" % divmod(int(self._RB["playingTime"]), 60))
				myDispString = myDispString.replace("%g", str(self._RB["genre"]))
				myDispString = myDispString.replace("%p", str(self._RB["path"]))
				myDispString = myDispString.replace("%r", "%d/5" % int(self._RB["rating"]))
				myDispString = myDispString.replace("%s", str(self._RB["filesize"]))
				myDispString = myDispString.replace("%t", str(self._RB["title"]))
				myDispString = myDispString.replace("%T", "%02d" % int(self._RB["track_number"]))
			else:
				# Set cover to default if not playing.
				self.setVar("cover", "gfx/coverDefault.png")
			# end if
		except Exception, error:
			self.handleException("Rhythmlet::refresh() : display string", error)

			# Display default cover.
			self.setVar("cover", "gfx/coverDefault.png")
			pass
		# end try


		try:
			if (self._RB.hasSongInfo()):
				# On new song: immediately set cover to default, set self._needCover to 1.
				if ((self.getVar("coverArtist") != self._RB["artist"]) or (self.getVar("coverAlbum") != self._RB["album"])):
					self.setVar("coverArtist", self._RB["artist"])
					self.setVar("coverAlbum", self._RB["album"])
					#self.setVar("cover", "gfx/coverDefault.png")

					self._needCover = 1
				# end if

				# Update watchers.
				self.setVar("album", escape(self._RB["album"]))
				self.setVar("artist", escape(self._RB["artist"]))
				self.setVar("backgroundColour", myBackgroundColour)
				self.setVar("bitrate", str(self._RB["bitrate"]) + " kbps")
				self.setVar("duration", ("%d:%02d" % divmod(int(myDuration), 60)))
				self.setVar("durationSecs", self._RB["duration"])
				self.setVar("elapsedTime", ("%d:%02d" % divmod(int(self._RB["playingTime"]), 60)))
				self.setVar("filePath", escape(self._RB["path"]))
				self.setVar("filesize", str(self._RB["filesize"]))
				self.setVar("filesizeMB", str("%02.02f MB" % (int(self._RB["filesize"]) / 1024.00 / 1024.00)))
				self.setVar("genre", escape(self._RB["genre"]))
				self.setVar("playCount", str(self._RB["play_count"]))
				self.setVar("playing", str(self._RB["playing"]))
				self.setVar("playingTime", self._RB["playingTime"])
				self.setVar("rating", ("%d/5" % self._RB["rating"]))
				self.setVar("title", escape(self._RB["title"]))
				self.setVar("trackNumber", ("%02d" % self._RB["track_number"]))
			# end if

			self.setVar("dispString", escape(myDispString))
			self.setVar("duration", ("%d:%02d" % divmod(int(myDuration), 60)))
			self.setVar("imgPlayPause", myImgPlayPause)
			self.setVar("imgRepeat", myImgRepeat)
			self.setVar("imgShuffle", myImgShuffle)
			self.setVar("progress", myProgress)
			self.setVar("rating1", myRating1)
			self.setVar("rating2", myRating2)
			self.setVar("rating3", myRating3)
			self.setVar("rating4", myRating4)
			self.setVar("rating5", myRating5)
			self.setVar("scrolledDispString", self.scrollText(self.getVar("dispString"), int(self._get_config("scrollChars"))))
			self.setVar("showCover", myShowCover)
		except Exception, error:
			self.handleException("Rhythmlet::refresh() : watchers", error)

			pass

		# Send output to gDesklets
		self.sendVars()

		return 1
	#-------------------------------------------------------------------


	def getCover_thread(self):
	#-------------------------------------------------------------------
		"""Check for cover, resize, and display it."""

		# Sleep for 1 second, allowing time for refresh to occur.
		time.sleep(1)

		while (1):
			# Break out of thread if desklet is stopped.
			if (self._is_stopped()):
				break
			# end if

			# Sleep if we need not fetch the cover.
			if ((self.getVar("showCover") == 0) or (self._needCover == 0)):
				time.sleep(4)

				continue
			# end if

			self.getCover()
		# end while
	#-------------------------------------------------------------------

	def getCover(self):
	#-------------------------------------------------------------------
		"""Check for cover, resize, and display it."""

		#
		# NOTE:
		#  Do not use self._RB here. The Rhythmbox class uses Bonobo;
		#  Threading angers Bonobo and when Bonobo gets angry, he kills.
		#

		# Initialise variables
		myCover		= ""
		tryLocal	= self._get_config("tryLocal")
		tryAmazon	= self._get_config("tryAmazon")
		saveAmazon	= self._get_config("saveAmazon")
		needResize	= 0
		resizeFailed= 0

		# Need to get new cover
		self.dPrint("Cover is needed for album: " + str(self.getVar("artist")) + " - " + str(self.getVar("album")))
		self._needCover = 0

		# Use bad hack to bypass caching.
		# (NOTE: Should be deprecated with gDesklets 0.30pre?)
		if (self.getVar("imgPath") != "/tmp/rhythmlet.png"):
			self.setVar("imgPath", "/tmp/rhythmlet.png")
		else:
			self.setVar("imgPath", "/tmp/rhythmlet-uncache.png")
		# end if
		myImgPath = self.getVar("imgPath")

		# Set the default link.
		self.setVar("link", "http://www.allmusic.com")


		# -------------------------------------------------------------------
		# Check for cover image locally.
		# -------------------------------------------------------------------
		if (tryLocal):
			self.dPrint("Checking for cover locally...")

			(localCover, candidates, filenames, songPath) = self._RLCover.getCover_local(self.getVar("artist"), self.getVar("album"), self.getVar("filePath"))
			if (localCover):
				self.dPrint("  Found cover locally.")
				self.dPrint("  (%s)" % localCover)

				myCover = localCover
				tryAmazon = 0			# no need to try Amazon
				needResize = 1
			else:
				self.dPrint("  Cover was not found locally.")
			# end if
		# end if


		# -------------------------------------------------------------------
		# Check Amazon.com for cover.
		# -------------------------------------------------------------------
		if (tryAmazon):
			self.dPrint("Checking for cover from Amazon...")

			# SOAPpy is needed to fetch images from Amazon.
			if (HAVE_SOAP == 0):
				self.dPrint("  [!!!] SOAPpy isn't installed; Abort Amazon check")

				self.setVar("cover", "gfx/coverNoSoap.png")
				self.setVar("link", "http://pywebsvcs.sourceforge.net/")
				needResize = 1

				return
			# end if

			(myCover, myLink) = self._RLCover.getCover_Amazon(self.getVar("artist"), self.getVar("album"), self.getVar("filePath"), self._get_config("saveAmazon"), myImgPath)

			if (myCover != "gfx/coverDefault.png"):
				needResize = 1
				self.setVar("link", myLink)
			# end if
		# end if


		# -------------------------------------------------------------------
		# Resize cover.
		# -------------------------------------------------------------------
		# ... only if necessary.
		if ((HAVE_PIL == 0) or (needResize == 0) or (myCover == "")):
			resizeFailed = 1
		# end if

		# Copy image to /tmp so it can be resized.
		if (resizeFailed == 0):
			if (myImgPath != myCover):
				#self._RLCover.copyFile(os.path.join(self._get_path(), myCover), myImgPath)
				imgConvert = Image.open(os.path.join(self._get_path(), myCover)).save(myImgPath)
			# end if

			# Resize cover.
			try:
				(newWidth, newHeight) = self._RLCover.resizeImage(myImgPath, self._get_config("coverSize"), self._get_config("showOutline"))
				self.setVar("cover", myImgPath)
			except Exception, error:
				self.handleException("getCover_thread(): resize, etc.", error)

				resizeFailed = 1
			# end try
		# end if

		if (resizeFailed):
			self.setVar("cover", "gfx/coverDefault.png")
		# end if
	#-------------------------------------------------------------------


	def doScrobbler_thread(self):
	#-------------------------------------------------------------------
		"""Update AudioScrobbler.com with client data (thread)."""

		# Sleep for 1 second, allowing time for refresh to occur.
		time.sleep(1)

		while (1):
			# Break out of thread if desklet is stopped.
			if (self._is_stopped()):
				break
			# end if

			# Run doScrobbler() only if user enables AudioScrobbler support and has entered a username and password.
			if (self._get_config("enableAudioScrobbler") and 
			(self._get_config("asUsername") and self._get_config("asPassword"))):
				self.doScrobbler()
			# end if

			# Only run every 5 seconds.
			time.sleep(5)
		# end while
	#-------------------------------------------------------------------

	def doScrobbler(self):
	#-------------------------------------------------------------------
		"""Update AudioScrobbler.com with client data."""

		#
		# NOTE:
		#  Do not use self._RB here
		#  Rhythmbox class uses Bonobo
		#  Threading angers Bonobo
		#  ?
		#
		#  This code is ugly and will be split away from Rhythmlet later.
		#  TODO: fork, clean code, GTK interface, notification icon, etc.
		#

		try:
			# Perform handshake if needed.
			self._Scrobbler.doHandshake()


			########################
			# Refresh client data. #
			########################
			# Collect data for AudioScrobbler.
			if ((self._oldArtist != self.getVar("artist")) or (self._oldTitle != self.getVar("title"))):
				self._oldArtist	= self.getVar("artist")
				self._oldTitle	= self.getVar("title")
				self._oldTime	= int(self.getVar("playingTime"))

				self._trackQueued = 0
			# end if

			# Discard song if skipping has been detected.
			if ((not self._trackQueued) and
			self.getVar("playing") and
			((int(self.getVar("playingTime")) - self._oldTime) > (int(time.time()) - self._lastRefresh + 5))):
				self._Scrobbler.scrobPrint("  Skipping detected; not submitting.")
				self._trackQueued = 1
			# end if

			# Update times
			self._lastRefresh = int(time.time())
			self._oldTime = int(self.getVar("playingTime"))

			# Only queue according to guidelines.
			if ((not self._trackQueued) and
			((int(self.getVar("playingTime")) > (int(self.getVar("durationSecs")) / 2)) or int(self.getVar("playingTime")) >= 240) and
			self.getVar("playing")):
				self._trackQueued = 1
				self._Scrobbler._runQueue = 1

				if (int(self.getVar("durationSecs")) < 30):
					self._Scrobbler.scrobPrint("  [!!] Duration is less than 30 seconds; not queuing.")
					return(1)
				if (int(self.getVar("durationSecs")) > 1800):
					self._Scrobbler.scrobPrint("  [!!] Duration is greater than 30 minutes; not queuing.")
					return(1)
				if (self.getVar("artist").lower() == "unknown"):
					self._Scrobbler.scrobPrint("  [!!] Artist is unknown; not queuing.")
					return(1)
				if (self.getVar("title").lower() == "unknown"):
					self._Scrobbler.scrobPrint("  [!!] Title is unknown; not queuing.")
					return(1)
				# end if

				# Format time correctly for AudioScrobbler.
				T = time.gmtime(time.time() - int(self.getVar("playingTime")))
				queueTime = u"%04d-%02d-%02d %02d:%02d:%02d" % (T[0], T[1], T[2], T[3], T[4], T[5])

				# Generate POST data for updates:
				#	u - username
				#	s - md5 response
				#	a - artist
				#	t - title
				#	b - album
				#	m - musicbrainz id
				#	l - length (secs)
				#	i - time (UTC)
				indexNumber = len(self._Scrobbler._queue)
				postData = ("a[%d]=%s&t[%d]=%s&b[%d]=%s&m[%d]=%s&l[%d]=%s&i[%d]=%s") % (
					indexNumber, urllib.quote_plus(self._Scrobbler.utf8Enc(unescape(self.getVar("artist").strip()))),
					indexNumber, urllib.quote_plus(self._Scrobbler.utf8Enc(unescape(self.getVar("title").strip()))),
					indexNumber, urllib.quote_plus(self._Scrobbler.utf8Enc(unescape(self.getVar("album").strip()))),
					indexNumber, "",
					indexNumber, urllib.quote_plus(self._Scrobbler.utf8Enc(str(self.getVar("durationSecs")))),
					indexNumber, urllib.quote_plus(queueTime)
				)

				# Add it to the queue.
				self._Scrobbler.addToQueue(postData)
			# end if


			# Run queue if necessary.
			self._Scrobbler.runQueue()
		except Exception, error:
			self.handleException("__init__::doScrobbler()", error)
	#-------------------------------------------------------------------


	def call_action(self, call, path, args = []):
	#-------------------------------------------------------------------
		"""Handle actions initiated from display."""

		if (call == "back"):
			self.dPrint("Rhythmlet::back()")
			self._RB.previousTrack()
		#---------------------------------------------------------------
		elif (call == "next"):
			self.dPrint("Rhythmlet::next()")
			self._RB.nextTrack()
		#---------------------------------------------------------------
		elif (call == "playPause"):
			self.dPrint("Rhythmlet::playPause()")
			self._RB.playPause()
		#---------------------------------------------------------------


		elif (call == "setSeekPos"):
			#self.dPrint("setSeekPos(" + str(args[0]) + ")")
			self._seekPos = args[0]
		#---------------------------------------------------------------
		elif (call == "setSeekScroll"):
			self.dPrint("Rhythmlet::setSeekScroll(" + str(args[0]) + ")")

			if (self._RB.hasSongInfo()):
				if (args[0] == 0):
					myNewTime = self._RB["playingTime"] - int(self._get_config("scrollSecs"))
				elif (args[0] == 1):
					myNewTime = self._RB["playingTime"] + int(self._get_config("scrollSecs"))
				# end if
			
				if (myNewTime < 0):
					myNewTime = 0
				# end if

				self._RB["playingTime"] = myNewTime

				# Don't lag on gauge update:
				self.setVar("progress", (100 * myNewTime / self._RB["duration"]))
			# end if
		#---------------------------------------------------------------
		elif (call == "setSeek"):
			self.dPrint("Rhythmlet::setSeek()")

			if (self._RB.hasSongInfo()):
				self._RB["playingTime"] = ((float(self._seekPos)/float(self._seekLength)) * int(self._RB["duration"]))

				# don't lag on gauge update:
				self.setVar("progress", (100 * self._RB["playingTime"] / self._RB["duration"]))
			# end if
		#---------------------------------------------------------------


		elif (call == "rateTo1"):
			self.dPrint("Rhythmlet::rateTo1()")
			self._RB["rating"] = 1
		#---------------------------------------------------------------
		elif (call == "rateTo2"):
			self.dPrint("Rhythmlet::rateTo2()")
			self._RB["rating"] = 2
		#---------------------------------------------------------------
		elif (call == "rateTo3"):
			self.dPrint("Rhythmlet::rateTo3()")
			self._RB["rating"] = 3
		#---------------------------------------------------------------
		elif (call == "rateTo4"):
			self.dPrint("Rhythmlet::rateTo4()")
			self._RB["rating"] = 4
		#---------------------------------------------------------------
		elif (call == "rateTo5"):
			self.dPrint("Rhythmlet::rateTo5()")
			self._RB["rating"] = 5
		#---------------------------------------------------------------


		elif (call == "launchBrowser"):
			self.dPrint("Rhythmlet::launchBrowser()")
			url = self.getVar("link")
			url = urllib.quote(url, ":/")
			self.dPrint("opening link: " + str(url))

			threading.Thread(target=os.system, args=("gnome-open \"%s\"" % (url),)).start()
		#---------------------------------------------------------------


		elif (call == "toggleShuffle"):
			self.dPrint("Rhythmlet::toggleShuffle()")

			if (self._RB["shuffle"]):
				self._RB["shuffle"] = 0
			else:
				self._RB["shuffle"] = 1
			# end if
		#---------------------------------------------------------------
		elif (call == "toggleRepeat"):
			self.dPrint("Rhythmlet::toggleRepeat()")

			if (self._RB["repeat"]):
				self._RB["repeat"] = 0
			else:
				self._RB["repeat"] = 1
			# end if
		#---------------------------------------------------------------


		elif (call == "handleFile"):
			self.dPrint("Rhythmlet::handleFile():")

			for file in args[0]:
				deleteAfterCopy = 0

				# Use bad hack to bypass caching.
				if (self.getVar("imgPath") != "/tmp/rhythmlet.png"):
					self.setVar("imgPath", "/tmp/rhythmlet.png")
				else:
					self.setVar("imgPath", "/tmp/rhythmlet-uncache.png")
				# end if
				myImgPath = self.getVar("imgPath")

				self.dPrint("  \"" + file + "\"")

				if ((file[0:7] == "http://") or (file[0:6] == "ftp://")):
					file = file.replace(" ", "%20")
					self.dPrint("    Downloading \"" + file + "\"...")
					file = urllib.urlretrieve(file, )[0]
					deleteAfterCopy = 1
				# end if

				try:
					for ext in self._RLCover.fileExtensions:
						# Check if file is an acceptable image file.
						if (file[-len(ext):] == ext):

							filenames = ["%s-%s.%s" % (self.normalizeString(self._RB["artist"]),
														self.normalizeString(self._RB["album"]),
														ext)
										]
							file = file.replace('file://', '')

							# First fetch the song path and do an extra check
							# that we really have _no_ cover art...
							(localCover, candidates, preference, songPath) = self._RLCover.getCover_local(self.getVar("artist"), self.getVar("album"), self.getVar("filePath"))
							if (localCover or candidates):
								filenames = []
							# end if

							for filename in filenames:
								# Save image in same directory as current playing song.
								self._RLCover.copyFile(file, os.path.join(songPath, filename))

								if (HAVE_PIL == 1):
									# Copy cover image to /tmp so it can be resized.
									if (self._RLCover.copyFile(file, myImgPath, deleteAfterCopy)):
										# Resize it.
										(newWidth, newHeight) = self._RLCover.resizeImage(myImgPath, self._get_config("coverSize"), self._get_config("showOutline"))
									# end if
								# end if

								print "Setting new cover: " + myImgPath
								self.setVar("cover", myImgPath)
							# end for
						else:
							self._RB.handleFile(file)
						# end if
					# end for
				except Exception, error:
					self.handleException("Rhythmlet::handleFile()", error)

					print error
				# end try
			# end for
	#-------------------------------------------------------------------
		# end if
	#-------------------------------------------------------------------


	def get_configurator(self):
    #-------------------------------------------------------------------
		configurator = self._new_configurator()

		configurator.set_name(_("Configuration"))
		configurator.add_title(_("Covers:"))
		configurator.add_checkbox(_("Show covers"), "showCover", _("Display cover images"))
		configurator.add_checkbox(_("Check for covers locally"), "tryLocal", _("Check for cover images in the same directory as current audio file"))
		if (HAVE_SOAP):
			configurator.add_checkbox(_("Fetch covers from Amazon.com"), "tryAmazon", _("Check Amazon.com for cover images"))
			configurator.add_checkbox(_("Store covers from Amazon.com locally"), "saveAmazon", _("Store downloaded cover images from Amazon.com locally"))
		if (HAVE_PIL):
			configurator.add_spin(_("Cover size:"), "coverSize", _("Size of cover in pixels"), 25, 250)
		configurator.add_checkbox(_("Outline covers"), "showOutline", _("Display outline on cover images"))
		
		configurator.add_title(_("Display:"))
		configurator.add_entry(_("Display string:"), "dispString", _("%a - artist\n%A - album\n%c - play count\n%d - duration\n%e - elapsed time\n%g - genre\n%p - path\n%r - rating\n%s - filesize\n%t - title\n%T - track number\n---\n%N - new line"))
		configurator.add_entry(_("Background colour:"), "backgroundColour", _("Colour of background (#rrggbb; or #rrggbbaa, where aa represents the alpha channel)"))

		configurator.add_title(_("Behaviour:"))
		configurator.add_spin(_("Text scroll limit (chars):"), "scrollChars", _("Number of characters to display when scrolling text"), 10, 1000)
		configurator.add_spin(_("Scroll by (secs):"), "scrollSecs", _("Number of seconds used when adjusting seek position on mouse scroll"), 1, 60)

		configurator.add_title(_("Audioscrobbler.com:"))
		configurator.add_entry(_("Username:"), "asUsername", _("Username to use when updating Audioscrobbler.com"))
		configurator.add_entry(_("Password:"), "asPassword", _("Password to use when updating Audioscrobbler.com"), passwd = 1)
		configurator.add_checkbox(_("Enable EXPERIMENTAL Audioscrobbler support"), "enableAudioScrobbler", _("Whether to update Audioscrobbler.com with current track information.\nSee audioscrobbler.com for more details."))

		return configurator
	#-------------------------------------------------------------------

	def configWatcher(self, name = "", value = ""):
	#-------------------------------------------------------------------
		# Print only non-sensitive information to debug.
		if (name != "asPassword"):
			self.dPrint("Configurator change: " + name + " = " + value)
		else:
			self.dPrint("Configurator change: asPassword = [hidden]")
		# end if

		# Cover stuff:
		if ((name == "coverSize") or (name == "showOutline")):
			# force update of cover.
			self.dPrint("  Forcing update of cover")

			self.getCover()
		# end if

		if (name == "showCover"):
			if (value == "false"):
				# Hide cover
				self.dPrint("  Hiding cover.")

				self.setVar("coverSizeX", 0)
				self.setVar("coverSizeY", 0)
			else:
				# Foce update of cover.
				self.dPrint("  Forcing update of cover")

				self.getCover()
			# end if

			# Send vars immediately (don't wait for refresh() to do it)
			self.sendVars()
		# end if


		# AudioScrobbler stuff:
		if (name == "asUsername"):
			self._Scrobbler.setUsername(value)
		if (name == "asPassword"):
			# 'value' doesn't hold the clear-text password.
			self._Scrobbler.setPassword(self._get_config("asPassword"))
		# end if
	#-------------------------------------------------------------------


	def scrollText(self, input, limit=0):
	#-------------------------------------------------------------------
		"""Make text wrap around itself (1 char per second.)"""

		# Based on code from CornerXMMS, written by
		# Stefano Mozzi <stefano at mompracem dot org>

		try:
			retVal = []
			arrayInput = input.split("\n")

			for i in arrayInput:
				if (type(i) != types.UnicodeType):
					i = i.decode("utf8")

				tmpLimit = limit
				if (limit == 0):
					tmpLimit = len(i)

				if (len(i) <= tmpLimit):
					retVal.append(i)
				else:
					tmpStr = "%s  *  " % i
					tmpStr = tmpStr.replace("&amp;", "&")
					x = int(time.time()) % len(tmpStr)

					tmpStr = "%s%s%s" % (tmpStr[x:],
										tmpStr,
										tmpStr[:x])
					retVal.append(escape(tmpStr[0:tmpLimit]))
				# end if

			return(string.join(retVal, "\n"))
		except Exception, error:
			self.handleException("__init__.py::scrollText()", error)
	#-------------------------------------------------------------------


	def getVar(self, var):
	#-------------------------------------------------------------------
		return(self._dictVars[var])
	#-------------------------------------------------------------------
	def setVar(self, var, data):
	#-------------------------------------------------------------------
		if (self._dictVars[var] != data):
			self._dictVars[var] = data
		# end if

		# If new cover is set, set also coverSize{X,Y} to appropriate values.
		if (var == "cover"):
			(newWidth, newHeight) = self._RLCover.getImageSize(data)
			if (data[0:4] != "gfx/"):
				self._dictVars['coverSizeX'] = newWidth
				self._dictVars['coverSizeY'] = newHeight
			else:
				self._dictVars['coverSizeX'] = 50
				self._dictVars['coverSizeY'] = 50
		# end if
	#-------------------------------------------------------------------
	def sendVars(self):
	#-------------------------------------------------------------------
		output = self._new_output()

		for i in self._dictVars.keys():
			data = self._dictVars[i]
			if (data != self._dictVarsOld[i]):
				#self.dPrint("output.set('" + i + "', '" + str(self._dictVars[i]) + "')")
				output.set(i, data)

				self._dictVarsOld[i] = self._dictVars[i]
			# end if
		# end for

		self._send_output(output)
	#-------------------------------------------------------------------


	def unfuckBonobo(self):
	#-------------------------------------------------------------------
	# this seems to break the sensor out after a Rhythmbox crash
	#  (but not always)
		self._RB.unfuckBonobo()

		return 1
	#-------------------------------------------------------------------
#-----------------------------------------------------------------------


def new_sensor(args):
#-----------------------------------------------------------------------
    return Rhythmlet(*args)
#-----------------------------------------------------------------------
