/*
 *      Copyright (C) 1998 Claus-Justus Heine.

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *
 *      This file contains some signal testing program for ftape
 *
 *      Compile with "gcc -o ftsignal ftsignal.o"
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/mtio.h>
#include <signal.h>
#include <getopt.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>

#ifndef DEFTAPE
#define DEFTAPE "/dev/tape"	/* default tape device */
#endif /* DEFTAPE */

static const char *short_options = "b:d:f:s:";
static const struct option long_options[] =
{
	{"delay", 1, 0, 'd'},
	{"file", 1, 0, 'f'},
	{"help", 0, 0, 'h'},
	{"signal", 1, 0, 's'},
};

size_t bufsize = 100 << 10;
int killdelay = 10;
int trysignal = SIGUSR1;

int got_signal;

void handler(int sig)
{
	time_t tm = time(NULL);

	printf("HANDLER: Pid %d got signal %d at %s",
	       getpid(), sig, ctime(&tm));
	got_signal = sig;
	(void)signal(trysignal, handler);
}

void child(void)
{
	time_t tm;

	printf("CHILD: pid %d\n", getpid());
	sleep(1);
	kill(getppid(), trysignal);
	pause();
	if (got_signal == trysignal) {
		printf("CHILD: Parent is ready, sleeping %d seconds\n",
		       killdelay);
		sleep(killdelay);
	} else {
		fprintf(stderr, "CHILD: caught signal %d\n", got_signal);
		exit(1);
	}
	tm = time(NULL);
	printf("CHILD: Killing parent again at %s", ctime(&tm));
	kill(getppid(), trysignal);
	printf("CHILD: Bye bye\n");
	exit(0);	
}

void usage(FILE *out)
{
	fprintf(out, "Usage:\n"
		"ftsignal [-d, --delay=SECONDS] [-f, --file=TAPEDEV]\n"
		"         [-s --signal=SIGNR]\n");
}

int main(int argc, char *argv[])
{
	time_t tm;
	int result;
	int pd;
	char *tape = (getenv("TAPE") && *getenv("TAPE")) ?
		getenv("TAPE") : DEFTAPE;
	int tapefd;
	const struct mtop top = { MTRETEN, 0 };

	while (1) {
		int option_index = 0;
		int c;

		c = getopt_long(argc, argv, short_options,
				long_options, &option_index);
		if (c == -1) {
			break;
		}
		
		switch (c) {
		case 'b':
			bufsize = atoi(optarg);
				break;
		case 'd':
			killdelay = atoi(optarg);
			break;
		case 'f':
			tape = optarg;
			break;
		case 'h':
			usage(stdout);
			exit(0);
		case 's':
			trysignal = atoi(optarg);
			break;			
		default:
			fprintf(stderr,
				"PARENT: "
				"Unexpected result from getopt(): %c\n", c);
			usage(stderr);
			exit(1);
			break;
		}
	}
	if (optind < argc) {
		fprintf(stderr, "PARENT: %s takes no non-option arguments\n",
			argv[0]);
		exit(1);
	}
	if (!tape) {
		fprintf(stderr, "PARENT: No tape device specified\n");
	}
	(void)signal(trysignal, handler);

	printf("PARENT: will use signal %d\n", trysignal);
	printf("PARENT: Child will kill us %d seconds "
	       "from start of sending \"MTRETEN\"\n", killdelay);
	
	switch(pd = fork()) {
	case -1:
		perror("Couldn't create child process");
		exit(1);
	case 0:
		child();
		exit(0);
	default:
		printf("PARENT: Spawned child %d (self: %d)\n", pd, getpid());
		pause();
		if (got_signal == trysignal) {
			printf("PARENT: Child is ready\n");
		} else {
			fprintf(stderr, "PARENT: unexpected signal %d\n",
				got_signal);
			exit(1);
		}
		got_signal = 0;
		break;
	}

	tapefd = open(tape, O_RDONLY);
	if (tapefd == -1) {
		perror("PARENT: Couldn't open the tape device");
		exit(1);
	}
	kill(pd, trysignal);
	tm = time(NULL);
	printf("PARENT: Sending \"MTRETEN\" command to tape drive at %s", 
	       ctime(&tm));
	result = ioctl(tapefd, MTIOCTOP, &top);
	tm = time(NULL);
	printf("PARENT: \"MTRETEN\" finished at %s", ctime(&tm));

	if (result == -1) {
		perror("PARENT: Error sending \"MTRETEN\" to tape");
		exit(1);
	} else {
		printf("PARENT: Successfully retensioned the tape cartridge\n");
	}
	if (close(tapefd) == -1) {
		perror("PARENT: Error closing tape device");
		exit(1);
	}
	if (got_signal == 0) {
		fprintf(stderr,
			"PARENT: expected to catch signal %d while "
			"reading from tape, but got none\n", trysignal);
		exit(1);
	} else if (got_signal == trysignal) {
		printf("PARENT: expectedly got signal %d while "
		       "retensioning cartridge\n", got_signal);
		exit(0);
	} else {
		fprintf(stderr,
			"PARENT: expected to catch signal %d while "
			"retensioning tape, but got %d\n",
			trysignal, got_signal);
		exit(1);
	}
	return 0;
}
