/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  File-Roller
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <gtk/gtk.h>
#include "file-data.h"
#include "file-utils.h"
#include "fr-process.h"
#include "fr-command.h"


static void fr_command_class_init  (FRCommandClass *class);

static void fr_command_init        (FRCommand *afile);

static void fr_command_destroy     (GtkObject *object);


enum {
	START,
        DONE,
        LAST_SIGNAL
};


static GtkObjectClass *parent_class = NULL;
static guint fr_command_signals[LAST_SIGNAL] = { 0 };


static void
base_fr_command_list (FRCommand *comm)
{
}


static void
base_fr_command_add (FRCommand *comm,
		     GList *file_list,
		     gchar *base_dir,
		     gboolean update)
{
}


static void
base_fr_command_delete (FRCommand *comm,
			GList *file_list)
{
}


static void
base_fr_command_extract (FRCommand *comm,
			 GList *file_list,
			 char *dest_dir,
			 gboolean overwrite,
			 gboolean skip_older,
			 gboolean junk_paths)
{
}


static void
base_fr_command_uncompress (FRCommand *comm)
{
}


static void
base_fr_command_recompress (FRCommand *comm)
{
}


static void
base_fr_command_handle_error (FRCommand *comm, 
			      FRProcError *error)
{
}


static void 
fr_command_class_init (FRCommandClass *class)
{
        GtkObjectClass *object_class;

        parent_class = gtk_type_class (GTK_TYPE_OBJECT);
        object_class = (GtkObjectClass*) class;

	fr_command_signals[START] =
                gtk_signal_new ("start",
                                GTK_RUN_LAST,
                                object_class->type,
                                GTK_SIGNAL_OFFSET (FRCommandClass, start),
                                gtk_marshal_NONE__INT,
                                GTK_TYPE_NONE, 1,
				GTK_TYPE_INT);
	fr_command_signals[DONE] =
                gtk_signal_new ("done",
                                GTK_RUN_LAST,
                                object_class->type,
                                GTK_SIGNAL_OFFSET (FRCommandClass, done),
                                gtk_marshal_NONE__INT_POINTER,
                                GTK_TYPE_NONE, 2,
				GTK_TYPE_INT,
				GTK_TYPE_POINTER);
	gtk_object_class_add_signals (object_class, fr_command_signals, 
                                      LAST_SIGNAL);

        object_class->destroy = fr_command_destroy;

	class->list           = base_fr_command_list;
	class->add            = base_fr_command_add;
	class->delete         = base_fr_command_delete;
	class->extract        = base_fr_command_extract;

	class->uncompress     = base_fr_command_uncompress;
	class->recompress     = base_fr_command_recompress;

	class->handle_error   = base_fr_command_handle_error;

	class->start          = NULL;
	class->done           = NULL;
}


static void 
fr_command_start (FRProcess *process,
		  gpointer data)
{
	FRCommand *comm = FR_COMMAND (data);
	gtk_signal_emit (GTK_OBJECT (comm), fr_command_signals[START], 
			 comm->action);
}


static void 
fr_command_done (FRProcess *process,
		 FRProcError *error, 
		 gpointer data)
{
	FRCommand *comm = FR_COMMAND (data);
	/*comm->file_list = g_list_reverse (comm->file_list); FIXME */

	if (error->type != FR_PROC_ERROR_NONE) 
		fr_command_handle_error (comm, error);

	gtk_signal_emit (GTK_OBJECT (comm), fr_command_signals[DONE], 
			 comm->action, 
			 error);
}


static void 
fr_command_init (FRCommand *comm)
{
	comm->filename = NULL;
	comm->e_filename = NULL;
	comm->file_list = NULL;
}


static void 
fr_command_destroy (GtkObject *object)
{
        FRCommand* comm;

        g_return_if_fail (object != NULL);
        g_return_if_fail (IS_FR_COMMAND (object));
  
	comm = FR_COMMAND (object);

	if (comm->filename != NULL)
		g_free (comm->filename);

	if (comm->e_filename != NULL)
		g_free (comm->e_filename);

	if (comm->file_list != NULL) {
		g_list_foreach (comm->file_list, 
				(GFunc) file_data_free, 
				NULL);
		g_list_free (comm->file_list);
	}

	gtk_signal_disconnect_by_data (GTK_OBJECT (comm->process), comm);
	gtk_object_unref (GTK_OBJECT (comm->process));

	/* Chain up */
        if (GTK_OBJECT_CLASS (parent_class)->destroy)
                (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


GtkType
fr_command_get_type ()
{
        static guint fr_command_type = 0;

        if (!fr_command_type) {
                GtkTypeInfo fr_command_info = {
                        "FRCommand",
                        sizeof (FRCommand),
                        sizeof (FRCommandClass),
                        (GtkClassInitFunc) fr_command_class_init,
                        (GtkObjectInitFunc) fr_command_init,
                        /* reserved_1 */ NULL,
                        /* reserved_2 */ NULL,
                        (GtkClassInitFunc) NULL,
                };
                fr_command_type = gtk_type_unique (gtk_object_get_type(),
						   &fr_command_info);
        }

        return fr_command_type;
}


void
fr_command_construct (FRCommand *comm,
		      FRProcess *process,
		      char *fr_command_name)
{
	fr_command_set_filename (comm, fr_command_name);

	gtk_object_ref (GTK_OBJECT (process));
	comm->process = process;
	gtk_signal_connect (GTK_OBJECT (comm->process), "start",
			    GTK_SIGNAL_FUNC (fr_command_start),
			    comm);
	gtk_signal_connect (GTK_OBJECT (comm->process), "done",
			    GTK_SIGNAL_FUNC (fr_command_done),
			    comm);
}


void
fr_command_set_filename (FRCommand *comm,
			 char *filename)
{
	g_return_if_fail (IS_FR_COMMAND (comm));

	if (comm->filename != NULL)
		g_free (comm->filename);

	if (comm->e_filename != NULL)
		g_free (comm->e_filename);

	if (! g_path_is_absolute (filename)) {
		char *current_dir;
		current_dir = g_get_current_dir ();
		comm->filename = g_strconcat (current_dir, 
					      "/", 
					      filename, 
					      NULL);
		g_free (current_dir);
	} else 
		comm->filename = g_strdup (filename);

	comm->e_filename = shell_escape (comm->filename);

#ifdef DEBUG
	g_print ("filename : %s\n", comm->filename);
	g_print ("e_filename : %s\n", comm->e_filename);
#endif
}


void
fr_command_list (FRCommand *comm)
{
	g_return_if_fail (IS_FR_COMMAND (comm));

	if (comm->file_list != NULL) {
		g_list_foreach (comm->file_list, (GFunc) file_data_free, NULL);
		g_list_free (comm->file_list);
		comm->file_list = NULL;
	}

	comm->action = FR_ACTION_LIST;
	FR_COMMAND_CLASS (GTK_OBJECT (comm)->klass)->list (comm);
}


void
fr_command_add (FRCommand *comm,
		GList *file_list,
		gchar *base_dir,
		gboolean update)
{
	comm->action = FR_ACTION_ADD;
	FR_COMMAND_CLASS (GTK_OBJECT (comm)->klass)->add (comm, 
							  file_list,
							  base_dir,
							  update);
}


void
fr_command_delete (FRCommand *comm,
		   GList *file_list)
{
	comm->action = FR_ACTION_DELETE;
	FR_COMMAND_CLASS (GTK_OBJECT (comm)->klass)->delete (comm, file_list);
}


void
fr_command_extract (FRCommand *comm,
		    GList *file_list,
		    char *dest_dir,
		    gboolean overwrite,
		    gboolean skip_older,
		    gboolean junk_paths)
{
	comm->action = FR_ACTION_EXTRACT;
	FR_COMMAND_CLASS (GTK_OBJECT (comm)->klass)->extract (comm, 
							      file_list, 
							      dest_dir,
							      overwrite,
							      skip_older,
							      junk_paths);
}


void
fr_command_uncompress (FRCommand *comm)
{
	FR_COMMAND_CLASS (GTK_OBJECT (comm)->klass)->uncompress (comm);
}


void
fr_command_recompress (FRCommand *comm)
{
	FR_COMMAND_CLASS (GTK_OBJECT (comm)->klass)->recompress (comm);
}


void
fr_command_handle_error (FRCommand *comm,
			 FRProcError *error)
{
	FR_COMMAND_CLASS (GTK_OBJECT (comm)->klass)->handle_error (comm, 
								   error);
}

