/*
    ext2_journal.c -- ext2 resize journalling support
    Copyright (C) 1999, 2000 Lennert Buytenhek <buytenh@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char _ext2_journal_c[] = "$Id: ext2_journal.c,v 1.11 2000/10/10 18:34:33 adilger Exp $";

#define USE_EXT2_IS_DATA_BLOCK
#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ext2.h"

#define JOURNAL_SIG0 0x2b863b7a		/* 2b863b7aecd79fc1bf59c55b45fe16d9 is the md5sum of		*/
#define JOURNAL_SIG1 0xecd79fc1		/* "This program is dedicated to Marloes van der Valk\n"	*/
#define JOURNAL_SIG2 0xbf59c55b
#define JOURNAL_SIG3 0x45fe16d9


struct ext2_journal
{
	ino_t		 ino;
	int		 dirty;
	__u32		*buf;
	int		 ptr;
	int		 size;
	blk_t		 block[4];
};






static int create_journal_inode(struct ext2_fs *fs, struct ext2_journal *j)
{
	struct ext2_inode inode;
	ino_t		  inodenum;

	if (!fs->sb.s_free_inodes_count) {
		fprintf(stderr, "need at least 1 free inode for a journal!\n");
		return 0;
	}

	inodenum = ext2_find_free_inode(fs);

	memset(&inode, 0, sizeof(inode));
	inode.i_mode = 0100400;
	inode.i_size = 4 << fs->logsize;
	inode.i_atime = JOURNAL_SIG0;
	inode.i_ctime =	JOURNAL_SIG1;
	inode.i_mtime =	JOURNAL_SIG2;
	inode.i_dtime = JOURNAL_SIG3;
	inode.i_links_count = 1;
	inode.i_blocks = 8 << fs->logsize;
	inode.i_block[0] = j->block[0];
	inode.i_block[1] = j->block[1];
	inode.i_block[2] = j->block[2];
	inode.i_block[3] = j->block[3];

	ext2_write_inode(fs, inodenum, &inode);
	ext2_set_inode_state(fs, inodenum, 1, 1);

	return 1;
}

static ino_t find_journal_inode(struct ext2_fs *fs)
{
	ino_t i;

	for (i = 1; i < fs->sb.s_inodes_count; i++) {
		struct ext2_inode inode;

		ext2_read_inode(fs, i, &inode);
		if (inode.i_atime == JOURNAL_SIG0 &&
		    inode.i_ctime == JOURNAL_SIG1 &&
		    inode.i_mtime == JOURNAL_SIG2) {
			if (inode.i_dtime != JOURNAL_SIG3)
				fprintf(stderr,
					"you ran e2fsck after ext2resize crashed? Hm.\n");

			return i;
		}
	}

	return 0;
}










int ext2_journal_replay(struct ext2_fs *fs)
{
	return 0;
}

int ext2_journal_invalidate(struct ext2_fs *fs)
{
	struct ext2_journal *j;

	j = fs->journal;
	return 0;
}

int ext2_journal_open(struct ext2_fs *fs)
{
	return 0;
}

int ext2_journal_create(struct ext2_fs *fs)
{
	return 0;
}

int ext2_journal_flush(struct ext2_fs *fs)
{
	struct ext2_journal *j;

	j = fs->journal;
	if (j->dirty) {
		blk_t block;
		unsigned char *buf;
		int i;

		buf = (unsigned char *)j->buf;

		for (i = 1; i < 4; i++) {
			block = j->block[i];
			ext2_bcache_flush_range(fs, block, 1);
			ext2_write_blocks(fs, buf + (i<<fs->logsize), block, 1);
		}

		ext2_sync(fs);

		block = j->block[0];
		ext2_bcache_flush_range(fs, block, 1);
		ext2_write_blocks(fs, buf, block, 1);

		ext2_sync(fs);

		j->dirty = 0;
	}

	return 0;
}

int ext2_journal_write(struct ext2_fs *fs, __u32 *data, int num)
{
	int i;
	struct ext2_journal *j;

	j = fs->journal;
	if (j->ptr + num >= j->size)
		return 0;

	for (i = 0; i < num; i++)
		j->buf[j->ptr++] = data[i];

	return 1;
}





int ext2_journal_init(struct ext2_fs *fs)
{
	struct ext2_journal *j;

	if ((j = (struct ext2_journal *)
	     malloc(sizeof(struct ext2_journal))) == NULL) {
		fprintf(stderr, "MAError!\n");
		return 0;
	}

	j->size = 1 << fs->logsize;

	if ((j->buf = (__u32 *)malloc(j->size << 2)) == NULL) {
		fprintf(stderr, "MAError!\n");
		return 0;
	}

	j->ino = find_journal_inode(fs);
	j->dirty = 0;
	j->ptr = 0;

	printf("journal: %li\n", j->ino);

	fs->journal = j;

	return 1;
}

void ext2_journal_deinit(struct ext2_fs *fs)
{
	struct ext2_journal *j;

	j = fs->journal;
	free(j->buf);
	free(j);
}


#if 0
int ext2_journal_init(struct ext2_fs *fs)
{
	ino_t journal;

	if (ext2_journal_size > 256) {
		fprintf(stderr, "max journal size is currently 256\n");
		ext2_journal_size = 256;
	}

	if ((journal = find_journal_inode(fs)) != 0) {
		fprintf(stderr, "we have a journal already!\n");
		fprintf(stderr, "replay it now\n");
		fprintf(stderr, "delete it now\n");
		return 1;
	}

	printf("creating journal\n");

	if ((journal = create_journal_inode(fs)) == 0) {
		fprintf(stderr, "error creating journal file!\n");
		return 0;
	}

	printf("journal is %li\n", journal);

	return 1;
}

void ext2_journal_deinit(struct ext2_fs *fs)
{
}
#endif
