/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/********************************************************************
*
*  File: storage.c
*
*  Purpose:  File defining details of storage implementation.
*            All machine-dependent gory details should be here
*            (purely private details) or in storage.h
*            (for inclusion in other source files that need to know).
*                
*        This version has element ids implemented as longs.
*/
#ifdef WIN32
#include <windows.h>
#endif

#include "include.h"


struct blocklist_struct *blocklist[NUMELEMENTS]; /* list of allocated blocks */
int blockcount[NUMELEMENTS];  /* how many blocks allocated */
int blockmax[NUMELEMENTS];  /* length of blocklist */

/* individual indirect block pointer arrays, handy for debugging */
INDIRECT_TYPE *vibase;
INDIRECT_TYPE *eibase;
INDIRECT_TYPE *fibase;
INDIRECT_TYPE *bibase;
INDIRECT_TYPE *feibase;

/* unified block references, indexed by element type */
INDIRECT_TYPE *ibase[NUMELEMENTS]; /* indirect */
int ialloc[NUMELEMENTS]; /* allocated length of ibase's */


struct webstruct web = {
{ { VERTEX, 1, 0, NULL, NULL, 0, NULLID, NULLID, 0, NULLID },
 { EDGE  , 2, 0, NULL, NULL, 0, NULLID, NULLID, 0, NULLID },
 { FACET , 3, 0, NULL,  NULL, 0, NULLID, NULLID, 0, NULLID },
 { BODY  , 0, 0, NULL,  NULL, 0, NULLID, NULLID, 0, NULLID },
 { FACETEDGE, 0,0,  NULL, NULL, 0, NULLID, NULLID, 0, NULLID } },
 {0,0,0,0,0},3,2,LINEAR
     };


element_id
  NULLVERTEX  =  VERTEX  << TYPESHIFT, 
  NULLEDGE     =  EDGE  << TYPESHIFT,
  NULLFACET    =  FACET  << TYPESHIFT,
  NULLBODY     =  BODY  << TYPESHIFT,  
  NULLFACETEDGE =  (unsigned long)FACETEDGE << TYPESHIFT;


struct element *elptr(id)
element_id id;
{
  int type = id_type(id);
  return (struct element *)(ibase[type][id & OFFSETMASK]);
}

int oid(id)
element_id id;
{ return inverted(id) ? -(ordinal(id)+1) : (ordinal(id) +1) ;
}
/* handy for debugging */
struct vertex * Vptr(id) vertex_id id; { return vptr(id); }
struct edge * Eptr(id) edge_id id; { return eptr(id); }
struct facet * Fptr(id) facet_id id; { return fptr(id); }
struct body * Bptr(id) body_id id; { return bptr(id); }
struct facetedge * Feptr(id) facetedge_id id; { return feptr(id); }

/***********************************************************************
*
*  function: expand()
*
*  purpose:  Increase size of element structures.  Works only for
*                indexed or indirect id. Will also decrease size.
*/
void expand(type,newsize)
int type;  /* VERTEX, etc. */
int newsize; /* new size of structure */
{ char *newblock;
  int oldsize = web.sizes[type];
  int count = web.skel[type].maxcount;
  int i,n;
  char *new,*old;

  ENTER_GRAPH_MUTEX;
  /* Don't mess with structures while graph thread using them */
  LEAVE_GRAPH_MUTEX;

  /* round up newsize to alignment for doubles */
  newsize = ((newsize+sizeof(REAL)-1)/sizeof(REAL))*sizeof(REAL);
  if ( newsize == web.sizes[type] ) return; /* don't have to expand */
  web.sizes[type] = newsize;
  if ( count == 0 ) return; /* don't have to reallocate space */

  for ( n = 0 ; n < blockcount[type] ; n++ )
  { INDIRECT_TYPE *iptr;
     char *spot;
     struct blocklist_struct *b = blocklist[type] + n;
     int copysize;

     newblock = mycalloc(b->count,newsize);
     copysize = (newsize < oldsize) ? newsize : oldsize;
     for ( i = 0, old = (char*)(b->blockptr), new = newblock; 
             i < b->count ; i++, old += oldsize, new += newsize )
         memcpy(new,old,copysize);
     myfree((char*)b->blockptr);
     b->blockptr = (struct element*)newblock;
     /* update indirect pointers */
     for ( i=0,spot=newblock, iptr=ibase[type]+b->start_ord ; i<b->count ;
              i++,spot+=newsize,iptr++ )
        *iptr = (struct element *)spot;
  }

  parallel_update_flag[type] = 1;
} 

/***********************************************************************
*
*  function: extend()
*
*  purpose: allocate more empty element structures
*/

void extend(type)
int type;
{
  int number;  /* of new items to allocate */
  char *newblock = NULL;
  INDIRECT_TYPE *newiblock = NULL;
  element_id id,oldfreehead;
  struct element *new = NULL;
  int k;
  int allocsize;
  long  oldnum = web.skel[type].maxcount;
  long  newnum;
  int neword;

  ENTER_GRAPH_MUTEX;
  /* Don't mess with structures while graph thread using them */
  LEAVE_GRAPH_MUTEX;

  if ( blockcount[type] >= blockmax[type] )
  { blocklist[type] = (struct blocklist_struct*)
      kb_realloc((char*)(blocklist[type]),
         (blockcount[type]+BATCHSIZE)*sizeof(struct blocklist_struct),
         blockcount[type]*sizeof(struct blocklist_struct));
     blockmax[type] += BATCHSIZE;
  }
  /* calculate number of structures to fit in block size just under 2^n */
  allocsize = BATCHSIZE*web.sizes[type];
  for ( k = 0x100 ; k < allocsize ; k <<= 1 );
  number = (k-16)/web.sizes[type]; /* maybe room for block header */
  newnum = web.skel[type].maxcount + number;

  newblock = mycalloc(number,web.sizes[type]);
  blocklist[type][blockcount[type]].start_ord = oldnum ;
  blocklist[type][blockcount[type]].count = number ;
  blocklist[type][blockcount[type]++].blockptr = (struct element *)newblock;
  while ( newnum > ialloc[type] )
  {
    if ( ibase[type] == NULL )
    { newiblock = (INDIRECT_TYPE*)mycalloc(number,sizeof(INDIRECT_TYPE));
      ialloc[type] = number;
    }
    else
    { newiblock = (INDIRECT_TYPE*)kb_realloc((char*)(ibase[type]),
        2*ialloc[type]*sizeof(INDIRECT_TYPE),ialloc[type]*sizeof(INDIRECT_TYPE));
      ialloc[type] *= 2;
    }
    ibase[type] = web.skel[type].ibase = newiblock;
    switch(type)
    {    case VERTEX: vibase = newiblock; break;
         case EDGE:    eibase = newiblock; break;
         case FACET:  fibase = newiblock; break;
         case BODY:    bibase = newiblock; break;
         case FACETEDGE: feibase = newiblock; break;
    }
  }
  /* add to freelist */
    neword = web.skel[type].maxcount;  
    id = ((long)type << TYPESHIFT) | VALIDMASK | neword; 
    oldfreehead = web.skel[type].free;
    web.skel[type].free = id;
    for ( k = 0 ; k < number ; k++ )
      { new = (struct element *)(newblock + k*web.sizes[type]); 
         ibase[type][neword] = new;
         neword++;
         id = ((long)type << TYPESHIFT) | VALIDMASK | neword; 
         new->forechain = id;
      }
    new->forechain = oldfreehead;

    web.skel[type].maxcount += number;
}

/************************************************************************
*
* function: new_element()
*
* purpose: Allocate new element from freelist
*
*/

element_id new_element(type,parent)
int type;
element_id parent; /* for inherited stuff */
{
  element_id newid;
  struct element *new,*last;
  ORDTYPE ord;
  
  newid = web.skel[type].free;
  
  if ( !valid_id(newid) ) /* free list empty */
    {
      extend(type);
      newid = web.skel[type].free;
    }
  new = elptr(newid);

  /* remove from free chain */
  web.skel[type].free = new->forechain;

  /* clean out old info */
  ord = ordinal(newid);
  memset((char *)new,0,web.sizes[type]);
  if ( ord > web.skel[type].max_ord ) web.skel[type].max_ord = ord;
  new->original = NO_ORIGINAL;


  if ( match_id_flag )
  { /* link in numerical order */
     int i;
     struct element *prev,*next;
     for ( i = ord-1 ; i >= 0 ; i-- )
     { prev = ibase[type][i];
        if ( prev->attr & ALLOCATED )
        { if ( valid_id(prev->forechain) )
          { next = elptr(prev->forechain);
             next->backchain = newid;
          }
          new->forechain = prev->forechain;
          prev->forechain = newid;
          new->backchain = prev->self_id;
          break;
        }
     }
     if ( i < 0 )  /* no predecessor */
        { new->forechain = web.skel[type].used;
          if ( valid_id(web.skel[type].used) )
          { next = elptr(web.skel[type].used);
             next->backchain = newid;
          }
          web.skel[type].used = newid;
        }
     if ( !valid_id(new->forechain) ) web.skel[type].last = newid;
  }
  else
  { /* add to end of in-use chain */
     new->forechain = NULLID;
     new->backchain = web.skel[type].last;
     if ( valid_id(web.skel[type].last) )
     {
        last = elptr(web.skel[type].last);     /* find end of in-use chain */
        last->forechain = newid;
     }
     else
     {        
        web.skel[type].used = newid;
     }
     web.skel[type].last = newid;
  }

  new->attr = ALLOCATED | NEWELEMENT;

  new->self_id = newid;
  web.skel[type].count++;  

  /* inherited named methods from parent */
  if ( valid_id(parent) )
  { struct element *e_ptr = elptr(parent);
     int *instlist = (int*)((char*)e_ptr + get_meth_offset(id_type(parent)));
     int i;
     for ( i = 0 ; i < (int)e_ptr->method_count ; i++ )
     { if ( METH_INST[instlist[i]].type <= type )
          apply_method(newid,METH_INST[instlist[i]].name);
     }
  }

  return newid;
}

void free_element(id)
element_id id;
{
  struct element *ptr;
  int type = id_type(id);

  if ( !valid_id(id) )
    {
      sprintf(errmsg,"Internal error: Trying to free invalid element type %d id %d \n",
            type,ordinal(id)+1);
      kb_error(1311,errmsg,WARNING);
      return;
    }
     
  if ( type == EDGE ) /* remove from vertex lists */
    { remove_vertex_edge(get_edge_tailv(id),id);
      remove_vertex_edge(get_edge_headv(id),inverse_id(id));
      if ( web.modeltype == QUADRATIC )
             free_element(get_edge_midv(id));
      if ( web.modeltype == LAGRANGE )
         kb_error(1312,"No free_element in Lagrange model yet.\n",RECOVERABLE );
    }
    
  ptr = elptr(id);
  if ( !(ptr->attr & ALLOCATED) )
    {
      sprintf(errmsg,"Internal error: Trying to free unallocated element type %d id %d \n",
            type,ordinal(id)+1);
      kb_error(1313,errmsg,WARNING);

      return;
    }
  ptr->attr &= ~ALLOCATED;

  /* remove from in-use list */
  if ( valid_id(ptr->forechain) )
     elptr(ptr->forechain)->backchain = ptr->backchain;
  else
     web.skel[type].last = ptr->backchain;

  if ( valid_id(ptr->backchain) )
     elptr(ptr->backchain)->forechain = ptr->forechain;
  else
     web.skel[type].used = ptr->forechain;

  /* add to discard list */
  /* save forechain for generators */
  ptr->backchain = web.skel[type].discard;
  web.skel[type].discard = id & (TYPEMASK | VALIDMASK | OFFSETMASK);
                                 /* clear old bits, keep only needed */

  web.skel[type].count--; 

}

/* reclaim element from discard list */
void unfree_element(id)
element_id id;
{
  struct element *ptr;
  int type = id_type(id);
  element_id *id_ptr;

  if ( !valid_id(id) )
    {
      sprintf(errmsg,"Internal error: Trying to unfree invalid id %08lX \n",id);
      kb_error(1314,errmsg,RECOVERABLE);

    }
    
  ptr = elptr(id);
  if ( ptr->attr & ALLOCATED )
    {
      sprintf(errmsg,"Internal error: Trying to unfree allocated element id %08lX \n",id);
      kb_error(1315,errmsg,RECOVERABLE);

    }
  ptr->attr |= ALLOCATED;

  /* remove from discard list */
  for ( id_ptr = &web.skel[type].discard ; valid_id(*id_ptr) ; 
                        id_ptr = &elptr(*id_ptr)->backchain )
     if ( equal_element(*id_ptr,id) )
        { *id_ptr = elptr(*id_ptr)->backchain;
          break;
        }
  if ( !valid_id(*id_ptr) )
     kb_error(1316,"Internal error: Could not find deleted element on discard list.\n",RECOVERABLE);


  /* add to start of in-use chain */
  if ( valid_id (web.skel[type].used ) )
     { struct element *used = elptr(web.skel[type].used);
        used->backchain = id;
     }
  ptr->forechain = web.skel[type].used;
  web.skel[type].used = id;
  ptr->backchain = NULLID;
  if ( !valid_id(web.skel[type].last ) )
     web.skel[type].last = id;

  web.skel[type].count++;  
}

/* index as id */
element_id get_ordinal_id(type,ord)
int type; /* type of element */
int ord;  /* ordinal of element */
{ element_id id;

  if ( (type < 0) || (type > NUMELEMENTS) ) return NULLID;
  if ( (ord < 0) || (ord > web.skel[type].max_ord)) return NULLID;
  id = (type << TYPESHIFT) | VALIDMASK | ord;
  if ( get_attr(id) & ALLOCATED ) return id;
  else return NULLID;
}

int generate_all(type,idptr)  /* re-entrant */
int type;
element_id *idptr;
{
  struct element *ptr;

  if ( !valid_id(*idptr) ) /* first time */
     { *idptr = web.skel[type].used;
        if ( !valid_id(*idptr) ) return 0;
     }
  else
     { ptr = elptr(*idptr);
        do
          { /* may have to get off discard list */
             *idptr = ptr->forechain;
             if ( !valid_id(*idptr) ) return 0;
             ptr = elptr(*idptr);
            }
        while ( !(ptr->attr & ALLOCATED) );
     }

  return 1;
}

void memory_report()
{
    long mem = 0;
    int k;

    mem = 0;
    for ( k = 0 ; k < NUMELEMENTS ; k++ )
      mem += web.skel[k].count*web.sizes[k];

    sprintf(msg,"Vertices: %ld    Edges: %ld    Facets: %ld    Facetedges: %ld    Memory: %ld\n",
		web.skel[0].count,web.skel[1].count,web.skel[2].count,web.skel[4].count,
      mem);
    outstring(msg);

#if defined(_WIN32) && defined(_HEAPOK)
#ifndef _HEAPINFO
#define _HEAPINFO _heapinfo
#endif
			if ( _heapchk() != _HEAPOK )
            kb_error(1317,"Internal error: Corrupt heap! Memory is trashed.\n",WARNING);
         else if ( memdebug )
				{ struct _HEAPINFO hinfo;
              int b_use=0,b_free=0;
              long mem_use=0,mem_free=0;

              hinfo._pentry = NULL;
              while ( _heapwalk(&hinfo) == _HEAPOK )
                 if (hinfo._useflag)
                    { 
#ifdef HEAPLIST
                      sprintf(msg,"%p %5ld ",hinfo._pentry,hinfo._size);
                      outstring(msg);
#endif
                      b_use++; mem_use+= hinfo._size;
                    }
                 else { b_free++; mem_free += hinfo._size; }
              outstring("\n");
             sprintf(msg,"blocks in use: %d    memory in use: %ld \n",
                    b_use,mem_use);
             outstring(msg);
             sprintf(msg,"blocks free:    %d    memory free:    %ld \n",
                    b_free,mem_free);
             outstring(msg);
            }
#endif

#if defined(M_MXFAST) && defined(IRIS)
  if ( memdebug )
     { char *ptr;
        struct mallinfo m;
        /* using libmalloc.a for debugging */
        ptr = malloc(10);
        if ( ptr == NULL )
          erroutstring("Bad heap.\n");
        else myfree(ptr);
        m = mallinfo();
        sprintf(msg,"Arena %d     Ordblocks: %d    Orduse: %d     Ordfree: %d\n",
            m.arena,m.ordblks,m.uordblks,m.fordblks);
        outstring(msg);
        sprintf(msg,"Small blocks: %d Small use: %d Small free: %d\n",
            m.smblks,m.usmblks,m.fsmblks);
        outstring(msg);
}
#endif

#if defined(SUNXX)
    if ( memdebug )
      if ( malloc_verify() != 1 )
         kb_error(1318,"Internal error: Malloc_verify() failed.\n.",RECOVERABLE);

#endif

#ifdef XENIX386
    if ( memdebug )
      { struct mallinfo minfo;
         minfo = mallinfo();
         printf("arena:  %d    blocks: %d     memory in use: %d    free: %d\n",
             minfo.arena,minfo.ordblks,minfo.uordblks,minfo.fordblks);
      }
#endif

}

/**************************************************************************
*
* function: reset_skeleton()
*
* purpose: Clean out old surface and initialize empty web.
*
*/

void reset_skeleton()
{
  int i,j;
  struct boundary *bdry;
  struct surf_energy *surfen;
  struct quantity *quan;
  int type;

  if ( gauss1Dpt ) { myfree((char *)gauss1Dpt); gauss1Dpt = NULL; }
  if ( gauss1Dwt ) { myfree((char *)gauss1Dwt); gauss1Dwt = NULL; }

  /* deallocate boundary specs */
  for ( i = 0, bdry = web.boundaries ; i < BDRYMAX ; i++, bdry++ )
     if ( bdry->coordf[0] )
        {
          for ( j = 0 ; j < SDIM ; j++ )
             free_expr(bdry->coordf[j]);
          myfree((char *)bdry->coordf[0]);
        }

  /* deallocate constraint specs */
  for ( i = 0 ; i < MAXCON ; i++ )
     if ( web.constraint_addr[i] ) constraint_free(GETCONSTR(i));

  /* deallocate surface energy specs */
  for ( i = 0, surfen = web.surfen ; i < web.surfen_count ; i++, surfen++ )
     { if ( (surfen->envect[0]) && !(surfen->attr & USURPED_BY_QUANTITY) )
          { for ( j = 0 ; j < SDIM ; j++ )
                 free_expr(surfen->envect[j]);
             myfree((char *)surfen->envect[0]);
          }
     }

  /* deallocate quantity specs */
  for ( i = 0, quan = web.quants ; i < web.quantity_count ; i++, quan++ )
     { if ( (quan->attr & IN_USE) && !(quan->attr & USURPED_BY_QUANTITY) )
          { for ( j = 0 ; j < SDIM ; j++ )
                 free_expr(quan->quanvect[j]);
             myfree((char *)quan->quanvect[0]);
          }
     }

  /* deallocate metric expressions */
  if ( web.metric_flag )
     for ( i = 0 ; i < SDIM ; i++ )
        for ( j = 0 ; j < SDIM ; j++ )
          free_expr(&web.metric[i][j]);

    for ( i = 0 ; i < NUMELEMENTS ; i++ )
     {
        for ( j = 0 ; j < blockcount[i] ; j++ )  
          myfree((char *)blocklist[i][j].blockptr);  
        if ( blocklist[i] ) myfree((char*)blocklist[i]);
        blocklist[i] = NULL; blockmax[i] = blockcount[i] = 0;
        if ( web.skel[i].ibase ) myfree((char *)web.skel[i].ibase);
        ialloc[i] = 0; ibase[i] = NULL;
     }

  if ( web.inverse_periods )  free_matrix(web.inverse_periods);

  memset((char *)&web,0,sizeof(web));  /* total clean out */

  web.sizes[VERTEX] = sizeof(struct vertex);
  web.sizes[EDGE] = sizeof(struct edge);
  web.sizes[FACET] = sizeof(struct facet);
  web.sizes[BODY] = sizeof(struct body);
  web.sizes[FACETEDGE] = sizeof(struct facetedge);

  for ( i = 0 ; i < NUMELEMENTS ; i++ ) web.skel[i].max_ord = -1;

  web.skel[EDGE].dimension = 1;
  web.skel[FACET].dimension = 2;
  web.skel[BODY].dimension = 3;

  /* set up permanent attributes, empty to start with */
  /* have REAL attributes first, so can align on 8-byte */
  /* Be sure the order given here is same as order in skeleton.h */
  /* vertex */
  add_attribute(VERTEX,"x",REAL_ATTR,0,0,NULL);
  add_attribute(VERTEX,"p",REAL_ATTR,0,0,NULL);
  add_attribute(VERTEX,"__force",REAL_ATTR,0,0,NULL);
  add_attribute(VERTEX,"__velocity",REAL_ATTR,0,0,NULL);
  add_attribute(VERTEX,"__v_boundary",INTEGER_ATTR,0,0,NULL);
  add_attribute(VERTEX,"star",INTEGER_ATTR,0,0,NULL);
  add_attribute(VERTEX,"__v_constraint_list",UCHAR_ATTR,0,0,NULL);
  /* edge */
  add_attribute(EDGE,"star",REAL_ATTR,0,0,NULL);
  add_attribute(EDGE,"density",REAL_ATTR,0,0,NULL);
  add_attribute(EDGE,"__e_vertices",ELEMENTID_ATTR,0,0,NULL);
  add_attribute(EDGE,"__e_boundary",INTEGER_ATTR,0,0,NULL);
  add_attribute(EDGE,"__wrap_list",INTEGER_ATTR,0,0,NULL);
  add_attribute(EDGE,"__surfen_map",INTEGER_ATTR,0,0,NULL);
  add_attribute(EDGE,"__num_quant_map",INTEGER_ATTR,0,0,NULL);
  add_attribute(EDGE,"__e_constraint_list",UCHAR_ATTR,0,0,NULL);
  /* facet */
  add_attribute(FACET,"__f_constraint_list",UCHAR_ATTR,0,0,NULL);
  add_attribute(FACET,"__f_vertices",ELEMENTID_ATTR,FACET_VERTS,0,NULL);
  add_attribute(FACET,"__f_boundary",INTEGER_ATTR,0,0,NULL);
  add_attribute(FACET,"__surfen_map",INTEGER_ATTR,0,0,NULL);
  add_attribute(FACET,"__num_quant_map",INTEGER_ATTR,0,0,NULL);
  add_attribute(FACET,"tag",INTEGER_ATTR,0,0,NULL);
  add_attribute(FACET,"__body_list",ELEMENTID_ATTR,0,0,NULL);
  add_attribute(FACET,"__next_vfacet_list",ELEMENTID_ATTR,0,0,NULL);
  add_attribute(FACET,"__next_bfacet_list",ELEMENTID_ATTR,0,0,NULL);
  add_attribute(FACET,"phase",INTEGER_ATTR,0,0,NULL);
  /* body */
  add_attribute(BODY,"__b_constraint_list",UCHAR_ATTR,0,0,NULL);

  /* all */
  for ( type = 0 ; type < NUMELEMENTS ; type++ )
     web.meth_attr[type] = add_attribute(type,"__method_list",INTEGER_ATTR,0,0,NULL);
}

/***********************************************************************
*
* function: free_discards()
*
* purpose: totally free up discard list.  to be called only when
*          no lists are being used.
*/
void free_discards()
{ int type;

  for ( type = 0 ; type < NUMELEMENTS ; type++ )
     { element_id id = web.skel[type].discard;
        while ( valid_id(id) )
          { struct element *ptr = elptr(id);
             web.skel[type].discard = ptr->backchain;
             ptr->forechain = web.skel[type].free;
             ptr->backchain = NULLID;
             web.skel[type].free = id;
             id = web.skel[type].discard;
          }
     }
}

/**************************************************************************
*
* function: move_to_free_front()
*
* purpose: get particular id element to front of free list so id will 
*             match datafile number.  Called only during datafile.
*/

void move_to_free_front(type,id)
int type; /* element type */
int id;    /* element id    */
{ int ord = id - 1;
  element_id eid;
  struct element *pptr,*bbptr;
  element_id prev;

  if ( !match_id_flag ) return; /* for old way */
  while ( id > web.skel[type].maxcount ) extend(type);
  /* linear search through free list */
  eid = web.skel[type].free;
  prev = NULLID;
  while ( valid_id(eid) )
  { if ( ordinal(eid) == ord ) break;
     prev = eid;
     eid = elptr(eid)->forechain;
  }
  if ( !valid_id(eid) )
  {  kb_error(2372,"Internal error: Cannot find element in free list.\n",DATAFILE_ERROR);
      return;
  }
  if ( eid == web.skel[type].free ) return; /* already in place */
  /* now cut and paste */
  bbptr = elptr(eid);
  pptr = elptr(prev);
  pptr->forechain = bbptr->forechain; /* cut */
  bbptr->forechain = web.skel[type].free; /* paste */
  web.skel[type].free = eid;
}

/*********************************************************************
* 
* Function: reorder_storage()
*
* Purpose: order storage of element structures according to value
* of element extra attribute order_key (real). Meant to order storage
* in cache and swap friendly way.  Invoked by command reorder_storage.
*
* Elements generators work through forechain pointer, so we must
* re-order storage and re-link forechain pointers.
**********************************************************************/

static int key_offset; /* offset of key in element structure */
static char * keynames[NUMELEMENTS] = {"vertex_order_key",
    "edge_order_key","facet_order_key","body_order_key",
    "facetedge_order_key"};

int esort (a,b)  /* sort routine */
char *a, *b;
{ if ( *(REAL*)(a+key_offset) < *(REAL*)(b+key_offset) ) return -1;
  if ( *(REAL*)(a+key_offset) > *(REAL*)(b+key_offset) ) return  1;
  return 0;
}

void reorder_storage()
{ struct element *newblock[NUMELEMENTS];
  int i,j,n;

  /* allocate single-block space for each type  */
  for ( n = 0 ; n < NUMELEMENTS ; n++ )
     newblock[n] = (struct element *)mycalloc(web.skel[n].count+1,web.sizes[n]);
  
  /* copy element structures. */
  for ( n = 0 ; n < NUMELEMENTS ; n++ )
  { element_id id = web.skel[n].used;
     char *spot = (char*)(newblock[n]);
     while ( valid_id(id) )
     { struct element *ep = elptr(id);
        memcpy(spot,(char*)ep,web.sizes[n]);
        id = ep->forechain;
        spot += web.sizes[n];
     }
  }

  /* sort */
  for ( i = 0 ; i < NUMELEMENTS ; i++ )
  { struct extra *ex;
     int k;
     
          if ( web.skel[i].count == 0 ) continue;
     key_offset = -1; /* sentinel value */
     for ( k = 0, ex = web.skel[i].extras ;
                 k < web.skel[i].extra_count ; k++ , ex++ )
          if ( stricmp(keynames[i], ex->name) == 0 )
          { key_offset = ex->offset;
             break;
          }
     if ( key_offset < 0 )
     { for ( n = 0 ; n < NUMELEMENTS ; n++ ) myfree((char*)newblock[n]);
        kb_error(2749,"reorder_storage: Key_order attribute not defined.\n",RECOVERABLE);
     }
     qsort((char*)newblock[i],web.skel[i].count,web.sizes[i],FCAST esort);
  }

  /* reset ibase array of pointers and list links */
  for ( i = 0 ; i < NUMELEMENTS ; i++ )
  { struct element *ep,*nextep;
     int k;
 
          if  ( web.skel[i].count == 0 ) continue;

     web.skel[i].used = newblock[i]->self_id;
     newblock[i]->backchain = NULLID;
     for ( k = 0, ep = newblock[i] ; k < web.skel[i].count-1 ; k++ )
     { nextep = (struct element *)((char*)ep + web.sizes[i]);
        ep->forechain = nextep->self_id;
        nextep->backchain = ep->self_id;
        ibase[i][ordinal(ep->self_id)] = ep;
        ep = nextep;
     }
     ibase[i][ordinal(ep->self_id)] = ep;
     ep->forechain = NULLID;
     web.skel[i].last = ep->self_id;
  }

  /* free old storage and empty free list */
  for ( i = 0 ; i < NUMELEMENTS ; i++ )
     {
        for ( j = 0 ; j < blockcount[i] ; j++ )
          myfree((char *)blocklist[i][j].blockptr);
        blockmax[i] = blockcount[i] = 0;
        web.skel[i].free = NULLID;
        web.skel[i].maxcount = web.skel[i].count;
     }

  /* establish new */
  for ( i = 0 ; i < NUMELEMENTS ; i++ )
  {  if ( web.skel[i].count == 0 ) continue;
      blocklist[i][0].blockptr = newblock[i];
      blocklist[i][0].count = web.skel[i].count;
      blockmax[i] = blockcount[i] = 1;
  }

}

/*************************************************************************
*
* Function: renumber_all()
*
* Purpose: Renumber elements according to lined list order.
*
*/

#define copy_sign(id1,id2)     (((id1)&~SIGNMASK) | ((id2) & SIGNMASK))
void renumber_all()
{ int type;
  struct element *ep;
  element_id id,newid;
  int k;
  int dim,off,dima,offa,dimb,offb,dimc,offc;
  struct element **newibase[NUMELEMENTS];

  for ( type = VERTEX ; type <= FACETEDGE ; type++ )
     newibase[type] = 
         (struct element **)mycalloc(web.skel[type].maxcount,
                  sizeof(element_id *)); 

  for ( type = VERTEX ; type <= FACETEDGE ; type++ )
  { element_id count;

     /* reset self-id  */
     count = 0;
     id = web.skel[type].used;
     while ( valid_id(id) )
     { ep = elptr(id);
        ep->self_id = (ep->self_id & (~OFFSETMASK)) | count;
        count++;
        id = ep->forechain;
     }

  }
 /* reset mutual links within structures */
  FOR_ALL_FACETEDGES(id)
  { struct facetedge *ep = feptr(id);
    ep->fe_edge_id = copy_sign(elptr(ep->fe_edge_id)->self_id,ep->fe_edge_id);
    ep->fe_facet_id = copy_sign(elptr(ep->fe_facet_id)->self_id,ep->fe_facet_id);
    ep->nextedge[0] = copy_sign(elptr(ep->nextedge[0])->self_id,ep->nextedge[0]);
    ep->nextedge[1] = copy_sign(elptr(ep->nextedge[1])->self_id,ep->nextedge[1]);
    ep->nextfacet[0] = copy_sign(elptr(ep->nextfacet[0])->self_id,ep->nextfacet[0]);
    ep->nextfacet[1] = copy_sign(elptr(ep->nextfacet[1])->self_id,ep->nextfacet[1]);
  }

  FOR_ALL_VERTICES(id)
  { struct vertex *ep = vptr(id);
    ep->e_id = copy_sign(elptr(ep->e_id)->self_id,ep->e_id);
  }

  off = web.skel[EDGE].extras[E_VERTICES_ATTR].offset;  /* endpoints */
  dim = web.skel[EDGE].extras[E_VERTICES_ATTR].dim;  
  FOR_ALL_EDGES(id)    
  { struct edge *ep = eptr(id);
    vertex_id *vp;
    ep->fe_id = copy_sign(elptr(ep->fe_id)->self_id,ep->fe_id);
    ep->next_vedge[0] = copy_sign(elptr(ep->next_vedge[0])->self_id,ep->next_vedge[0]);
    ep->next_vedge[1] = copy_sign(elptr(ep->next_vedge[1])->self_id,ep->next_vedge[1]);
    vp = (vertex_id*)((char*)ep + off);
    for ( k = 0 ; k < dim ; k++ )
      vp[k] = vptr(vp[k])->self_id;
  }

  off = web.skel[FACET].extras[F_VERTICES_ATTR].offset;  /* endpoints */
  dim = web.skel[FACET].extras[F_VERTICES_ATTR].dim;  
  offa = web.skel[FACET].extras[F_BODY_LIST_ATTR].offset;  /* bodies */
  dima = web.skel[FACET].extras[F_BODY_LIST_ATTR].dim;  
  offb = web.skel[FACET].extras[F_NEXT_VFACET_ATTR].offset;  /* links */
  dimb = web.skel[FACET].extras[F_NEXT_VFACET_ATTR].dim;  
  offc = web.skel[FACET].extras[F_NEXT_BFACET_ATTR].offset;  /* links */
  dimc = web.skel[FACET].extras[F_NEXT_BFACET_ATTR].dim;  
  FOR_ALL_FACETS(id)  
  { struct facet *ep = fptr(id);
    element_id *p;
    ep->fe_id = copy_sign(elptr(ep->fe_id)->self_id,ep->fe_id);
    p = (element_id*)((char*)ep + off);
    for ( k = 0 ; k < dim ; k++ )
      if ( valid_id(p[k]) )
         p[k] = copy_sign(elptr(p[k])->self_id,p[k]);
    p = (element_id*)((char*)ep + offa);
    for ( k = 0 ; k < dima ; k++ )
      if ( valid_id(p[k]) )
         p[k] = copy_sign(elptr(p[k])->self_id,p[k]);
    p = (element_id*)((char*)ep + offb);
    for ( k = 0 ; k < dimb ; k++ )
      if ( valid_id(p[k]) )
         p[k] = copy_sign(elptr(p[k])->self_id,p[k]);
    p = (element_id*)((char*)ep + offc);
    for ( k = 0 ; k < dimc ; k++ )
      if ( valid_id(p[k]) )
         p[k] = copy_sign(elptr(p[k])->self_id,p[k]);
  }

  FOR_ALL_BODIES(id)    
  { struct body * ep = bptr(id);
     ep->fe_id = copy_sign(elptr(ep->fe_id)->self_id,ep->fe_id);
  }

  /* now new pointers in newibase */
  for ( type = VERTEX ; type <= FACETEDGE ; type++ )
  {  int count = 0;
      if ( web.skel[type].count == 0 ) continue; 
      FOR_ALL_ELEMENTS(type,id)
          newibase[type][count++] = elptr(id);
  } 


  /* fix up links */
  for ( type = VERTEX ; type <= FACETEDGE ; type++ )
  { if ( web.skel[type].count == 0 ) continue;
     if ( web.skel[type].count == 1 )
     { newibase[type][0]->forechain = NULLID;
        newibase[type][0]->backchain = NULLID;
        continue;
     }
     newibase[type][0]->forechain = newibase[type][0]->self_id+1;
     newibase[type][0]->backchain = NULLID;
     for ( k = 1 ; k < web.skel[type].count-1; k++ )
     {
        newibase[type][k]->forechain = newibase[type][k]->self_id+1;
        newibase[type][k]->backchain = newibase[type][k]->self_id-1;
     }
     newibase[type][k]->forechain = NULLID;
     newibase[type][k]->backchain = newibase[type][k]->self_id - 1;
     web.skel[type].used = newibase[type][0]->self_id;
     web.skel[type].last = newibase[type][k]->self_id;
  }

  /* fix up freelists */
  for ( type = VERTEX ; type <= FACETEDGE ; type++ )
  { int count;
     count = web.skel[type].count;
     id = web.skel[type].free;
     if ( !valid_id(id) ) continue;
     newid = (id & (~OFFSETMASK)) | count;
     web.skel[type].free = newid;

     while ( valid_id(id) )
     { ep = elptr(id);
        id = ep->forechain;
        ep->forechain = ++newid;
        newibase[type][count++] = ep;
     } 
     ep->forechain = NULLID;
  }

  /* miscellaneous elements */
  if ( valid_id(web.zoom_v) ) web.zoom_v = vptr(web.zoom_v)->self_id;
  if ( pickvnum ) pickvnum = ordinal(vibase[pickvnum-1]->self_id)+1;
  if ( pickenum ) pickenum = ordinal(eibase[pickenum-1]->self_id)+1;
  if ( pickfnum ) pickfnum = ordinal(fibase[pickfnum-1]->self_id)+1;

  /* swap ibase to new */
  for ( type = VERTEX ; type <= FACETEDGE ; type++ )
  { myfree((char*)ibase[type]);
     ibase[type] = newibase[type];
  }
  vibase = ibase[VERTEX];
  eibase = ibase[EDGE];
  fibase = ibase[FACET];
  bibase = ibase[BODY];
  feibase = ibase[FACETEDGE];
}

