/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/**********************************************************************
*
*  File: userfunc.c
*
*  Purpose: Lets user compile own function definitions for use in
*           expressions.  Function passed pointer to coordinate array.
*           User should also provide derivative function.
*           After defining functions, add names to arrays at bottom
*           of file.  Names in this file don't mattter; first function
*           in array is usr1 in datafile or queries, etc. Use in 
*           datafile does NOT use arguments, they are implicit.
*           Example: (usr1 + usr3)/usr10.
*
*           Also has functions for handling dynamic load libraries.
*/

#include "include.h"

/**************************************************************************/

/* usr1 as defined here gives conformal metric for 3-sphere in 
    stereographic projection */

REAL usr1 ARGS((REAL *));
REAL usr1_deriv ARGS((REAL *,REAL *));
REAL usr1_seconds ARGS((REAL *,REAL *,REAL **));

REAL usr1(x)
REAL *x; /* incoming parameters */
{ REAL denom;
  denom = 4+x[0]*x[0]+x[1]*x[1]+x[2]*x[2]; 
  return 16/denom/denom;
}

REAL usr1_deriv(x,partials)
REAL *x; /* incoming parameters */
REAL *partials; /* outgoing partial derivatives */
{ REAL denom,cube;
  int i;

  denom = 4+x[0]*x[0]+x[1]*x[1]+x[2]*x[2]; 
  cube = denom*denom*denom;
  for ( i = 0 ; i < SDIM ; i++ )
     partials[i] = -64/cube*x[i]; 

  return 16/denom/denom;
}

REAL usr1_seconds(x,partials,seconds)
REAL *x; /* incoming parameters */
REAL *partials; /* outgoing partial derivatives */
REAL **seconds; /* outgoing second derivatives */
{ REAL denom,cube,quart;
  int i,j;

  denom = 4+x[0]*x[0]+x[1]*x[1]+x[2]*x[2]; 
  cube = denom*denom*denom;
  quart = cube*denom;
  for ( i = 0 ; i < SDIM ; i++ )
     partials[i] = -64/cube*x[i]; 
  for ( i = 0 ; i < SDIM ; i++ )
  { for ( j = 0 ; j < SDIM ; j++ )
        seconds[i][j] = 384*x[i]*x[j]/quart;
     seconds[i][i] -= 64/cube;
  }

  return 16/denom/denom;
}

/***************************************************************************/

/* Another example of a user function, which is a polynomial in x,y,z.     */
/* This function is referred to as usr2 in expressions.                    */

static REAL usr_poly ARGS((REAL *));
static REAL usr_poly_grad ARGS((REAL *,REAL *));
static REAL usr_poly_hess ARGS((REAL *,REAL *,REAL **));

static REAL usr_poly(x)
REAL *x; /* incoming parameters */
{ 
    return x[0]*x[0] + x[1]*x[2] + x[2]*x[2]*x[2];
}

static REAL usr_poly_grad(x,partials)
REAL *x; /* incoming parameters */
REAL *partials; /* outgoing partial derivatives */
{
  partials[0] = 2*x[0];
  partials[1] = x[2];
  partials[2] = x[1] + 3*x[2]*x[2];

  return x[0]*x[0] + x[1]*x[2] + x[2]*x[2]*x[2];
}

static REAL usr_poly_hess(x,partials,seconds)
REAL *x; /* incoming parameters */
REAL *partials; /* outgoing partial derivatives */
REAL **seconds; /* outgoing second derivatives */
{ 
  partials[0] = 2*x[0];
  partials[1] = x[2];
  partials[2] = x[1] + 3*x[2]*x[2];

  seconds[0][0] = 2.0;
  seconds[0][1] = seconds[1][0] = 0.0;
  seconds[0][2] = seconds[2][0] = 0.0;
  seconds[1][1] = 0.0;
  seconds[1][2] = seconds[2][1] = 1.0;
  seconds[2][2] = 6*x[2];

  return x[0]*x[0] + x[1]*x[2] + x[2]*x[2]*x[2];
}

/**************************************************************************/

/* Add your functions to these arrays; this is how they will be invoked! */
#ifdef NOPROTO
REAL (*userfunc[])() = {usr1,usr_poly};
REAL (*userfunc_deriv[])() = {usr1_deriv,usr_poly_grad};
REAL (*userfunc_seconds[])() = {usr1_seconds,usr_poly_hess};
#else
REAL (*userfunc[])(REAL*) = {usr1,usr_poly};
REAL (*userfunc_deriv[])(REAL*,REAL*) = {usr1_deriv,usr_poly_grad};
REAL (*userfunc_seconds[])(REAL*,REAL*,REAL**) = {usr1_seconds,usr_poly_hess};
#endif



/* A user defined attribute function.  Undocumented. */
/* Use "user_attr" in queries like length or area or id */

REAL user_attribute(id)
element_id id;
{
  /* a sample smorgasbord */
  switch ( id_type(id) )
  { case VERTEX: return get_coord(id)[0];
     case EDGE:    return get_edge_length(id);
     case FACET:  return get_facet_area(id);
     case BODY:    return get_body_volume(id);
     case FACETEDGE: return (REAL)(ordinal(id)+1);
  }
  return 0.0;
}

/*********************************************************************
**********************************************************************

             D Y N A M I C     L O A D     L I B R A R I E S

**********************************************************************
*********************************************************************/

/*********************************************************************
*
* function: load_library()
*
* purpose: Find and load dynamic library.  Searches current directory,
*              EVOLVERPATH, default library path.
*/

void load_library(libname)
char *libname;
{  
#ifdef ENABLE_DLL
  int k;
  char *env;
  char path[200];
  int len;
  void *fd;

  for ( k = 0 ; k < MAX_DLL ; k++ )
     if ( dll_list[k].name == NULL ) break;

  if ( k >= MAX_DLL )
     kb_error(1993,"Too many dynamic load libraries.\n",DATAFILE_ERROR);


  env = getenv("EVOLVERPATH");

  /* try current directory first */
  strcpy(path,"./");
  strncpy(path+2,libname,sizeof(path)-2);
  while ( (fd = dlopen(path,RTLD_NOW)) == NULL)
     { /* try paths in EVOLVERPATH */
        if ( env == NULL ) break;
        len = strcspn(env,ENVPATHCHAR);
        if ( len == 0 ) break;
        strncpy(path,env,len);
        path[len] = PATHCHAR;
        strncpy(path+len+1,libname,sizeof(path)-len-2);
        if ( env[len] == 0 ) env = NULL; /* end of EVOLVERPATH */
        else env += len+1;
     } 
  /* try given name */
  if ( ! fd )
  { strncpy(path,libname,sizeof(path));
     fd = dlopen(path,RTLD_NOW);
  }
  
  if ( ! fd )
  { sprintf(errmsg,"Cannot open dynamic library %s. Reason:\n",libname);
     strncpy(errmsg+strlen(errmsg),dlerror(),sizeof(errmsg)-strlen(errmsg)-2);
     kb_error(1991,errmsg,DATAFILE_ERROR);
  }

  dll_list[k].name = kb_calloc(1,strlen(libname)+4);
  strcpy(dll_list[k].name,libname);

  dll_list[k].handle = fd;

#else

  kb_error(1992,"This Evolver not compiled for dynamic load libraries.\n",
     DATAFILE_ERROR);
#endif

}

/*************************************************************************
*
* function: unload_libraries
*
* purpose: unload dynamic link libraries
*/

void unload_libraries()
{
  int k;

  for ( k = 0 ; k < MAX_DLL ; k++ )
     if ( dll_list[k].name )
     { myfree(dll_list[k].name); dll_list[k].name = NULL;
#ifdef ENABLE_DLL
        dlclose(dll_list[k].handle); dll_list[k].handle = NULL;
#endif
     }
}

/*********************************************************************
*
* function: search_libraries()
*
* purpose: find function name in dynamic load libraries.
*
* return: pointer to function. NULL if not found.
*/

dll_func_type search_libraries(funcname)
char *funcname;
{ 

#ifdef ENABLE_DLL
  int i;
  dll_func_type f;

  for ( i = 0 ; i < MAX_DLL ; i++ )
    if ( dll_list[i].handle ) 
     { f = dlsym(dll_list[i].handle,funcname);
        if ( f ) return f;
     }
#endif
  return NULL;
}
