/************************************************************
*  This file is part of the Surface Evolver source code.    *
*  Programmer:  Ken Brakke, brakke@susqu.edu                *
************************************************************/


/*****************************************************************
*
*  File: evaltree.c
*
*  Purpose: To execute expression and command trees.
*          Tree nodes are stored in postorder linear form
*          for fast postorder execution.
*    
*/

#include "include.h" 
#include "ytab.h"


/*****************************************************************
*
*  Function eval()
*
*  Purpose: runtime evaluation of expressions and commands.
*    The big switch statement is split between two functions to
*    get functions small enough for DOS compilers. All nodes
*    manipulating the stack are in the eval() function, and 
*    others in other_stuff().  This should not impose any speed
*    penalty on expressions.
*
*  Notes: The evaluation stack is a local variable, to permit
*  parallel evaluations on shared memory machines.  To prevent
*  eval() from getting too long, many cases have been moved
*  to evalmore(), which is called by the default case of eval().
*  But for efficiency, most frequently evaluated cases should
*  be in eval().
*  
*/


REAL eval(ex,params,self_id)
struct expnode *ex;   /* expression tree */
REAL *params;         /* vector of parameters */
element_id self_id;   /* reference element, if any */
{ struct treenode *node,*finish;
  int k,n;
  REAL x,y;
  element_id id;
  REAL *bins;
  REAL hi;
  REAL lo;
  REAL val;
  int old_flag = iterate_flag;
  facet_id f_id;
  REAL vect[MAXCOORD];
  REAL stack[MAXSTACK];
  REAL *stacktop = stack;
  REAL *newstack = stack;
  REAL *stack_sentinel = &stack[MAXSTACK-5];
  int stackmax = MAXSTACK;
  facetedge_id fe;
  vertex_id v_id;
  int refine_count;
  int delete_count;
  int dissolve_count;
  int fix_count,unfix_count;
  int equi_count;
  int recalc_flag;
  int update_display_flag;
  struct boundary *bdry;
  REAL *histo_data;
  int histo_max;
  int histo_count;
  element_id q_id = self_id;  /* innermost loop element */

#define STACKMAGIC 12738546.2341341
  *stack_sentinel = STACKMAGIC;  /* overflow check */

  if ( !params )
  { refine_count = 0; delete_count = 0;
     dissolve_count = 0; fix_count = 0,unfix_count = 0;
     equi_count = 0;
     recalc_flag = 0;
     update_display_flag = 0;
  }  

  if ( ex == NULL ) return 0.0;

  if ( ex->start == NULL )
  { kb_error(1253,"Trying to evaluate null expression.\n",WARNING);
    return 0.0;
  }

  if ( !breakflag ) iterate_flag = 2; /* for interrupt handler */

  finish = ex->root+1;
  for ( node = ex->start+1 ; node != finish ; node++ )
  {
    switch ( node->type )
    {
      struct treenode *where,*enode,*eroot;

      case ACOMMANDEXPR_: break;

      case SHOW_:
      case SHOW_EXPR_:
      { int etype; /* element type */  
        where = node + node->op1.intval;
        if ( where->type == WHERE_ ) /* condition */
          { /* copy over expression */
             enode = where + where->left; /* NEXT */
             eroot = where + where->right;
             etype = enode[enode->left].op1.intval;
             show_expr[etype] = show_expr_table + etype;
             tree_copy(show_expr[etype],eroot);
             /* can use first slot to record type of element */
             /* element location */
             show_expr[etype]->start->op2.intval = enode->op2.intval;
             enode += enode->left; /* INIT */
             show_expr[etype]->start->op1.intval = enode->op1.intval; 
          }
         else
            { etype =  where[where->left].op1.intval;
              tree_copy(show_expr[etype],NULL);
              show_expr[etype] = NULL;
            }
         /* save for dump */ 
         tree_copy(show_command+etype,node+node->op1.intval+1);
         if ( (node->type == SHOW_) && !OOGL_flag && !go_display_flag)
            do_show();
         else update_display();
         node += node->op1.intval; /* skip over expression */
         break;
       }

      case UNREDEFINE_SINGLE_:
        free_expr(&single_redefine[node->op1.intval]);
        break;
          
      case REDEFINE_SINGLE_:
        tree_copy(&single_redefine[node->op1.intval],node+node->op2.intval);
        single_redefine[node->op1.intval].flag = USERCOPY;
        node += node->op2.intval; /* skip over procedure */
        break;

      case SET_PROCEDURE_:
        free_expr(&globals[node->op1.intval].value.proc);
        tree_copy(&globals[node->op1.intval].value.proc,
                                node+node->op2.intval);
        globals[node->op1.intval].proc_timestamp = proc_timestamp++;
        node += node->op2.intval; /* skip over procedure */
        break;

      case RETURN_: stacktop = newstack; goto the_exit;

         /********************/
         /* repeated command */
         /********************/

      case REPEAT_INIT_:
        node->op1.intval = gocount = (int)*(stacktop--); 
        node->op2.intval = 0;
        /* initialize count */
        if ( gocount <= 0 ) node += node->op3.intval[0]; /* skip loop */
        break;

      case REPEAT_:
      { struct treenode *rnode = node + node->left;
        temp_free_all();
        gocount = rnode->op1.intval - ++rnode->op2.intval;
#if defined(MAC_APP) || defined(MAC_CW)
        break_check();
#endif
        if ( breakflag == BREAKREPEAT ) { breakflag = 0; break; }
        if ( (gocount > 0) && !breakflag )
          { node = rnode;  /* do again */
             /* loop increment will step over REPEAT_INIT_ node */
          }
      }
      break;


      /*******************/
      /* flow of control */
      /*******************/

      case IFTEST_:
      case COND_TEST_:
        if ( *(stacktop--) == 0. )
        { /* jump */
          node += node->op1.intval;
        }
        break;

      case IF_:
      case COND_EXPR_:
         /* did first command, so skip second */
         node += node->op1.intval;
         break;

      case CONTINUE_:
         node += node->op1.intval; /* back to loop top */
         stacktop = newstack + node->stackpos;
         node += node->op3.intval[1]; /* jump to expr */
         break;

      case BREAK_:
         node += node->op1.intval; /* back to loop top */
         stacktop = newstack + node->stackpos;
         node += node->op3.intval[0]; /* jump to end */
         break;

      case WHILE_TOP_:
         /* jumptest if expr false */
         node->stackpos = stacktop - newstack - 1;
         if ( (*(stacktop--) == 0.) || breakflag )
           node += node->op1.intval;
         break;

      case WHILE_END_:
         /* loop back */
         node += node->op1.intval;
         break;

      case DO_TOP_: 
         node->stackpos = stacktop - newstack;
         break;

      case DO_ENTRY_: 
         break;

      case DO_END_:
         /* loop back if true */
         if ( *(stacktop--) && !breakflag )
           node += node->op1.intval;
         break;

      
         /*******************/
         /* aggregate verbs */
         /*******************/

      case LIST_:
        switch ( node->op2.intval )
        { 
           case VERTEX:
             vertex_dump(positive_id(q_id),outfd);
             break;

           case EDGE:
             edge_dump(positive_id(q_id),outfd);
             break;

           case FACET:
             facet_dump(positive_id(q_id),outfd);
             break;

           case BODY:
             body_dump(positive_id(q_id),outfd);
             break;

           case FACETEDGE:
             facetedge_dump(positive_id(q_id),outfd);
             break;

           default: kb_error(1255,"Bad LIST element type.\n",RECOVERABLE);

         }
        node += node->op1.intval - 1;  /* back to start of loop */
        break;

      /***********************
      * commands with counts *
      ***********************/
      case REFINE_:
         switch ( node->op2.intval )
         { 
            case EDGE:
              refine_count += edge_refine(q_id);
              break;
            case FACET:
              face_triangulate(q_id,FACET_EDGES);
              refine_count++;
              break;
            default: kb_error(1256,"Bad refine element type.\n",RECOVERABLE);

         }
         node += node->op1.intval - 1;  /* back to start of loop */
         break;

      case EDGESWAP_:
         equi_count += edgeswap(q_id); 
         node += node->op1.intval - 1;  /* back to start of loop */
         recalc_flag = 1;
         break;

      case DISSOLVE_:
         switch ( node->op2.intval )
         {  case VERTEX: 
              dissolve_count += dissolve_vertex(q_id); break;
            case EDGE:
              dissolve_count += dissolve_edge(q_id); break;
            case FACET:
              dissolve_count += dissolve_facet(q_id); break;
            case BODY:
              dissolve_count += dissolve_body(q_id); break;
          }
          node += node->op1.intval - 1;  /* back to start of loop */
          break;

      case FIX_: 
         if ( !(get_attr(q_id)&FIXED) )
            { set_attr(q_id,FIXED); fix_count++; }
         node += node->op1.intval - 1;  /* back to start of loop */
         break;

      case UNFIX_:
         if ( get_attr(q_id) & FIXED )
            { unset_attr(q_id,FIXED); unfix_count++; }
         node += node->op1.intval - 1;  /* back to start of loop */
         break;

      case VERTEX_AVERAGE_:
         if ( new_vertex_average(q_id,VOLKEEP) ) recalc_flag = 1;
         node += node->op1.intval - 1;  /* back to start of loop */
         break;

      case RAW_VERTEX_AVERAGE_:
         if ( new_vertex_average(q_id,NOVOLKEEP) ) recalc_flag = 1;
         node += node->op1.intval - 1;  /* back to start of loop */
         break;

      case RAWEST_VERTEX_AVERAGE_:
         if ( new_vertex_average(q_id,RAWEST) ) recalc_flag = 1;
         node += node->op1.intval - 1;  /* back to start of loop */
         break;

      case DELETE_:
         switch ( node->op2.intval )
         { 
            case EDGE:
             { int elimcount = eliminate_edge(q_id);
                if ( elimcount ) free_element(q_id);
                delete_count += elimcount;
             }
              break;
            case FACET:
                delete_count += eliminate_facet(q_id);
                break;

            default: kb_error(1257,"Bad delete element type.\n",
              RECOVERABLE);
         }
         node += node->op1.intval - 1;  /* back to start of loop */
         break;

         /***************/
         /* expressions */
         /***************/

      case REPLACECONST:
         *stacktop = node->op1.real;
        break;

      case PUSHCONST:
         *++stacktop = node->op1.real;
        break;

      case PUSHDELTA_:
        { struct global *g = globals + node->op1.intval;
          *++stacktop = g->delta;
          break;
        }

      case PUSHGLOBAL_:
      case STRINGGLOBAL_:
        { struct global *g = globals + node->op1.intval;
          if ( g->flags & QUANTITY_NAME )
            { struct gen_quant *q = GEN_QUANTS + g->value.quant;
              if ( (q->flags & Q_INFO) && (q->timestamp < graph_timestamp) )
             calc_quants(Q_INFO);
             *++stacktop = q->value;
            }
          else if ( g->flags & METHOD_NAME )
            { struct method_instance *mi = METH_INST + g->value.meth_inst;
              if ( (GEN_QUANTS[mi->quant].flags & Q_INFO) 
                      && (mi->timestamp < graph_timestamp) )
             calc_quants(Q_INFO);
             *++stacktop = mi->value;
            }
          else if ( g->flags & QUANTITY_MODULUS )
             *++stacktop = GEN_QUANTS[g->value.quant].modulus; 
          else if ( g->flags & METHOD_MODULUS )
             *++stacktop = METH_INST[g->value.meth_inst].modulus;
          else if ( g->flags & QUANTITY_TARGET )
             *++stacktop = GEN_QUANTS[g->value.quant].target;
          else if ( g->flags & FILE_VALUES )
          *++stacktop = g->value.file.values[int_val];
          else if ( g->flags & STRINGVAL )
          { int pp = (sizeof(REAL)+sizeof(char*)-1)/sizeof(char*);
             int nn; 
             stacktop++;
             for ( nn = 0 ; nn < pp ; nn++ )
                 ((char **)stacktop)[nn] = g->value.string;
          }
          else
          *++stacktop = g->value.real;
          }
        break;

      case PUSHPI:
         *++stacktop = M_PI;
        break;

      case PUSHE:
         *++stacktop = M_E;
        break;

      case PUSHG:
         *++stacktop = web.gravflag ? web.grav_const : 0.0;
        break;

      case PUSHPARAM:
         *++stacktop = params[node->op1.intval];
        break;

      case USERFUNC:
        *++stacktop = (*userfunc[node->op1.intval])(params);
        break;

      case DYNAMIC_LOAD_FUNC_:
        if ( ! params )
        { sprintf(errmsg,"Must use dynamic load function %s in context with parameters.\n",globals[node->op2.intval].name);
          kb_error(1993,errmsg,RECOVERABLE);
        } else
        (*node->op1.funcptr)(FUNC_VALUE,params,(struct dstack*)(++stacktop));
        break;

      case GT_:
        stacktop--;
        stacktop[0] = (REAL)(stacktop[0] > stacktop[1]);
        break;

      case LT_:
        stacktop--;
        stacktop[0] = (REAL)(stacktop[0] < stacktop[1]);
        break;

      case LE_:
        stacktop--;
        stacktop[0] = (REAL)(stacktop[0] <= stacktop[1]);
        break;

      case GE_:
        stacktop--;
        stacktop[0] = (REAL)(stacktop[0] >= stacktop[1]);
        break;

      case NE_:
        stacktop--;
        stacktop[0] = (REAL)(stacktop[0] != stacktop[1]);
        break;

      case EQ_:
        stacktop--;
        stacktop[0] = (REAL)(stacktop[0] == stacktop[1]);
        break;

      case AND_: /* short-circuit */
        if ( *stacktop == 0.0 )
          node += node->op1.intval; /* leave 0 as result */
        else stacktop--; /* pop, second operand will be result */ 
        break;

      case CONJUNCTION_END:    /* get proper 1 for true */
        if ( *stacktop ) *stacktop = 1.0; break;

      case OR_: /* short-circuit */
        if ( *stacktop != 0.0 )
          node += node->op1.intval; /* leave as result */
        else stacktop--;  /* pop */
        break;

      case NOT_:
        stacktop[0] = (REAL)(!stacktop[0]);
        break;

      case PLUS:
        stacktop--;
         stacktop[0] += stacktop[1];
        break;

      case MINUS:
      case EQUATE:
        stacktop--;
         stacktop[0] -= stacktop[1];
        break;

      case TIMES:
        stacktop--;
         stacktop[0] *= stacktop[1];
        break;

      case DIVIDE:
        stacktop--;
        if ( stacktop[1] == 0.0 ) 
          kb_error(1258,"Division by zero.\n",RECOVERABLE);

         stacktop[0] /= stacktop[1];
        break;

      case REALMOD:
        stacktop--;
        if ( stacktop[1] == 0.0 ) 
          kb_error(1259,"Modulus base zero.\n",RECOVERABLE);
        stacktop[0] = stacktop[0] - floor(stacktop[0]/stacktop[1])
                              *stacktop[1];
        break;

      case IMOD_:
        stacktop--;
        if ( stacktop[1] == 0.0 ) 
          kb_error(1260,"Modulus base zero.\n",RECOVERABLE);
        stacktop[0] = floor(stacktop[0]) - 
            floor(floor(stacktop[0])/floor(stacktop[1]))
                              *floor(stacktop[1]);
        break;

      case IDIV_:
        stacktop--;
        if ( (int)stacktop[1] == 0 ) 
          kb_error(1261,"Division by zero.\n",RECOVERABLE);
        stacktop[0] = (REAL)((int)(stacktop[0])/(int)(stacktop[1]));
        break;

      case INTPOW:
         /* cases n = 0,1,2 taken care of in parsing */
         x = *stacktop;
         k = node->op1.intval < 0 ? -node->op1.intval : node->op1.intval;
         for ( n = 1 ; n < k ; n++ )
         *stacktop *= x;
         if ( node->op1.intval < 0 )
         { if ( *stacktop == 0.0 ) 
           { sprintf(errmsg,"Negative power (%f) of zero.\n",
              (DOUBLE)*stacktop);
             kb_error(1262,errmsg,RECOVERABLE);
           }
           *stacktop = 1/(*stacktop);
         } 
         break;

      case POW:
        stacktop--;
        if ( (stacktop[0] < 0.0) && (floor(stacktop[1]) != stacktop[1]) )
          kb_error(3919,"Non-integer power of a negative number.\n",
             RECOVERABLE);
        if ( stacktop[0] == 0.0 )
        { if ( stacktop[1] >= 0.0 ) *stacktop = 0.0;
          else *stacktop = 1e30;
        }
        else  *stacktop = pow(stacktop[0],stacktop[1]);
        /* Note: pow recognizes integer powers, so OK for neg x */
        break;

      case MAXIMUM_:
        stacktop--;
         *stacktop = (stacktop[0] > stacktop[1]) ? stacktop[0] : stacktop[1];
        break;

      case MINIMUM_:
        stacktop--;
         *stacktop = (stacktop[0] < stacktop[1]) ? stacktop[0] : stacktop[1];
        break;

      case WRAP_COMPOSE_:
          stacktop--;
          *stacktop = (REAL)(*sym_compose)((unsigned int)stacktop[0],(unsigned int)stacktop[1]);
          break;

      case WRAP_INVERSE_:
          *stacktop = (REAL)(*sym_inverse)((unsigned int)stacktop[0]);
           break;

      case ATAN2_:
        stacktop--;
         *stacktop = atan2(stacktop[0],stacktop[1]);
        break;

      case SQR:
         *stacktop *= *stacktop;
        break;

      case SQRT:
        if ( *stacktop < 0.0 )
        { if ( *stacktop > -1e-10 ) *stacktop = 0.0;
          else kb_error(1263,"Square root of negative number.\n",RECOVERABLE);

        }
        else *stacktop = sqrt(*stacktop);
        break;

      case CEIL_:
          *stacktop = ceil(*stacktop);
          break;

      case FLOOR_:
          *stacktop = floor(*stacktop);
          break;

      case ABS:
         *stacktop = fabs(*stacktop);
        break;

      case SIN:
         *stacktop = sin(*stacktop);
        break;

      case COS:
         *stacktop = cos(*stacktop);
        break;

      case TAN:
         *stacktop = tan(*stacktop);
        break;

      case EXP:
         *stacktop = exp(*stacktop);
        break;

      case SINH:
         *stacktop = (exp(*stacktop)-exp(-*stacktop))/2;
        break;

      case COSH:
         *stacktop = (exp(*stacktop)+exp(-*stacktop))/2;
        break;

      case TANH:
         y = exp(*stacktop);
         *stacktop = (y*y-1)/(y*y+1) ;
        break;

      case ASINH:
         *stacktop = log(*stacktop + sqrt(*stacktop*(*stacktop) + 1));
         break;

      case ACOSH:
         *stacktop = 2*log(sqrt(*stacktop+1) + sqrt(*stacktop - 1)) 
                - log(2.0);
         break;

      case ATANH:
         *stacktop = log(*stacktop+1)/2 - log(1-*stacktop)/2; 
         break;

      case LOG:
         if ( *stacktop <= 0.0 )
             kb_error(1264,"log of zero or negative number.\n", RECOVERABLE );

         *stacktop = log(*stacktop);
        break;

      case ASIN:
         if ( *stacktop > 1.0 ) *stacktop = asin(1.0);
         else if ( *stacktop < -1.0 ) *stacktop = asin(-1.0);
         else *stacktop = asin(*stacktop);
        break;

      case ACOS:
         if ( *stacktop > 1.0 ) *stacktop = 0.0;
         else if ( *stacktop < -1.0 ) *stacktop = M_PI;
         else *stacktop = acos(*stacktop);
        break;

      case ATAN:
         *stacktop = atan(*stacktop);
        break;
      
      case CHS:
         *stacktop = -*stacktop;
        break;
      
      case INV:
         *stacktop = 1/(*stacktop);
        break;

      /* here are attributes for queries */
      case COORD_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        switch ( id_type(id) )
        { case VERTEX:
             *++stacktop = get_coord(id)[node->op2.intval];
             break;
          case EDGE:
             get_edge_side(id,vect);
             *++stacktop = vect[node->op2.intval];
             break;
          case FACET:
             get_facet_normal(id,vect);
             *++stacktop = vect[node->op2.intval];
             break;
         }
        break;

      case INDEXED_COORD_:
      { int k = (int)*stacktop - 1;  /* 1 based indexing */
        if ( k < 0 || k >= SDIM )
        { sprintf(errmsg,"Invalid index %d for x; must be between 1 and %d.\n",
            k+1,SDIM);
          kb_error(4589,errmsg,RECOVERABLE );
        }
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        switch( id_type(id) )
         { case VERTEX:
             *stacktop = get_coord(id)[k];
             break;
          case EDGE:
             get_edge_side(id,vect);
             *stacktop = vect[k];
             break;
          case FACET:
             get_facet_normal(id,vect);
             *stacktop = vect[k];
             break;
          default: 
             kb_error(3329,"Can't have indexed x on this element.\n",
                RECOVERABLE);
         }
       }
       break;

      case GET_VERTEXNORMAL_:     
         { MAT2D(normal,MAXCOORD,MAXCOORD);
           int k = (int)*stacktop - 1;  /* 1 based indexing */
           REAL mag;
           if ( k < 0 || k >= SDIM )
             { sprintf(errmsg,"Invalid index %d for facetnormal; must be between 1 and %d.\n",
                 k+1,SDIM);
               kb_error(4590,errmsg,RECOVERABLE );
             }
           if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
             else id = q_id;
           new_calc_vertex_normal(id,normal); 
           mag = sqrt(SDIM_dot(normal[0],normal[0]));
           *stacktop = mag == 0.0 ? 0.0 : normal[0][k]/mag;
         }
        break;

      case PARAM_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
         *++stacktop = get_param(id)[node->op2.intval];
        break;


      case GET_SQ_MEAN_CURV_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
         *++stacktop = vertex_sq_mean_curvature(id);
        break;

      case GET_FIXEDVOL_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
         *++stacktop = get_battr(id)&FIXEDVOL ? 1.0 : 0.0;
        break;

      case GET_LENGTH_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        calc_edge(id);
         *++stacktop = get_edge_length(id);
        break;

      case GET_DIHEDRAL_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        if ( id_type(id) == EDGE ) *++stacktop = dihedral(id);
        else if ( id_type(id) == VERTEX ) *++stacktop = vertex_angle(id);
        else *++stacktop = 0.0;
        break;

      case GET_ORIENTATION_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        *++stacktop = (get_attr(id) & NEGBOUNDARY) ? -1.0 : 1.0;
        break;

      case VALENCE_:
      case GET_VALENCE_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        switch ( id_type(id) )
         { case VERTEX:
              *++stacktop = (REAL)get_vertex_evalence(id);
              break;
            case EDGE:
              *++stacktop = (REAL)get_edge_valence(id);
              break;
            case FACET:
              *++stacktop = (REAL)get_facet_valence(id);
              break;
            case BODY:
              *++stacktop = (REAL)get_body_valence(id);
              break;
         }
        break;

      case GET_EDGE_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
         *++stacktop = (REAL)(ordinal(get_fe_edge(id))+1);
        break;

      case GET_FACET_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
         *++stacktop = (REAL)(ordinal(get_fe_facet(id))+1);
        break;

      case AREA_:
      case GET_AREA_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        *++stacktop = get_facet_area(id);
        break;

      case GET_WRAP_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        *++stacktop = (REAL)get_edge_wrap(id); 
        break;

      case GET_PRESSURE_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        switch ( id_type(id) )
          { 
             case BODY: *++stacktop = get_body_pressure(id); break;
             default: kb_error(1267,"Pressure only for bodies.\n",RECOVERABLE);

          }
        break;

      case GET_USERATTR_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        *++stacktop = user_attribute(id);
        break;

      case GET_QUANTITY_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
         if ( !valid_id(id) )
         { sprintf(errmsg,"Quantity name '%s' needs attribute like .value\n",
         GEN_QUANTS[node->op2.intval].name);
        kb_error(3121,errmsg, RECOVERABLE);
        }
        *++stacktop = quantity_attribute(id,node->op2.intval);
        break;

      case GET_INSTANCE_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        if ( !valid_id(id) )
        { sprintf(errmsg,"Instance name '%s' needs attribute like .value\n",
             METH_INST[node->op2.intval].name);
          kb_error(3122,errmsg, RECOVERABLE);
        }
        *++stacktop = instance_attribute(id,node->op2.intval);
        break;

      case GET_PHASE_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        switch ( id_type(id) )
          { 
             case FACET: *++stacktop = (REAL)get_f_phase(id); break;
             case BODY: *++stacktop = (REAL)get_b_phase(id); break;
             default: kb_error(1268,"Phase of wrong type element.\n",RECOVERABLE);

          }
        break;

      case DENSITY_:
      case GET_DENSITY_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        switch ( id_type(id) )
          { case EDGE: *++stacktop = get_edge_density(id); break;
             case FACET: *++stacktop = get_facet_density(id); break;
             case BODY: *++stacktop = get_body_density(id); break;
             default: kb_error(1269,"Density of wrong type element.\n",RECOVERABLE);

          }
        break;

      case GET_STAR_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        switch ( id_type(id) ) 
          { case VERTEX: *++stacktop = get_vertex_star(id); break;
             case EDGE: *++stacktop = get_edge_star(id); break;
             default: *++stacktop = 0.0; break;
          }
        break;

      case VOLUME_:
      case GET_VOLUME_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        if ( (volume_timestamp < global_timestamp) ||
             (get_body_voltimestamp(id) < global_timestamp) )
        calc_content(Q_FIXED|Q_INFO);
        *++stacktop = get_body_volume(id);
        break;

      case GET_VOLCONST_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        *++stacktop = get_body_volconst(id);
        break;

      case GET_TARGET_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        *++stacktop = get_body_fixvol(id);
        break;

      case ID_:
      case GET_ID_:
      case GET_OID_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        if ( (node->type == GET_OID_) && inverted(id) )
          *++stacktop = -(REAL)(ordinal(id)+1);
        else *++stacktop = (REAL)(ordinal(id)+1);
        break;

      case ORIGINAL_:
      case GET_ORIGINAL_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        *++stacktop = (REAL)get_original(id);
        break;

      case GET_COLOR_:
      case GET_FRONTCOLOR_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        switch ( id_type(id) )
         { case EDGE:  *++stacktop = (REAL)get_edge_color(id); break;
            case FACET:  *++stacktop = (REAL)get_facet_color(id); break;
            default: *++stacktop = 0.0;
         }
        break;

      case GET_BACKCOLOR_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        switch ( id_type(id) )
         { case FACET:  *++stacktop = (REAL)get_facet_backcolor(id); break;
            default: *++stacktop = 0.0;
         }
        break;

      case GET_FRONTBODY_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        switch ( id_type(id) )
         { case EDGE: 
             fe = get_edge_fe(id);
             if ( !valid_id(fe) ) {*++stacktop = 0.0; break;}
             f_id = get_fe_facet(fe);
             if ( inverted(f_id) ) f_id = get_fe_facet(get_next_facet(fe));
             if ( inverted(f_id) ) { *++stacktop = 0.0; break;}
             *++stacktop = (REAL)ordinal(get_facet_body(f_id))+1; 
             break;
           case FACET:  
              *++stacktop = (REAL)ordinal(get_facet_body(id))+1; 
              break;
            default: *++stacktop = 0.0;
         }
        break;

      case GET_BACKBODY_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        switch ( id_type(id) )
         { case FACET:  *++stacktop =
              (REAL)ordinal(get_facet_body(inverse_id(id)))+1;
            break;
           case EDGE: 
             fe = get_edge_fe(id);
             if ( !valid_id(fe) ) {*++stacktop = 0.0; break;}
             f_id = get_fe_facet(fe);
             if ( !inverted(f_id) ) f_id = get_fe_facet(get_next_facet(fe));
             if ( !inverted(f_id) ) { *++stacktop = 0.0; break;}
             *++stacktop = (REAL)ordinal(get_facet_body(inverse_id(f_id)))+1; 
             break;
            default: *++stacktop = 0.0;
         }
        break;

      case TAG_:
      case GET_TAG_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        *++stacktop = (REAL)get_tag(id);
        break;

      case BARE_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        *++stacktop = (get_attr(id) & BARE_NAKED) ? 1.0 : 0.0;
        break;

      case GET_MIDV_:
        if ( web.modeltype != QUADRATIC )
          kb_error(1992,"Cannot do MIDV except in QUADRATIC model.\n",
             RECOVERABLE);
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        *++stacktop = ordinal(get_edge_midv(id)) + 1.0;
        break;

      case FIXED_:
      case GET_FIXED_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        *++stacktop = (get_attr(id) & FIXED) ? 1.0 : 0.0;
        break;

      case GET_NO_DISPLAY_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        *++stacktop = (get_attr(id) & NODISPLAY) ? 1.0 : 0.0;
        break;

      case GET_NO_REFINE_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        *++stacktop = (get_attr(id) & NO_REFINE) ? 1.0 : 0.0;
        break;

      case GET_TRIPLE_PT_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        *++stacktop = (get_attr(id) & TRIPLE_PT) ? 1.0 : 0.0;
        break;

      case GET_TETRA_PT_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        *++stacktop = (get_attr(id) & TETRA_PT) ? 1.0 : 0.0;
        break;

      case GET_AXIAL_POINT_:
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        *++stacktop = (get_attr(id) & AXIAL_POINT) ? 1.0 : 0.0;
        break;

      case GET_EXTRA_ATTR_:
       { struct extra *ext;

         n = node->op3.intval[0];
         if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
         else id = q_id;
         ext = web.skel[id_type(id)].extras + n;
         if ( id_type(id) != node->op2.intval )
         { sprintf(errmsg,"Attribute %s is %s attribute, not %s attribute.\n",
             web.skel[node->op2.intval].extras[n].name,typenames[node->op2.intval],
             typenames[id_type(id)]);
           kb_error(5387,errmsg,RECOVERABLE);
         }
         if ( ext->code.start ) eval(&ext->code,NULL,id);
         if ( ext->dim == 1)
          switch ( ext->type )
          { case REAL_ATTR: *++stacktop = *(REAL*)get_extra(id,n); break;
             case INTEGER_ATTR: 
             *++stacktop = (REAL)*(int*)get_extra(id,n); break;
             case ULONG_ATTR: 
             *++stacktop = (REAL)*(unsigned long*)get_extra(id,n); break;
             case UCHAR_ATTR: 
             *++stacktop = (REAL)*(unsigned char*)get_extra(id,n); break;
          }
         else /* dimensioned */
         { int j = (int)*(stacktop--) - 1;  /* 1 based index */
            if ( j >= ext->dim )
            { sprintf(errmsg,"Attribute %s index is %d; maximum is %d.\n",
                 ext->name,j+1,ext->dim);
              kb_error(1270,errmsg,RECOVERABLE);
            }
            if ( j < 0 )
            { sprintf(errmsg,"Attribute %s index zero or negative: %d.\n",
              ext->name,j+1);
            kb_error(1271,errmsg,RECOVERABLE);
            }

            switch ( ext->type )
            { case REAL_ATTR: *++stacktop = ((REAL*)get_extra(id,n))[j]; 
              break;
              case INTEGER_ATTR: 
             *++stacktop = (REAL)((int*)get_extra(id,n))[j];
              break;
              case ULONG_ATTR: 
             *++stacktop = (REAL)((unsigned long*)get_extra(id,n))[j];
              break;
              case UCHAR_ATTR: 
             *++stacktop = (REAL)((unsigned char*)get_extra(id,n))[j];
              break;
            }
         }
       }
       break;

      case ON_CONSTRAINT_:
      { int testcon = (int)*(stacktop--);
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        switch(id_type(id))
         { case VERTEX: *++stacktop = (REAL)v_on_constraint(id,testcon); break;
            case EDGE  : *++stacktop = (REAL)e_on_constraint(id,testcon); break;
            case FACET : *++stacktop = (REAL)f_on_constraint(id,testcon); break;
            default: kb_error(1272,"Can't do constraints on this type element.\n",

             RECOVERABLE);
         }
       }
     break;

      case HIT_CONSTRAINT_:
       { 
         int testcon = (int)*(stacktop--);
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        switch(id_type(id))
         { case VERTEX: *++stacktop = (REAL)get_v_constraint_status(id,testcon); 
              break;
            default: kb_error(1273,"Can do hit_constraints only on vertices.\n",

             RECOVERABLE);
         }
        }
        break;

      case ON_BOUNDARY_:
       { struct boundary *b=NULL;
         int testb = (int)*(stacktop--);
        if ( node->op1.intval ) id = (node+node->op1.intval)->op2.id;
        else id = q_id;
        switch(id_type(id))
         { case VERTEX: b = get_boundary(id); break;
            case EDGE  : b = get_edge_boundary(id); break;
            case FACET : b = get_facet_boundary(id); break;
            default: kb_error(1274,"Can't do boundary on this type element.\n",

             RECOVERABLE);
         }
        *++stacktop = (b == web.boundaries+testb) ? 1.0 : 0.0;
        }
        break;

     case SELF_ELEMENT_:
        if ( !valid_id(self_id) )
          kb_error(6343,"No element for SELF to refer to.\n",RECOVERABLE);
        node->op2.id = self_id;
        break;

     case SYMBOL_:
        node->op2.id = (node+node->op3.intval[0])->op2.id;
        break;

     case SYMBOL_ELEMENT_:
        node->op2.id = (node+node->op3.intval[0])->op2.id;
        break;

     case INDEXED_SUBTYPE_:  /* like ee.vertex[1] */
        { 
          int ord = (int)*(stacktop--) - 1;  /* which one */
          element_id parent = node[node->left].op2.id;
          int ptype = id_type(parent);
          element_id first; /* sentinel for looping */

          if ( ord < 0 )
             kb_error(1275,"Element index must be positive.\n",RECOVERABLE);

          id=NULLID;
          switch ( ptype )
          { case VERTEX:
             switch ( node->op1.intval /* subtype */ )
             { case EDGE:
              if ( vedge_timestamp < top_timestamp ) make_vedge_lists();
              id = first = get_vertex_edge(parent);
              if ( !valid_element(id) )
              { sprintf(errmsg,"Vertex %d has no edges.\n",ordinal(parent)+1);
                  kb_error(1276,errmsg,RECOVERABLE);
              }
              for ( n = 1 ; n <= ord ; n++ )
              { id = get_next_tail_edge(id);
                 if ( equal_id(id,first) )
                kb_error(1277,"Edge index exceeds valence of vertex.\n",RECOVERABLE);
              }
              node->op2.intval = id;
              break;
              
            case FACET:
              if ( vfacet_timestamp < top_timestamp ) make_vfacet_lists();
              id = first = get_vertex_facet(parent);
              if ( !valid_element(id) )
              { sprintf(errmsg,"Vertex %d has no facets.\n",ordinal(parent)+1);
                 kb_error(1278,errmsg,RECOVERABLE);
              }
              for ( n = 1 ; n <= ord ; n++ )
              { id = get_next_vertex_facet(parent,id);
                 if ( equal_id(id,first) )
                kb_error(1279,"Facet index exceeds facet valence of vertex.\n",RECOVERABLE);
              }
              node->op2.intval = id;
              break;

             }
             break;

         case EDGE:
             switch ( node->op1.intval /* subtype */ )
             { case VERTEX:
              if ( ord >= web.skel[EDGE].ctrlpts )
                 kb_error(1280,
                "Index exceeds number of vertices on an edge.\n",
                  RECOVERABLE);
              if ( web.modeltype == LAGRANGE )
              { node->op2.id = get_edge_vertices(parent)
                  [inverted(parent) ? web.skel[EDGE].ctrlpts-ord-1 : ord]; 
              }
              else 
                 switch ( ord )
                 { case 0: node->op2.id = get_edge_tailv(parent); break;
                case 1: node->op2.id = get_edge_headv(parent); break;
                case 2: node->op2.id = get_edge_midv(parent); break;
                 } 
              break;
            case FACET:
              id = first = get_edge_fe(parent);
              if ( !valid_element(id) )
                 kb_error(1281,"Edge has no facets.\n",COMMAND_ERROR);

              for ( n = 1 ; n <= ord ; n++ )
              { id = get_next_facet(id);
                 if ( equal_id(id,first) )
                kb_error(1282,"Facet index exceeds number of facets on edge.\n",COMMAND_ERROR);

              }
             node->op2.id = get_fe_facet(id);
             break;

             case FACETEDGE: 
              id = first = get_edge_fe(parent);
              if ( !valid_element(id) )
                 kb_error(1281,"Edge has no facets.\n",COMMAND_ERROR);

              for ( n = 1 ; n <= ord ; n++ )
              { id = get_next_facet(id);
                 if ( equal_id(id,first) )
                kb_error(1282,"Index exceeds valence.\n",COMMAND_ERROR);

              }
             node->op2.id = id;
             break;
              }
             break;
          
         case FACET:
             switch ( node->op1.intval /* subtype */ )
             { case VERTEX:
              if ( web.representation == SIMPLEX )
              { node->op2.id = get_facet_vertices(parent)[ord];
                 break;
              }
              id = first = get_facet_fe(parent);
              if ( !valid_element(id) )
                 kb_error(1283,"Facet has no edges.\n",COMMAND_ERROR);

              for ( n = 1 ; n <= ord ; n++ )
              { id = get_next_edge(id);
                 if ( equal_id(id,first) )
                kb_error(1284,"Vertex index exceeds number of vertices on facet.\n",COMMAND_ERROR);

              }
             node->op2.id = get_fe_tailv(id);
             break;
            case EDGE:
              id = first = get_facet_fe(parent);
              if ( !valid_element(id) )
                 kb_error(1285,"Facet has no edges.\n",COMMAND_ERROR);

              for ( n = 1 ; n <= ord ; n++ )
              { id = get_next_edge(id);
                 if ( equal_id(id,first) )
                kb_error(1286,"Edge index exceeds number of edges on facet.\n",COMMAND_ERROR);

              }
             node->op2.id = get_fe_edge(id);
             break;
            case BODY:
              switch ( ord )
              { case 0: node->op2.id = get_facet_body(parent); break;
                 case 1: node->op2.id = get_facet_body(inverse_id(parent)); 
                  break;
                 default: kb_error(1287,"Illegal facet body index; must be 1 or 2.\n",RECOVERABLE);

              }
              }
             break;
          
         case BODY:
             switch ( node->op1.intval /* subtype */ )
             { case FACET:
             if ( bfacet_timestamp < top_timestamp )  make_bfacet_lists();
             id = first = get_fe_facet(get_body_fe(parent));
             for ( n = 1 ; n <= ord ; n++ )
              {
                 id = get_next_body_facet(id);
                 if ( equal_id(id,first) )
                kb_error(1288,"Facet index exceeds number of facets on body.\n",COMMAND_ERROR);

              }
             node->op2.id = id;
            break;
             }
             break;

         case FACETEDGE:
             switch ( node->op1.intval /* subtype */ )
             { case EDGE: node->op2.id = get_fe_edge(parent); break;
            case FACET: node->op2.id = get_fe_facet(parent); break;
             }
             break;
          }
        } break;  /* end INDEXED_SUBTYPE */

     case INDEXED_ELEMENT_:
       { int ord = (int)*(stacktop--);
         node->op2.id = get_ordinal_id(node->op1.intval,abs(ord)-1);
         if ( !valid_id(node->op2.id) ) 
         { sprintf(errmsg,"%s index %d is not valid.\n",
              typenames[node->op1.intval],ord);
            kb_error(1289,errmsg,RECOVERABLE);
         }
         if ( ord < 0 ) invert(node->op2.id);
         break;
          }

     case QUALIFIED_ATTRIBUTE:
          break; /* just a no-op in execution */

         /****************************/
         /* aggregate initialization */
         /****************************/

     case SET_INIT_: 
        web.where_count = 0;
         node->stackpos = stacktop - newstack + 3; /* in case not set later */
        break;

     case AGGREGATE_INIT_:
        aggregate_depth++;
        web.where_count = 0;
        node->stackpos = stacktop - newstack + 3; /* in case not set later */
        switch ( node->op1.intval )
          {  case FOREACH_:
             case SET_ATTRIBUTE_LOOP_:
                break;

              case MAX_:
                *++stacktop = -MAXDOUBLE; /* max of empty set */
                break;

              case MIN_:
                *++stacktop = MAXDOUBLE; /* min of empty set */
                break;

              case SUM_: 
                *++stacktop = 0.0; /* sum */
                break;

              case AVG_:
                *++stacktop = 0.0; /* count */
                *++stacktop = 0.0; /* sum */
                break;

              case COUNT_:
                *++stacktop = 0.0; /* count */
                break;

              case HISTOGRAM_:
              case LOGHISTOGRAM_:
                *++stacktop = 0.0; /* first phase counter */
                *++stacktop = MAXDOUBLE;  /* min */
                *++stacktop = -MAXDOUBLE; /* max */
                for ( n = 0 ; n < HISTBINS+1 ; n++ )
                  *++stacktop = 0.0;
                histo_max = web.skel[EDGE].count + 5;
                histo_data = (REAL*)mycalloc(histo_max,sizeof(REAL));
                histo_count = 0;
                break;

              case LIST_: /* print column headers */
                switch ( node->op2.intval )
                { case VERTEX:
                  if ( SDIM == 2 )
     outstring("// Id             X              Y\n");
                else
     outstring("// Id             X              Y             Z\n");
                    break;
                 case EDGE:
                    outstring("// Id    endpoints\n");
                    break;
                 case FACET:
                    if ( web.representation == SIMPLEX )
                      outstring("// Id      vertices\n");
                    else outstring("// Id    edges\n");
                    break;
                 case BODY:
                    outstring("// Id    facets\n");
                    break;
                 case FACETEDGE:
                    outstring("//  id     edge  facet  prevedge nextedge    prevfacet nextfacet\n");
                    break;
                }
              break;

              case REFINE_: 
                break;
              
              case DISSOLVE_:
                break;

              case DELETE_:
                break;
              
              case FIX_:
                break;

              case UNFIX_:
                break;

              case EDGESWAP_:
                break;

              case VERTEX_AVERAGE_: break;
              case RAW_VERTEX_AVERAGE_: break;
              case RAWEST_VERTEX_AVERAGE_: break;

              default: kb_error(1290,"Bad aggregate type.\n",RECOVERABLE);
            }
         break;
         /* end case AGGREGATE_INIT */

        /************************/
        /* next element in loop */
        /************************/
         
     case SINGLE_ELEMENT_: 
          q_id = node[node->left].op2.id;
          if ( stacktop[-1] > 0.0 ) /* done */
              node += node->op1.intval;
          else stacktop[-1] = 1.0;
          break;

     case NEXT_VERTEX_: /* all vertices */
       { 
         vertex_id *vp;
         vp = (element_id *)(stacktop - 0);
         if ( !valid_id(*vp) || breakflag )
                  { /* done */ 
                 node += node->op1.intval - 1;
                 break;
                  }
         q_id = *vp;
         node->op2.id = q_id; /* local variable */
         /* check sentinel; set for next time around */
         if ( equal_id(*vp,*(vertex_id*)(stacktop-1)) ) *vp = NULLID;
         else *vp = vptr(*vp)->forechain;

         if ( !valid_element(q_id) ) node--;    /* skip body of loop */
        }
        break; 

     case NEXT_EDGE_VERTEX_: /* edge vertices */
       {
         int *numptr;
         vertex_id *vp;
         numptr = (int *)(stacktop - 1);
         if ( *numptr== web.skel[EDGE].ctrlpts ) 
            { /* done */
              node += node->op1.intval - 1;
              break;
            }
         vp = *(element_id **)(stacktop - 0);
         q_id = vp[*numptr];
         node->op2.id = q_id; /* local variable */
         (*numptr)++;

          if ( !valid_element(q_id) ) node--;    /* skip body of loop */
         break;
       }

      case NEXT_FACET_VERTEX_: /* facet vertices */
        {
         int *numptr;
         facetedge_id *fe_ptr;
         facet_id *f_ptr;
         if ( (web.representation == SIMPLEX) ||
             (web.modeltype==LAGRANGE) )
         { numptr = (int *)(stacktop - 1);
            if ( *numptr >= web.skel[FACET].ctrlpts )
            { node += node->op1.intval - 1;
              break;
            }
            f_ptr = (facet_id *)(stacktop);
            q_id = get_facet_vertices(*f_ptr)[*numptr];
            (*numptr)++;
         }
         else
         { fe_ptr = (element_id *)(stacktop - 0);
            if ( !valid_id(*fe_ptr) ) /* see if done */
            { node += node->op1.intval - 1;
              break;
            }
            if ( id_type(*fe_ptr) == VERTEX )
            { /* last vertex in string unclosed facet */
              q_id = *fe_ptr; *fe_ptr = NULLID;
            }
            else
            { edge_id e_id = get_fe_edge(*fe_ptr);
              q_id = get_edge_tailv(e_id);
              *fe_ptr = get_next_edge(*fe_ptr);
              if ( !valid_id(*fe_ptr) ) 
                *fe_ptr = get_edge_headv(e_id); /* end of string facet */
              else if ( equal_id(*fe_ptr,*(element_id *)(stacktop-1)))
                *fe_ptr = NULLID; /* last one */
            }
         }
         node->op2.id = q_id; /* local variable */
          if ( !valid_element(q_id) ) node--;    /* skip body of loop */
         break;
        }

     case NEXT_BODY_VERTEX_: /* body vertices */
        break;

     case NEXT_EDGE_: /* all edges */
       { 
         edge_id *ep; 
         ep = (element_id *)(stacktop - 0);
         if ( !valid_id(*ep) || breakflag )
                  {  /* done */
                 node += node->op1.intval - 1;
                 break;
                  }
         q_id = *ep;
         node->op2.id = q_id; /* local variable */
         /* check sentinel; set for next time around */
         if ( equal_id(*ep,*(edge_id*)(stacktop-1)) ) *ep = NULLID;
         else  *ep = eptr(*ep)->forechain; 

         if ( !valid_element(q_id) ) node--;    /* skip body of loop */
          }
       break;

     case NEXT_VERTEX_EDGE_: /* vertex edges */
       { 
         edge_id *e_ptr;
         e_ptr = (element_id *)(stacktop - 0);
         if ( !valid_id(*e_ptr) ) /* see if done */
            { 
              node += node->op1.intval - 1;
              break;
            }
         q_id = *e_ptr;
         node->op2.id = q_id; /* local variable */
         if ( get_attr(*(element_id *)(stacktop-2)) & (Q_MIDPOINT|Q_MIDEDGE) )
            *e_ptr = NULLID; /* last one */
         else
         { *e_ptr = get_next_tail_edge(*e_ptr);
             if ( equal_id(*e_ptr,get_vertex_edge(get_edge_tailv(*e_ptr))))
            *e_ptr = NULLID; /* last one */
          }

          if ( !valid_element(q_id) ) node--;    /* skip body of loop */
          break;
        }

     case NEXT_FACET_EDGE_: /* facet edges */
       { 
         facetedge_id *fe_ptr;
         fe_ptr = (element_id *)(stacktop - 0);
         if ( !valid_id(*fe_ptr) ) /* see if done */
            { 
              node += node->op1.intval - 1;
              break;
            }
         q_id = get_fe_edge(*fe_ptr);
         node->op2.id = q_id; /* local variable */
         *fe_ptr = get_next_edge(*fe_ptr);
         if ( web.representation == STRING )
         {
          if ( equal_id(*fe_ptr,*(element_id *)(stacktop-1)))
            *fe_ptr = NULLID; /* last one */
          }
          else
          { if ( ++(*(int*)(stacktop-1)) == FACET_VERTS )
            *fe_ptr = NULLID; /* last one */
          }
          if ( !valid_element(q_id) ) node--;    /* skip body of loop */
          break;
        }

     case NEXT_BODY_EDGE_: /* body edges */
        break;

     case NEXT_FACET_:  /* all facets */
       { 
         facet_id *fp; 
         fp = (element_id *)(stacktop - 0);
         if ( !valid_id(*fp)|| breakflag  )
            { node += node->op1.intval - 1; /* skip to end */
              break;
            }
          q_id = *fp;
          node->op2.id = q_id; /* local variable */
          /* check sentinel; set for next time around */
          if ( equal_id(*fp,*(facet_id*)(stacktop-1)) ) *fp = NULLID;
          else *fp = fptr(*fp)->forechain;

         if ( !valid_element(q_id) ) node--;    /* skip body of loop */
        }
        break;

     case NEXT_EDGE_FACET_:  /* all facets on edge */
       { 
         facetedge_id *fe_ptr;
         fe_ptr = (element_id *)(stacktop - 0);
         if ( !valid_id(*fe_ptr) ) /* see if done */
            { node += node->op1.intval - 1;
              break;
            }
          q_id = get_fe_facet(*fe_ptr);
          node->op2.id = q_id; /* local variable */
          *fe_ptr = get_next_facet(*fe_ptr);
          if ( equal_element(*fe_ptr,get_edge_fe(get_fe_edge(*fe_ptr))))
                  *fe_ptr = NULLID; /* last one */

          if ( !valid_element(q_id) ) node--;    /* skip body of loop */
            break;
        }

     case NEXT_VERTEX_FACET_: /* facets adjacent to vertex */
       { 
         facet_id *f_ptr;
         f_ptr = (element_id *)(stacktop - 0);
         if ( !valid_id(*f_ptr) ) /* see if done */
            { 
              node += node->op1.intval - 1;
              break;
            }
         q_id = *f_ptr;
         node->op2.id = q_id; /* local variable */
         v_id = *(vertex_id*)(stacktop-2);
         if ( get_vattr(v_id) & Q_MIDPOINT )
         { facetedge_id start_fe = get_vertex_fe(v_id);
            fe = *(facetedge_id*)(stacktop-1);
            fe = get_next_facet(fe);
            *(facetedge_id*)(stacktop-1) = fe;
            if ( equal_id(fe,start_fe) ) *f_ptr = NULLID;
            else *f_ptr = get_fe_facet(fe);
         }
         else if ( get_vattr(v_id) & Q_MIDFACET ) *f_ptr = NULLID;
         else 
         { *f_ptr=get_next_vertex_facet(*(vertex_id*)(stacktop-2),*f_ptr);
            if ( equal_element(*f_ptr,
             get_first_vertex_facet(*(vertex_id *)(stacktop-2))))
            *f_ptr = NULLID; /* last one */
         }
         if ( !valid_element(q_id) ) node--;    /* skip body of loop */
        }
        break;

     case NEXT_BODY_FACET_: /* facets on body */
       { 
         facet_id *f_ptr;
         f_ptr = (element_id *)(stacktop - 0);
         if ( !valid_id(*f_ptr) ) /* see if done */
            { 
              node += node->op1.intval - 1;
              break;
            }
         q_id = *f_ptr;
         node->op2.id = q_id; /* local variable */
         *f_ptr = get_next_body_facet(*f_ptr);
          if ( equal_id(*f_ptr,*(element_id *)(stacktop-1)))
            *f_ptr = NULLID; /* last one */

          if ( !valid_element(q_id) ) node--;    /* skip body of loop */
        }

        break;

     case NEXT_BODY_: /* all bodies */
       { 
         body_id *bp; 
         bp = (element_id *)(stacktop - 0);
         if ( !valid_id(*bp)|| breakflag  )
          { node += node->op1.intval - 1;
             break;
          }  
          q_id = *bp;
          node->op2.id = q_id; /* local variable */
          /* check sentinel; set for next time around */
          if ( equal_id(*bp,*(body_id*)(stacktop-1)) ) *bp = NULLID;
          else *bp = bptr(*bp)->forechain;

          if ( !valid_element(q_id) ) node--;    /* skip body of loop */
         }
         break;

         case NEXT_FACET_BODY_: /* both bodies on facet */
         { 
            int *numptr = (int *)(stacktop - 1);
            f_id = *(facet_id *)(stacktop - 2); /* facet */
            (*numptr)++;
            if ( *numptr == 3 ) /* see if done */
            { /* done */
              node += node->op1.intval - 1;
              break;
            }
            if ( *numptr == 1 ) /* first */
            { 
              q_id = get_facet_body(f_id);
              node->op2.id = q_id; /* local variable */
              if ( !valid_id(q_id) )
                node--;  /* try again */
             }
             else /* *numptr == 2 */
             { q_id = get_facet_body(inverse_id(f_id));
                node->op2.id = q_id; /* local variable */
                if ( !valid_id(q_id) )
                node--;  /* try again */
             }

             break;
         }

     case NEXT_VERTEX_BODY_: /* bodies adjacent to vertex */
        break;
     case NEXT_EDGE_BODY_: /* bodies adjacent to edge */
        break;

     case NEXT_FACETEDGE_: /* all facetedgs */
       { 
         facetedge_id *fep;
         fep = (element_id *)(stacktop - 0);
         if ( !valid_id(*fep)|| breakflag  )
                  { /* done */ 
                 node += node->op1.intval - 1;
                 break;
                  }
         q_id = *fep;
         node->op2.id = q_id; /* local variable */
         /* check sentinel; set for next time around */
         if ( equal_id(*fep,*(facetedge_id*)(stacktop-1)) ) *fep = NULLID;
         else *fep = feptr(*fep)->forechain;

          if ( !valid_element(q_id) ) node--;    /* skip body of loop */
        }
        break; 

        /***********************/
        /* aggregate condition */
        /***********************/

     case WHERE_: /* see if element qualifies */
          if ( *(stacktop--) == 0.0 )
          node += node->left-1;  /* no, go back to generator */
          else node->op2.intval++; /* where count */
          break;  /* else continue with current element */

     case SINGLE_ASSIGN_:
          /* restore old q_id */
          q_id = *(element_id *)(stacktop - 2);
         stacktop -= 3;  /* erase locals */
         recalc_flag = 1;
         break;


        /*****************/
        /* aggregate end */
        /*****************/

     case AGGREGATE_END_:
          aggregate_depth--;
          /* restore old q_id */
          q_id = *(element_id *)(stacktop - 2);
          where = node+node->right;
          where = where+where->left;
          if ( where->type == WHERE_ )
          { web.where_count = where->op2.intval;
            where->op2.intval = 0; /* reset where count */
          }
          switch ( node->op1.intval )
          {
            case FOREACH_:
            case SET_ATTRIBUTE_LOOP_:
              stacktop -= 3;  /* erase locals */
              break; 

            case MAX_:  case MIN_: case SUM_: case COUNT_:
              stacktop -= 3;  /* erase locals */
              break;

            case AVG_:
              stacktop -= 4;  /* erase locals */
              if ( stacktop[0] > 0.0 )
             stacktop[0] = stacktop[1]/stacktop[0];
              /* else leave avg as 0 */
              break;

             case LOGHISTOGRAM_:
             case HISTOGRAM_:
              { int i;
             REAL binsize;
             bins = stacktop - (HISTBINS+1) + 1 - 3;

             if ( histo_count == 0 )
                { 
                  outstring("No qualifying values.\n");
                  stacktop -= HISTBINS+1 + 3 + 3;
                  break;
                }
             /* find hi and lo */
             hi = -MAXDOUBLE; lo = MAXDOUBLE;
             for ( i = 0 ; i < histo_count ; i++ )
             { if ( histo_data[i] < lo ) lo = histo_data[i];
                if ( histo_data[i] > hi ) hi = histo_data[i];
             }
             if ( hi == lo )
              { sprintf(msg,"%10.5g - %10.5g     %d\n",
                     (DOUBLE)hi,(DOUBLE)lo,histo_count);
                  outstring(msg);
                  stacktop -= HISTBINS+1 + 3 + 3;
                  break;
                }
             if (node->op1.intval == HISTOGRAM_ )
                 binsize = (hi-lo)/HISTBINS;/* binsize */
              else /* first bin for 0 values for log histogram */
              { if ( hi <= 0.0 ) binsize = 0.0;
                else 
                 { hi = log(hi);
                   if ( lo <= 0.0 ) 
                   { binsize = 1.0; lo = hi-(HISTBINS-1); }
                   else { lo = log(lo); 
                          binsize = (hi-lo)/(HISTBINS-1);
                        }
                 }
              }
             binsize *= 1.00000000001; /* for rounding margin */
             if ( lo != 0.0 ) lo -= 1e-10*binsize;
             /* construct histogram */
             for ( i = 0 ; i < histo_count ; i++ )
             { val = histo_data[i];
                if ( !is_finite(val) ) k = HISTBINS;  /* NaN */ 
                else if ( node->op1.intval == HISTOGRAM_ )
                  k = (int)((val-lo)/binsize);
                else if ( val <= 0.0 ) k = 0;
                else
                 { val = log(val);
                   k = (int)((val-lo)/binsize)+1;
                 }
                bins[k] += 1.0;
             }
             /* print histogram */
             if (node->op1.intval == HISTOGRAM_ )
              for ( n = 0 ; n < HISTBINS ; n++ )
                { sprintf(msg,"%10.5g - %10.5g     %d\n",
                  (DOUBLE)(lo + n*binsize),(DOUBLE)(lo + (n+1)*binsize),
                     (int)bins[n]);
                  outstring(msg);
                }
              else /* log histogram */
                { int zeroes = (int)bins[0];
                  if ( zeroes )
                  { sprintf(msg,"          <= 0.0     %d\n",zeroes);
                    outstring(msg);
                  }
                  for ( n = 1 ; n < HISTBINS ; n++ )
                  { sprintf(msg,"%10.5g - %10.5g     %d\n",
                     (DOUBLE)(exp(lo + (n-1)*binsize)),
                     (DOUBLE)(exp(lo + n*binsize)),
                     (int)bins[n]);
                    outstring(msg);
                  }
                }
              if ( bins[HISTBINS] > 0.0 ) 
                 { sprintf(msg,"NaN          %d\n",(int)bins[HISTBINS]);
                   outstring(msg);
                  }

              temp_free((char*)histo_data);
              stacktop -= HISTBINS+1 + 3 + 3;  /* erase locals */
              break;
            }

             case LIST_:
             stacktop -= 3;  /* erase locals */
             break;

             case SET_COLOR_: 
             case SET_FRONTCOLOR_: 
             case SET_BACKCOLOR_: 
             case SET_OPACITY_:
             stacktop -= 3;  /* erase locals */
             update_display_flag = 1;
             break;

             case SET_FIXED_: case SET_NO_REFINE_: case SET_NO_DISPLAY_:
             case SET_DENSITY_: case UNSET_DENSITY_: 
             case SET_CONSTRAINT_: case UNSET_CONSTRAINT_:
             case SET_VOLUME_: case SET_PRESSURE_: case UNSET_BOUNDARY_:
             case UNSET_FIXED_: case UNSET_PRESSURE_: case UNSET_VOLUME_:
             case SET_NAMED_QUANTITY_: case UNSET_NO_REFINE_: 
             case UNSET_NAMED_QUANTITY_: case UNSET_NO_DISPLAY_:
             case SET_METHOD_INSTANCE_: case UNSET_METHOD_INSTANCE_:
             case UNSET_TRIPLE_PT_: case UNSET_TETRA_PT_:
             stacktop -= 3;  /* erase locals */
             recalc_flag = 1;
             break;

             case SET_FRONTBODY_:
             case SET_BACKBODY_:
             case UNSET_FRONTBODY_:
             case UNSET_BACKBODY_:
             case UNSET_FACET_BODY_:
                 make_bfacet_lists();  /* so lists legal */
                 stacktop -= 3;  /* erase locals */
                 recalc_flag = 1;
                 break;

              case REFINE_:
                stacktop -= 3;  /* erase locals */
                web.refine_count=refine_count;
                if ( refine_count ) recalc_flag = 1;
                break;

              case DISSOLVE_:
                stacktop -= 3;  /* erase locals */
                web.dissolve_count = dissolve_count;
                if ( dissolve_count ) recalc_flag = 1;
                break;

              case DELETE_:
                stacktop -= 3;  /* erase locals */
                web.delete_count = delete_count;
                if ( delete_count ) recalc_flag = 1;
                break;

              case FIX_:
                stacktop -= 3;  /* erase locals */
                if ( fix_count ) recalc_flag = 1;
                break;

              case UNFIX_:
                stacktop -= 3;  /* erase locals */
                if ( unfix_count ) recalc_flag = 1;
                break;

              default:
                stacktop -= 3;  /* erase locals */
                break; 
          } break; /* end AGGREGATE_END */

        /*********************/
        /* attribute setting */
        /* aggregate verbs    */
        /*********************/

    case SET_FIXED_:
        set_attr(q_id,FIXED);
        /* go back to next element generator */
        node += node->op1.intval - 1;  /* back to start of loop */
        break;

    case UNSET_FIXED_:
        unset_attr(q_id,FIXED);
        /* go back to next element generator */
        node += node->op1.intval - 1;  /* back to start of loop */
        break;

    case SET_NO_DISPLAY_:
        set_attr(q_id,NODISPLAY);
        /* go back to next element generator */
        node += node->op1.intval - 1;  /* back to start of loop */
        break;

    case UNSET_NO_DISPLAY_:
        unset_attr(q_id,NODISPLAY);
        /* go back to next element generator */
        node += node->op1.intval - 1;  /* back to start of loop */
        break;

    case SET_NO_REFINE_:
        set_attr(q_id,NO_REFINE);
        /* go back to next element generator */
        node += node->op1.intval - 1;  /* back to start of loop */
        break;

    case UNSET_NO_REFINE_:
        unset_attr(q_id,NO_REFINE);
        /* go back to next element generator */
        node += node->op1.intval - 1;  /* back to start of loop */
        break;

    case UNSET_PRESSURE_:
        unset_attr(q_id,PRESSURE);
        /* go back to next element generator */
        node += node->op1.intval - 1;  /* back to start of loop */
        break;

    case UNSET_CONSTRAINT_:
        k = (int)*(stacktop--);
        if ( (k < 0) || (k >= MAXCON) || !web.constraint_addr[k])
        { sprintf(errmsg,"Illegal constraint number: %d.\n",k);
          kb_error(1291,errmsg,RECOVERABLE);
        }
        switch ( id_type(q_id) )
        { case VERTEX:
             unset_v_constraint_map(q_id,k);
             break;
          case EDGE:
             unset_e_constraint_map(q_id,k);
             break;
          case FACET:
             unset_f_constraint_map(q_id,k);
             break;
          default: 
             kb_error(1292,"Bad element type for constraint.\n",RECOVERABLE);

        }
        /* go back to next element generator */
        node += node->op1.intval - 1;  /* back to start of loop */
        break;

    case UNSET_BOUNDARY_:
        k = (int)*(stacktop--);
        bdry = web.boundaries+k;
        if ( get_attr(q_id) & BOUNDARY )
        switch ( id_type(q_id) )
        { case VERTEX:
             if ( get_boundary(q_id) == web.boundaries+k )
              { set_boundary_num(q_id,0);
                unset_attr(q_id,BOUNDARY);
              }
             break;
          case EDGE:
             if ( get_edge_boundary(q_id) == web.boundaries+k )
              { set_edge_boundary_num(q_id,0);
                unset_attr(q_id,BOUNDARY);
              }
             break;
          case FACET:
             if ( get_facet_boundary(q_id) == web.boundaries+k )
              { set_facet_boundary_num(q_id,0);
                unset_attr(q_id,BOUNDARY);
              }
             break;
          default: 
             kb_error(1293,"Bad element type for boundary.\n",RECOVERABLE);

         }
        if ( bdry->attr & CON_ENERGY )
             unapply_method(q_id,bdry->energy_method);
        if ( bdry->attr & CON_CONTENT )
          { unapply_method(q_id,bdry->content_method_1);
            unapply_method(q_id,bdry->content_method_2);
          }
        /* go back to next element generator */
        node += node->op1.intval - 1;  /* back to start of loop */
        break;

    case UNSET_VOLUME_:
    case UNSET_TARGET_:
        if ( get_attr(q_id) & FIXEDVOL)
         { unset_attr(q_id,FIXEDVOL);
           if (everything_quantities_flag )
           { struct gen_quant *q = GEN_QUANTS + bptr(q_id)->volquant;
             q->target = 0.0;
             if ( !(q->flags & Q_INFO) )
             { q->flags &= ~(Q_FIXED|Q_ENERGY);
                q->flags |= Q_INFO;
                reset_quant_bits();
             }
           }
         }
        /* go back to next element generator */
        node += node->op1.intval - 1;  /* back to start of loop */
        break;

    case UNSET_FACET_BODY_:
        if ( id_type(q_id) != FACET  )
          kb_error(1294,"Trying to unset body of non-facet.\n",RECOVERABLE);
        set_facet_body(q_id,NULLID);
        set_facet_body(inverse_id(q_id),NULLID);
        /* go back to next element generator */
        node += node->op1.intval - 1;  /* back to start of loop */
        break;

   case UNSET_FRONTBODY_:
   case UNSET_BACKBODY_:
       set_body(node->type==UNSET_FRONTBODY_?q_id:inverse_id(q_id),NULLID);
       recalc_flag = 1;
       break;

    case UNSET_DENSITY_:
        unset_attr(q_id,DENSITY);
        /* go back to next element generator */
        node += node->op1.intval - 1;  /* back to start of loop */
        break;

    case UNSET_TRIPLE_PT_:
        unset_attr(q_id,TRIPLE_PT);
        /* go back to next element generator */
        node += node->op1.intval - 1;  /* back to start of loop */
        break;

    case UNSET_TETRA_PT_:
        unset_attr(q_id,TETRA_PT);
        /* go back to next element generator */
        node += node->op1.intval - 1;  /* back to start of loop */
        break;

    case UNSET_AXIAL_POINT_:
        unset_attr(q_id,AXIAL_POINT);
        /* go back to next element generator */
        node += node->op1.intval - 1;  /* back to start of loop */
        break;

    case SET_DENSITY_:
      { REAL density = *(stacktop--);
         if ( density != 1.0 ) set_attr(q_id,DENSITY);
         switch ( node->op2.intval )
            { case EDGE: set_edge_density(q_id,density); break;
              case FACET: set_facet_density(q_id,density); break;
              case BODY: set_body_density(q_id,density); break;
            }
         /* go back to next element generator */
         node += node->op1.intval - 1;  /* back to start of loop */
        }
      break;

    case SET_VOLUME_: 
      { REAL v = *(stacktop--);
        if ( get_attr(q_id) & PRESSURE )
            kb_error(2014,"Must unset body pressure before fixing target.\n",
            RECOVERABLE);
        set_attr(q_id,FIXEDVOL);
        set_body_fixvol(q_id,v); 
         /* go back to next element generator */
         node += node->op1.intval - 1;  /* back to start of loop */
        break;
      }

    case SET_PRESSURE_: 
      { REAL p = *(stacktop--);
        if ( get_attr(q_id) & FIXEDVOL )
            kb_error(2013,"Must unset body target before fixing pressure.\n",
            RECOVERABLE);
        set_attr(q_id,PRESSURE);
        set_body_pressure(q_id,p); 
        if ( everything_quantities_flag )
        { struct gen_quant *q = GEN_QUANTS + get_body_volquant(q_id);
          q->modulus = -p;
          if ( !(q->flags & Q_ENERGY) )
          { q->flags &= ~(Q_INFO|Q_FIXED);
             q->flags |= Q_ENERGY;
             reset_quant_bits();
          }
        }
         /* go back to next element generator */
         node += node->op1.intval - 1;  /* back to start of loop */
        break;
      }

    case SET_OPACITY_:
        facet_alpha = *(stacktop)--;
         /* go back to next element generator */
         node += node->op1.intval - 1;  /* back to start of loop */
        break;

    case SET_CONSTRAINT_:
       { int con = (int)*(stacktop--);
         if ( (con<0) || (con>=MAXCON) || !web.constraint_addr[con])
         { sprintf(errmsg,"Illegal constraint number: %d.\n",con);
            kb_error(1295,errmsg,RECOVERABLE);
         }
         switch ( node->op2.intval )
            { case VERTEX: set_v_constraint_map(q_id,con); break;
              case EDGE: set_e_constraint_map(q_id,con); break;
              case FACET: set_f_constraint_map(q_id,con); break;
            }
         /* go back to next element generator */
         node += node->op1.intval - 1;  /* back to start of loop */
       }
       break;

    case SET_NAMED_QUANTITY_:
       { int qnum = (int)*(stacktop--);
         apply_quantity(q_id,qnum);
         /* go back to next element generator */
         node += node->op1.intval - 1;  /* back to start of loop */
        }
        break;

    case UNSET_NAMED_QUANTITY_:
       { int qnum = (int)*(stacktop--);
         unapply_quantity(q_id,qnum);
         /* go back to next element generator */
         node += node->op1.intval - 1;  /* back to start of loop */
       }
       break;

    case SET_METHOD_INSTANCE_:
       { int qnum = (int)*(stacktop--);
         apply_method(q_id,METH_INST[qnum].name);
         /* go back to next element generator */
         node += node->op1.intval - 1;  /* back to start of loop */
       }
       break;

    case UNSET_METHOD_INSTANCE_:
       { int qnum = (int)*(stacktop--);
         unapply_method(q_id,qnum);
         /* go back to next element generator */
         node += node->op1.intval - 1;  /* back to start of loop */
       }
       break;

    case SET_COLOR_:
       { int color = (int)*(stacktop--);
         switch ( node->op2.intval )
            { case EDGE: set_edge_color(q_id,color); break;
              case FACET: set_facet_color(q_id,color); break;
            }
         /* go back to next element generator */
         node += node->op1.intval - 1;  /* back to start of loop */
         update_display_flag = 1;
       }
       break;

    case SET_FRONTCOLOR_:
      { int color = (int)*(stacktop--);
         set_facet_frontcolor(q_id,color);
         /* go back to next element generator */
         node += node->op1.intval - 1;  /* back to start of loop */
         update_display_flag = 1;
        }
        break;

    case SET_BACKCOLOR_:
      { int color = (int)*(stacktop--);
         set_facet_backcolor(q_id,color);
         /* go back to next element generator */
         node += node->op1.intval - 1;  /* back to start of loop */
         update_display_flag = 1;
        }
        break;

    case SET_TAG_:
       { int tag = (int)*(stacktop--);
         switch ( node->op2.intval )
            { 
              case FACET: set_tag(q_id,tag); break;
            }
         /* go back to next element generator */
         node += node->op1.intval - 1;  /* back to start of loop */
        }
       break;

    case SET_COORD_:
         get_coord(q_id)[node->op2.intval] = *(stacktop--);
         if ( node->op2.intval <= SDIM ) recalc_flag = 1;
         /* go back to next element generator */
         node += node->op1.intval - 1;  /* back to start of loop */
        break;

    case SET_PARAM_:
        if ( get_vattr(q_id) & BOUNDARY )
          { struct boundary *boundary = get_boundary(q_id);
             REAL *param = get_param(q_id);
             REAL *xx = get_coord(q_id);
             if ( node->op2.intval > boundary->pcount )
             { sprintf(errmsg,"Parameter number is %d; maximum is %d.\n",
                node->op2.intval,boundary->pcount); 
            kb_error(1296,errmsg,RECOVERABLE);
             }
             param[node->op2.intval] = *(stacktop--);
             for ( k = 0 ; k < SDIM ; k++ )
            xx[k] = eval(boundary->coordf[k],param,q_id);
             recalc_flag = 1;
          }
         else
          { /* probably just handy for storage */
             REAL *param = get_param(q_id);
             param[node->op2.intval] = *(stacktop--);
          }
         /* go back to next element generator */
         node += node->op1.intval - 1;  /* back to start of loop */
        break;

        /************************/
        /* numerical aggregates */
        /************************/

    case MAX_: 
       { REAL *aggrptr;

         val = *(stacktop--);
         aggrptr = (REAL *)(stacktop - 3);
         if ( val > *aggrptr ) *aggrptr = val;
         /* go back to next element generator */
         node += node->op1.intval - 1;  /* back to start of loop */
       }
       break;

    case MIN_: 
       {
         REAL *aggrptr;

         val = *(stacktop--);
         aggrptr = (REAL *)(stacktop - 3);
         if ( val < *aggrptr ) *aggrptr = val;
         /* go back to next element generator */
         node += node->op1.intval - 1;  /* back to start of loop */
       }
       break;

    case SUM_: 
         stacktop--;
         stacktop[-3] += stacktop[1];
         /* go back to next element generator */
         node += node->op1.intval - 1;  /* back to start of loop */
         break;

    case AVG_: 
         stacktop--;
         stacktop[-3] += stacktop[1]; /* sum */
         stacktop[-4] += 1.0; /* count */
         /* go back to next element generator */
         node += node->op1.intval - 1;  /* back to start of loop */
         break;

    case COUNT_: 
          stacktop--;
          stacktop[-3] += 1.0;
            /* go back to next element generator */
              node += node->op1.intval - 1;  /* back to start of loop */
          break;

    case HISTOGRAM_: 
    case LOGHISTOGRAM_:
          if ( histo_count >= histo_max )
          { histo_data = (REAL*)kb_realloc((char*)histo_data,
            2*histo_max*sizeof(REAL),histo_max*sizeof(REAL));
             histo_max *= 2;
          }
          histo_data[histo_count++] = *(stacktop--);
          node += node->op1.intval - 1;  /* back to start of loop */
        break;

    case FOREACH_:
    case SET_ATTRIBUTE_LOOP_:
        node += node->op1.intval - 1;  /* back to start of loop */
        break;

    case FINISHED:
        goto the_exit; 

    default:
        stacktop = other_stuff(node,stacktop,&recalc_flag,
            &update_display_flag,q_id,newstack);
        break;
      
  }      

  if ( breakflag == BREAKFULL )
     kb_error(2345,"Command aborted due to user interrupt.\n",RECOVERABLE);

  if ( *stack_sentinel != STACKMAGIC ) 
     { if ( stackmax == MAXSTACK )  /* stack was on runtime stack */
       { stackmax *= 2;  /* double size */
         newstack = (REAL*)temp_calloc(stackmax,sizeof(REAL) );
         memcpy(newstack,stack,stackmax*sizeof(REAL));
         stacktop = newstack + (stacktop-stack);
         stack_sentinel = &newstack[stackmax-5];
       } 
       else    /* stack is temp memory */
       { int n = stacktop - newstack;
         newstack = (REAL *)temp_realloc((char*)newstack,2*stackmax*sizeof(REAL),
                            stackmax*sizeof(REAL));
         stacktop = newstack + n; 
         stackmax *= 2;
         stack_sentinel = &newstack[stackmax-5];
       }
      *stack_sentinel = STACKMAGIC;
    }
  }
the_exit:

  if ( !params )
  {
     if ( fix_count )
     { sprintf(msg,"Fixed: %d\n",fix_count); outstring(msg); }
     if ( unfix_count )
     { sprintf(msg,"Unfixed: %d\n",unfix_count); outstring(msg); }
     if ( refine_count )
     { sprintf(msg,"Refined: %d\n",refine_count); outstring(msg); }
     if ( dissolve_count )
     { sprintf(msg,"Dissolved: %d\n",dissolve_count); outstring(msg); }
     if ( delete_count )
     { sprintf(msg,"Deleted: %d\n",delete_count); outstring(msg); }
     if ( equi_count )
     { sprintf(msg,"Swapped: %d\n",equi_count); outstring(msg); }

     if ( recalc_flag ) recalc();
     else if ( update_display_flag ) update_display();

     if ( aggregate_depth == 0 ) free_discards();
  }
  iterate_flag = old_flag; 
  if ( stacktop == newstack+1 ) return *(stacktop--);
  else if ( stacktop != newstack )
  { 
      sprintf(errmsg,"Internal error: Stack misalignment by %d in eval()\n",stacktop-newstack);
      kb_error(1298,errmsg, RECOVERABLE);
  }

  return 0.0;
} /* end eval() */
