/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: main.c
 */

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <getopt.h>
#include <locale.h>

#include <pthread.h>
#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "pixmap.h"
#include "main_window.h"
#include "logging.h"
#include "message.h"
#include "commit.h"
#include "views.h"
#include "main.h"

static pthread_t main_thread_tid;
static engine_mode_t open_mode = -1;
static debug_level_t debug_level = -1;
static gchar *log_name = NULL;
static gchar *node_name = NULL;
static GtkWidget *startup_window = NULL;
static GtkWidget *startup_label = NULL;

static void show_help(char **argv)
{
	printf("\nEnterprise Volume Management System %s\n", VERSION);
	printf("International Business Machines  %s\n\n", DATE);

	printf("Usage: %s [OPTION...] \n\n", argv[0]);

	printf
	    ("  -d, --log-level=LEVEL\t\tLogging output level where LEVEL is one of the following:\n");
	printf("  \t\t\t\t\tcritical | serious | error | warning | default |\n");
	printf("  \t\t\t\t\tdetails | debug | extra | entry-exit | everything\n");
	printf("  -?, --help\t\t\tShow this help message\n");
	printf
	    ("  -m, --mode=ENGINE-MODE\tEngine mode where ENGINE-MODE is one of the following:\n");
	printf("  \t\t\t\t\treadwrite | readonly\n");
	printf("  -n, --node-name=NODENAME\tAdminister the given node in a cluster\n");
	printf("  -l, --log-name=FILENAME\tFile to send logging output to\n\n");
}

/*
 *
 *   static gint parse_options (int argc, char **argv)
 *
 *   Description:
 *      This routine parses the command line options. It then sets the
 *      appropriate program variables accordingly.
 *
 *   Entry:
 *      argc - count of command line arguments
 *      argv - array of program command line argument strings
 *
 *   Exit:
 *      Returns 0 is parsing was successful or -1 if a problem was encountered.
 *
 */
static gint parse_options(int argc, char **argv)
{

	int c;
	gint rc = SUCCESS;
	char *short_opts = "h?d:m:l:n:";
	struct option long_opts[] = {
		{"help", no_argument, NULL, 'h'},
		{"log-level", required_argument, NULL, 'd'},
		{"mode", required_argument, NULL, 'm'},
		{"log-name", required_argument, NULL, 'l'},
		{"node-name", required_argument, NULL, 'n'},
		{NULL, 0, NULL, 0}
	};

	while (rc == SUCCESS && (c = getopt_long(argc, argv, short_opts, long_opts, NULL)) != EOF) {
		switch (c) {
		case 'm':
			if (g_strcasecmp(optarg, "readwrite") == 0)
				open_mode = ENGINE_READWRITE;
			else if (g_strcasecmp(optarg, "readonly") == 0)
				open_mode = ENGINE_READONLY;
			else {
				g_warning(_("Unrecognized mode argument -- \"%s\"\n\n"), optarg);
				rc = -1;
			}

			break;

		case 'd':
			if (g_strcasecmp(optarg, "critical") == 0)
				debug_level = CRITICAL;
			else if (g_strcasecmp(optarg, "serious") == 0)
				debug_level = SERIOUS;
			else if (g_strcasecmp(optarg, "error") == 0)
				debug_level = ERROR;
			else if (g_strcasecmp(optarg, "warning") == 0)
				debug_level = WARNING;
			else if (g_strcasecmp(optarg, "default") == 0)
				debug_level = DEFAULT;
			else if (g_strcasecmp(optarg, "details") == 0)
				debug_level = DETAILS;
			else if (g_strcasecmp(optarg, "debug") == 0)
				debug_level = DEBUG;
			else if (g_strcasecmp(optarg, "extra") == 0)
				debug_level = EXTRA;
			else if (g_strcasecmp(optarg, "entry-exit") == 0)
				debug_level = ENTRY_EXIT;
			else if (g_strcasecmp(optarg, "everything") == 0)
				debug_level = EVERYTHING;
			else {
				g_warning(_("Unrecognized debug level argument -- \"%s\"\n\n"),
					  optarg);
				rc = -1;
			}

			break;

		case 'l':
			log_name = g_strdup(optarg);
			break;

		case 'n':
			node_name = g_strdup(optarg);
			break;

		default:
			g_warning(_("Unrecognized option -- \"%c\"\n\n"), c);
		case 'h':
		case '?':
			rc = -1;
			break;
		}
	}

	if (rc != SUCCESS)
		show_help(argv);

	return rc;
}

/*
 *   void discovery_status_callback (gchar *)
 *
 *   Description:
 *       This routine updates the splash screen message label with the
 *       supplied status screen.
 *
 *   Entry:
 *       status - the status string
 *
 *   Exit:
 *       Returns nothing
 */
void discovery_status_callback(gchar * status)
{
	if (startup_label && GTK_WIDGET_MAPPED(startup_label)) {
		gdk_threads_enter();
		gtk_label_set_text(GTK_LABEL(startup_label), g_strchomp(status));
		gdk_threads_leave();
	}
}

/*
 *
 *   void display_startup_window (void)
 *
 *   Description:
 *      This routine simply displays a window without decorations that
 *      contains a label with text that indicates that EVMS is coming up.
 *      threads.
 *
 *   Entry:
 *      Nothing
 *
 *   Exit:
 *      Returns nothing
 *
 */
void display_startup_window(void)
{
	startup_window = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_window_set_position(GTK_WINDOW(startup_window), GTK_WIN_POS_CENTER);

	startup_label = gtk_label_new(_("EVMS is examining your system. Please wait..."));

	gtk_container_add(GTK_CONTAINER(startup_window), startup_label);
	gtk_misc_set_padding(GTK_MISC(startup_label), 10, 10);

	gtk_widget_realize(startup_window);
	gdk_window_set_decorations(startup_window->window, 0);

	gtk_widget_show(startup_label);
	gtk_widget_show(startup_window);

	/*
	 * Since this is our first window, take advantage of it to
	 * initialize the standard pixmaps before any other windows
	 * such as message windows use them.
	 */
	create_standard_pixmaps(startup_window);
}

/*
 *
 *   inline void destroy_startup_window (void)
 *
 *   Description:
 *      This routine destroys the startup window.
 *
 *   Entry:
 *      Nothing
 *
 *   Exit:
 *      Returns nothing
 *
 */
inline void destroy_startup_window(void)
{
	if (startup_window) {
		gtk_widget_destroy(startup_window);
		startup_window = NULL;
	}
}

/*
 *
 *   void* main_event_loop_thread (void *)
 *
 *   Description:
 *      This routine executes the gtk_main () function to process
 *      the GTK main event loop. Having the event loop on its own
 *      thread allows us to inject events while we are asynchronously
 *      processing other things in the open engine and commit changes
 *      threads.
 *
 *   Entry:
 *      not_used - just like it sounds
 *
 *   Exit:
 *      Returns nothing
 *
 */
void *main_event_loop_thread(void *not_used)
{
	main_thread_tid = pthread_self();

	gdk_threads_enter();
	display_startup_window();
	gtk_main();
	gdk_threads_leave();

	return NULL;
}

/*
 *
 *   gint check_engine_version (pthread_t)
 *
 *   Description:
 *      This routine checks to see that we are running a version
 *      of the engine that we like. If not, we display a message
 *      and wait for the user to acknowledge before exiting.
 *
 *   Entry:
 *      tid - thread id of main event loop thread
 *
 *   Exit:
 *      Returns SUCCESS if we are OK with the version. Otherwise,
 *      we'll return EPERM.
 *
 */
gint check_engine_version(pthread_t tid)
{
	gint rc = EPERM;
	evms_version_t version;

	evms_get_api_version(&version);

	if (REQUIRED_ENGINE_VER_MAJOR == version.major &&
	    ((version.minor > REQUIRED_ENGINE_VER_MINOR) ||
	     (REQUIRED_ENGINE_VER_MINOR == version.minor
	      && REQUIRED_ENGINE_VER_PATCHLEVEL <= version.patchlevel))) {
		rc = SUCCESS;
	} else {
		gchar *error_message;
		GtkWidget *window;

		sleep(1);

		error_message =
		    g_strdup_printf(_
				    ("Version %u.%u.%u or later of the EVMS Engine API required. Detected version %u.%u.%u."),
				    REQUIRED_ENGINE_VER_MAJOR, REQUIRED_ENGINE_VER_MINOR,
				    REQUIRED_ENGINE_VER_PATCHLEVEL, version.major, version.minor,
				    version.patchlevel);
		gdk_threads_enter();

		destroy_startup_window();

		window = create_informational_message_window(error_message, FALSE);

		gtk_widget_show(window);

		gdk_threads_leave();

		g_free(error_message);

		/*
		 * Wait for main loop thread to exit before returning.
		 */
		pthread_join(tid, NULL);
	}

	return rc;
}

/*
 *
 *   gint open_engine (pthread_t)
 *
 *   Description:
 *      This routine allows the opening of the engine to get processed
 *      on the initial thread while we have a separate thread sit in the
 *      GTK main event loop. This is in case we receive message callbacks
 *      and need to display messages during the discovery phase.
 *
 *      Once the engine discovery is complete, we block this thread until
 *      the main event loop thread exits. Blocking this thread allows the
 *      VFS to have a valid pid registered for the lock on the EVMS block
 *      device.
 *
 *   Entry:
 *      tid - thread id of main event loop thread
 *
 *   Exit:
 *      Returns return code from evms_open_engine
 *
 */
gint open_engine(pthread_t tid)
{
	gint rc;
	GtkWidget *window;

	get_ui_callback_table()->status = discovery_status_callback;
	rc = evms_open_engine(node_name, open_mode, get_ui_callback_table(), debug_level, log_name);
	get_ui_callback_table()->status = NULL;

	if (rc != SUCCESS) {
		gchar *error_message;

		log_error(_("%s: evms_open_engine() failed with return code %d.\n"), __FUNCTION__,
			  rc);
		log_error("%s: %s\n", __FUNCTION__, evms_strerror(ABS(rc)));

		error_message =
		    g_strdup_printf(_("Received the following error opening EVMS engine: %s"),
				    evms_strerror(ABS(rc)));
		/*
		 * Give any scheduled idle function a chance to run before displaying the last message
		 */

		sleep(1);

		gdk_threads_enter();

		destroy_startup_window();

		window = create_informational_message_window(error_message, FALSE);

		gtk_widget_show(window);

		gdk_threads_leave();

		g_free(error_message);
	} else {
		gdk_threads_enter();

		destroy_startup_window();

		get_ui_callback_table()->status = commit_status_callback;

		window = create_main_window();

		set_main_window_title(GTK_WINDOW(window), node_name);

		/*
		 * Store away the widget ids of some useful main window widgets
		 */
		set_main_window_id(window);
		set_status_bar_id(gtk_object_get_data(GTK_OBJECT(window), "main_status_bar"));
		set_progress_bar_id(gtk_object_get_data(GTK_OBJECT(window), "main_progress_bar"));
		set_menu_bar_id(gtk_object_get_data(GTK_OBJECT(window), "main_menubar"));
		set_button_toolbar_id(gtk_object_get_data
				      (GTK_OBJECT(window), "icon_button_toolbar"));

		set_status_bar_message(_("Ready."));

		gtk_widget_show(window);
		set_message_window_transient_for(GTK_WINDOW(window));

		gdk_threads_leave();
	}

	pthread_join(tid, NULL);

	if (rc == SUCCESS)
		evms_close_engine();

	return rc;
}

/*
 *
 *   int main (int, char **)
 *
 *   Description:
 *      This routine is the main routine. It determines if the user has root
 *      privileges. If the user does not have root privileges then we prompt
 *      for the root password in order to run as root. It creates the main
 *      window and common pixmaps and enters the main event loop.
 *
 *   Entry:
 *      argc - count of command line arguments
 *      argv - array of program command line argument strings
 *
 *   Exit:
 *      Returns -1 when command line usage is displayed.
 *
 */
int main(int argc, char *argv[])
{
	gint rc;

	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);

	if (geteuid() == 0) {
		g_thread_init(NULL);
		gtk_init(&argc, &argv);

		rc = parse_options(argc, argv);

		if (rc == SUCCESS) {
			rc = pthread_create(&main_thread_tid, NULL, main_event_loop_thread, NULL);

			if (rc == SUCCESS) {
				rc = check_engine_version(main_thread_tid);

				if (rc == SUCCESS)
					rc = open_engine(main_thread_tid);
			} else
				log_error("%s: Unable to start main event loop thread. rc == %d.\n",
					  __FUNCTION__, rc);
		}
	} else {
		rc = EPERM;
		g_warning(_("This command must be run with root privilege.\n"));
	}

	return rc;
}

inline gboolean is_main_event_loop_thread(void)
{
	return pthread_equal(main_thread_tid, pthread_self());
}

engine_mode_t get_open_mode(void)
{
	return open_mode;
}

debug_level_t get_debug_level(void)
{
	return debug_level;
}

gchar *get_log_name(void)
{
	return log_name;
}
