/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdrivelink.so
 *
 *   File: dl_sn.c
 */

#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>

#include <plugin.h>

#include "dl_common.h"


/*
 *  Function: dl_gen_parent_serial_number
 *
 *  Called to get a serial number that can be used for a drivelink
 *  parent object.  This routine generates a new serial number and
 *  also registers it with the engine, returning the SN to the caller.
 *
 *  A parent serial number is different than a childs serial number
 *  in that it needs to be unique amongst all drivelink parent objects
 *  and so we need to register its name to guarantee this.  A child
 *  serial number is only unique amongst the child objects comprising 
 *  the drivelink and so its serial number is NOT registered with 
 *  the engine.
 *
 *  Returns: BAD_SERIAL_NUMBER ... if unable to provide a serial number
 *                              ( otherwise )
 *           A new parent serial number that is registered with the 
 *           engine and OK to use.
 */
u_int32_t dl_gen_parent_serial_number( u_int32_t  guess )
{
        int    rc=EINVAL;
        sn_t   sn = {PREFIX_STRING, guess, 0x00};
        char  *cptr = (char *)&sn.serial_number;
        int    attempts = 0;
        struct timeval  tv;

        do {
                // dont allow any zeroes in the serial number itself, which would
                // NULL terminate the char string too soon!
                if (cptr[0] == 0x00)  cptr[0] = 0x01;
                if (cptr[1] == 0x00)  cptr[1] = 0x02;
                if (cptr[2] == 0x00)  cptr[2] = 0x03;
                if (cptr[3] == 0x00)  cptr[3] = 0x04;

                rc = EngFncs->register_name( (char *) &sn );

                if (rc) {  // Again ?
                        rc = gettimeofday(&tv,NULL);
                        if (rc==0) {
                                sn.serial_number += (u_int32_t) tv.tv_usec;
                                rc = -1; 
                        }
                        else {
                                ++sn.serial_number;
                        }
                }

                ++attempts;

        } while( rc != 0  &&  attempts < 25 );

        if (rc) {
                return BAD_SERIAL_NUMBER;
        }
        else {
                return sn.serial_number;
        }
}


/*
 *  Function: dl_gen_child_serial_number
 *
 *  Called to get a serial number that is Ok to use in the specified
 *  ordering table, i.e. it is unused so far and not a duplicate.
 *
 *  We start with a guess... 0x100 plus the current drive link count. Then,
 *  increment the guess till we hit a serial number that is not a dup of
 *  any existing serial numbers.
 *
 *  Returns: BAD_SERIAL_NUMBER ... if unable to provide a serial number
 *                              ( otherwise )
 *           A new child serial number that is Ok to use in the specified
 *           drive link ordering table
 */
u_int32_t dl_gen_child_serial_number( storage_object_t *drivelink ) 
{
        u_int32_t      guess;
        u_int32_t      serial_number = BAD_SERIAL_NUMBER;
        drivelink_private_data_t *pdata=NULL;

        if (dl_isa_drivelink(drivelink)==TRUE) {
                pdata = (drivelink_private_data_t *)drivelink->private_data;
        }       
        
        if (pdata != NULL) {

                guess = 0x100 + pdata->drive_link_count;

                do {
                        if (dl_isa_dup_child_sn(drivelink, guess) == FALSE) {
                                serial_number = guess;
                        }
                        else {
                                ++guess;
                        }

                } while (serial_number == BAD_SERIAL_NUMBER);

        }

        return serial_number;
}

