/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdos.so
 *
 *   File: segs.h
 */
#ifndef SEGS_HEADER
#define SEGS_HEADER

#include "defsegmgr.h"
#include "ptables.h"

extern list_anchor_t     Disk_PrivateData_List;


boolean            isa_partition_number(char *number);
char *             get_disk_name( storage_object_t *seg );
int                get_name_for_disk_segment( DISKSEG *seg );
DISKSEG *          allocate_disk_segment( LOGICALDISK *ld );
void               free_disk_segment( DISKSEG *seg);

int                create_disk_private_data( LOGICALDISK  *ld );
int                delete_disk_private_data( LOGICALDISK *ld );
DISK_PRIVATE_DATA *get_disk_private_data( LOGICALDISK *ld );

boolean            disk_has_extended_partition( LOGICALDISK *ld );
boolean            seglist_has_mbr_segment( list_anchor_t  seglist );
DISKSEG *          get_mbr_from_seglist( list_anchor_t  seglist );

int                get_first_unused_ptable_entry( list_anchor_t  seglist, DISKSEG *ebr );
int                get_ptable_entry_count( list_anchor_t  seglist, DISKSEG *ebr );

DISKSEG *          get_first_freespace_seg_in_list( list_anchor_t seglist );

boolean            freespace_follows_seg( DISKSEG *seg );

DISKSEG *          get_freespace_following_seg( DISKSEG *seg );
DISKSEG *          find_freespace_in_seglist( list_anchor_t  seglist );
int                find_freespace_on_disk( LOGICALDISK *ld );

int                remove_diskseg_from_list( list_anchor_t  seglist, DISKSEG *seg );
void *             insert_diskseg_into_list( list_anchor_t  seglist, DISKSEG *seg);
void *             insert_diskseg_into_ordered_list( list_anchor_t  seglist, DISKSEG *seg);
int                merge_adjacent_freedisksegs_in_list( list_anchor_t seglist );

int                create_mbr_For_Disk( LOGICALDISK *ld, char *DiskName, boolean isa_os2_disk );
DISKSEG *          build_mbr_disk_segment( LOGICALDISK *ld );
DISKSEG *          build_ebr_disk_segment( LOGICALDISK      *ld,
                                           Partition_Record *part,
                                           DISKSEG          *ebr,
                                           lba_t             ebr_lba,
                                           u_int32_t         ptable_index,
                                           boolean           primary_partition_flag );


DISKSEG *          build_diskseg_from_partition_record( LOGICALDISK      *ld,
                                                        Partition_Record *part,
                                                        DISKSEG          *ebr,
                                                        u_int32_t         ptable_index,
                                                        boolean           primary_partition_flag );


int                fixup_logical_partition_names( LOGICALDISK *ld );
void               fixup_EBR_Names( LOGICALDISK *ld );
int                fixup_EBR_Chain( LOGICALDISK *ld );
void               fixup_EBR_Sizes( LOGICALDISK *ld );
void               fixup_disk_extd_partition_dimensions( LOGICALDISK *ld );
void               fixup_disk_extd_partition_anchor( LOGICALDISK *ld );

int                create_logical_partition( LOGICALDISK *ld, DISKSEG *seg, DLA_Entry *dla, DISKSEG *freespace, sector_count_t offset );
int                create_primary_partition( LOGICALDISK *ld, DISKSEG *seg, DLA_Entry *dla );

boolean            seg_is_within_or_adjacant_to_extended_partition( LOGICALDISK *ld, DISKSEG *seg );
boolean            seg_is_within_the_extended_partition( LOGICALDISK *ld, DISKSEG *seg );


boolean            seg_is_volitile( DISKSEG *seg );
int                diskseg_to_container_segment( DISKSEG *seg );
void               revert_container_segment( DISKSEG *seg );
boolean            seg_is_within_container_segment( DISKSEG *seg );
boolean            seg_overlaps_container_segment( DISKSEG *seg );
int                remove_container_seg_overlap( DISKSEG *seg);


DISKSEG *          only_child( DISKSEG *seg );

//boolean            dos_prune_deactivate_segments( list_anchor_t list );


/*
 *  Called to return the Logical Disk that a segment belongs to.
 *
 *  Can be called with any object.
 *
 *  Returns NULL on failure.
 */
static inline LOGICALDISK * get_logical_disk( storage_object_t *obj )
{
        LOGICALDISK  *ld=NULL;
        SEG_PRIVATE_DATA *pdata;

        if (obj) {

                if ( obj->plugin != Seg_My_PluginRecord_Ptr ) {
                        ld = (LOGICALDISK *) obj;
                }
                else {
                        pdata = (SEG_PRIVATE_DATA *) obj->private_data;

                        if (pdata != NULL) {

                                if ( pdata->signature == DOS_SEG_MGR_PDATA_SIGNATURE ) {
                                        ld = pdata->logical_disk;
                                }
                        }

                }

        }

        return ld;
}


/*
 *     LBA addresses are converted into CHS addressees by this formula:
 *
 *        Sector    = ( LBA MOD Sectors Per Track ) + 1
 *        Head      = ( LBA DIV Sectors Per Track ) MOD Heads Per Cylinder
 *        Cylinder  = ( LBA DIV Sectors Per Track ) DIV Heads Per Cylinder
 */
static inline int LBAtoCHS( LOGICALDISK *ld, lba_t  lba, chs_t *chs )
{
 u_int32_t          sectors_per_track;
 u_int32_t          drive_heads;
 u_int32_t          sectors_per_cylinder;
 int                rc;
 DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data( ld );


    if ( (chs != NULL) &&
         (disk_pdata != NULL) &&
         (ld->geometry.sectors_per_track > 0)  ) {

        memset(chs, 0, sizeof(chs_t));

        sectors_per_track    = ld->geometry.sectors_per_track * disk_pdata->vsectors_per_block;
        drive_heads          = ld->geometry.heads;
        sectors_per_cylinder = sectors_per_track * drive_heads;

        chs->sector   = ( lba % sectors_per_track ) + 1;
        chs->head     = ( lba / sectors_per_track ) % drive_heads;
        chs->cylinder =   lba / sectors_per_cylinder;

        rc = 0;
    }
    else {
        rc = EINVAL;
    }

    return rc;
}



/*
 *     CHS addresses are converted into LBA addresses by the following formula:
 *
 *        LBA = (Sector - 1) + (Head * Sectors Per Track) + (Cylinder * Heads Per Cylinder * Sectors Per Track)
 *
 */
static inline int CHStoLBA( LOGICALDISK *ld, chs_t *chs, lba_t *callers_lba )
{
    u_int32_t          sectors_per_track;
    u_int32_t          drive_heads;
    u_int32_t          sectors_per_cylinder;
    lba_t              lba = 0;
    int                rc;
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data( ld );


    if (  ( chs != NULL ) &&
          ( callers_lba != NULL ) &&
          ( disk_pdata != NULL ) &&
          ( ld->geometry.sectors_per_track > 0 ) ) {

        sectors_per_track    = ld->geometry.sectors_per_track * disk_pdata->vsectors_per_block;
        drive_heads          = ld->geometry.heads;
        sectors_per_cylinder = sectors_per_track * drive_heads;

        lba = (chs->sector - 1 ) + (chs->head * sectors_per_track) + (chs->cylinder * sectors_per_cylinder );

        *callers_lba = lba;

        rc = 0;
    }
    else {
        rc = EINVAL;
    }

    return rc;
}




/*
 *     A disk can be addressed by LBA or by CHS values. A partition table entry
 *     uses the following field sizes:
 *
 *     LBA - 32 bits
 *
 *     CHS - 24 bits  ( cylinders - 10 bits,  heads - 8 bits, sectors - 6 bits )
 *
 *     So, the CHS value has maximums of: 1024 cylinders, 255 heads, 63 sectors
 *
 *     Since disks have grown beyond 1024 cylinders in size, there is a special
 *     convention used in partition records to signify this.
 *
 *     If a partition record has an LBA that wont fit into either the starting
 *     or ending CHS values ... then ... use the LBA field in the partition record
 *     and calculate the ending address as LBA+SIZE.  The partition record
 *     signifys this by placing max values in the CHS fields.
 *
 *     The below routine was written to intercept calls to the LBAtoCHS() conversion
 *     routine ... when the returned CHS values are to be placed in a partition record.
 *
 *     This was done so we'd have a chance to convert CHS values to MAXIMUMS when
 *     we see that the LBA is above 1024 cylinders ... OR ... that the disk is
 *     using LBA addressing for all partition records.
 */

static inline int LBA_to_Ptable_CHS( LOGICALDISK *ld, lba_t  lba, chs_t  *chs )
{
    int  rc = 0;
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data( ld );

    if (disk_pdata->flags & DISK_USES_LBA_ADDRESSING ) {
        chs->cylinder = MAX_CYLINDERS;
        chs->head     = ld->geometry.heads-1;
        chs->sector   = ld->geometry.sectors_per_track;
    }
    else {

        rc = LBAtoCHS( ld, lba, chs );

        if (rc==0) {

            /* if above 1024 cylinder limit of a partition table entry ... we need to
             * max out the CHS values to indicate this situation. We also need to
             * do this if we are using LBA addressing on this disk.
             */
            if ( chs->cylinder > MAX_CYLINDERS ) {

                if (MAX_CYLINDERS > ld->geometry.cylinders) {
                    chs->cylinder = ld->geometry.cylinders-1;
                }
                else {
                    chs->cylinder = MAX_CYLINDERS;
                }

                chs->head     = ld->geometry.heads-1;
                chs->sector   = ld->geometry.sectors_per_track;

            }

        }

    }

    return rc;
}



/*
 *  Returns TRUE if the LBA starts on a cylinder boundary
 */
static inline boolean  starts_on_cylinder_boundary( LOGICALDISK *ld, lba_t lba )
{
    chs_t  chs;

    if ( LBAtoCHS( ld, lba, &chs ) ) {
        return TRUE;
    }
    else {

        if ( ( chs.sector == 1 ) && ( chs.head == 0 ) ) {
            return TRUE;
        }
        else {
            return FALSE;
        }
    }

}


/*
 *  Returns TRUE if the LBA ends on a cylinder boundary
 */
static inline boolean  ends_on_cylinder_boundary( LOGICALDISK *ld, lba_t lba )
{
    chs_t  chs;
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data( ld );


    if ( LBAtoCHS( ld, lba, &chs ) ) {
        return TRUE;
    }
    else {

        if ( ( chs.sector == (ld->geometry.sectors_per_track * disk_pdata->vsectors_per_block) ) &&
             ( chs.head   == ld->geometry.heads-1 ) ) {
            return TRUE;
        }
        else {
            return FALSE;
        }

    }

}


/*
 *  Called to calculate the cylinder size for the specified
 *  storage object. Storage object might be either DISK or
 *  SEGMENT type of object.
 */
static inline sector_count_t get_cylinder_size( storage_object_t *obj )
{
    u_int32_t          sectors_per_track=0;
    u_int32_t          drive_heads=0;
    sector_count_t     sectors_per_cylinder=0;
    LOGICALDISK       *ld=NULL;
    DISK_PRIVATE_DATA *disk_pdata=NULL;


    if ( obj ) {

        if (obj->object_type == DISK) {
            ld = obj;
        }
        else {
            ld = get_logical_disk(obj);
        }

        if (ld) {

            disk_pdata           = get_disk_private_data( ld );

            sectors_per_track    = ld->geometry.sectors_per_track * disk_pdata->vsectors_per_block;
            drive_heads          = ld->geometry.heads;
            sectors_per_cylinder = (sector_count_t) sectors_per_track * drive_heads;

        }

    }


    return sectors_per_cylinder;
}


/*
 *  Returns the specified LBA rounded up to a track boundary.
 */
static inline lba_t  roundup_to_track_boundary( LOGICALDISK *ld, lba_t lba )
{
    lba_t              new_lba = lba;
    sector_count_t     extra_sectors=0;
    sector_count_t     sectors_per_track=0;
    DISK_PRIVATE_DATA *disk_pdata;

    disk_pdata = get_disk_private_data( ld );

    if ( disk_pdata ) {

        sectors_per_track = ld->geometry.sectors_per_track * disk_pdata->vsectors_per_block;

        if (sectors_per_track) {

            extra_sectors = lba % sectors_per_track;

            if ( extra_sectors != 0) {
                new_lba = lba + ( sectors_per_track - extra_sectors ) - 1;
            }

        }

    }

    return new_lba;
}


/*
 *  Returns the specified LBA rounded down to a track boundary.
 */
static inline lba_t  rounddown_to_track_boundary( LOGICALDISK *ld, lba_t lba )
{
    lba_t              new_lba = lba;
    sector_count_t     extra_sectors=0;
    sector_count_t     sectors_per_track=0;
    DISK_PRIVATE_DATA *disk_pdata;

    disk_pdata = get_disk_private_data( ld );

    if ( disk_pdata ) {

        sectors_per_track = ld->geometry.sectors_per_track * disk_pdata->vsectors_per_block;

        if (sectors_per_track) {

            extra_sectors = lba % sectors_per_track;

            if ( extra_sectors != 0) {
                new_lba = lba - extra_sectors;
            }

        }

    }

    return new_lba;
}



/*
 *  Returns the specified LBA rounded up to a cylinder boundary.
 */
static inline lba_t  roundup_to_cylinder_boundary( LOGICALDISK *ld, lba_t lba )
{
    lba_t              new_lba = lba;
    sector_count_t     extra_sectors=0;
    sector_count_t     sectors_per_cylinder;


    sectors_per_cylinder = get_cylinder_size(ld);

    if ( sectors_per_cylinder ) {

        extra_sectors = lba % sectors_per_cylinder;

        if ( extra_sectors != 0) {
            new_lba = lba + ( sectors_per_cylinder - extra_sectors) - 1;
        }

    }

    return new_lba;
}


/*
 *  Returns the specified LBA rounded down to a cylinder boundary.
 */
static inline lba_t  rounddown_to_cylinder_boundary( LOGICALDISK *ld, lba_t lba )
{
    lba_t            new_lba=lba;
    sector_count_t   extra_sectors=0;
    sector_count_t   sectors_per_cylinder;



    sectors_per_cylinder = get_cylinder_size(ld);

    if (sectors_per_cylinder) {

        extra_sectors = lba % sectors_per_cylinder;

        if ( extra_sectors != 0) {
            new_lba = lba - extra_sectors;
        }

    }

    return new_lba;
}


/*
 *  Tests if the stated LBA is below the 1024 cylinder limit on
 *  the drive.
 */
static inline boolean below_1024_cylinder_limit(LOGICALDISK *ld, lba_t  lba )
{
    chs_t   chs;

    if ( LBAtoCHS( ld, lba, &chs ) ) {
       return TRUE;
    }
    else {

        if (chs.cylinder < MAX_CYLINDERS ) {
            return TRUE;
        }
        else {
            return FALSE;
        }

    }

}



/*
 *  Called to test if we have private data for a disk.
 */
static inline boolean i_can_modify_disk( LOGICALDISK *ld )
{
    DISK_PRIVATE_DATA  *disk_pdata;

    if (ld) {

        disk_pdata = get_disk_private_data( ld );

        if (disk_pdata) {

            if (disk_pdata->signature == DOS_SEG_MGR_PDATA_SIGNATURE) {
                return TRUE;
            }

        }
    }

    return FALSE;
}


/*
 *  Called to test if we own the specified disk segment and can
 *  work with the segment.
 */
static inline boolean i_can_modify_seg( DISKSEG *seg )
{
    if (seg) {

        if (seg->plugin == Seg_My_PluginRecord_Ptr ) {

            if (seg->private_data) {

                if ( ((SEG_PRIVATE_DATA *)seg->private_data)->signature == DOS_SEG_MGR_PDATA_SIGNATURE ) {

                    if ( ((SEG_PRIVATE_DATA *)seg->private_data)->logical_disk ) {

                        return TRUE;

                    }

                }

            }

        }
    }

    return FALSE;
}



static inline boolean disk_move_pending( storage_object_t *object )
{
        LOGICALDISK        *ld=get_logical_disk(object);
        DISK_PRIVATE_DATA  *disk_pdata=NULL;

        if ( ld ) {

                disk_pdata = get_disk_private_data(ld);

                if (disk_pdata) {

                        if (disk_pdata->flags & DISK_HAS_MOVE_PENDING) {
                                return TRUE;
                        }

                }

        }

        return FALSE;
}


#endif
