/*----------------------------------------------------------------------
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: display.c
 *---------------------------------------------------------------------*/

/*----------------------------------------------------------------------
 * Change History:
 *
 * 9/2001  John Stiles   Initial version.
 *
 *---------------------------------------------------------------------*/


/* Identify this file. */
#define DISPLAY_C    1

/*----------------------------------------------------------------------
 * Necessary include files
 *---------------------------------------------------------------------*/
#include <stdlib.h>                    /* malloc, free                     */
#include <string.h>                    /* strcat, strcpy                   */
#include <stdio.h>                     /* printf                           */
#include <ctype.h>                     /* isprint                          */

#include "evmsn.h"

/*----------------------------------------------------------------------
 * Global Variables
 *---------------------------------------------------------------------*/


/*----------------------------------------------------------------------
 * There are No Private Constants
 * There are No Private Type Definitions
 * There are No Private Global Variables.
 *---------------------------------------------------------------------*/


/*----------------------------------------------------------------------
 * Local Function Prototypes
 *---------------------------------------------------------------------*/


/*----------------------------------------------------------------------
 * Public Functions
 *---------------------------------------------------------------------*/

/*----------------------------------------------------------------------
 *
 *   Function Name:  draw_frame
 *
 *   Descriptive Name:  Draws the startup frame for this app.
 *
 *   Input:  none
 *
 *   Output:  none
 *
 *   Error Handling:  none needed
 *
 *   Side Effects:  Displays the outside frame and app. title.  Screen
 *                   is cleared as part of this routine.
 *
 *   Notes:
 *
 *---------------------------------------------------------------------*/
void draw_frame( void )
{
  int size, xoffset;
  char stitle[] = _( "EVMS Administration Tool" );

  clear_screen();

  // Screen Title
  size = strlen( stitle );
  if ( size < GL_screen_cols )
    xoffset = ( GL_screen_cols - size ) / 2;  /* centers it on screen */
  else {
    xoffset = XPOS_TITLE;
  }
  print_string( xoffset, YPOS_TITLE, COLOR_TITLE, stitle );

  // Draw the outside box
  size = GL_screen_lines - ( YPOS_BOX1BOTTOM + YPOS_BOX1TOP - 1 );
  drawbleft( XPOS_BOX1LEFT, YPOS_BOX1TOP, size );
  drawbright( GL_screen_cols - XPOS_BOX1RIGHT, YPOS_BOX1TOP, size );
  size = GL_screen_cols - XPOS_BOX1RIGHT - 1;
  drawhline( XPOS_BOX1LEFT + 1, YPOS_BOX1TOP, size );
  drawhline( XPOS_BOX1LEFT + 1, GL_screen_lines - YPOS_BOX1BOTTOM, size );
}


/*----------------------------------------------------------------------
 *
 *   Function Name:  draw_fnkeys
 *
 *   Descriptive Name:  Draws the info for the function keys.
 *
 *   Input:  int == which key context is in use.
 *
 *   Output:  none
 *
 *   Error Handling:  none needed
 *
 *   Side Effects:  Displays the function key definitions.
 *
 *   Notes:  draw_keyloop is a private function, only used here.
 *
 *---------------------------------------------------------------------*/
void draw_keyloop( int key_count, char * keyname[] )
{
  int i, size, xoffset;

  // Draw the function key line
  xoffset = XPOS_FUNCKEYS;

  for ( i = 0; i < key_count ; i++ ) {
    size = strlen( keyname[i] );
    if ( xoffset + size > GL_screen_cols )
      break;

    print_string( xoffset, GL_screen_lines - YPOS_FUNCKEYS, COLOR_FKEYS, keyname[i] );
    xoffset += size + 2;
  }
}


void draw_fnkeys( int key_context )
{
  char *blanks;
  static char *defnames[] = {  _( "F1=help" ), _( "2=volumes" ), _( "4=select view" ), _( "5=commit changes" ), _( "10=quit" ) },
              *selnames[] = {  _( "F1=help" ), _( "2=next" ), _( "3=prev" ), _( "10=quit" ) };

  // first blank the line
  blanks = malloc( GL_screen_cols - 1 );
  if ( blanks ) {
    memset( blanks, ' ', GL_screen_cols - 2 );
    blanks[GL_screen_cols - 2] = '\0';
    print_string( XPOS_FUNCKEYS, GL_screen_lines - YPOS_FUNCKEYS, COLOR_DEFAULT, blanks );
    free( blanks );
  }
  else {
    log_error( "%s: could not malloc storage for blanks buffer.\n", __FUNCTION__ );
  }

  switch ( key_context ) {
    case KEYS_PREV :
      draw_keyloop( 4, selnames );
      break;
    default:
      draw_keyloop( 5, defnames );
      break;
  }
}


/*----------------------------------------------------------------------
 *
 *   Function Name:  draw_box2
 *
 *   Descriptive Name:  Draws main panel frame for this app.
 *
 *   Input:  title, char *, == NULL no title.
 *
 *   Output:  none
 *
 *   Error Handling:  none needed
 *
 *   Side Effects:  Displays the panel frame, with panel title.
 *
 *   Notes:
 *
 *---------------------------------------------------------------------*/
void draw_box2( char * title )
{
  int size, xoftitle, hold_pos = 0;
  char *hold;

  // Draw the box
  print_string( XPOS_BOX2LEFT + 1, YPOS_BOX2TOP, COLOR_TITLE, " " );

  size = CALC_SCREEN2_LINES + 3;
  drawbleft( XPOS_BOX2LEFT, YPOS_BOX2TOP, size );
  drawbright( GL_screen_cols - XPOS_BOX2RIGHT, YPOS_BOX2TOP, size );

  size = GL_screen_cols - XPOS_BOX2LEFT - XPOS_BOX2RIGHT - 1;
  drawhline( XPOS_BOX2LEFT + 1, GL_screen_lines - YPOS_BOX2BOTTOM, size );
  // Box Title
  if ( title ) {
    xoftitle = strlen( title );
    if ( xoftitle < size - 3 ) {
      xoftitle += XPOS_BOX2LEFT + 1;
      print_string( XPOS_BOX2LEFT + 1, YPOS_BOX2TOP, COLOR_TITLE, title );
      drawhline( xoftitle, YPOS_BOX2TOP, size - xoftitle + 2 );
    }
    else {
      hold_pos = size - 5;

      hold = malloc( hold_pos );
      if ( hold ) {
        strncpy( hold, title, hold_pos );
        hold[hold_pos - 1] = '\0';
        print_string( XPOS_BOX2LEFT + 1, YPOS_BOX2TOP, COLOR_TITLE, hold );
        free( hold );
      }
      else {
        log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
      }
    }
  }
  else
    drawhline( XPOS_BOX2LEFT + 1, YPOS_BOX2TOP, size );
}


/*----------------------------------------------------------------------
 *
 *   Function Name:  draw_box3
 *
 *   Descriptive Name:  Draws the secondary panel frame for this app.
 *
 *   Input:  title, char *, == NULL no title.
 *
 *   Output:  none
 *
 *   Error Handling:  none needed
 *
 *   Side Effects:  Displays the panel frame, with panel title.
 *
 *   Notes:
 *
 *---------------------------------------------------------------------*/
void draw_box3( char * title )
{
  int size, xoftitle, hold_pos = 0;
  char *hold;

  // Draw the box
  print_string( XPOS_BOX3LEFT + 1, YPOS_BOX3TOP, COLOR_TITLE, " " );

  size = CALC_SCREEN3_LINES + 3;
  drawbleft( XPOS_BOX3LEFT, YPOS_BOX3TOP, size );
  drawbright( GL_screen_cols - XPOS_BOX3RIGHT, YPOS_BOX3TOP, size );

  size = GL_screen_cols - XPOS_BOX3LEFT - XPOS_BOX3RIGHT - 1;
  drawhline( XPOS_BOX3LEFT + 1, GL_screen_lines - YPOS_BOX3BOTTOM, size );
  // Box Title
  if ( title ) {
    xoftitle = strlen( title );
    if ( xoftitle < size - 3 ) {
      xoftitle += XPOS_BOX3LEFT + 1;
      print_string( XPOS_BOX3LEFT + 1, YPOS_BOX3TOP, COLOR_TITLE, title );
      drawhline( xoftitle, YPOS_BOX3TOP, size - xoftitle + XPOS_BOX3LEFT + 1 );
    }
    else {
      hold_pos = size - 5;

      hold = malloc( hold_pos );
      if ( hold ) {
        strncpy( hold, title, hold_pos );
        hold[hold_pos - 1] = '\0';
        print_string( XPOS_BOX3LEFT + 1, YPOS_BOX3TOP, COLOR_TITLE, hold );
        free( hold );
      }
      else {
        log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
      }
    }
  }
  else
    drawhline( XPOS_BOX3LEFT + 1, YPOS_BOX3TOP, size );
}


/*----------------------------------------------------------------------
 *
 *   Function Name:  draw_box4
 *
 *   Descriptive Name:  Draws the tertiary panel frame for this app.
 *
 *   Input:  title, char *, == NULL no title.
 *
 *   Output:  none
 *
 *   Error Handling:  none needed
 *
 *   Side Effects:  Displays the panel frame, with panel title.
 *
 *   Notes:
 *
 *---------------------------------------------------------------------*/
void draw_box4( char * title )
{
  int size, xoftitle, hold_pos = 0;
  char *hold;

  // Draw the box
  print_string( XPOS_BOX4LEFT + 1, YPOS_BOX4TOP, COLOR_TITLE, " " );

  size = CALC_SCREEN4_LINES + 3;
  drawbleft( XPOS_BOX4LEFT, YPOS_BOX4TOP, size );
  drawbright( GL_screen_cols - XPOS_BOX4RIGHT, YPOS_BOX4TOP, size );

  size = GL_screen_cols - XPOS_BOX4LEFT - XPOS_BOX4RIGHT - 1;
  drawhline( XPOS_BOX4LEFT + 1, GL_screen_lines - YPOS_BOX4BOTTOM, size );
  // Box Title
  if ( title ) {
    xoftitle = strlen( title );
    if ( xoftitle < size - 3 ) {
      xoftitle += XPOS_BOX4LEFT + 1;
      print_string( XPOS_BOX4LEFT + 1, YPOS_BOX4TOP, COLOR_TITLE, title );
      drawhline( xoftitle, YPOS_BOX4TOP, size - xoftitle + XPOS_BOX4LEFT + 1 );
    }
    else {
      hold_pos = size - 5;

      hold = malloc( hold_pos );
      if ( hold ) {
        strncpy( hold, title, hold_pos );
        hold[hold_pos - 1] = '\0';
        print_string( XPOS_BOX4LEFT + 1, YPOS_BOX4TOP, COLOR_TITLE, hold );
        free( hold );
      }
      else {
        log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
      }
    }
  }
  else
    drawhline( XPOS_BOX4LEFT + 1, YPOS_BOX4TOP, size );
}


/*----------------------------------------------------------------------
 *
 *   Function Name:  draw_help_panel
 *
 *   Descriptive Name:  Draws help panel based on screen context.
 *
 *   Input:  title, char *, == NULL no title.
 *
 *   Output:  none
 *
 *   Error Handling:  none needed
 *
 *   Side Effects:  Displays the help panel.
 *
 *   Notes:
 *
 *---------------------------------------------------------------------*/
void draw_help_panel( int screen_context )
{
  draw_status_line( "Help function not yet implemented, press any key to continue." );
  get_inputc();
}


/*----------------------------------------------------------------------
 *
 *   Function Name:  draw_status_line
 *
 *   Descriptive Name:  Draws the status line for this app.
 *
 *   Input:  status_text == NULL draw Ready message,
 *                       != NULL draw text string as message
 *
 *   Output:  none
 *
 *   Error Handling:  none needed
 *
 *   Side Effects:  Displays the status line.
 *
 *   Notes:  Status line may be truncated due to screen size, recommend
 *            using text strings < 80 characters.
 *
 *---------------------------------------------------------------------*/
void draw_status_line( char * status_text )
{
  char * draw_text;
  char hold = '\0', readymsg[] = _( "Ready." );
  int  hold_pos = 0;

  // first blank the line
  draw_text = malloc( GL_screen_cols - 1 );
  if ( draw_text ) {
    memset( draw_text, ' ', GL_screen_cols - 2 );
    draw_text[GL_screen_cols - 2] = '\0';
    print_string( XPOS_STATUS, GL_screen_lines - YPOS_STATUS, COLOR_DEFAULT, draw_text );
    free( draw_text );
  }
  else {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
  }

  if ( status_text == NULL )
    draw_text = readymsg;
  else
    draw_text = status_text;

  // Check if string will fit within the screen
  if ( strlen( draw_text ) > GL_screen_cols ) {
    hold_pos = GL_screen_cols - 1;
    hold = status_text[hold_pos];
    status_text[hold_pos] = '\0';
  }

  print_string( XPOS_STATUS, GL_screen_lines - YPOS_STATUS, COLOR_DEFAULT, draw_text );

  if ( hold_pos )
    status_text[hold_pos] = hold;
}


/*----------------------------------------------------------------------
 *
 *   Function Name:  draw_sector_size
 *
 *   Descriptive Name:  convert a sector_count_t to a printable string.
 *
 *   Input:  sector_count_t size  - the size to convert.
 *           char * output_string - where to put the string.
 *
 *   Output:  the character string
 *
 *   Error Handling:  none
 *
 *   Side Effects:  the character string
 *
 *   Notes:  data format:  double 6.1 xB
 *
 *---------------------------------------------------------------------*/
void draw_sector_size( sector_count_t size, char * output_string )
{
  double fscaled = size;

  if ( size < 5 ) {
    fscaled *= 512.0;
    sprintf( output_string, "%6.1f  B", fscaled );
  }
  else  if ( size < 4096 ) {
      fscaled /= 2.0;
      sprintf( output_string, "%6.1f KB", fscaled );
    }
    else  if ( size < 4194304 ) {
        fscaled /= 2048.0;
        sprintf( output_string, "%6.1f MB", fscaled );
      }
      else  if ( size < 4294967296 ) {
          fscaled /= 2097152.0;
          sprintf( output_string, "%6.1f GB", fscaled );
        }
        else  if ( size < 4398046511104 ) {
            fscaled /= 2097152.0 * 1024.0;
            sprintf( output_string, "%6.1f TB", fscaled );
          }
          else  {
              fscaled /= 2097152.0 * 1024.0 * 1024.0;
              sprintf( output_string, "%6.1f PB", fscaled );
            }
}


/*----------------------------------------------------------------------
 *
 *   Function Name:  draw_value_type
 *                   draw_value_units
 *
 *   Descriptive Name:  convert the value fields to a printable string.
 *
 *   Input:
 *      draw_value_type( value_type_t   Value_Type == type of field to print.
 *                       value_format_t Suggested_Format == print format desired.
 *                       value_t *      Value  == pointer to the actual value.
 *                       int    buffer_length == size of buffer.
 *                       char * output_string == where to put the string.
 *
 *     draw_value_units( value_unit_t Units   == units field to convert.
 *                       char * output_string == where to put the string.
 *
 *   Output:  the character string
 *
 *   Error Handling:  none
 *
 *   Side Effects:  the character string
 *
 *   Notes:  data format varies with the type
 *
 *---------------------------------------------------------------------*/
void draw_value_type( value_type_t   Value_Type,
                      value_format_t Suggested_Format,
                      value_t *      Value,
                      int       buffer_length,
                      char    * output_string )
{
  switch ( Value_Type ) {

    case EVMS_Type_String :
      if ( Value->s ) {
        if ( strlen( Value->s ) > buffer_length ) {
          strncpy( output_string, Value->s, buffer_length - 2 );
          output_string[buffer_length - 2] = LINE_CONTINUED_CHAR ;
          output_string[buffer_length - 1] = '\0';
        }
        else
          strcpy( output_string, Value->s );
      }
      else {
        output_string[0] = '\0';
      }
      break;

    case EVMS_Type_Boolean :
      if ( Value->bool )
        sprintf( output_string, "True" );
      else
        sprintf( output_string, "False" );
      break;

    case EVMS_Type_Char :
      if ( Suggested_Format == EVMS_Format_Hex )
        sprintf( output_string, "0x%x", Value->c );
      else
        sprintf( output_string, "%c", Value->c );
      break;

    case EVMS_Type_Unsigned_Char :
      if ( Suggested_Format == EVMS_Format_Hex )
        sprintf( output_string, "0x%x", Value->uc );
      else
        sprintf( output_string, "%c", Value->uc );
      break;

    case EVMS_Type_Real32 :
      if( Value->r32 < 5000 )
        sprintf( output_string, "%f", Value->r32 );
      else
        sprintf( output_string, "%e", Value->r32 );
      break;

    case EVMS_Type_Real64 :
      if( Value->r64 < 5000 )
        sprintf( output_string, "%f", Value->r64 );
      else
        sprintf( output_string, "%e", Value->r64 );
      break;

    case EVMS_Type_Int :
      if ( Suggested_Format == EVMS_Format_Hex )
        sprintf( output_string, "0x%x", Value->i );
      else
        sprintf( output_string, "%d", Value->i );
      break;

    case EVMS_Type_Int8 :
      if ( Suggested_Format == EVMS_Format_Hex )
        sprintf( output_string, "0x%x", Value->i8 );
      else
        sprintf( output_string, "%d", Value->i8 );
      break;

    case EVMS_Type_Int16 :
      if ( Suggested_Format == EVMS_Format_Hex )
        sprintf( output_string, "0x%x", Value->i16 );
      else
        sprintf( output_string, "%d", Value->i16 );
      break;

    case EVMS_Type_Int32 :
      if ( Suggested_Format == EVMS_Format_Hex )
        sprintf( output_string, "0x%x", Value->i32 );
      else
        sprintf( output_string, "%d", Value->i32 );
      break;

    case EVMS_Type_Int64 :
//      if ( Suggested_Format == EVMS_Format_Hex )
//      printf( "0x%x", Value->i64 );
//    else
        sprintf( output_string, "%Ld", Value->i64 );
      break;

    case EVMS_Type_Unsigned_Int :
      if ( Suggested_Format == EVMS_Format_Hex )
        sprintf( output_string, "0x%x", Value->ui );
      else
        sprintf( output_string, "%d", Value->ui );
      break;

    case EVMS_Type_Unsigned_Int8 :
      if ( Suggested_Format == EVMS_Format_Hex )
        sprintf( output_string, "0x%x", Value->ui8 );
      else
        sprintf( output_string, "%d", Value->ui8 );
      break;

    case EVMS_Type_Unsigned_Int16 :
      if ( Suggested_Format == EVMS_Format_Hex )
        sprintf( output_string, "0x%x", Value->ui16 );
      else
        sprintf( output_string, "%d", Value->ui16 );
      break;

    case EVMS_Type_Unsigned_Int32 :
      if ( Suggested_Format == EVMS_Format_Hex )
        sprintf( output_string, "0x%x", Value->ui32 );
      else
        sprintf( output_string, "%d", Value->ui32 );
      break;

    case EVMS_Type_Unsigned_Int64 :
//      if ( Suggested_Format == EVMS_Format_Hex )
//      printf( "0x%x", Value->ui64 );
//    else
        sprintf( output_string, "%Ld", Value->ui64 );
      break;

    default:
      log_error( "%s: unknown data type for field value\n", __FUNCTION__ );
      break;
  }
}


int draw_value_units( value_unit_t Units, char * output_string )
{
  int rc = 0;

  switch ( Units ) {

    case EVMS_Unit_None:
      rc = 1;
      break;

    case EVMS_Unit_Disks:
      sprintf( output_string, _( "disks" ));
      break;

    case EVMS_Unit_Sectors:
      sprintf( output_string, _( "sectors" ));
      break;

    case EVMS_Unit_Segments:
      sprintf( output_string, _( "segments" ));
      break;

    case EVMS_Unit_Regions:
      sprintf( output_string, _( "regions" ));
      break;

    case EVMS_Unit_Percent:
      sprintf( output_string, _( "percent" ));
      break;

    case EVMS_Unit_Milliseconds:
      sprintf( output_string, _( "milliseconds" ));
      break;

    case EVMS_Unit_Microseconds:
      sprintf( output_string, _( "microseconds" ));
      break;

    case EVMS_Unit_Bytes:
      sprintf( output_string, _( "bytes" ));
      break;

    case EVMS_Unit_Kilobytes:
      sprintf( output_string, _( "KB" ));
      break;

    case EVMS_Unit_Megabytes:
      sprintf( output_string, _( "MB" ));
      break;

    case EVMS_Unit_Gigabytes:
      sprintf( output_string, _( "GB" ));
      break;

    case EVMS_Unit_Terabytes:
      sprintf( output_string, _( "TB" ));
      break;

    case EVMS_Unit_Petabytes :
      sprintf( output_string, _( "PB" ));
      break;

    default:
      rc = 1;
      break;
  }
  return rc;
}


int comp_value_type( value_type_t Value_Type, value_t *Value1, value_t *Value2 )
{
  int rc = 0;

  switch ( Value_Type ) {

    case EVMS_Type_String :
      if ( Value1->s && Value2->s ) {
        if ( strcmp( Value1->s, Value2->s ))
          rc = 1;
      }
      else {
        log_error( "%s: at least one string missing for compare\n", __FUNCTION__ );
        rc = 1;
      }
      break;

    case EVMS_Type_Boolean :
      if ( Value1->bool != Value2->bool )
        rc = 1;
      break;

    case EVMS_Type_Char :
      if ( Value1->c > Value2->c )
        rc = 1;
      else if ( Value1->c < Value2->c )
        rc = -1;
      break;

    case EVMS_Type_Unsigned_Char :
      if ( Value1->uc > Value2->uc )
        rc = 1;
      else if ( Value1->uc < Value2->uc )
        rc = -1;
      break;

    case EVMS_Type_Real32 :
      if ( Value1->r32 > Value2->r32 )
        rc = 1;
      else if ( Value1->r32 < Value2->r32 )
        rc = -1;
      break;

    case EVMS_Type_Real64 :
      if ( Value1->r64 > Value2->r64 )
        rc = 1;
      else if ( Value1->r64 < Value2->r64 )
        rc = -1;
      break;

    case EVMS_Type_Int :
      if ( Value1->i > Value2->i )
        rc = 1;
      else if ( Value1->i < Value2->i )
        rc = -1;
      break;

    case EVMS_Type_Int8 :
      if ( Value1->i8 > Value2->i8 )
        rc = 1;
      else if ( Value1->i8 < Value2->i8 )
        rc = -1;
      break;

    case EVMS_Type_Int16 :
      if ( Value1->i16 > Value2->i16 )
        rc = 1;
      else if ( Value1->i16 < Value2->i16 )
        rc = -1;
      break;

    case EVMS_Type_Int32 :
      if ( Value1->i32 > Value2->i32 )
        rc = 1;
      else if ( Value1->i32 < Value2->i32 )
        rc = -1;
      break;

    case EVMS_Type_Int64 :
      if ( Value1->i64 > Value2->i64 )
        rc = 1;
      else if ( Value1->i64 < Value2->i64 )
        rc = -1;
      break;

    case EVMS_Type_Unsigned_Int :
      if ( Value1->ui > Value2->ui )
        rc = 1;
      else if ( Value1->ui < Value2->ui )
        rc = -1;
      break;

    case EVMS_Type_Unsigned_Int8 :
      if ( Value1->ui8 > Value2->ui8 )
        rc = 1;
      else if ( Value1->ui8 < Value2->ui8 )
        rc = -1;
      break;

    case EVMS_Type_Unsigned_Int16 :
      if ( Value1->ui16 > Value2->ui16 )
        rc = 1;
      else if ( Value1->ui16 < Value2->ui16 )
        rc = -1;
      break;

    case EVMS_Type_Unsigned_Int32 :
      if ( Value1->ui32 > Value2->ui32 )
        rc = 1;
      else if ( Value1->ui32 < Value2->ui32 )
        rc = -1;
      break;

    case EVMS_Type_Unsigned_Int64 :
      if ( Value1->ui64 > Value2->ui64 )
        rc = 1;
      else if ( Value1->ui64 < Value2->ui64 )
        rc = -1;
      break;

    default:
      log_error( "%s: unknown data type for field value\n", __FUNCTION__ );
      break;
  }
  return rc;
}


int copy_value_string( value_type_t Value_Type, value_t *OutValue, char *InString )
{
  int rc = 0;

  if ( InString ) {
    if ( Value_Type == EVMS_Type_String ) {
      OutValue->s = malloc( strlen( InString ));
      if ( OutValue->s )
        strcpy( OutValue->s, InString );
      else {
        log_error( "%s: could not malloc the string to copy in to\n", __FUNCTION__ );
        rc = 1;
      }
    }
    else  if ( Value_Type == EVMS_Type_Char )
      OutValue->c = InString[0];
    else  if ( Value_Type == EVMS_Type_Unsigned_Char )
      OutValue->uc = InString[0];
  }
  else {
    log_error( "%s: there is no string to copy from\n", __FUNCTION__ );
    rc = 1;
  }
  return rc;
}


int copy_value_type( value_type_t Value_Type, value_t *OutValue, value_t *InValue )
{
  int rc = 0;

  if ( Value_Type == EVMS_Type_String ) {
    if ( InValue->s ) {
      OutValue->s = malloc( strlen( InValue->s ));
      if ( OutValue->s )
        strcpy( OutValue->s, InValue->s );
      else {
        log_error( "%s: could not malloc the string to copy in to\n", __FUNCTION__ );
        rc = 1;
      }
    }
    else {
      log_error( "%s: there is no string to copy from\n", __FUNCTION__ );
      rc = 1;
    }
  }
  else
    memcpy( OutValue, InValue, sizeof( value_t ));

  return rc;
}


void free_value_type( value_type_t Value_Type, value_t *Value_to_Free )
{
  if ( Value_Type == EVMS_Type_String ) {
    if ( Value_to_Free->s )
      free( Value_to_Free->s );
    else
      log_error( "%s: there is no string to free\n", __FUNCTION__ );
  }
}


sector_count_t convert_to_sector_count( value_type_t Value_Type, value_t *Value )
{
  sector_count_t new_sector_value = 0;

  switch ( Value_Type ) {

    case EVMS_Type_Boolean :
      if ( Value->bool )
        new_sector_value = 1;
      else
        new_sector_value = 0;
      break;

    case EVMS_Type_Char :
      new_sector_value = Value->c;
      break;

    case EVMS_Type_Unsigned_Char :
      new_sector_value = Value->uc;
      break;

    case EVMS_Type_Real32 :
      new_sector_value = Value->r32;
      break;

    case EVMS_Type_Real64 :
      new_sector_value = Value->r64;
      break;

    case EVMS_Type_Int :
      new_sector_value = Value->i;
      break;

    case EVMS_Type_Int8 :
      new_sector_value = Value->i8;
      break;

    case EVMS_Type_Int16 :
      new_sector_value = Value->i16;
      break;

    case EVMS_Type_Int32 :
      new_sector_value = Value->i32;
      break;

    case EVMS_Type_Int64 :
      new_sector_value = Value->i64;
      break;

    case EVMS_Type_Unsigned_Int :
      new_sector_value = Value->ui;
      break;

    case EVMS_Type_Unsigned_Int8 :
      new_sector_value = Value->ui8;
      break;

    case EVMS_Type_Unsigned_Int16 :
      new_sector_value = Value->ui16;
      break;

    case EVMS_Type_Unsigned_Int32 :
      new_sector_value = Value->ui32;
      break;

    case EVMS_Type_Unsigned_Int64 :
      new_sector_value = Value->ui64;
      break;

    default:
      log_error( "%s: cannot convert the field value type given\n", __FUNCTION__ );
      break;
  }

  return new_sector_value;
}


int convert_from_sector_count( value_type_t Value_Type, sector_count_t sector_value, value_t *Value )
{
  int rc = 0;

  switch ( Value_Type ) {

    case EVMS_Type_Boolean :
      if ( sector_value )
        Value->bool = 1;
      else
        Value->bool = 0;
      break;

    case EVMS_Type_Char :
    case EVMS_Type_Unsigned_Char :
    case EVMS_Type_Int8 :
    case EVMS_Type_Unsigned_Int8 :
      Value->ui8 = ( u_int8_t )sector_value;
      break;

    case EVMS_Type_Real32 :
      Value->r32 = sector_value * 1.0;
      break;

    case EVMS_Type_Real64 :
      Value->r64 = sector_value * 1.0;
      break;

    case EVMS_Type_Int :
      Value->i = ( int )sector_value;
      break;

    case EVMS_Type_Int16 :
      Value->i16 = ( int16_t )sector_value;
      break;

    case EVMS_Type_Int32 :
      Value->i32 = ( int32_t )sector_value;
      break;

    case EVMS_Type_Int64 :
      Value->i64 = ( int64_t )sector_value;
      break;

    case EVMS_Type_Unsigned_Int :
      Value->ui = ( u_int )sector_value;
      break;

    case EVMS_Type_Unsigned_Int16 :
      Value->ui16 = ( u_int16_t )sector_value;
      break;

    case EVMS_Type_Unsigned_Int32 :
      Value->ui32 = ( u_int32_t )sector_value;
      break;

    case EVMS_Type_Unsigned_Int64 :
      Value->ui64 = ( u_int64_t )sector_value;
      break;

    default:
      log_error( "%s: cannot convert the field value type given\n", __FUNCTION__ );
      rc = 1;
      break;
  }
  return rc;
}


/*----------------------------------------------------------------------
 *
 *   Function Name:  text_entry
 *                   number_entry
 *
 *   Descriptive Name:  allow manupulation of a text entry field on screen.
 *
 *   Input:  int argc    - The number of parameters on the command line.
 *           char * argv - An array of pointers to the text of each
 *                          parameter found on the command line.
 *
 *   Output: If Success :  0, there is a text string in the char buffer
 *
 *           If Failure :  !0 if the user cancelled ( by pressing ESC key )
 *
 *   Error Handling:  If an error occurs, all memory allocated by this
 *                     program is freed, and an error message is output
 *                     to the user.
 *
 *   Side Effects:  Displays a window to enter textons, and/or volumes
 *                   in the system may be altered.  Nodes may be
 *                   created or removed from the /dev tree.
 *
 *   Notes:
 *
 *---------------------------------------------------------------------*/
int text_entry( int x, int y, int length, char * input_string )
{
  int i, cursor_pos = 0, insert_mode = 0, keyval = 0;

  // Check if dimensions are within the screen
  if ( x + 20 < GL_screen_cols && y < GL_screen_lines ) {

    cursor_pos = 0;
    do  {
      print_string( x, y, COLOR_DEFAULT, input_string );
      move_cursor( x + cursor_pos, y );
      keyval = get_inputc();

      switch ( keyval ) {

        case KEY_ESC :
        case KEY_ENTER :
          break;

        case KEY_BACKSPACE :
          if ( cursor_pos > 0 ) {
            cursor_pos--;

            for ( i = cursor_pos; i < length - 1; i++ )
              input_string[i] = input_string[i + 1];

            input_string[length - 1] = ' ';
          }
          break;

        case KEY_CURSORLEFT :
          if ( cursor_pos > 0 )
            cursor_pos--;
          break;

        case KEY_CURSORRIGHT :
          if ( cursor_pos < length - 1 )
            cursor_pos++;
          break;

        // case KEY_END :
        //   cursor_pos = strlen( input_string );
        //   break;

        // case KEY_HOME :
        //   cursor_pos = 0;
        //   break;

        case KEY_DELETE :
          for ( i = cursor_pos; i < length - 1; i++ )
            input_string[i] = input_string[i + 1];

          input_string[length - 1] = ' ';
          break;

        case KEY_INSERT :
          if ( insert_mode == 0 )
            insert_mode = 1;
          else
            insert_mode = 0;
          break;

        default:
          if ( keyval != '\'' && isprint( keyval )) {
            if ( insert_mode == 1 ) {
              for ( i = length - 1; i > cursor_pos; i-- )
                input_string[i] = input_string[i - 1];
            }

            input_string[cursor_pos] = keyval;
            if ( cursor_pos < length - 1 )
              cursor_pos++;
          }
          break;
      }

    } while ( keyval != KEY_ESC && keyval != KEY_ENTER );

    if ( keyval == KEY_ESC )
      return 2;

    else {
      cursor_pos = length - 1;
      do {
        if ( input_string[cursor_pos] != ' ' && input_string[cursor_pos] != '\0' )
          break;

        cursor_pos--;
      } while ( cursor_pos > 0 );

      input_string[cursor_pos + 1] = '\0';
    }
  }
  else {
    log_error( "%s: string %s is off the screen:  x == %i (max == %i), y == %i (max == %i).\n", __FUNCTION__, input_string, x, GL_screen_cols, y, GL_screen_lines );
    return 1;
  }

  return 0;
}


int number_entry( int x, int y, int length, char * input_string )
{
  int i, cursor_pos = 0, insert_mode = 0, keyval = 0;

  // Check if dimensions are within the screen
  if ( x + 15 < GL_screen_cols && y < GL_screen_lines ) {

    cursor_pos = 0;
    do  {
      print_string( x, y, COLOR_DEFAULT, input_string );
      move_cursor( x + cursor_pos, y );
      keyval = get_inputc();

      switch ( keyval ) {

        case KEY_ESC :
        case KEY_ENTER :
          break;

        case KEY_BACKSPACE :
          if ( cursor_pos > 0 ) {
            cursor_pos--;

            for ( i = cursor_pos; i < length - 1; i++ )
              input_string[i] = input_string[i + 1];

            input_string[length - 1] = ' ';
          }
          break;

        case KEY_CURSORLEFT :
          if ( cursor_pos > 0 )
            cursor_pos--;
          break;

        case KEY_CURSORRIGHT :
          if ( cursor_pos < length - 1 )
            cursor_pos++;
          break;

        // case KEY_END :
        //   cursor_pos = strlen( input_string );
        //   break;

        // case KEY_HOME :
        //   cursor_pos = 0;
        //   break;

        case KEY_DELETE :
          for ( i = cursor_pos; i < length - 1; i++ )
            input_string[i] = input_string[i + 1];

          input_string[length - 1] = ' ';
          break;

        case KEY_INSERT :
          if ( insert_mode == 0 )
            insert_mode = 1;
          else
            insert_mode = 0;
          break;

        default:
          if ( isdigit( keyval )) {
            if ( insert_mode == 1 ) {
              for ( i = length - 1; i > cursor_pos; i-- )
                input_string[i] = input_string[i - 1];
            }

            input_string[cursor_pos] = keyval;
            if ( cursor_pos < length - 1 )
              cursor_pos++;
          }
          break;
      }

    } while ( keyval != KEY_ESC && keyval != KEY_ENTER );

    if ( keyval == KEY_ESC )
      return 2;

    else {
      cursor_pos = length - 1;
      do {
        if ( input_string[cursor_pos] != ' ' && input_string[cursor_pos] != '\0' )
          break;

        cursor_pos--;
      } while ( cursor_pos > 0 );

      input_string[cursor_pos + 1] = '\0';
    }
  }
  else {
    log_error( "%s: string %s is off the screen:  x == %i (max == %i), y == %i (max == %i).\n", __FUNCTION__, input_string, x, GL_screen_cols, y, GL_screen_lines );
    return 1;
  }

  return 0;
}

