/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LvmUtils
 * File: evms_vgscan.c
 *
 *	Emulates LVM's 'vgscan' utility using the EVMS Engine. All options
 *	and several status messages are based on the original vgscan command
 *	from Heinz Mauelshagen and Sistina Software (www.sistina.com).
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <libgen.h>
#include <frontend.h>

typedef struct cmd_options_s {
	int createdevs;
	int debug;
	int help;
	int verbose;
	int version;
} cmd_options_t;

static char * cmd = NULL;
static cmd_options_t opts;

#include "helpers/get_lvmregmgr.c"
#include "helpers/open_engine.c"


static int showheader( void )
{
	// VERSION and DATE are defined in the top-level make.rules
	printf("Enterprise Volume Management System\n");
	printf("International Business Machines  %s\n", DATE);
	printf("LVM Emulation Utilities %s\n\n", VERSION);
	return 0;
}


static int showhelp( void )
{
	showheader();
	printf("\n");
	printf("%s -- Volume Group Scan\n\n", cmd);
	printf("Synopsis:\n");
	printf("---------\n\n");
	printf("%s\n", cmd);
	printf("\t[-c/--createdevs]\n");
	printf("\t[-d/--debug]\n");
	printf("\t[-h/-?/--help]\n");
	printf("\t[-v/--verbose]\n");
	printf("\t[-V/--version]\n");
	return 0;
}


static int vgscan_end(	handle_array_t		* a,
			handle_object_info_t	* b )
{
	if (a) evms_free(a);
	if (b) evms_free(b);
	evms_close_engine();
	return 0;
}


static int parse_options( int		argc,
			char		** argv )
{
	int		c;
	char		* short_opts = "cdh?vV";
	struct option	long_opts[] = {
				{ "createdevs",	no_argument, NULL, 'c'},
				{ "debug",	no_argument, NULL, 'd'},
				{ "help",	no_argument, NULL, 'h'},
				{ "verbose",	no_argument, NULL, 'v'},
				{ "version",	no_argument, NULL, 'V'},
				{ NULL, 0, NULL, 0} };

	while ( (c = getopt_long(argc, argv, short_opts,
				long_opts, NULL)) != EOF ) {
		switch (c) {
		case 'c':
			opts.createdevs++;
			break;
		case 'd':
			opts.debug++;
			opts.verbose++;
			break;
		case 'h':
		case '?':
			opts.help++;
			break;
		case 'v':
			opts.verbose++;
			break;
		case 'V':
			opts.version++;
			break;
		default:
			printf("%s -- unrecognized option \"%c\"\n\n", cmd, c);
			return EINVAL;
		}
	}

	return 0;
}


int main( int argc, char * argv[] )
{
	plugin_handle_t		lvmregmgr;
	handle_array_t		* container_array = NULL;
	handle_object_info_t	* container_info = NULL;
	engine_mode_t		engine_mode = ENGINE_READONLY;
	int			num_vg = 0;
	int			log_level = DEFAULT;
	int			rc,i;

	memset(&opts, 0, sizeof(cmd_options_t));
	cmd = basename(argv[0]);

	// Get the command line options.
	rc = parse_options(argc, argv);
	if (rc) {
		showhelp();
		return rc;
	}
	if ( opts.help ) {
		showhelp();
		return 0;
	}
	if ( opts.version ) {
		showheader();
		return 0;
	}
	if ( opts.verbose ) {
		log_level = DEBUG;
	}
	if ( opts.debug ) {
		log_level = ENTRY_EXIT;
	}
	if ( opts.createdevs ) {
		engine_mode = ENGINE_READWRITE;
	}

	printf("%s -- reading all physical volumes (this may take a while...)\n", cmd);    

	// Open the engine.
	rc = open_engine(engine_mode, log_level);
	if (rc) {
		return rc;
	}

	// Get the handle for the LVM region manager.
	rc = get_lvmregmgr(&lvmregmgr);
	if (rc) {
		vgscan_end(container_array, container_info);
		return rc;
	}

	// Get the list of containers for the LVM plugin.
	if ( opts.verbose ) {
		printf("%s -- Getting list of LVM containers\n", cmd);
	}
	rc = evms_get_container_list(lvmregmgr, &container_array);
	if (rc) {
		printf("%s -- Error getting container list (%d)\n", cmd, rc);
		vgscan_end(container_array, container_info);
		return rc;
	}


	for ( i = 0; i < container_array->count; i++ ) {
		if ( opts.verbose ) {
			printf("%s -- Getting info for container handle %u\n", cmd, container_array->handle[i]);
		}
		rc = evms_get_info(container_array->handle[i], &container_info);
		if (rc) {
			printf("%s -- Error getting info for container handle %u (%d)\n", cmd, container_array->handle[i], rc);
			vgscan_end(container_array, container_info);
			return rc;
		}

		num_vg++;
		printf("%s -- Found volume group \"%s\"\n", cmd, container_info->info.container.name);
		evms_free(container_info);
		container_info = NULL;
	}

	if ( num_vg == 0 ) {
		printf("%s -- No volume groups found.\n", cmd);
	}

	if ( opts.createdevs ) {
		// Create all necessary device files by performing a commit.
		rc = evms_commit_changes(NULL);
		if (rc) {
			printf("%s -- Error committing changes to disk (%d)\n", cmd, rc);
		}
	}

	vgscan_end(container_array, container_info);
	return 0; 
}

