/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LvmUtils
 * File: evms_lvscan.c
 *
 *	Emulates LVM's 'lvscan' utility using the EVMS Engine. All options
 *	and several status messages are based on the original lvscan command
 *	from Heinz Mauelshagen and Sistina Software (www.sistina.com).
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <libgen.h>
#include <frontend.h>

typedef struct cmd_options_s {
	int blockdevice;
	int debug;
	int help;
	int verbose;
	int version;
} cmd_options_t;

static char * cmd = NULL;
static cmd_options_t opts;

#include "helpers/get_lvmregmgr.c"
#include "helpers/open_engine.c"
#include "helpers/print_size.c"


static int showheader( void )
{
	// VERSION and DATE are defined in the top-level make.rules
	printf("Enterprise Volume Management System\n");
	printf("International Business Machines  %s\n", DATE);
	printf("LVM Emulation Utilities %s\n\n", VERSION);
	return 0;
}


static int showhelp( void )
{
	showheader();
	printf("\n");
	printf("%s - scan for logical volumes\n\n", cmd);
	printf("%s [-b|--blockdevice] [-d|--debug] [-D|--disk]\n", cmd);
	printf("\t[-h|--help] [-v|--verbose] [-V|--version]\n");
	return 0;
}


static int lvscan_end(	handle_array_t		* a,
			handle_object_info_t	* b,
			handle_object_info_t	* c,
			handle_object_info_t	* d )
{
	if (a) evms_free(a);
	if (b) evms_free(b);
	if (c) evms_free(c);
	if (d) evms_free(d);
	evms_close_engine();
	return 0;
}


static int parse_options( int		argc,
			char		** argv )
{
	int		c;
	char		* short_opts = "bdDh?vV";
	struct option	long_opts[] = {
				{ "blockdevice",no_argument, NULL, 'b'},
				{ "debug",	no_argument, NULL, 'd'},
				{ "disk",	no_argument, NULL, 'D'},
				{ "help",	no_argument, NULL, 'h'},
				{ "verbose",	no_argument, NULL, 'v'},
				{ "version",	no_argument, NULL, 'V'},
				{ NULL, 0, NULL, 0} };

	while ( (c = getopt_long(argc, argv, short_opts,
				long_opts, NULL)) != EOF ) {
		switch (c) {
		case 'b':
			opts.blockdevice++;
			break;
		case 'd':
			opts.debug++;
			opts.verbose++;
			break;
		case 'D':
			// -D is ignored by EVMS
			break;
		case 'h':
		case '?':
			opts.help++;
			break;
		case 'v':
			opts.verbose++;
			break;
		case 'V':
			opts.version++;
			break;
		default:
			printf("%s -- unrecognized option \"%c\"\n\n", cmd, c);
			return EINVAL;
		}
	}

	return 0;
}


int main( int argc, char * argv[] )
{
	object_handle_t			lvmregmgr;
	handle_array_t			* container_array = NULL;
	handle_object_info_t		* container_info = NULL;
	handle_object_info_t		* region_info = NULL;
	handle_object_info_t		* lv_info = NULL;
	char				* str1;
	long				current_size = 0;
	long				total_size = 0;
	int				current_minor = 0;
	int				num_vg = 0;
	int				num_active_lv = 0;
	int				num_inactive_lv = 0;
	int				log_level = DEFAULT;
	int				i, j, rc;

	memset(&opts, 0, sizeof(cmd_options_t));
	cmd = basename(argv[0]);

	// Get the command line options.
	rc = parse_options(argc, argv);
	if (rc) {
		showhelp();
		return rc;
	}
	if ( opts.help ) {
		showhelp();
		return 0;
	}
	if ( opts.version ) {
		showheader();
		return 0;
	}
	if ( opts.verbose ) {
		log_level = DEBUG;
	}
	if ( opts.debug ) {
		log_level = ENTRY_EXIT;
	}

	// Open the engine.
	rc = open_engine(ENGINE_READONLY, log_level);
	if (rc) {
		return rc;
	}

	// Get the handle for the LVM region manager.
	rc = get_lvmregmgr(&lvmregmgr);
	if (rc) {
		evms_close_engine();
		return rc;
	}

	// Get the list of containers for the LVM plugin.
	if ( opts.verbose ) {
		printf("%s -- Getting list of LVM VG's\n", cmd);
	}
	rc = evms_get_container_list(lvmregmgr, &container_array);
	if (rc) {
		printf("%s -- Error getting container list (%d)\n", cmd, rc);
		lvscan_end(container_array, container_info, region_info, lv_info);
		return rc;
	}

	// Examine each LVM container.
	for ( i = 0; i < container_array->count; i++ ) {

		// Get the info for this container.
		rc = evms_get_info(container_array->handle[i], &container_info);
		if (rc) {
			printf("%s -- Error getting info for container handle %u (%d)\n", cmd, container_array->handle[i], rc);
			lvscan_end(container_array, container_info, region_info, lv_info);
			return rc;
		}
		if ( opts.verbose ) {
			printf("%s -- Got info for container %s\n", cmd, container_info->info.container.name);
		}

		if ( strcmp(container_info->info.container.name, "lvm/LVM_Unassigned_Group") ) {
			num_vg++;
		}

		// Examine each region in this container.
		for ( j = 0; j < container_info->info.container.objects_produced->count; j++ ) {
			current_minor = 0;

			// Get the info for this region.
			rc = evms_get_info(container_info->info.container.objects_produced->handle[j], &region_info);
			if (rc) {
				printf("%s -- Error getting info for region handle %u (%d)\n", cmd, container_info->info.container.objects_produced->handle[j], rc);
				lvscan_end(container_array, container_info, region_info, lv_info);
				return rc;
			}
			if ( opts.verbose ) {
				printf("%s -- Got info for region %s\n", cmd, region_info->info.region.name);
			}

			// Skip freespace regions.
			if ( strstr(region_info->info.region.name, "Freespace") ) {
				evms_free(region_info);
				region_info = NULL;
				continue;
			}

			current_size = region_info->info.region.size;
			total_size += current_size;

			// See if this region is an EVMS volume.
			if ( region_info->info.region.volume ) {
				rc = evms_get_info(region_info->info.region.volume, &lv_info);
				if (rc) {
					num_inactive_lv++;
					printf("%s -- Error getting volume info for region %s\n", cmd, region_info->info.region.name);
					printf("%s -- inactive\t\t\"%s\" [%s]", cmd, region_info->info.region.name, str1 = print_size(current_size,0));
				}
				else {
					current_minor = lv_info->info.volume.minor_number;
					if ( opts.verbose ) {
						printf("%s -- Got info for volume %s\n", cmd, lv_info->info.volume.name);
					}
					evms_free(lv_info);
					lv_info = NULL;
					num_active_lv++;
					printf("%s -- ACTIVE\t\t\"%s\" [%s]", cmd, region_info->info.region.name, str1 = print_size(current_size,0));
					if ( opts.blockdevice && current_minor ) {
						printf(" %d:%d", EVMS_MAJOR, current_minor);
					}
				}
			}
			else {
				num_inactive_lv++;
				printf("%s -- inactive\t\t\"%s\" [%s]", cmd, region_info->info.region.name, str1 = print_size(current_size,0));
			}
			printf("\n");
			free(str1);
			evms_free(region_info);
			region_info = NULL;
		}
		evms_free(container_info);
		container_info = NULL;
	}


	if ( num_active_lv + num_inactive_lv ) {
		printf("%s -- %d logical volumes with %s total in %d volume groups\n",
			cmd, num_active_lv + num_inactive_lv, str1 = print_size(total_size,0), num_vg);
		if ( num_active_lv ) {
			printf("%s -- %d active logical volumes\n", cmd, num_active_lv);
		}
		if ( num_inactive_lv ) {
			printf("%s -- %d inactive logical volumes\n", cmd, num_inactive_lv);
		}
		free(str1);
	}
	else {
		printf("%s -- no logical volumes found.\n", cmd);
	}

	lvscan_end(container_array, container_info, region_info, lv_info);
	return 0;
}

