/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LvmUtils
 * File: evms_lvcreate.c
 *
 *	Emulates LVM's 'lvcreate' utility using the EVMS Engine. All options
 *	and several status messages are based on the original lvcreate command
 *	from Heinz Mauelshagen and Sistina Software (www.sistina.com).
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <libgen.h>
#include <ctype.h>
#include <frontend.h>

typedef struct cmd_options_s {
	u_int32_t	chunk_size;
	int		contiguous;
	int		debug;
	int		help;
	u_int32_t	extents;
	u_int32_t	lv_size;
	char		* lv_name;
	int		permission;
	int		snapshot;
	char		* snapshot_org;
	u_int32_t	stripes;
	u_int32_t	stripe_size;
	int		verbose;
	int		version;
	int		zero;
} cmd_options_t;

static char * cmd = NULL;
static cmd_options_t opts;

// Must be included after above definitions.
#include "helpers/get_container_handle.c"
#include "helpers/get_freespace.c"
#include "helpers/get_lvmregmgr.c"
#include "helpers/get_number.c"
#include "helpers/get_region_handle.c"
#include "helpers/open_engine.c"
#include "helpers/remove_duplicates.c"


static int showheader( void )
{
	// VERSION and DATE are defined in the top-level make.rules
	printf("Enterprise Volume Management System\n");
	printf("International Business Machines  %s\n", DATE);
	printf("LVM Emulation Utilities %s\n\n", VERSION);
	printf("%s -- initialize a logical volume for use by EVMS\n",cmd);
	return 0;
}


static int showhelp( void )
{
	showheader();
	printf("\n");
	printf("%s [-A|--autobackup {y|n}] [-C|--contiguous {y|n}] [-d|--debug]\n", cmd);
	printf("\t[-h|--help] [-i|--stripes Stripes [-I|--stripesize StripeSize]]\n");
	printf("\t{-l|--extents LogicalExtentsNumber |\n");
	printf("\t -L|--size LogicalVolumeSize[kKmMgGtT]} [-n|--name LogicalVolumeName]\n");
	printf("\t[-p|--permission {r|rw}] [-r|--readahead ReadAheadSectors]\n");
	printf("\t[-v|--verbose] [-Z|--zero {y|n}] [--version]\n");
	printf("\tVolumeGroupName [PhysicalVolumePath...]\n");
	printf("\n");
	printf("%s -s|--snapshot [-c|--chunksize ChunkSize]\n", cmd);
	printf("\t{-l|--extents LogicalExtentsNumber |\n");
	printf("\t -L|--size LogicalVolumeSize[kKmMgGtT]}\n");
	printf("\t-n|--name SnapshotLogicalVolumeName\n");
	printf("\tLogicalVolume[Path] [PhysicalVolumePath...]\n");
	return 0;
}


static int lvcreate_end( option_array_t	* a )
{
	if (a) free(a);
	evms_close_engine();
	return 0;
}


static int parse_options( int argc, char ** argv )
{
	int		c;
	char		* short_opts = "A:C:c:dh?i:I:l:L:n:p:r:svVZ:";
	char		* leftover = NULL;
	struct option	long_opts[] = {
				{ "autobackup",	required_argument, NULL, 'A'},
				{ "contiguous", required_argument, NULL, 'C'},
				{ "chunksize",	required_argument, NULL, 'c'},
				{ "debug",	no_argument,       NULL, 'd'},
				{ "help",	no_argument,       NULL, 'h'},
				{ "extents",	required_argument, NULL, 'l'},
				{ "name",	required_argument, NULL, 'n'},
				{ "permission",	required_argument, NULL, 'p'},
				{ "readahead",	required_argument, NULL, 'r'},
				{ "size",	required_argument, NULL, 'L'},
				{ "snapshot",	no_argument,       NULL, 's'},
				{ "stripes", 	required_argument, NULL, 'i'},
				{ "stripesize",	required_argument, NULL, 'I'},
				{ "verbose",	no_argument,       NULL, 'v'},
				{ "version",	no_argument,       NULL, 'V'},
				{ NULL, 0, NULL, 0} };

	// Zero option defaults to TRUE
	opts.zero = TRUE;

	while ( (c = getopt_long(argc, argv, short_opts,
				long_opts, NULL)) != EOF ) {
		switch (c) {
		case 'A':
			// -A is ignored by EVMS.
			if ( optarg[0] != 'n' && optarg[0] != 'y' ) {
				printf( "%s -- ERROR option %c argument \"%s\"\n", cmd, c, optarg);
				return EINVAL;
			}
			break;
		case 'C':
			if ( optarg[0] != 'n' && optarg[0] != 'y' ) {
				printf( "%s -- ERROR option %c argument \"%s\"\n", cmd, c, optarg);
			}
			opts.contiguous = (optarg[0] == 'y') ? TRUE : FALSE;
			break;
		case 'c':
			if ( (opts.chunk_size = strtoul(optarg, NULL, 0)) <= 0 ) {
				printf("%s -- ERROR option %c argument \"%s\"\n", cmd, c, optarg);
				return EINVAL;
			}
			// Convert kB to sectors
			opts.chunk_size = (opts.chunk_size * 1024) / EVMS_VSECTOR_SIZE;
			break;
		case 'd':
			opts.debug = TRUE;
			opts.verbose = TRUE;
			break;
		case 'h':
		case '?':
			opts.help = TRUE;
			break;
		case 'i':
			if ( (opts.stripes = strtoul(optarg, NULL, 0)) <= 0 ) {
				printf("%s -- ERROR option %c argument \"%s\"\n", cmd, c, optarg);
				return EINVAL;
			}
			break;
		case 'I':
			if ( (opts.stripe_size = strtoul(optarg, NULL, 0)) <= 0 ) {
				printf("%s -- ERROR option %c argument \"%s\"\n", cmd, c, optarg);
				return EINVAL;
			}
			// Convert kB to sectors
			opts.stripe_size = (opts.stripe_size * 1024) / EVMS_VSECTOR_SIZE;
			break;
		case 'l':
			// Only specify -l or -L
			if ( opts.lv_size ) {
				printf("%s -- -L option already given\n\n", cmd);
				return EINVAL;
			}
			if ( (opts.extents = strtoul(optarg, &leftover, 0)) <= 0 ) {
				printf("%s -- ERROR option %c argument \"%s\"\n", cmd, c, optarg);
				return EINVAL;
			}
			break;
		case 'L':
			// Only specify -l or -L
			if ( opts.extents ) {
				printf("%s -- -l option already given\n\n", cmd);
				return EINVAL;
			}
			if ( (opts.lv_size = get_number(optarg)) <= 0 ) {
				printf("%s -- ERROR option %c argument \"%s\"\n", cmd, c, optarg);
				return EINVAL;
			}
			// Convert kB to sectors
			opts.lv_size = (opts.lv_size * 1024) / EVMS_VSECTOR_SIZE;
			break;
		case 'n':
			opts.lv_name = optarg;
			break;
		case 'p':
			opts.permission = (!strcmp(optarg, "r")) ? TRUE : FALSE;
			break;
		case 'r':
			// -r is ignored by EVMS
			break;
		case 's':
			opts.snapshot = TRUE;
			break;
		case 'v':
			opts.verbose = TRUE;
			break;
		case 'V':
			opts.version = TRUE;
			break;
		case 'Z':
			if ( optarg[0] != 'n' && optarg[0] != 'y' ) {
				printf( "%s -- ERROR option %c argument \"%s\"\n", cmd, c, optarg);
				return EINVAL;
			}
			opts.zero = (optarg[0] == 'y') ? TRUE : FALSE;
			break;
		default:
			printf("%s -- unrecognized option \"%c\"\n\n", cmd, c);
			return EINVAL;
		}
	}

	return 0;
}


int verify_lvcreate_options( cmd_options_t * opts )
{
	// Must specify a size
	if ( ! opts->extents && ! opts->lv_size ) {
		printf("%s -- please enter either option -l or -L\n\n", cmd);
		return EINVAL;
	}
	// Must specify a name
	if ( ! opts->lv_name ) {
		printf("%s -- please enter a volume name using the -n option\n\n", cmd);
		return EINVAL;
	}
	// Snapshot volumes cannot be contiguous or striped
	if ( opts->snapshot ) {
		if ( opts->stripes ) {
			printf("%s -- cannot specify both -s and -i\n\n", cmd);
			return EINVAL;
		}
		if ( opts->contiguous ) {
			printf("%s -- cannot specify both -s and -C\n\n", cmd);
			return EINVAL;
		}
	}
	// Must specify striping if stripe_size is set
	if ( opts->stripe_size && ! opts->stripes ) {
		printf("%s -- -I option requires -i option\n\n", cmd);
		return EINVAL;
	}
	// Must specify snapshotting if chunk_size is set
	if ( opts->chunk_size && ! opts->snapshot ) {
		printf("%s -- -c option requires -s option\n\n", cmd);
	}

	// Any other checks???

	return 0;
}


int set_lvcreate_options(	option_array_t	* creation_options,
				cmd_options_t	* opts,
				char		** pv_names )
{
	int i = 0;
	int j;

	// Set the name option
	creation_options->count++;
	creation_options->option[i].is_number_based	= TRUE;
	creation_options->option[i].number		= 0;
	creation_options->option[i].value.s		= malloc(strlen(opts->lv_name)+1);
	strcpy(creation_options->option[i].value.s, opts->lv_name);
	if ( opts->verbose ) {
		printf("%s -- Setting LV Name option: %s\n", cmd, opts->lv_name);
	}
	i++;

	// Set the appropriate size option.
	if ( opts->extents ) {
		creation_options->count++;
		creation_options->option[i].is_number_based	= TRUE;
		creation_options->option[i].number		= 1;	// Extents
		creation_options->option[i].value.ui32		= opts->extents;
		if ( opts->verbose ) {
			printf("%s -- Setting Extents option: %d\n", cmd, opts->extents);
		}
		i++;
	}
	else {
		creation_options->count++;
		creation_options->option[i].is_number_based	= TRUE;
		creation_options->option[i].number		= 2;	// LV Size
		creation_options->option[i].value.ui32		= opts->lv_size;
		if ( opts->verbose ) {
			printf("%s -- Setting Size option: %d sectors\n", cmd, opts->lv_size);
		}
		i++;
	}

	// Set striping options
	if ( opts->stripes ) {
		creation_options->count++;
		creation_options->option[i].is_number_based	= TRUE;
		creation_options->option[i].number		= 3;
		creation_options->option[i].value.ui32		= opts->stripes;
		if ( opts->verbose ) {
			printf("%s -- Setting Stripes option: %d\n", cmd, opts->stripes);
		}
		i++;

		if ( opts->stripe_size ) {
			creation_options->count++;
			creation_options->option[i].is_number_based	= TRUE;
			creation_options->option[i].number		= 4;
			creation_options->option[i].value.ui32		= opts->stripe_size;
			if ( opts->verbose ) {
				printf("%s -- Setting Stripe Size option: %d sectors\n", cmd, opts->stripe_size);
			}
			i++;
		}
	}

	// Set snapshotting options
	if ( opts->snapshot ) {
		creation_options->count++;
		creation_options->option[i].is_number_based	= TRUE;
		creation_options->option[i].number		= 5;
		creation_options->option[i].value.i		= TRUE;
		if ( opts->verbose ) {
			printf("%s -- Setting Snapshot option\n", cmd);
		}
		i++;

		if ( opts->chunk_size ) {
			creation_options->count++;
			creation_options->option[i].is_number_based	= TRUE;
			creation_options->option[i].number		= 6;
			creation_options->option[i].value.ui32		= opts->chunk_size;
			if ( opts->verbose ) {
				printf("%s -- Setting Chunk Size option: %d sectors\n", cmd, opts->chunk_size);
			}
			i++;
		}
		else {
			printf("%s -- INFO: using default snapshot chunk size of 64kB for \"%s\"\n", cmd, opts->lv_name);
		}

		creation_options->count++;
		creation_options->option[i].is_number_based	= TRUE;
		creation_options->option[i].number		= 7;
		creation_options->option[i].value.s		= malloc(strlen(opts->snapshot_org)+1);
		opts->snapshot_org = translate_dev_name(opts->snapshot_org);
		strcpy(creation_options->option[i].value.s, opts->snapshot_org);
		if ( opts->verbose ) {
			printf("%s -- Setting Snapshot Original option: %s\n", cmd, opts->snapshot_org);
		}
		i++;
	}

	if ( opts->contiguous ) {
		creation_options->count++;
		creation_options->option[i].is_number_based	= TRUE;
		creation_options->option[i].number		= 8;
		creation_options->option[i].value.i		= TRUE;
		if ( opts->verbose ) {
			printf("%s -- Setting Contiguous option\n", cmd);
		}
		i++;
	}

	if ( opts->permission ) {
		creation_options->count++;
		creation_options->option[i].is_number_based	= TRUE;
		creation_options->option[i].number		= 9;
		creation_options->option[i].value.i		= TRUE;
		if ( opts->verbose ) {
			printf("%s -- Setting Read-Only option\n", cmd);
		}
		i++;
	}

	creation_options->count++;
	creation_options->option[i].is_number_based	= TRUE;
	creation_options->option[i].number		= 10;
	creation_options->option[i].value.i		= opts->zero;
	if ( opts->verbose ) {
		printf("%s -- Setting Zero option: %d\n", cmd, opts->zero);
	}
	i++;

	if ( pv_names[0] ) {
		creation_options->count++;
		creation_options->option[i].is_number_based	= TRUE;
		creation_options->option[i].number		= 11;
		creation_options->option[i].value.list = malloc(2500);
		for ( j = 0; pv_names[j]; j++ ) {
			creation_options->option[i].value.list->count++;
			creation_options->option[i].value.list->value[j].s = malloc(strlen(pv_names[j])+1);
			pv_names[j] = translate_dev_name(pv_names[j]);
			strcpy(creation_options->option[i].value.list->value[j].s, pv_names[j]);
			if ( opts->verbose ) {
				printf("%s -- Setting PV Name %d option: %s\n", cmd, j, pv_names[j]);
			}
		}
		i++;
	}

	return 0;
}


int main( int argc, char * argv[] )
{
	option_array_t		* creation_options	= NULL;
	plugin_handle_t		lvmregmgr		= 0;
	object_handle_t		container		= 0;
	object_handle_t		snapshot_org		= 0;
	handle_object_info_t	* snapshot_org_info	= NULL;
	handle_array_t		freespace_region;
	handle_array_t		* new_region_array	= NULL;
	char			* vg_name		= NULL;
	char			* pv_names[256]		= {NULL};
	int			number_of_pvs		= 0;
	int			log_level		= DEFAULT;
	int			rc;

	memset(&opts, 0, sizeof(cmd_options_t));
	cmd = basename(argv[0]);

	// Get the command line options.
	rc = parse_options(argc, argv);
	if (rc) {
		showhelp();
		return rc;
	}
	if ( opts.help ) {
		showhelp();
		return 0;
	}
	if ( opts.version ) {
		showheader();
		return 0;
	}
	if ( opts.verbose ) {
		log_level = DEBUG;
	}
	if ( opts.debug ) {
		log_level = ENTRY_EXIT;
	}

	// Verify that options are valid
	rc = verify_lvcreate_options(&opts);
	if (rc) {
		return rc;
	}

	// Memory for the segment and option arrays.
	creation_options = (option_array_t*)malloc(1000);
	if ( ! creation_options ) {
		printf("%s -- Memory allocation error.\n", cmd);
		lvcreate_end(creation_options);
		return ENOMEM;
	}
	memset(creation_options, 0, 1000);

	// Open the EVMS Engine.
	rc = open_engine(ENGINE_READWRITE, log_level);
	if (rc) {
		lvcreate_end(creation_options);
		return rc;
	}

	// Get the handle for the LVM region manager.
	rc = get_lvmregmgr(&lvmregmgr);
	if (rc) {
		lvcreate_end(creation_options);
		return rc;
	}

	if ( opts.snapshot ) {
		// Creating a snapshot. Check for snapshot original
		if ( optind == argc ) {
			printf("%s -- please enter logical volume name to be snapshotted\n\n", cmd);
			lvcreate_end(creation_options);
			return EINVAL;
		}
		opts.snapshot_org = argv[optind];
		optind++;
		printf("%s -- WARNING: the snapshot will be automatically disabled once it gets full\n", cmd);

		// Find the container for the snapshot original.
		rc = get_region_handle(opts.snapshot_org, &snapshot_org);
		if (rc) {
			lvcreate_end(creation_options);
			return rc;
		}
		rc = evms_get_info(snapshot_org, &snapshot_org_info);
		if (rc) {
			printf("%s -- Error getting info for snapshot original %s (%d)\n", cmd, opts.snapshot_org, rc);
			lvcreate_end(creation_options);
			return rc;
		}
		container = snapshot_org_info->info.region.producing_container;
		evms_free(snapshot_org_info);
	}
	else {
		// Creating a regular volume. Check for volume group name.
		if ( optind == argc ) {
			printf("%s -- please enter a volume group name\n", cmd);
			lvcreate_end(creation_options);
			return EINVAL;
		}
		vg_name = argv[optind];
		if ( opts.verbose ) {
			printf("%s -- Creating volume in group %s\n", cmd, vg_name);
		}
		optind++;

		// Check for PV names
		if ( optind != argc) {
			number_of_pvs = argc - optind;
	
			// Copy the PV names from the command line, detecting any duplicates.
			rc = remove_duplicates( &argv[optind], pv_names, number_of_pvs);
			if (rc) {
				printf("%s -- Duplicate PVs specified. Please only specify each PV once\n", cmd);
				lvcreate_end(creation_options);
				return rc;
			}
		}

		// Get the handle for the target volume group
		rc = get_container_handle(vg_name, &container);
		if (rc) {
			lvcreate_end(creation_options);
			return rc;
		}
	}

	// Set up the option array
	set_lvcreate_options(creation_options, &opts, pv_names);

	// Get the handle for the target freespace region
	rc = get_freespace_for_container(container, &(freespace_region.handle[0]));
	if (rc) {
		lvcreate_end(creation_options);
		return rc;
	}
	freespace_region.count = 1;

	// Create the new region
	rc = evms_create(lvmregmgr,
			&freespace_region,
			creation_options,
			&new_region_array);
	if (rc) {
		printf("%s -- Error creating region %s (%d)\n", cmd, opts.lv_name, rc);
		lvcreate_end(creation_options);
		return rc;
	}
	if ( opts.verbose ) {
		printf("%s -- New region created\n", cmd);
	}

	// Turn the region into a volume. This is only necessary for
	// non-snapshot volumes. The engine automatically creates
	// compatibilities out of snapshots.
	if ( ! opts.snapshot ) {
		rc = evms_create_compatibility_volume(new_region_array->handle[0]);
		if (rc) {
			printf("%s -- Error creating compatibility volume from region %s (%d)\n", cmd, opts.lv_name, rc);
			lvcreate_end(creation_options);
			return rc;
		}
	}
	if ( opts.verbose ) {
		printf("%s -- New compatibility volume created\n", cmd);
	}

	// Write everything to disk.
	rc = evms_commit_changes(NULL);
	if (rc) {
		printf("%s -- Error committing changes to disk (%d)\n", cmd, rc);
		lvcreate_end(creation_options);
		return rc;
	}
	if ( opts.verbose ) {
		printf("%s -- New volume committed to disk\n", cmd);
	}


	printf("%s -- logical volume \"%s\" successfully created\n\n", cmd, opts.lv_name);
	lvcreate_end(creation_options);
	return 0;
}

