/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: embedded.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>
#include <linux/evms/evms_user.h>

#include "ptables.h"
#include "segs.h"
#include "embedded.h"
#include "bsd.h"
#include "solarisX86.h"
#include "unixware.h"


/*
 *  Called to allocate a segment storage object
 */
DISKSEG *  alloc_diskseg_object( LOGICALDISK *ld )
{
    int      rc;
    DISKSEG *seg=NULL;


    LOGENTRY();

    rc = SegEngFncs->allocate_segment( NULL, &seg );
    if (rc==0) {

        if (rc == 0) {

            seg->plugin      = Seg_My_PluginRecord_Ptr;
            seg->object_type = SEGMENT;

            memcpy(&seg->geometry, &ld->geometry, sizeof(geometry_t) );

            seg->private_data = calloc(1, sizeof(SEG_PRIVATE_DATA));
            if (seg->private_data) {

                ((SEG_PRIVATE_DATA *)seg->private_data)->signature = DEFAULT_SEG_MGR_PDATA_SIGNATURE;
                ((SEG_PRIVATE_DATA *)seg->private_data)->logical_disk = ld;

            }
            else {
                SegEngFncs->free_segment( seg );
                seg = NULL;
            }
        }
    }

    LOGEXIT();
    return seg;
}


/*
 * Builds a disk segment matching the embedded partition info
 */
DISKSEG * build_segment_for_embedded_partition( LOGICALDISK       *ld,
                                                DISKSEG           *msdos_seg,
                                                u_int32_t          start,
                                                u_int32_t          size,
                                                u_int32_t          sys_id,
                                                u_int32_t          ptable_index,
                                                u_int32_t          minor )
{
    DISKSEG           *seg        = NULL;
    SEG_PRIVATE_DATA  *pdata      = NULL;
    int                rc;
    void              *handle;

    LOGENTRY();

    seg = alloc_diskseg_object( ld );
    if (seg) {


        rc = InsertObject (  seg->child_objects,
                            (int)              sizeof(DISKSEG),
                            (void *)           msdos_seg,
                            (TAG)              SEGMENT_TAG,
                            (void *)           NULL,
                            (Insertion_Modes)  InsertBefore,
                            (BOOLEAN)          TRUE,
                            (void **)         &handle );


        if (rc == DLIST_SUCCESS) {

            pdata = (SEG_PRIVATE_DATA *) seg->private_data;

            seg->size              = size;
            seg->start             = start;
            seg->data_type         = DATA_TYPE;

            pdata->sys_id          = sys_id;
            pdata->part_number     = minor;
            pdata->ptable_index    = ptable_index;

        }
        else {
            free(seg->private_data);
            SegEngFncs->free_segment( seg );
            seg = NULL;
        }


    }

    LOGEXIT();
    return seg;
}



/*
 *  Returns DISKSEG matching query information
 */
DISKSEG * get_matching_segment( dlist_t seglist, lba_t start, sector_count_t size )
{
    DISKSEG          *seg;
    int               rc;

    rc = GoToStartOfList( seglist );
    if (rc == DLIST_SUCCESS) {

        rc = GetObject( seglist,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **)&seg );

        while (rc == DLIST_SUCCESS) {

            if ( (seg->start == start) && (seg->size == size) ) return seg;

            rc = GetNextObject( seglist,sizeof(DISKSEG),SEGMENT_TAG,(void **)&seg );

        }

    }

    return NULL;
}


/*
 *  Called to remove the embedded partitions, found in the recovery_list, from
 *  the specified logical disk.  This routine is called when embedded partition
 *  discovery has failed for a certain primary partition we we need to recover
 *  the state of the drive prior to when embedded partition discovery failed.
 */
int  remove_embedded_partitions_from_disk( LOGICALDISK *ld, dlist_t recovery_list )
{
    DISKSEG          *seg;
    int               rc;

    LOGENTRY();

    rc = GoToStartOfList( recovery_list );
    if (rc == DLIST_SUCCESS) {

        rc = GetObject( recovery_list, sizeof(DISKSEG), SEGMENT_TAG,NULL,TRUE,(void **)&seg );

        while (rc == DLIST_SUCCESS) {

            LOG_DEBUG("removing %s\n", seg->name );
            rc = DeleteObject( ld->parent_objects, seg );

            if ((rc == DLIST_SUCCESS)||(rc == DLIST_OBJECT_NOT_FOUND)) {
                rc = GetNextObject( recovery_list,sizeof(DISKSEG),SEGMENT_TAG,(void **)&seg );
            }

        }

    }

    // throw away errors like ... DLIST_END_OF_LIST ... which are not real errors
    if ( ( rc == DLIST_EMPTY ) || ( rc == DLIST_END_OF_LIST ) || (rc == DLIST_OBJECT_NOT_FOUND)) {
    rc = DLIST_SUCCESS;
    }

    LOGEXITRC();
    return rc;
}


/*
 *  Called to look for an embedded partition scheme on a drive, examine
 *  the disk label, and return the geometry we discover.
 */
int read_embedded_geometry( LOGICALDISK *ld, geometry_t *geometry )
{
    struct bsd_disklabel       *bl;
    struct unixware_disklabel  *ul;
    char                        data[EVMS_VSECTOR_SIZE];
    Master_Boot_Record          mbr[EVMS_VSECTOR_SIZE];
    int                         rc = ENOSYS;
    struct plugin_functions_s  *dft;
    int                         i;
    Partition_Record           *part;
    BOOLEAN                     success=FALSE;

    LOGENTRY();


    // get disk manager function table
    dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
    if (dft==NULL) {
        LOGEXIT();
        return EINVAL;
    }

    // read mbr sector
    rc = dft->read( ld, 0, 1, (void *) &mbr );
    if (rc) {
        free(data);
        LOGEXIT();
        return rc;
    }

    // look for an embedded partition in the mbr
    for (i=0; (i < 4) && (success == FALSE); i++) {

        part = &mbr->Partition_Table[i];

        switch ( SYS_IND(part) ) {

            case BSD_PARTITION:
            case NETBSD_PARTITION:
            case OPENBSD_PARTITION:

                // read bsd info
                rc = dft->read( ld,
                                DISK_TO_CPU32( START_LBA(part) ) + BSD_DISKLABEL_PART_TABLE_SECTOR_OFFSET,
                                1,
                                (void *) &data );

                if (rc) {
                    LOGEXIT();
                    return rc;
                }

                // check for BSD magic number
                bl = (struct bsd_disklabel *) &data;

LOG_DEBUG("found bsd partition \n");

                if ( DISK_TO_CPU32(bl->d_magic) == BSD_DISKMAGIC) {
                    geometry->cylinders         = DISK_TO_CPU32(bl->d_ncylinders);
                    geometry->heads             = DISK_TO_CPU32(bl->d_ntracks);
                    geometry->sectors_per_track = DISK_TO_CPU32(bl->d_nsectors);

                    success = TRUE;
                }

                break;


            case UNIXWARE_PARTITION:

                // read UnixWare info
                rc = dft->read( ld,
                                DISK_TO_CPU32( START_LBA(part) ) + UNIXWARE_PART_TABLE_SECTOR_OFFSET,
                                1,
                                (void *) &data );

                if (rc) {
                    LOGEXIT();
                    return rc;
                }

                // check unixware magic number
                ul = (struct unixware_disklabel *) &data;

LOG_DEBUG("found unixware partition\n");

                if ( ( DISK_TO_CPU32(ul->d_magic)      == UNIXWARE_DISKMAGIC ) &&
                     ( DISK_TO_CPU32(ul->vtoc.v_magic) == UNIXWARE_DISKMAGIC2 ))  {
                    geometry->cylinders         = DISK_TO_CPU32(ul->d_ncylinders);
                    geometry->heads             = DISK_TO_CPU32(ul->d_ntracks);
                    geometry->sectors_per_track = DISK_TO_CPU32(ul->d_nsectors);

                    success = TRUE;
                }

                break;

            case SOLARIS_X86_PARTITION:

                // not implemented yet
                break;


            default:
                break;
        }

    }


    if (success == TRUE) {
        LOG_DEBUG("success:  C= %lld  H= %d  S= %d\n", geometry->cylinders, geometry->heads, geometry->sectors_per_track);
        rc = 0;
    }
    else {
        LOG_DEBUG("failure\n");
        if (rc == 0) rc = EINVAL;
    }

    LOGEXITRC();
    return rc;
}
