/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: option.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include <fullengine.h>
#include "task.h"
#include "option.h"
#include "engine.h"
#include "handlemgr.h"
#include "common.h"
#include "memman.h"

/*
 *
 *   int GetOptionCount(task_handle_t, int *)
 *
 *   Description:
 *      This routine retrieves the total possible plug-in options for a
 *      specific task. The count remains unchanged through the duration
 *      of a specific task.
 *
 *   Entry:
 *      handle - handle to task context created by CreateTask()
 *      count  - address to copy option count to
 *
 *   Exit:
 *      On success, returns a return code of 0 and *count is updated.
 *      On error, returns an error code < 0
 *
 */
int evms_get_option_count (task_handle_t handle, int * count) {
    int rc;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(handle,
                              &object,
                              &type);

        if (rc == DLIST_SUCCESS) {

            if (type == TASK_TAG) {
                task_context_t * task = (task_context_t *) object;

                if (count != NULL) {
                    *count = task->option_descriptors->count;
                    rc = 0;
                } else {
                    engine_write_log_entry(ERROR, "Can not return count through NULL pointer.\n");
                    rc = EINVAL;
                }
            } else {
                engine_write_log_entry(ERROR, "Not a task context handle.\n");
                rc = EINVAL;
            }
        } else {
            engine_write_log_entry(ERROR, "Error from translate_handle().\n");
            rc = EINVAL;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


void free_option_descriptor_contents(void * object) {

    int i;
    option_descriptor_t * descriptor = (option_descriptor_t *) object;

    LOG_PROC_ENTRY();

    if (descriptor->name != NULL) {
        free(descriptor->name);
    }
    if (descriptor->title != NULL) {
        free(descriptor->title);
    }
    if (descriptor->tip != NULL) {
        free(descriptor->tip);
    }
    if (descriptor->help != NULL) {
        free(descriptor->help);
    }

    /*
     * The constraint member contains either a pointer to a list structure or a
     * pointer to a range structure.  Regardless, we free whatever it points to
     * if allocated.
     */
    switch (descriptor->constraint_type) {
        case EVMS_Collection_List:
            /*
             * If the constraint type is for a list then we also check to see if
             * it is a string list.  If so, we free those strings before freeing
             * the list.
             */
            if (descriptor->constraint.list != NULL) {
                if (descriptor->type == EVMS_Type_String) {
                    for (i = 0; i < descriptor->constraint.list->count; i++) {
                        if (descriptor->constraint.list->value[i].s != NULL) {
                            free(descriptor->constraint.list->value[i].s);
                        }
                    }
                }
                free(descriptor->constraint.list);

            } else {
                engine_write_log_entry(WARNING, "Collection says it has a list but the list pointer is NULL.\n");
            }
            break;

        case EVMS_Collection_Range:
            if (descriptor->constraint.range != NULL) {
                free(descriptor->constraint.range);

            } else {
                engine_write_log_entry(WARNING, "Collection says it has a range but the range pointer is NULL.\n");
            }

        default:
            break;
    }

    /*
     * If the value is a list, free the list.  If it is a list of strings,
     * free each of the strings, too.
     */
    if (EVMS_OPTION_VALUE_IS_LIST (descriptor->flags) &&
        EVMS_OPTION_HAS_VALUE (descriptor->flags)) {
        if (descriptor->type == EVMS_Type_String) {
            for (i = 0; i < descriptor->value.list->count; i++) {
                if (descriptor->value.list->value[i].s != NULL) {
                    free(descriptor->value.list->value[i].s);
                }
            }
        }

        free(descriptor->value.list);

    } else {
        /* If the value is a string (not a collection of them), free the string. */
        if (descriptor->type == EVMS_Type_String &&
            descriptor->constraint_type == EVMS_Collection_None) {
            if (descriptor->value.s != NULL) {
                free(descriptor->value.s);
            }
        }
    }


    if (descriptor->group.group_name) {
        free(descriptor->group.group_name);
    }

    LOG_PROC_EXIT_VOID();
}


static int copy_to_new_string(const char * source, char * * target) {

    int rc = 0;
    char * string;

    LOG_PROC_ENTRY();

    if (source != NULL) {
        string = malloc(strlen(source) + 1);
        if (string != NULL) {
            strcpy(string, source);
            *target = string;
        } else {
            *target = NULL;
            rc = ENOMEM;
        }

    } else {
        *target = NULL;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}

static int duplicate_value_list (value_list_t *source, value_type_t type, value_list_t **target)
{
    int rc=0;

    if (source != NULL) {
        uint size;

        if (source->count == 0) {
            size = sizeof(source->count);
        } else {
            size = sizeof(value_list_t) + (sizeof(value_t) * (source->count - 1));
        }

        *target = calloc(1, size);
        if (*target != NULL) {
            if (type == EVMS_Type_String) {
                int i;

                /* Copy the count of strings. */
                (*target)->count = source->count;

                /* Get new copies of the strings for the target. */
                for (i = 0; (rc == 0) && (i < source->count); i++) {
                    rc = copy_to_new_string(source->value[i].s, &((*target)->value[i].s));
                }

            } else {
                /* Copy the value list. */
                memcpy((value_list_t *) *target, source, size);
            }

        } else {
            rc = ENOMEM;
        }
    }

    return rc;
}

static int deep_copy_option_descriptor(option_descriptor_t * target,
                                       option_descriptor_t * source) {
    int rc = 0;

    LOG_PROC_ENTRY();

    /* Copy the base structure to the target descriptor. */
    memcpy(target, source, sizeof(option_descriptor_t));

    /*
     * Wipe out the strings in the target descriptor.  In case of an error we
     * will clean up the target descriptor.  We don't want any pointers to the
     * original strings lying around and accidentally freed on the cleanup.
     */

    target->name  = NULL;
    target->title = NULL;
    target->tip   = NULL;
    target->help  = NULL;
    target->constraint.list  = NULL;
    target->group.group_name = NULL;

    if (source->type == EVMS_Type_String || EVMS_OPTION_VALUE_IS_LIST (source->flags))
        target->value.s = NULL;

    /*
     * Allocate new copies of the strings for the target descriptors and put
     * them into the target descriptor.
     */

    rc = copy_to_new_string(source->name, &target->name);

    if ( rc == 0) {
        rc = copy_to_new_string(source->title, &target->title);
    }

    if ( rc == 0) {
        rc = copy_to_new_string(source->tip, &target->tip);
    }

    if ( rc == 0) {
        rc = copy_to_new_string(source->help, &target->help);
    }

    switch (source->constraint_type) {
        case EVMS_Collection_List:
            if (source->constraint.list != NULL)
                rc = duplicate_value_list (source->constraint.list, source->type,
                                           &(target->constraint.list));

            break;

        case EVMS_Collection_Range:
            if (source->constraint.range != NULL) {
                target->constraint.range = malloc(sizeof(value_range_t));
                if (target->constraint.range != NULL) {
                    memcpy((value_range_t *) target->constraint.range, source->constraint.range, sizeof(value_range_t));
                } else {
                    rc = ENOMEM;
                }
            }
            break;

        default:
            break;
    }

    if (rc == 0) {
        rc = copy_to_new_string(source->group.group_name, &target->group.group_name);
    }

    /*
     * Lastly, if the source had an initial value we need to copy it
     * over if its a string or a list.
     */
    if (rc == 0)
    {
        if (EVMS_OPTION_HAS_VALUE (source->flags))
        {
            if (EVMS_OPTION_VALUE_IS_LIST (source->flags))
                rc = duplicate_value_list (source->value.list, source->type, &(target->value.list));
            else if (source->type == EVMS_Type_String)
            {
                target->value.s = malloc (source->size + 1);

                if (target->value.s != NULL)
                    strcpy (target->value.s, source->value.s);
                else
                    rc = ENOMEM;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}

/*
 *
 *   int GetOptionDescriptor(task_handle_t, INT32, option_descriptor_t *)
 *
 *   Description:
 *      This routine retrieves a full description of a backend option
 *      for the specified task context.
 *
 *   Entry:
 *      handle     - handle to task context created by CreateTask()
 *      index      - zero-based ordinal number used to access descriptor
 *      descriptor - address to copy option descriptor to
 *
 *   Exit:
 *      On success, returns a return code of 0 and *descriptor is updated
 *      On error, returns an error code < 0
 *
 */
int evms_get_option_descriptor (task_handle_t handle, u_int32_t index,
                                option_descriptor_t * *descriptor) {
    int rc;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(handle,
                              &object,
                              &type);

        if (rc == DLIST_SUCCESS) {

            if (type == TASK_TAG) {
                task_context_t * task = (task_context_t *) object;

                if (descriptor != NULL) {
                    if (index < task->option_descriptors->count) {
                        *descriptor = alloc_app_struct(sizeof(option_descriptor_t), free_option_descriptor_contents);
                        if (*descriptor != NULL) {
                            rc = deep_copy_option_descriptor(*descriptor, &(task->option_descriptors->option[index]));
                            if (rc != 0) {
                                evms_free(*descriptor);
                                *descriptor = NULL;
                            }
                        } else {
                            rc = ENOMEM;
                        }
                    } else {
                        engine_write_log_entry(ERROR, "Invalid option index %d.\n", index);
                        rc = EINVAL;
                    }
                } else {
                    engine_write_log_entry(ERROR, "Can not return descriptor through NULL pointer.\n");
                    rc = EINVAL;
                }
            } else {
                engine_write_log_entry(ERROR, "Not a task context handle.\n");
                rc = EINVAL;
            }
        } else {
            engine_write_log_entry(ERROR, "Error from translate_handle().\n");
            rc = EINVAL;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}

/*
 *
 *   static inline void GetValue(task_context_t *, INT32, value_t *)
 *
 *   Description:
 *      This private routine retrieves the current value in the option
 *      values array associated with the given task context and index
 *      and copies the value to the return value location.
 *
 *   Entry:
 *      task  - address of task context structure created by CreateTask()
 *      index - zero-based ordinal number used to access option value
 *      value - address to copy option value to
 *
 *   Exit:
 *      *value is updated.
 *
 */
static inline void GetValue (task_context_t *task, u_int32_t index, value_t *value) {

    LOG_PROC_ENTRY();

    switch (task->option_descriptors->option[index].type) {
        case EVMS_Type_String:
            value->s = task->option_descriptors->option[index].value.s;
            break;

        case EVMS_Type_Real32:
            value->r32 = task->option_descriptors->option[index].value.r32;
            break;

        case EVMS_Type_Real64:
            value->r64 = task->option_descriptors->option[index].value.r64;
            break;

        case EVMS_Type_Int8:
        case EVMS_Type_Char:
        case EVMS_Type_Unsigned_Int8:
        case EVMS_Type_Unsigned_Char:
            value->uc = task->option_descriptors->option[index].value.uc;
            break;

        case EVMS_Type_Int16:
        case EVMS_Type_Unsigned_Int16:
            value->ui16 = task->option_descriptors->option[index].value.ui16;
            break;

        case EVMS_Type_Int:
        case EVMS_Type_Int32:
        case EVMS_Type_Boolean:
        case EVMS_Type_Unsigned_Int:
        case EVMS_Type_Unsigned_Int32:
            value->ui32 = task->option_descriptors->option[index].value.ui32;
            break;

        case EVMS_Type_Int64:
        case EVMS_Type_Unsigned_Int64:
            value->ui64 = task->option_descriptors->option[index].value.ui64;
            break;

        default:
            engine_write_log_entry(ERROR, "Unknown value type.\n");
            break;
    }

    LOG_PROC_EXIT_VOID();
}

/*
 *
 *   static inline void copy_value (task_context_t *, INT32, value_t *)
 *
 *   Description:
 *      This private routine retrieves the current value in the option
 *      values array associated with the given task context and index
 *      and copies the value to the return value location.
 *
 *   Entry:
 *      task  - address of task context structure created by CreateTask()
 *      index - zero-based ordinal number used to access option value
 *      value - address to copy option value to
 *
 *   Exit:
 *      *value is updated.
 *
 */
static inline void copy_value (task_context_t *task, u_int32_t index, value_t *value) {

    LOG_PROC_ENTRY();

    if (EVMS_OPTION_VALUE_IS_LIST (task->option_descriptors->option[index].flags))
    {
        int           size;
        value_list_t *list;

        list = task->option_descriptors->option[index].value.list;
        size = sizeof (value_list_t) + ((list->count - 1) * sizeof (value_t));

        if (task->option_descriptors->option[index].type == EVMS_Type_String)
        {
            int i;

            for (i=0; i < list->count; i++)
            {
                strcpy (value->list->value[i].s, list->value[i].s);
            }
        }
        else
        {
            memcpy (value->list, list, size);
        }
    }
    else if (task->option_descriptors->option[index].type == EVMS_Type_String)
    {
        strcpy (value->s, task->option_descriptors->option[index].value.s);
    }
    else
    {
        memcpy (value, &(task->option_descriptors->option[index].value), sizeof (value_t));
    }

    LOG_PROC_EXIT_VOID();
}

/*
 *
 *   static inline int OptionNameToIndex(task_context_t *, const CHAR *, INT32 *)
 *
 *   Description:
 *      This private routine takes an option name, scans the option
 *      descriptors in the array and returns the index corresponding
 *      to the option descriptor entry matching the name.
 *
 *   Entry:
 *      task  - address of task context structure created by CreateTask()
 *      name  - address of NULL-terminated string containing option name
 *      index - address to copy matching option index to
 *
 *   Exit:
 *      On success, returns a return code of 0 and *index is updated.
 *      On error, returns an error code of EINVAL
 *
 */
static inline int OptionNameToIndex (task_context_t *task, const char *name,
                                     u_int32_t *index) {
    int i = 0;
    int rc = EINVAL;

    LOG_PROC_ENTRY();

    /*
     * Currently, don't care about search order so simply search from
     * max option index down to 0...
     */
    i = task->option_descriptors->count - 1;

    while ((i >= 0) && (rc != 0)) {
        if (strcasecmp(name, task->option_descriptors->option[i].name) == 0) {
            *index = i;
            rc = 0;
        } else {
            i--;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 *
 *   static inline int SetValue(task_context_t *, INT32, value_t *, INT32 *)
 *
 *   Description:
 *      This private routine calls the SetOption plug-in function
 *      to attempt to update the option value in the option values
 *      array corresponding to the given option index.
 *
 *   Entry:
 *      task  - address of task context structure created by CreateTask()
 *      index - zero-based ordinal number used to access option value
 *      value - address of new option value
 *      info  - address for returning side effect flags bitmap
 *
 *   Exit:
 *      On success, returns a return code of 0 and option value in options
 *      array and possibly *info and *value are updated.
 *      On error, returns an error code < 0
 *
 */
static inline int SetValue (task_context_t *task, u_int32_t index, value_t *value,
                            u_int32_t *info) {
    int rc;

    LOG_PROC_ENTRY();

    *info = 0;

    switch (GetPluginType(task->plugin->id)) {
        case EVMS_DEVICE_MANAGER:
        case EVMS_SEGMENT_MANAGER:
        case EVMS_REGION_MANAGER:
        case EVMS_FEATURE:
        case EVMS_ASSOCIATIVE_FEATURE:
            rc = task->plugin->functions.plugin->set_option(task,
                                                            index,
                                                            value,
                                                            info);
            break;

        case EVMS_FILESYSTEM_INTERFACE_MODULE:
            rc = task->plugin->functions.fsim->set_option(task,
                                                          index,
                                                          value,
                                                          info);
            break;

        case EVMS_CLUSTER_MANAGER_INTERFACE_MODULE:
        case EVMS_DISTRIBUTED_LOCK_MANAGER_INTERFACE_MODULE:
        default:
            rc = ENOSYS;
            break;
    }

    /*
     * If the plug-in accepted the value but fudged it, get the
     * fudged value from the option descriptor array in the
     * task context and reflect the change in the user's supplied
     * value buffer.
     */

    if (rc == 0 && *info == EVMS_Effect_Inexact)
        copy_value (task, index, value);

    LOG_PROC_EXIT_INT(rc);
    return rc;
}

/*
 *
 *   int SetOptionValue(task_handle_t, INT32, value_t *, INT32 *)
 *
 *   Description:
 *      This routine attempts to update the option value corresponding
 *      to the given option index.
 *
 *   Entry:
 *      handle - handle to task context created by CreateTask()
 *      index  - zero-based ordinal number used to access option value
 *      value  - address of new option value
 *      info   - address for returning side effect flags bitmap
 *
 *   Exit:
 *      On success, returns a return code of 0 and option value in task
 *      and possibly *value and *info are updated.
 *      On error, returns an error code < 0
 *
 */
int evms_set_option_value (task_handle_t handle, u_int32_t index, value_t *value, u_int32_t *info) {
    int rc;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(handle,
                              &object,
                              &type);

        if (rc == DLIST_SUCCESS) {

            if (type == TASK_TAG) {
                task_context_t * task = (task_context_t *) object;

                if (value != NULL) {
                    if (info != NULL) {
                        if (index < task->option_descriptors->count) {
                            rc = SetValue(task, index, value, info);
                        } else {
                            engine_write_log_entry(ERROR, "Invalid option index %d.\n", index);
                            rc = EINVAL;
                        }
                    } else {
                        engine_write_log_entry(ERROR, "Can not return info through NULL pointer.\n");
                        rc = EINVAL;
                    }
                } else {
                    engine_write_log_entry(ERROR, "Can not accept NULL value pointer.\n");
                    rc = EINVAL;
                }
            } else {
                engine_write_log_entry(ERROR, "Not a task context handle.\n");
                rc = EINVAL;
            }
        } else {
            engine_write_log_entry(ERROR, "Error from translate_handle().\n");
            rc = EINVAL;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}

/*
 *
 *   int GetOptionDescriptorByName(task_handle_t, const CHAR *, option_descriptor_t *)
 *
 *   Description:
 *      This routine retrieves a full description of a backend option
 *      for the specified task context.
 *
 *   Entry:
 *      handle     - handle to task context created by CreateTask()
 *      name       - address of NULL-terminated string containing option name
 *      descriptor - address to copy option descriptor to
 *
 *   Exit:
 *      On success, returns a return code of 0 and *descriptor is updated
 *      On error, returns an error code < 0
 *
 */
int evms_get_option_descriptor_by_name (task_handle_t handle, const char *name,
                                        option_descriptor_t * *descriptor) {
    int rc;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(handle,
                              &object,
                              &type);

        if (rc == DLIST_SUCCESS) {

            if (type == TASK_TAG) {
                task_context_t * task = (task_context_t *) object;

                if (*descriptor != NULL) {
                    if (name != NULL) {
                        u_int32_t index;

                        rc = OptionNameToIndex(task, name, &index);

                        if (rc == 0) {
                            *descriptor = alloc_app_struct(sizeof(option_descriptor_t), free_option_descriptor_contents);
                            if (*descriptor != NULL) {
                                rc = deep_copy_option_descriptor(*descriptor, &(task->option_descriptors->option[index]));
                                if (rc != 0) {
                                    evms_free(*descriptor);
                                    *descriptor = NULL;
                                }
                            } else {
                                rc = ENOMEM;
                            }
                        } else {
                            engine_write_log_entry(ERROR, "%s is not a known option name.\n", name);
                        }
                    } else {
                        engine_write_log_entry(ERROR, "Option name is NULL.\n");
                        rc = EINVAL;
                    }
                } else {
                    engine_write_log_entry(ERROR, "Can not return descriptor through NULL pointer.\n");
                    rc = EINVAL;
                }
            } else {
                engine_write_log_entry(ERROR, "Not a task context handle.\n");
                rc = EINVAL;
            }
        } else {
            engine_write_log_entry(ERROR, "Error from translate_handle().\n");
            rc = EINVAL;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}

/*
 *
 *   int GetOptionValueByName(task_handle_t, const CHAR *, value_t *)
 *
 *   Description:
 *      This routine retrieves the option value for the given task
 *      and option name and copies the resulting value to the location
 *      pointed to by the value pointer.
 *
 *   Entry:
 *      handle - handle to task context created by CreateTask()
 *      name   - address of NULL-terminated string containing option name
 *      value  - address to copy option value to
 *
 *   Exit:
 *      On success, returns a return code of 0 and *value is updated.
 *      On error, returns an error code < 0
 *
 */
int evms_get_option_value_by_name (task_handle_t handle, const char *name, value_t *value) {
    int rc;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(handle,
                              &object,
                              &type);

        if (rc == DLIST_SUCCESS) {

            if (type == TASK_TAG) {
                task_context_t * task = (task_context_t *) object;

                if (value != NULL) {
                    if (name != NULL) {
                        u_int32_t index;

                        rc = OptionNameToIndex(task, name, &index);

                        if (rc == 0) {
                            GetValue(task, index, value);
                        } else {
                            engine_write_log_entry(ERROR, "%s is not a known option name.\n", name);
                        }
                    } else {
                        engine_write_log_entry(ERROR, "Option name is NULL.\n");
                        rc = EINVAL;
                    }
                } else {
                    engine_write_log_entry(ERROR, "Can not return value through NULL pointer.\n");
                    rc = EINVAL;
                }
            } else {
                engine_write_log_entry(ERROR, "Not a task context handle.\n");
                rc = EINVAL;
            }
        } else {
            engine_write_log_entry(ERROR, "Error from translate_handle().\n");
            rc = EINVAL;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}

/*
 *
 *   int SetOptionValueByName(task_handle_t, const CHAR *, value_t *, INT32 *)
 *
 *   Description:
 *      This routine attempts to update the option value corresponding
 *      to the given option name.
 *
 *   Entry:
 *      handle - handle to task context created by CreateTask()
 *      name   - address of NULL-terminated string containing option name
 *      value  - address of new option value
 *      info   - address for returning side effect flags bitmap
 *
 *   Exit:
 *      On success, returns a return code of 0 and option value in task
 *      and possibly *value and *info are updated.
 *      On error, returns an error code < 0
 *
 */
int evms_set_option_value_by_name (task_handle_t handle, const char *name, value_t *value,
                                   u_int32_t *info) {
    int rc;
    void * object;
    object_type_t type;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        rc = translate_handle(handle,
                              &object,
                              &type);

        if (rc == DLIST_SUCCESS) {

            if (type == TASK_TAG) {
                task_context_t * task = (task_context_t *) object;

                if (value != NULL) {
                    if (name != NULL) {
                        if (info != NULL) {
                            u_int32_t index;

                            rc = OptionNameToIndex(task, name, &index);

                            if (rc == 0) {
                                rc = SetValue(task, index, value, info);
                            } else {
                                engine_write_log_entry(ERROR, "%s is not a known option name.\n", name);
                            }
                        } else {
                            engine_write_log_entry(ERROR, "Can not return info through NULL pointer.\n");
                            rc = EINVAL;
                        }
                    } else {
                        engine_write_log_entry(ERROR, "Option name is NULL.\n");
                        rc = EINVAL;
                    }
                } else {
                    engine_write_log_entry(ERROR, "Can not accept NULL value pointer.\n");
                    rc = EINVAL;
                }
            } else {
                engine_write_log_entry(ERROR, "Not a task context handle.\n");
                rc = EINVAL;
            }
        } else {
            engine_write_log_entry(ERROR, "Error from translate_handle().\n");
            rc = EINVAL;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}

