/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: dload.c
 *
 *
 * Functions:
 *
 *      MallocPluginRecord()
 *      FreePluginRecord()
 *      isa_valid_plugin_record()
 *      load_module()
 *      unload_module()
 *      get_sym_addr()
 *      getProcAddr()
 *      load_module_plugins()
 *      unload_plugin()
 *      load_plugins()
 *      unload_plugins()
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <dirent.h>
#include <dlfcn.h>
#include <unistd.h>
#include <errno.h>

#include <fullengine.h>
#include "dload.h"
#include "engine.h"
#include "message.h"
#include "internalAPI.h"


/* Forward references */
static int unload_plugin(plugin_record_t * pPlugRec, dlist_t plugin_list);


/*
 *  Called with a plug-in record to see if it is a duplicate of an existing
 *  plug-in record in the dlist_t. Matching them based on the plug-in record ID
 *  field allows for a quick comparison of: OEM id, EVMS plug-in Type and
 *  EVMS plug-in ID.  Plug-ins are duplicates if they have matching ID fields!
 *
 *  Returns: TRUE if a duplicate is found in the list
 *           FALSE if we don't find a duplicate plug-in in the list
 */
static BOOLEAN isa_duplicate_plugin( plugin_record_t * pPlugRec, dlist_t plugin_list ) {
    BOOLEAN           rc = FALSE;
    int               result;
    plugin_record_t * pr;

    LOG_PROC_ENTRY();

    result = GoToStartOfList( plugin_list );
    if (result == DLIST_SUCCESS) {

        result = GetObject( plugin_list, sizeof(plugin_record_t), PLUGIN_TAG, NULL, TRUE,(ADDRESS *) &pr );
        if (result == DLIST_SUCCESS) {

            do {

                if (pr) {

                    if (pr->id == pPlugRec->id) {

                        engine_user_message(NULL, NULL, "Duplicate Plug-ins detected\n"
                                                        "Already have loaded:\n"
                                                        "  long name: %-40s\n"
                                                        "  oem name: %-40s\n"
                                                        "  ID: 0x%08X\n"
                                                        "  version: %d.%d.%d\n"
                                                        "Currently loading:\n"
                                                        "  long name: %-40s\n"
                                                        "  oem name: %-40s\n"
                                                        "  ID: 0x%08X\n"
                                                        "  version: %d.%d.%d\n",
                                            pr->long_name,
                                            pr->oem_name,
                                            pr->id,
                                            pr->version.major, pr->version.minor, pr->version.patchlevel,
                                            pPlugRec->long_name,
                                            pPlugRec->oem_name,
                                            pPlugRec->id,
                                            pPlugRec->version.major, pPlugRec->version.minor, pPlugRec->version.patchlevel);

                        /*
                         * Check to see if the plug-in we are about to load is
                         * older than the one that is loaded.  If so,  return
                         * TRUE that this is a duplicate plug-in.
                         */
                        if (pr->version.major > pPlugRec->version.major) {
                            rc = TRUE;
                        } else {
                            if ((pr->version.major == pPlugRec->version.major) &&
                                (pr->version.minor > pPlugRec->version.minor)) {
                                rc = TRUE;
                            } else {
                                if ((pr->version.minor == pPlugRec->version.minor) &&
                                    (pr->version.patchlevel >= pPlugRec->version.patchlevel)) {
                                    rc = TRUE;
                                }
                            }
                        }

                        if (rc == FALSE) {
                            /*
                             * The plug-in we are about to load is newer than
                             * the one that is currently loaded.  Unload the
                             * current plug-in and allow the new one to be
                             * loaded.
                             */

                            engine_user_message(NULL, NULL, "Unloading version %d.%d.%d of the plug-in.\n",
                                                pr->version.major, pr->version.minor, pr->version.patchlevel);
                            DeleteObject(plugin_list, pr);
                            unload_plugin(pr, plugin_list);

                        } else {
                            engine_user_message(NULL, NULL, "Keeping version %d.%d.%d of the plug-in.\n",
                                                pr->version.major, pr->version.minor, pr->version.patchlevel);
                        }

                        break;

                    } else {
                        result = GetNextObject( plugin_list, sizeof(plugin_record_t), PLUGIN_TAG, (ADDRESS *) &pr );
                    }
                }

            } while ((result == DLIST_SUCCESS) && (pr != NULL));
        }
    }

    LOG_PROC_EXIT_BOOLEAN(rc);
    return rc;
}


/*
 *  Validates that a plug-in record is useable.  Does simple checking to
 *  see if we could access the plug-in Ok from the EVMS engine.
 */

static BOOLEAN isa_valid_plugin_record( plugin_record_t * pPlugRec ) {
    BOOLEAN result = FALSE;

    LOG_PROC_ENTRY();

    if (pPlugRec->long_name != NULL) {
        if (pPlugRec->short_name != NULL) {
            if (pPlugRec->oem_name != NULL) {
                if (pPlugRec->functions.plugin != NULL) {
                    if (pPlugRec->required_api_version.major == ENGINE_PLUGIN_API_MAJOR_VERION) {
                        if (pPlugRec->required_api_version.minor <= ENGINE_PLUGIN_API_MINOR_VERION) {
                            if (pPlugRec->required_api_version.minor == ENGINE_PLUGIN_API_MINOR_VERION) {
                                if (pPlugRec->required_api_version.patchlevel <= ENGINE_PLUGIN_API_PATCH_LEVEL) {
                                    result = TRUE;
                                } else {
                                    engine_user_message(NULL, NULL, "The plug-in %s in module %s failed to load.  It requires a plug-in API patch level (%d.%d.%d) which is greater than this Engine's plug-in API patch level (%d.%d.%d).\n",
                                                        pPlugRec->short_name, pPlugRec->so_record->name,
                                                        pPlugRec->required_api_version.major, pPlugRec->required_api_version.minor, pPlugRec->required_api_version.patchlevel,
                                                        ENGINE_PLUGIN_API_MAJOR_VERION, ENGINE_PLUGIN_API_MINOR_VERION, ENGINE_PLUGIN_API_PATCH_LEVEL);
                                }
                            } else {
                                result = TRUE;
                            }
                        } else {
                            engine_user_message(NULL, NULL, "The plug-in %s in module %s failed to load.  It requires a plug-in API minor version number (%d.%d.%d) which is greater than this Engine's plug-in API minor version number (%d.%d.%d).\n",
                                                pPlugRec->short_name, pPlugRec->so_record->name,
                                                pPlugRec->required_api_version.major, pPlugRec->required_api_version.minor, pPlugRec->required_api_version.patchlevel,
                                                ENGINE_PLUGIN_API_MAJOR_VERION, ENGINE_PLUGIN_API_MINOR_VERION, ENGINE_PLUGIN_API_PATCH_LEVEL);
                        }
                    } else {
                        engine_user_message(NULL, NULL, "The plug-in %s in module %s failed to load.  It requires a plug-in API major version number (%d.%d.%d) which does not match this Engine's plug-in API major version number (%d.%d.%d).\n",
                                            pPlugRec->short_name, pPlugRec->so_record->name,
                                            pPlugRec->required_api_version.major, pPlugRec->required_api_version.minor, pPlugRec->required_api_version.patchlevel,
                                            ENGINE_PLUGIN_API_MAJOR_VERION, ENGINE_PLUGIN_API_MINOR_VERION, ENGINE_PLUGIN_API_PATCH_LEVEL);
                    }
                } else {
                    engine_user_message(NULL, NULL, "The plug-in %s in module %s failed to load.  It did not provide a function table.\n",
                                        pPlugRec->short_name, pPlugRec->so_record->name);
                }
            } else {
                engine_user_message(NULL, NULL, "The plug-in %s in module %s failed to load.  It did not provide an oem name.\n",
                                    pPlugRec->short_name, pPlugRec->so_record->name);
            }
        } else {
            engine_user_message(NULL, NULL, "The plug-in %s in module %s failed to load.  It did not provide a short name.\n",
                                pPlugRec->short_name, pPlugRec->so_record->name);
        }
    } else {
        engine_user_message(NULL, NULL, "The plug-in %s in module %s failed to load.  It did not provide a long name.\n",
                            pPlugRec->short_name, pPlugRec->so_record->name);
    }

    LOG_PROC_EXIT_BOOLEAN(result);
    return result;
}


/*
 *  Two routines to load and unload Linux shared objects.
 */

static int load_module(char * soname, so_record_t * * pso_record) {

    int rc = 0;
    char * error = NULL;
    so_record_t * so_record = calloc(1, sizeof(so_record_t));

    LOG_PROC_ENTRY();

    *pso_record = NULL;

    if (so_record != NULL) {

        dlerror();
        so_record->handle = dlopen(soname, RTLD_NOW);

        error = dlerror();
        if (error == NULL) {
            so_record->name = strdup(soname);
            so_record->plugin_list = CreateList();

            if (so_record->plugin_list != NULL) {
                *pso_record = so_record;

            } else {
                LOG_CRITICAL("Error allocating memory for a plug-in list in a .so record.\n");
                free(so_record->name);
                free(so_record);
                rc = ENOMEM;
            }

        } else {
            engine_user_message(NULL, NULL, "Error loading %s\n", error);
            free (so_record);
            rc = ELIBBAD;
        }

    } else {
        LOG_CRITICAL("Error allocating memory for a .so record.\n");
        rc = ENOMEM;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static void unload_module(so_record_t * so_record) {

    uint count = 0;

    LOG_PROC_ENTRY();
    LOG_DEBUG("Unload module %s.\n", so_record->name);
    free(so_record->name);
    LOG_DEBUG("Issuing dlclose() on handle %d (%x).\n", so_record->handle, so_record->handle);
    dlclose(so_record->handle);
    GetListSize(so_record->plugin_list, &count);
    if (count != 0) {
        LOG_WARNING("Warning: Unloading module %s while %d plug-ins are still loaded from it.\n", so_record->name, count);
    }
    DestroyList(&so_record->plugin_list, FALSE);
    free(so_record);

    LOG_PROC_EXIT_VOID();
}


/*  Two routines which get the address of routines and variables
 *  in Linux shared objects.
 */

static ADDRESS get_sym_addr(module_handle_t handle, char * symname) {

    ADDRESS      symaddr;
    const char * error;

    LOG_PROC_ENTRY();

    LOG_DEBUG("Get address for symbol %s from module handle %d.\n", symname, handle);

    dlerror();
    symaddr = dlsym(handle, symname);
    if ((error = dlerror()) != NULL) {
        LOG_DEBUG("Error getting address for %s from module %p: %s\n", symname, handle, error);
        symaddr  = NULL;
    }

    LOG_PROC_EXIT_PTR(symaddr);
    return symaddr;
}


/*
 * Dummy functions for filling in function table entries that a plug-in
 * does not provide.
 */
static int return_ENOSYS() {
    return ENOSYS;
}


static int return_0() {
    return 0;
}


static int validate_plugin_functions(plugin_record_t * pPlugRec) {

    int rc = 0;

    LOG_PROC_ENTRY();

    /* A plug-in must support the following functions. */

    if ((pPlugRec->functions.plugin->setup_evms_plugin        == NULL) ||
        (pPlugRec->functions.plugin->discover                 == NULL) ||
        (pPlugRec->functions.plugin->add_sectors_to_kill_list == NULL) ||
        (pPlugRec->functions.plugin->commit_changes           == NULL) ||
        (pPlugRec->functions.plugin->read                     == NULL) ||
        (pPlugRec->functions.plugin->write                    == NULL)) {
        rc = ENOSYS;
    }

    /*
     * The remaining functions are not absolutely necessary for a plug-in to
     * function.  If the plug-in doesn't specify them, we supply a default.
     */

    if (pPlugRec->functions.plugin->can_delete == NULL) {
        pPlugRec->functions.plugin->can_delete = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.plugin->can_expand == NULL) {
        pPlugRec->functions.plugin->can_expand = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.plugin->can_expand_by == NULL) {
        pPlugRec->functions.plugin->can_expand_by = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.plugin->can_shrink == NULL) {
        pPlugRec->functions.plugin->can_shrink = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.plugin->can_shrink_by == NULL) {
        pPlugRec->functions.plugin->can_shrink_by = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.plugin->can_move == NULL) {
        pPlugRec->functions.plugin->can_move = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.plugin->can_set_volume == NULL) {
        pPlugRec->functions.plugin->can_set_volume = (void *) return_0;
    }
    if (pPlugRec->functions.plugin->set_volume == NULL) {
        pPlugRec->functions.plugin->set_volume = (void *) return_0;
    }
    if (pPlugRec->functions.plugin->create == NULL) {
        pPlugRec->functions.plugin->create = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.plugin->delete == NULL) {
        pPlugRec->functions.plugin->delete = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.plugin->expand == NULL) {
        pPlugRec->functions.plugin->expand = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.plugin->shrink == NULL) {
        pPlugRec->functions.plugin->shrink = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.plugin->move == NULL) {
        pPlugRec->functions.plugin->move = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.plugin->get_option_count == NULL) {
        pPlugRec->functions.plugin->get_option_count = (void *) return_0;
    }
    if (pPlugRec->functions.plugin->init_task == NULL) {
        pPlugRec->functions.plugin->init_task = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.plugin->set_option == NULL) {
        pPlugRec->functions.plugin->set_option = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.plugin->set_objects == NULL) {
        pPlugRec->functions.plugin->set_objects = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.plugin->get_info == NULL) {
        pPlugRec->functions.plugin->get_info = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.plugin->set_info == NULL) {
        pPlugRec->functions.plugin->set_info = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.plugin->get_plugin_info == NULL) {
        pPlugRec->functions.plugin->get_plugin_info = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.plugin->direct_plugin_communication == NULL) {
        pPlugRec->functions.plugin->direct_plugin_communication = (void *) return_ENOSYS;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static int validate_fsim_functions(plugin_record_t * pPlugRec) {

    int rc = 0;

    LOG_PROC_ENTRY();

    /* An FSIM must support the following functions. */

    if ((pPlugRec->functions.fsim->setup_evms_plugin == NULL) ||
        (pPlugRec->functions.fsim->is_this_yours     == NULL)) {
        rc = ENOSYS;
    }

    /*
     * The remaining functions are not absolutely necessary for a plug-in to
     * function.  If the plug-in doesn't specify them, we supply a default.
     */
    if (pPlugRec->functions.fsim->get_fs_size == NULL) {
        pPlugRec->functions.fsim->get_fs_size = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.fsim->get_fs_limits == NULL) {
        pPlugRec->functions.fsim->get_fs_limits = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.fsim->can_mkfs == NULL) {
        pPlugRec->functions.fsim->can_mkfs = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.fsim->can_unmkfs == NULL) {
        pPlugRec->functions.fsim->can_unmkfs = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.fsim->can_fsck == NULL) {
        pPlugRec->functions.fsim->can_fsck = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.fsim->can_defrag == NULL) {
        pPlugRec->functions.fsim->can_defrag = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.fsim->can_expand_by == NULL) {
        pPlugRec->functions.fsim->can_expand_by = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.fsim->can_shrink_by == NULL) {
        pPlugRec->functions.fsim->can_shrink_by = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.fsim->mkfs == NULL) {
        pPlugRec->functions.fsim->mkfs = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.fsim->unmkfs == NULL) {
        pPlugRec->functions.fsim->unmkfs = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.fsim->fsck == NULL) {
        pPlugRec->functions.fsim->fsck = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.fsim->defrag == NULL) {
        pPlugRec->functions.fsim->defrag = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.fsim->expand == NULL) {
        pPlugRec->functions.fsim->expand = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.fsim->shrink == NULL) {
        pPlugRec->functions.fsim->shrink = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.fsim->get_option_count == NULL) {
        pPlugRec->functions.fsim->get_option_count = (void *) return_0;
    }
    if (pPlugRec->functions.fsim->init_task == NULL) {
        pPlugRec->functions.fsim->init_task = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.fsim->set_option == NULL) {
        pPlugRec->functions.fsim->set_option = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.fsim->set_volumes == NULL) {
        pPlugRec->functions.fsim->set_volumes = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.fsim->get_volume_info == NULL) {
        pPlugRec->functions.fsim->get_volume_info = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.fsim->get_plugin_info == NULL) {
        pPlugRec->functions.fsim->get_plugin_info = (void *) return_ENOSYS;
    }
    if (pPlugRec->functions.fsim->direct_plugin_communication == NULL) {
        pPlugRec->functions.fsim->direct_plugin_communication = (void *) return_ENOSYS;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static int validate_container_functions(plugin_record_t * pPlugRec) {

    int rc = 0;

    LOG_PROC_ENTRY();

    /*
     * If a plug-in has a container functions table, all the function
     * addresses must be filled in.  No container functions are optional.
     */

    if ((pPlugRec->container_functions->can_create_container     == NULL) ||
        (pPlugRec->container_functions->can_delete_container     == NULL) ||
        (pPlugRec->container_functions->can_add_object           == NULL) ||
        (pPlugRec->container_functions->can_remove_object        == NULL) ||
        (pPlugRec->container_functions->create_container         == NULL) ||
        (pPlugRec->container_functions->add_object               == NULL) ||
        (pPlugRec->container_functions->transfer_object          == NULL) ||
        (pPlugRec->container_functions->remove_object            == NULL) ||
        (pPlugRec->container_functions->delete_container         == NULL) ||
        (pPlugRec->container_functions->commit_container_changes == NULL) ||
        (pPlugRec->container_functions->get_container_info       == NULL)) {

        rc = ENOSYS;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static int check_and_setup_plugin(dlist_t plugin_list, plugin_record_t * pPlugRec) {

    int  rc = 0;

    LOG_PROC_ENTRY();

    if (isa_valid_plugin_record( pPlugRec ) == TRUE) {

        /* Don't run with duplicate plug-ins loaded ! */
        if (isa_duplicate_plugin( pPlugRec, plugin_list ) == FALSE) {
            uint plugin_type;

            plugin_type = GetPluginType(pPlugRec->id);

            LOG_DEBUG("Call plug-in's setup_evms_plugin().\n");
            switch (plugin_type) {
                case EVMS_DEVICE_MANAGER:
                case EVMS_SEGMENT_MANAGER:
                case EVMS_REGION_MANAGER:
                case EVMS_FEATURE:
                case EVMS_ASSOCIATIVE_FEATURE:
                    {
                        rc = validate_plugin_functions(pPlugRec);
                        if (rc == 0) {

                            if (pPlugRec->container_functions != NULL) {
                                rc = validate_container_functions(pPlugRec);

                                if (rc != 0) {
                                    engine_user_message(NULL, NULL, "The %s plug-in in module %s failed to load.  It does not have a valid container functions table.\n", pPlugRec->short_name, pPlugRec->so_record->name);
                                }
                            }

                            if (rc == 0) {
                                rc = pPlugRec->functions.plugin->setup_evms_plugin(engine_mode, &engine_functions);

                                if (rc != 0) {
                                    engine_user_message(NULL, NULL, "The plug-in %s in module %s failed to load.  The plug-in's setup_evms_plugin() function failed with error code %d: %s.\n", pPlugRec->short_name, pPlugRec->so_record->name, rc, strerror(rc));
                                }
                            }

                        } else {
                            engine_user_message(NULL, NULL, "The %s plug-in in module %s failed to load.  It does not have a valid function table.\n", pPlugRec->short_name, pPlugRec->so_record->name);
                        }
                        break;
                    }

                case EVMS_FILESYSTEM_INTERFACE_MODULE:
                    {
                        rc = validate_fsim_functions(pPlugRec);
                        if (rc == 0) {
                            rc = pPlugRec->functions.fsim->setup_evms_plugin(engine_mode, &engine_functions);
                            if (rc != 0) {
                                engine_user_message(NULL, NULL, "The plug-in %s in module %s failed to load.  The plug-in's setup_evms_plugin() function failed with error code %d: %s.\n", pPlugRec->short_name, pPlugRec->so_record->name, rc, strerror(rc));
                            }
                        } else {
                            engine_user_message(NULL, NULL, "The %s plug-in in module %s failed to load.  It does not have a valid function table.\n", pPlugRec->short_name, pPlugRec->so_record->name);
                        }
                        break;
                    }

                default:
                    engine_user_message(NULL, NULL, "The %s plug-in in module %s failed to load.  It is of type of 0x%x which is not valid.\n", pPlugRec->short_name, pPlugRec->so_record->name, plugin_type);
                    rc = ENOSYS;
            }

        } else {
            rc = EEXIST;
        }

    } else {
        LOG_ERROR("Plug-in returned an invalid plug-in record.\n");
        rc = ENOSYS;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static int load_module_plugins(so_record_t * so_record, plugin_record_t * * ppPlugRec, dlist_t plugin_list) {

    int rc = 0;
    uint count = 0;
    plugin_record_t * pPlugRec = *ppPlugRec;

    LOG_PROC_ENTRY();

    while (pPlugRec != NULL) {
        pPlugRec->so_record = so_record;

        rc = check_and_setup_plugin(plugin_list, pPlugRec);

        if (rc == 0) {
            void * trash;
            rc = InsertObject(plugin_list,
                              sizeof(plugin_record_t),
                              pPlugRec,
                              PLUGIN_TAG,
                              NULL,
                              InsertAtStart,
                              TRUE,
                              &trash);

            if (rc == DLIST_SUCCESS) {
                rc = InsertObject(so_record->plugin_list,
                                  sizeof(plugin_record_t),
                                  pPlugRec,
                                  PLUGIN_TAG,
                                  NULL,
                                  AppendToList,
                                  FALSE,
                                  &trash);

                if (rc == DLIST_SUCCESS) {
                    LOG_DEFAULT("Loaded from %s.\n", so_record->name);
                    LOG_DEFAULT("  short name:  %s\n", pPlugRec->short_name);
                    LOG_DEFAULT("  long name:   %s\n", pPlugRec->long_name);
                    LOG_DEFAULT("  version:     %d.%d.%d\n", pPlugRec->version.major, pPlugRec->version.minor, pPlugRec->version.patchlevel);

                } else {
                    LOG_CRITICAL("InsertObject() to put plug-in record in .so record failed with return code %d.\n", rc);
                    DeleteObject(plugin_list, *ppPlugRec);
                }

            } else {
                LOG_CRITICAL("InsertObject() to put plug-in record in the PluginList failed with return code %d.\n", rc);
            }
        }

        /* Don't let any error stop the plug-in loading process. */
        rc = 0;

        /* Check if the .so has another plug-in record. */
        ppPlugRec++;
        pPlugRec = *ppPlugRec;
    }

    /*
     * If we didn't load any plug-ins from this module, then unload
     * the module.
     */
    GetListSize(so_record->plugin_list, &count);

    if (count == 0) {
        unload_module(so_record);
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 *   This routine is called to load plug-ins for EVMS. Since we have
 *   no idea which plug-ins are available on the system we load them
 *   in the following manner:
 *
 *     (1) Try and open the "well known" EVMS Plugin directory ... /usr/lib/evms/
 *     (2) For each file found in the plug-in directory we'll call load_plugin()
 *         which can recognize plug-ins and load them.
 *     (3) If load_plugin() returns success we'll add the plug-in to the plugin
 *         dlist_t by inserting it as an object.
 *
 *   Finally, If we find at least 1 plug-in we'll return success.
 */

int load_plugins(dlist_t plugin_list) {
    DIR           * pDir;
    struct dirent * pDirent;
    int             rc = 0;
    uint            plugins_loaded = 0;
    plugin_record_t * * ppPlugRec;
    char            so_name[256];

    LOG_PROC_ENTRY();

    if (plugin_list != NULL) {

        pDir = opendir(PluginDirectory);
        if (pDir != NULL) {

            pDirent = readdir(pDir);
            while ((rc == 0) && (pDirent != NULL)) {

                if ((strcmp(pDirent->d_name,".") != 0) &&
                    (strcmp(pDirent->d_name,"..") != 0)) {
                    so_record_t * so_record;

                    strcpy(so_name, PluginDirectory);
                    strcat(so_name, "/");
                    strcat(so_name, pDirent->d_name);

                    LOG_DETAILS("Module to load is %s\n", so_name);

                    rc = load_module(so_name, &so_record);

                    if (rc == 0) {
                        if (so_record != NULL) {

                            /* Check for an array of plug-in pointers. */
                            ppPlugRec = (plugin_record_t * *) get_sym_addr(so_record->handle, "evms_plugin_records");

                            if (ppPlugRec != NULL) {

                                /* Load the plug-in(s) in the module. */
                                rc = load_module_plugins(so_record, ppPlugRec, plugin_list);

                            } else {
                                /*
                                 * The module doesn't export the published name
                                 * for a plug-in record table.  It must not be
                                 * an Engine plug-in.  Unload the module.
                                 */
                                engine_user_message(NULL, NULL, "Failed to load module %s.  It does not export an \"evms_plugin_records\" variable.\n", so_record->name);
                                unload_module(so_record);
                            }

                        } else {
                            LOG_WARNING("load_module() failed.\n");
                        }

                    } else {
                        /*
                         * Don't let module load failures top the plug-in
                         * loading process.
                         */
                        rc = 0;
                    }
                }

                pDirent = readdir(pDir);
            }

            closedir(pDir);

        } else {
            rc = ENOENT;
        }

        /* Find out how many plug-ins got loaded. */
        GetListSize(plugin_list, &plugins_loaded);

        LOG_DEBUG("%s Plug-ins were loaded.\n", (plugins_loaded != 0) ? "" : "No ");
        LOG_DEBUG("Return code is %d.\n", rc);
        if (rc == 0) {
            if (plugins_loaded == 0) {
                rc = ENOENT;
            }
        } else {
            if (plugins_loaded != 0) {
                unload_plugins(plugin_list);
            }
        }

    } else {
        // plugin_list is NULL.
        rc = EINVAL;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 *   A plug-in is unloaded by calling unload_module to
 *   unload the shared object.  Before doing so we first
 *   need to call the plug-in's cleanup routine if it
 *   has one.
 */
static int unload_plugin(plugin_record_t * pPlugRec, dlist_t plugin_list) {

    uint count = 0;

    LOG_PROC_ENTRY();

    if (pPlugRec != NULL) {

        if (pPlugRec->functions.plugin->cleanup_evms_plugin != NULL) {
            pPlugRec->functions.plugin->cleanup_evms_plugin();
        }

        DeleteObject(plugin_list, pPlugRec);

        DeleteObject(pPlugRec->so_record->plugin_list, pPlugRec);

        /*
         * Unload the module if we have no other plug-ins loaded from
         * the module.
         */
        GetListSize(pPlugRec->so_record->plugin_list, &count);

        if (count == 0) {
            unload_module(pPlugRec->so_record);
        }
    }

    LOG_PROC_EXIT_INT(0);
    return 0;
}


/*
 *   Look in the EVMS Plugin dlist_t and unload any plug-ins found
 *   in the list by doing ...
 *         (1) extracting an object from the plug-in dlist_t
 *         (2) call unload_plugin() to unload the shared object
 *         (3) free the plug-in record's memory
 */

int unload_plugins(dlist_t plugin_list) {
    int               error = DLIST_SUCCESS;
    plugin_record_t  *pPlugRec;

    LOG_PROC_ENTRY();

    error = GoToStartOfList(plugin_list);
    if (error != DLIST_SUCCESS) {
        LOG_WARNING("GoToStartOfList returned error code %d.\n", error);
        return EIO;
    }

    do {
        error = ExtractObject(plugin_list,
                              sizeof(plugin_record_t),
                              PLUGIN_TAG,
                              NULL,
                              (ADDRESS *) &pPlugRec);

        if (error == DLIST_SUCCESS) {
            unload_plugin(pPlugRec, plugin_list);
            // BUGBUG
            // In the new scheme, plug-ins own their plugin_records so
            // we don't free them.  While we are running in a potentially
            // mixed environment we can't tell whether the plugin-record
            // is one we allocated or one that is in a plug-in.  So, just
            // don't free it.  Worst case is that this is a memory leak
            // for plugin-records that we allocated.  But process exit
            // will clean that up anyway.

            // free_plugin_record(pPlugRec);
        }

    } while (error == DLIST_SUCCESS);

    LOG_PROC_EXIT_INT(0);
    return 0;
}
