# -*- coding: utf-8 -*-

# Copyright (c) 2005 - 2008 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the VCS project helper for Subversion.
"""

from PyQt4.QtCore import *
from PyQt4.QtGui import *

from VCS.ProjectHelper import VcsProjectHelper

from E4Gui.E4Action import E4Action

class SvnProjectHelper(VcsProjectHelper):
    """
    Class implementing the VCS project helper for Subversion.
    """
    def __init__(self, vcsObject, projectObject, parent = None, name = None):
        """
        Constructor
        
        @param vcsObject reference to the vcs object
        @param projectObject reference to the project object
        @param parent parent widget (QWidget)
        @param name name of this object (string or QString)
        """
        VcsProjectHelper.__init__(self, vcsObject, projectObject, parent, name)
        
    def initActions(self):
        """
        Public method to generate the action objects.
        """
        self.vcsNewAct = E4Action(self.trUtf8('New from repository'),
                self.trUtf8('&New from repository...'), 0, 0, self, 'subversion_new')
        self.vcsNewAct.setStatusTip(self.trUtf8(
            'Create a new project from the VCS repository'
        ))
        self.vcsNewAct.setWhatsThis(self.trUtf8(
            """<b>New from repository</b>"""
            """<p>This creates a new local project from the VCS repository.</p>"""
        ))
        self.vcsNewAct.connectIt(SIGNAL('triggered()'), self._vcsCheckout)
        self.actions.append(self.vcsNewAct)
        
        self.vcsUpdateAct = E4Action(self.trUtf8('Update from repository'),
                self.trUtf8('&Update from repository'), 0, 0, self,
                'subversion_update')
        self.vcsUpdateAct.setStatusTip(self.trUtf8(
            'Update the local project from the VCS repository'
        ))
        self.vcsUpdateAct.setWhatsThis(self.trUtf8(
            """<b>Update from repository</b>"""
            """<p>This updates the local project from the VCS repository.</p>"""
        ))
        self.vcsUpdateAct.connectIt(SIGNAL('triggered()'), self._vcsUpdate)
        self.actions.append(self.vcsUpdateAct)
        
        self.vcsCommitAct = E4Action(self.trUtf8('Commit changes to repository'),
                self.trUtf8('&Commit changes to repository...'), 0, 0, self,
                'subversion_commit')
        self.vcsCommitAct.setStatusTip(self.trUtf8(
            'Commit changes to the local project to the VCS repository'
        ))
        self.vcsCommitAct.setWhatsThis(self.trUtf8(
            """<b>Commit changes to repository</b>"""
            """<p>This commits changes to the local project to the VCS repository.</p>"""
        ))
        self.vcsCommitAct.connectIt(SIGNAL('triggered()'), self._vcsCommit)
        self.actions.append(self.vcsCommitAct)
        
        self.vcsAddAct = E4Action(self.trUtf8('Add to repository'),
                self.trUtf8('&Add to repository...'), 0, 0, self, 'subversion_add')
        self.vcsAddAct.setStatusTip(self.trUtf8(
            'Add the local project to the VCS repository'
        ))
        self.vcsAddAct.setWhatsThis(self.trUtf8(
            """<b>Add to repository</b>"""
            """<p>This adds (imports) the local project to the VCS repository.</p>"""
        ))
        self.vcsAddAct.connectIt(SIGNAL('triggered()'), self._vcsImport)
        self.actions.append(self.vcsAddAct)
        
        self.vcsRemoveAct = E4Action(self.trUtf8(
                'Remove from repository (and disk)'),
                self.trUtf8('&Remove from repository (and disk)'),
                0, 0, self, 'subversion_remove')
        self.vcsRemoveAct.setStatusTip(self.trUtf8(
            'Remove the local project from the VCS repository (and  disk)'
        ))
        self.vcsRemoveAct.setWhatsThis(self.trUtf8(
            """<b>Remove from repository</b>"""
            """<p>This removes the local project from the VCS repository"""
            """ (and disk).</p>"""
        ))
        self.vcsRemoveAct.connectIt(SIGNAL('triggered()'), self._vcsRemove)
        self.actions.append(self.vcsRemoveAct)
        
        self.vcsLogAct = E4Action(self.trUtf8(
                'Show log'),
                self.trUtf8('Show &log'),
                0, 0, self, 'subversion_log')
        self.vcsLogAct.setStatusTip(self.trUtf8(
            'Show the log of the local project'
        ))
        self.vcsLogAct.setWhatsThis(self.trUtf8(
            """<b>Show log</b>"""
            """<p>This shows the log of the local project.</p>"""
        ))
        self.vcsLogAct.connectIt(SIGNAL('triggered()'), self._vcsLog)
        self.actions.append(self.vcsLogAct)
        
        self.svnLogLimitedAct = E4Action(self.trUtf8(
                'Show limited log'),
                self.trUtf8('Show limited log'),
                0, 0, self, 'subversion_log_limited')
        self.svnLogLimitedAct.setStatusTip(self.trUtf8(
            'Show a limited log of the local project'
        ))
        self.svnLogLimitedAct.setWhatsThis(self.trUtf8(
            """<b>Show limited log</b>"""
            """<p>This shows the log of the local project limited to a selectable"""
            """ number of entries.</p>"""
        ))
        self.svnLogLimitedAct.connectIt(SIGNAL('triggered()'), self.__svnLogLimited)
        self.actions.append(self.svnLogLimitedAct)
        
        self.svnLogBrowserAct = E4Action(self.trUtf8(
                'Show log browser'),
                self.trUtf8('Show log browser'),
                0, 0, self, 'subversion_log_browser')
        self.svnLogBrowserAct.setStatusTip(self.trUtf8(
            'Show a dialog to browse the log of the local project'
        ))
        self.svnLogBrowserAct.setWhatsThis(self.trUtf8(
            """<b>Show log browser</b>"""
            """<p>This shows a dialog to browse the log of the local project."""
            """ A limited number of entries is shown first. More can be"""
            """ retrieved later on.</p>"""
        ))
        self.svnLogBrowserAct.connectIt(SIGNAL('triggered()'), self.__svnLogBrowser)
        self.actions.append(self.svnLogBrowserAct)
        
        self.vcsDiffAct = E4Action(self.trUtf8('Show difference'),
                self.trUtf8('Show &difference'),
                0, 0, self, 'subversion_diff')
        self.vcsDiffAct.setStatusTip(self.trUtf8(
            'Show the difference of the local project to the repository'
        ))
        self.vcsDiffAct.setWhatsThis(self.trUtf8(
            """<b>Show difference</b>"""
            """<p>This shows the difference of the local project to the repository.</p>"""
        ))
        self.vcsDiffAct.connectIt(SIGNAL('triggered()'), self._vcsDiff)
        self.actions.append(self.vcsDiffAct)
        
        self.svnExtDiffAct = E4Action(self.trUtf8('Show difference (extended)'),
                self.trUtf8('Show difference (extended)'),
                0, 0, self, 'subversion_extendeddiff')
        self.svnExtDiffAct.setStatusTip(self.trUtf8(
            'Show the difference of revisions of the project to the repository'
        ))
        self.svnExtDiffAct.setWhatsThis(self.trUtf8(
            """<b>Show difference (extended)</b>"""
            """<p>This shows the difference of selectable revisions of the project.</p>"""
        ))
        self.svnExtDiffAct.connectIt(SIGNAL('triggered()'), self.__svnExtendedDiff)
        self.actions.append(self.svnExtDiffAct)
        
        self.svnUrlDiffAct = E4Action(self.trUtf8('Show difference (URLs)'),
                self.trUtf8('Show difference (URLs)'),
                0, 0, self, 'subversion_urldiff')
        self.svnUrlDiffAct.setStatusTip(self.trUtf8(
            'Show the difference of the project between two repository URLs'
        ))
        self.svnUrlDiffAct.setWhatsThis(self.trUtf8(
            """<b>Show difference (URLs)</b>"""
            """<p>This shows the difference of the project between"""
            """ two repository URLs.</p>"""
        ))
        self.svnUrlDiffAct.connectIt(SIGNAL('triggered()'), self.__svnUrlDiff)
        self.actions.append(self.svnUrlDiffAct)
        
        self.vcsStatusAct = E4Action(self.trUtf8('Show status'),
                self.trUtf8('Show &status'),
                0, 0, self, 'subversion_status')
        self.vcsStatusAct.setStatusTip(self.trUtf8(
            'Show the status of the local project'
        ))
        self.vcsStatusAct.setWhatsThis(self.trUtf8(
            """<b>Show status</b>"""
            """<p>This shows the status of the local project.</p>"""
        ))
        self.vcsStatusAct.connectIt(SIGNAL('triggered()'), self._vcsStatus)
        self.actions.append(self.vcsStatusAct)
        
        self.vcsTagAct = E4Action(self.trUtf8('Tag in repository'), 
                self.trUtf8('&Tag in repository...'),
                0, 0, self, 'subversion_tag')
        self.vcsTagAct.setStatusTip(self.trUtf8(
            'Tag the local project in the repository'
        ))
        self.vcsTagAct.setWhatsThis(self.trUtf8(
            """<b>Tag in repository</b>"""
            """<p>This tags the local project in the repository.</p>"""
        ))
        self.vcsTagAct.connectIt(SIGNAL('triggered()'), self._vcsTag)
        self.actions.append(self.vcsTagAct)
        
        self.vcsExportAct = E4Action(self.trUtf8('Export from repository'), 
                self.trUtf8('&Export from repository...'),
                0, 0, self, 'subversion_export')
        self.vcsExportAct.setStatusTip(self.trUtf8(
            'Export a project from the repository'
        ))
        self.vcsExportAct.setWhatsThis(self.trUtf8(
            """<b>Export from repository</b>"""
            """<p>This exports a project from the repository.</p>"""
        ))
        self.vcsExportAct.connectIt(SIGNAL('triggered()'), self._vcsExport)
        self.actions.append(self.vcsExportAct)
        
        self.vcsPropsAct = E4Action(self.trUtf8('Command options'),
                self.trUtf8('Command &options...'),0,0,self,
                'subversion_options')
        self.vcsPropsAct.setStatusTip(self.trUtf8('Show the VCS command options'))
        self.vcsPropsAct.setWhatsThis(self.trUtf8(
            """<b>Command options...</b>"""
            """<p>This shows a dialog to edit the VCS command options.</p>"""
        ))
        self.vcsPropsAct.connectIt(SIGNAL('triggered()'), self._vcsCommandOptions)
        self.actions.append(self.vcsPropsAct)
        
        self.vcsRevertAct = E4Action(self.trUtf8('Revert changes'),
                self.trUtf8('Re&vert changes'),
                0, 0, self, 'subversion_revert')
        self.vcsRevertAct.setStatusTip(self.trUtf8(
            'Revert all changes made to the local project'
        ))
        self.vcsRevertAct.setWhatsThis(self.trUtf8(
            """<b>Revert changes</b>"""
            """<p>This reverts all changes made to the local project.</p>"""
        ))
        self.vcsRevertAct.connectIt(SIGNAL('triggered()'), self._vcsRevert)
        self.actions.append(self.vcsRevertAct)
        
        self.vcsMergeAct = E4Action(self.trUtf8('Merge'),
                self.trUtf8('Mer&ge changes...'),
                0, 0, self, 'subversion_merge')
        self.vcsMergeAct.setStatusTip(self.trUtf8(
            'Merge changes of a tag/revision into the local project'
        ))
        self.vcsMergeAct.setWhatsThis(self.trUtf8(
            """<b>Merge</b>"""
            """<p>This merges changes of a tag/revision into the local project.</p>"""
        ))
        self.vcsMergeAct.connectIt(SIGNAL('triggered()'), self._vcsMerge)
        self.actions.append(self.vcsMergeAct)
        
        self.vcsSwitchAct = E4Action(self.trUtf8('Switch'),
                self.trUtf8('S&witch...'),
                0, 0, self, 'subversion_switch')
        self.vcsSwitchAct.setStatusTip(self.trUtf8(
            'Switch the local copy to another tag/branch'
        ))
        self.vcsSwitchAct.setWhatsThis(self.trUtf8(
            """<b>Switch</b>"""
            """<p>This switches the local copy to another tag/branch.</p>"""
        ))
        self.vcsSwitchAct.connectIt(SIGNAL('triggered()'), self._vcsSwitch)
        self.actions.append(self.vcsSwitchAct)
        
        self.vcsResolveAct = E4Action(self.trUtf8('Resolve conflicts'),
                self.trUtf8('Resolve con&flicts'),
                0, 0, self, 'subversion_resolve')
        self.vcsResolveAct.setStatusTip(self.trUtf8(
            'Resolve all conflicts of the local project'
        ))
        self.vcsResolveAct.setWhatsThis(self.trUtf8(
            """<b>Resolve conflicts</b>"""
            """<p>This resolves all conflicts of the local project.</p>"""
        ))
        self.vcsResolveAct.connectIt(SIGNAL('triggered()'), self.__svnResolve)
        self.actions.append(self.vcsResolveAct)
        
        self.vcsCleanupAct = E4Action(self.trUtf8('Cleanup'),
                self.trUtf8('Cleanu&p'),
                0, 0, self, 'subversion_cleanup')
        self.vcsCleanupAct.setStatusTip(self.trUtf8(
            'Cleanup the local project'
        ))
        self.vcsCleanupAct.setWhatsThis(self.trUtf8(
            """<b>Cleanup</b>"""
            """<p>This performs a cleanup of the local project.</p>"""
        ))
        self.vcsCleanupAct.connectIt(SIGNAL('triggered()'), self._vcsCleanup)
        self.actions.append(self.vcsCleanupAct)
        
        self.vcsCommandAct = E4Action(self.trUtf8('Execute command'),
                self.trUtf8('E&xecute command...'),
                0, 0, self, 'subversion_command')
        self.vcsCommandAct.setStatusTip(self.trUtf8(
            'Execute an arbitrary VCS command'
        ))
        self.vcsCommandAct.setWhatsThis(self.trUtf8(
            """<b>Execute command</b>"""
            """<p>This opens a dialog to enter an arbitrary VCS command.</p>"""
        ))
        self.vcsCommandAct.connectIt(SIGNAL('triggered()'), self._vcsCommand)
        self.actions.append(self.vcsCommandAct)
        
        self.svnTagListAct = E4Action(self.trUtf8('List tags'), 
                self.trUtf8('List tags...'),
                0, 0, self, 'subversion_list_tags')
        self.svnTagListAct.setStatusTip(self.trUtf8(
            'List tags of the project'
        ))
        self.svnTagListAct.setWhatsThis(self.trUtf8(
            """<b>List tags</b>"""
            """<p>This lists the tags of the project.</p>"""
        ))
        self.svnTagListAct.connectIt(SIGNAL('triggered()'), self.__svnTagList)
        self.actions.append(self.svnTagListAct)
        
        self.svnBranchListAct = E4Action(self.trUtf8('List branches'), 
                self.trUtf8('List branches...'),
                0, 0, self, 'subversion_list_branches')
        self.svnBranchListAct.setStatusTip(self.trUtf8(
            'List branches of the project'
        ))
        self.svnBranchListAct.setWhatsThis(self.trUtf8(
            """<b>List branches</b>"""
            """<p>This lists the branches of the project.</p>"""
        ))
        self.svnBranchListAct.connectIt(SIGNAL('triggered()'), self.__svnBranchList)
        self.actions.append(self.svnBranchListAct)
        
        self.svnListAct = E4Action(self.trUtf8('List repository contents'), 
                self.trUtf8('List repository contents...'),
                0, 0, self, 'subversion_contents')
        self.svnListAct.setStatusTip(self.trUtf8(
            'Lists the contents of the repository'
        ))
        self.svnListAct.setWhatsThis(self.trUtf8(
            """<b>List repository contents</b>"""
            """<p>This lists the contents of the repository.</p>"""
        ))
        self.svnListAct.connectIt(SIGNAL('triggered()'), self.__svnTagList)
        self.actions.append(self.svnListAct)
        
        self.svnPropSetAct = E4Action(self.trUtf8('Set Property'),
                self.trUtf8('Set Property...'),
                0, 0, self, 'subversion_property_set')
        self.svnPropSetAct.setStatusTip(self.trUtf8(
            'Set a property for the project files'
        ))
        self.svnPropSetAct.setWhatsThis(self.trUtf8(
            """<b>Set Property</b>"""
            """<p>This sets a property for the project files.</p>"""
        ))
        self.svnPropSetAct.connectIt(SIGNAL('triggered()'), self.__svnPropSet)
        self.actions.append(self.svnPropSetAct)
        
        self.svnPropListAct = E4Action(self.trUtf8('List Properties'),
                self.trUtf8('List Properties...'),
                0, 0, self, 'subversion_property_list')
        self.svnPropListAct.setStatusTip(self.trUtf8(
            'List properties of the project files'
        ))
        self.svnPropListAct.setWhatsThis(self.trUtf8(
            """<b>List Properties</b>"""
            """<p>This lists the properties of the project files.</p>"""
        ))
        self.svnPropListAct.connectIt(SIGNAL('triggered()'), self.__svnPropList)
        self.actions.append(self.svnPropListAct)
        
        self.svnPropDelAct = E4Action(self.trUtf8('Delete Property'),
                self.trUtf8('Delete Property...'),
                0, 0, self, 'subversion_property_delete')
        self.svnPropDelAct.setStatusTip(self.trUtf8(
            'Delete a property for the project files'
        ))
        self.svnPropDelAct.setWhatsThis(self.trUtf8(
            """<b>Delete Property</b>"""
            """<p>This deletes a property for the project files.</p>"""
        ))
        self.svnPropDelAct.connectIt(SIGNAL('triggered()'), self.__svnPropDel)
        self.actions.append(self.svnPropDelAct)
        
        self.svnRepoBrowserAct = E4Action(self.trUtf8('Repository Browser'),
                self.trUtf8('Repository Browser...'),
                0, 0, self, 'subversion_repo_browser')
        self.svnRepoBrowserAct.setStatusTip(self.trUtf8(
            'Show the Repository Browser dialog'
        ))
        self.svnRepoBrowserAct.setWhatsThis(self.trUtf8(
            """<b>Repository Browser</b>"""
            """<p>This shows the Repository Browser dialog.</p>"""
        ))
        self.svnRepoBrowserAct.connectIt(SIGNAL('triggered()'), self.__svnRepoBrowser)
        self.actions.append(self.svnRepoBrowserAct)
    
    def initMenu(self, menu):
        """
        Public method to generate the VCS menu.
        
        @param menu reference to the menu to be populated (QMenu)
        """
        menu.clear()
        
        act = menu.addAction(self.vcs.vcsName(), self._vcsInfoDisplay)
        font = act.font()
        font.setBold(True)
        act.setFont(font)
        menu.addSeparator()
        
        self.vcsUpdateAct.addTo(menu)
        self.vcsCommitAct.addTo(menu)
        menu.addSeparator()
        self.vcsNewAct.addTo(menu)
        self.vcsExportAct.addTo(menu)
        menu.addSeparator()
        self.vcsAddAct.addTo(menu)
        self.vcsRemoveAct.addTo(menu)
        menu.addSeparator()
        self.vcsTagAct.addTo(menu)
        if self.vcs.otherData["standardLayout"]:
            self.svnTagListAct.addTo(menu)
            self.svnBranchListAct.addTo(menu)
        else:
            self.svnListAct.addTo(menu)
        menu.addSeparator()
        self.vcsLogAct.addTo(menu)
        if self.vcs.versionStr >= '1.2.0':
            self.svnLogLimitedAct.addTo(menu)
        self.svnLogBrowserAct.addTo(menu)
        menu.addSeparator()
        self.vcsStatusAct.addTo(menu)
        menu.addSeparator()
        self.vcsDiffAct.addTo(menu)
        self.svnExtDiffAct.addTo(menu)
        self.svnUrlDiffAct.addTo(menu)
        menu.addSeparator()
        self.vcsRevertAct.addTo(menu)
        self.vcsMergeAct.addTo(menu)
        self.vcsResolveAct.addTo(menu)
        menu.addSeparator()
        self.vcsSwitchAct.addTo(menu)
        menu.addSeparator()
        self.svnPropSetAct.addTo(menu)
        self.svnPropListAct.addTo(menu)
        self.svnPropDelAct.addTo(menu)
        menu.addSeparator()
        self.vcsCleanupAct.addTo(menu)
        menu.addSeparator()
        self.vcsCommandAct.addTo(menu)
        self.svnRepoBrowserAct.addTo(menu)
        menu.addSeparator()
        self.vcsPropsAct.addTo(menu)
    
    def __svnResolve(self):
        """
        Private slot used to resolve conflicts of the local project.
        """
        self.vcs.svnResolve(self.project.ppath)
        
    def __svnPropList(self):
        """
        Private slot used to list the properties of the project files.
        """
        self.vcs.svnListProps(self.project.ppath, True)
        
    def __svnPropSet(self):
        """
        Private slot used to set a property for the project files.
        """
        self.vcs.svnSetProp(self.project.ppath, True)
        
    def __svnPropDel(self):
        """
        Private slot used to delete a property for the project files.
        """
        self.vcs.svnDelProp(self.project.ppath, True)
        
    def __svnTagList(self):
        """
        Private slot used to list the tags of the project.
        """
        self.vcs.svnListTagBranch(self.project.ppath, True)
        
    def __svnBranchList(self):
        """
        Private slot used to list the branches of the project.
        """
        self.vcs.svnListTagBranch(self.project.ppath, False)
        
    def __svnExtendedDiff(self):
        """
        Private slot used to perform a svn diff with the selection of revisions.
        """
        self.vcs.svnExtendedDiff(self.project.ppath)
        
    def __svnUrlDiff(self):
        """
        Private slot used to perform a svn diff with the selection of repository URLs.
        """
        self.vcs.svnUrlDiff(self.project.ppath)
        
    def __svnLogLimited(self):
        """
        Private slot used to perform a svn log --limit.
        """
        self.vcs.svnLogLimited(self.project.ppath)
        
    def __svnLogBrowser(self):
        """
        Private slot used to browse the log of the current project.
        """
        self.vcs.svnLogBrowser(self.project.ppath)
        
    def __svnRepoBrowser(self):
        """
        Private slot to open the repository browser.
        """
        self.vcs.svnRepoBrowser(projectPath = self.project.ppath)
