/* 
   elmo - ELectronic Mail Operator

   Copyright (C) 2003 rzyjontko

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; version 2.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

   ----------------------------------------------------------------------
   
   String manipulation routines.
   
*/
/****************************************************************************
 *    IMPLEMENTATION HEADERS
 ****************************************************************************/

#include <string.h>

#include "linech.h"
#include "xmalloc.h"

/****************************************************************************
 *    IMPLEMENTATION PRIVATE DEFINITIONS / ENUMERATIONS / SIMPLE TYPEDEFS
 ****************************************************************************/

#define MIN(a,b) (((a)<(b))?(a):(b))

/****************************************************************************
 *    IMPLEMENTATION PRIVATE CLASS PROTOTYPES / EXTERNAL CLASS REFERENCES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE STRUCTURES / UTILITY CLASSES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION REQUIRED EXTERNAL REFERENCES (AVOID)
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE DATA
 ****************************************************************************/

static void
copy_string (chtype *dest, const char *from, int len, int mask)
{
  while (len){
    *dest = (*from & ~A_ATTRIBUTES) | mask;
    dest++;
    from++;
    len--;
  }
}



static void
write_spaces (linech_t *line, int len)
{
  int i;
  
  for (i = len; i < line->filled - len; i++){
    line->str[i] = ' ';
  }
  line->filled = len;
}

/****************************************************************************
 *    INTERFACE DATA
 ****************************************************************************/

linech_t *
linech_create (int size)
{
  linech_t *result = xmalloc (sizeof (linech_t));

  result->filled    = size;
  result->size      = size;
  result->str       = xmalloc ((size + 1) * sizeof (chtype));
  result->str[size] = 0;
  result->mask      = 0;

  linech_put_str_len (result, NULL, 0);
  
  return result;
}



void
linech_destroy (linech_t *line)
{
  if (line == NULL || line->str == NULL)
    return;
  xfree (line->str);
  xfree (line);
}



void
linech_clear (linech_t *line)
{
  line->filled = line->size;

  linech_put_str_len (line, NULL, 0);
}



void
linech_put_str_len (linech_t *line, const char *str, int len)
{
  len = MIN (len, line->size - 1);
  copy_string (line->str, str, len, line->mask);
  write_spaces (line, len);
}



void
linech_set_mask (linech_t *line, int mask)
{
  line->mask |= mask;
}



void
linech_unset_mask (linech_t *line, int mask)
{
  line->mask &= ~mask;
}



void
linech_add_char (linech_t *line, char c)
{
  if (line->filled >= line->size - 1)
    return;
  line->str[line->filled] = c;
  line->filled++;
}



void
linech_add_str_len (linech_t *line, const char *str, int len)
{
  len = MIN (len, line->size - 1 - line->filled);
  copy_string (line->str + line->filled, str, len, line->mask);
  line->filled += len;
}



void
linech_add_str_pad (linech_t *line, const char *str, int pad_len)
{
  int i;
  int len;

  len = (str) ? strlen (str) : 0;

  if (len < pad_len){
    linech_add_str_len (line, str, len);
    for (i = 0; i < pad_len - len; i++)
      linech_add_char (line, ' ');
  }
  else {
    linech_add_str_len (line, str, pad_len);
  }
}

/****************************************************************************
 *    IMPLEMENTATION PRIVATE FUNCTION PROTOTYPES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE FUNCTIONS
 ****************************************************************************/
/****************************************************************************
 *    INTERFACE FUNCTIONS
 ****************************************************************************/
/****************************************************************************
 *    INTERFACE CLASS BODIES
 ****************************************************************************/
/****************************************************************************
 *
 *    END MODULE linech.c
 *
 ****************************************************************************/
