package de.lmu.ifi.dbs.elki.math.statistics.distribution;

/*
 This file is part of ELKI:
 Environment for Developing KDD-Applications Supported by Index-Structures

 Copyright (C) 2012
 Ludwig-Maximilians-Universität München
 Lehr- und Forschungseinheit für Datenbanksysteme
 ELKI Development Team

 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU Affero General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU Affero General Public License for more details.

 You should have received a copy of the GNU Affero General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
import org.junit.Test;

import de.lmu.ifi.dbs.elki.JUnit4Test;

/**
 * Unit test for the Normal distribution in ELKI.
 * 
 * The reference values were computed using GNU R and SciPy.
 * 
 * @author Erich Schubert
 */
public class TestLogNormalDistribution extends AbstractDistributionTest implements JUnit4Test {
  public static final double[] P_CDFPDF = { //
  1e-10, 1e-05, 0.1, 0.1234567, 0.2, 0.271828182846, 0.3, 0.314159265359, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0, 1.1, 1.2, 1.3, 1.4, 1.5, 1.6, 1.7, 1.8, 1.9, 2.0, 2.71828182846, 3.14159265359 //
  };

  public static final double[] SCIPY_LOGNORM_CDF_0_1 = { //
  1.28417563064343188586946855334205034593745303926717e-117, // 0.000000
  5.67797929684089614274940495423182419605501044534498e-31, // 0.000010
  1.06510993417001289390588780747748387511819601058960e-02, // 0.100000
  1.82253062118146111347805060631799278780817985534668e-02, // 0.123457
  5.37603104516631360998424327135580824688076972961426e-02, // 0.200000
  9.63582248817356212811091609182767570018768310546875e-02, // 0.271828
  1.14300045049151532960962640572688542306423187255859e-01, // 0.300000
  1.23461564770417142611336203117389231920242309570312e-01, // 0.314159
  1.79757213895785417090422697583562694489955902099609e-01, // 0.400000
  2.44108595785582749293496362952282652258872985839844e-01, // 0.500000
  3.04736582510231668319278242051950655877590179443359e-01, // 0.600000
  3.60667582622649085521970846457406878471374511718750e-01, // 0.700000
  4.11711891857454936882731999503448605537414550781250e-01, // 0.800000
  4.58044872785658863101332372025353834033012390136719e-01, // 0.900000
  5.00000000000000000000000000000000000000000000000000e-01, // 1.000000
  5.37965771424616989726530391635606065392494201660156e-01, // 1.100000
  5.72334808836768349493695495766587555408477783203125e-01, // 1.200000
  6.03479689632151705680485065386164933443069458007812e-01, // 1.300000
  6.31742607836675351684618817671434953808784484863281e-01, // 1.400000
  6.57432169485154149768391107500065118074417114257812e-01, // 1.500000
  6.80823787674826652605020171904470771551132202148438e-01, // 1.600000
  7.02161792699145248519698725431226193904876708984375e-01, // 1.700000
  7.21662252065084608432243840070441365242004394531250e-01, // 1.800000
  7.39515971034705121844865516322897747159004211425781e-01, // 1.900000
  7.55891404214417250706503637047717347741127014160156e-01, // 2.000000
  8.41344746068542925776512220181757584214210510253906e-01, // 2.718282
  8.73839466286804089634188130730763077735900878906250e-01, // 3.141593
  };

  public static final double[] SCIPY_LOGNORM_PDF_0_1 = { //
  2.96247992534875871443606538046919616608519359260632e-106, // 0.000000
  6.58561599261676419576702545979628229597991165133025e-25, // 0.000010
  2.81590189015268332894947889144532382488250732421875e-01, // 0.100000
  3.62394132775150645819906003453070297837257385253906e-01, // 0.123457
  5.46267870758180063006648197188042104244232177734375e-01, // 0.200000
  6.28312639712034104455540273193037137389183044433594e-01, // 0.271828
  6.44203257359199588094611499400343745946884155273438e-01, // 0.300000
  6.49600535087859753957673092372715473175048828125000e-01, // 0.314159
  6.55444168060311471712964248581556603312492370605469e-01, // 0.400000
  6.27496077115924366651711352460552006959915161132812e-01, // 0.500000
  5.83573822594503965888179664034396409988403320312500e-01, // 0.600000
  5.34794832076919912644541454938007518649101257324219e-01, // 0.700000
  4.86415781111553480187126297096256166696548461914062e-01, // 0.800000
  4.40815685912026689052822803205344825983047485351562e-01, // 0.900000
  3.98942280401432702863218082711682654917240142822266e-01, // 1.000000
  3.61031261229040001392576186844962649047374725341797e-01, // 1.100000
  3.26972024074255740444527873478364199399948120117188e-01, // 1.200000
  2.96496370636105011087124694313388317823410034179688e-01, // 1.300000
  2.69276228949932749934248477075016126036643981933594e-01, // 1.400000
  2.44973651710509943146831801641383208334445953369141e-01, // 1.500000
  2.23265447430299024400568441706127487123012542724609e-01, // 1.600000
  2.03854259497871476147778935228416230529546737670898e-01, // 1.700000
  1.86472448538908153192750205562333576381206512451172e-01, // 1.800000
  1.70882238241215372864800770003057550638914108276367e-01, // 1.900000
  1.56874019278981091662927838115138001739978790283203e-01, // 2.000000
  8.90160549159514924433267424319637939333915710449219e-02, // 2.718282
  6.59491245186608343598066994672990404069423675537109e-02, // 3.141593
  };

  public static final double[] GNUR_LOGNORM_CDF_0_1 = { //
  1.28417563064351817736427131462082407099073717791586e-117, // 0.000000
  5.67797929684099160620728708239478087513287207088142e-31, // 0.000010
  1.06510993417001341432293060051961219869554042816162e-02, // 0.100000
  1.82253062118146215431213619240224943496286869049072e-02, // 0.123457
  5.37603104516631430387363366207864601165056228637695e-02, // 0.200000
  9.63582248817956288355901506292866542935371398925781e-02, // 0.271828
  1.14300045049151519083174832758231787011027336120605e-01, // 0.300000
  1.23461564770430590187721975325985113158822059631348e-01, // 0.314159
  1.79757213895785389334847081954649183899164199829102e-01, // 0.400000
  2.44108595785582721537920747323369141668081283569336e-01, // 0.500000
  3.04736582510231668319278242051950655877590179443359e-01, // 0.600000
  3.60667582622649085521970846457406878471374511718750e-01, // 0.700000
  4.11711891857454936882731999503448605537414550781250e-01, // 0.800000
  4.58044872785658863101332372025353834033012390136719e-01, // 0.900000
  5.00000000000000000000000000000000000000000000000000e-01, // 1.000000
  5.37965771424616989726530391635606065392494201660156e-01, // 1.100000
  5.72334808836768349493695495766587555408477783203125e-01, // 1.200000
  6.03479689632151705680485065386164933443069458007812e-01, // 1.300000
  6.31742607836675351684618817671434953808784484863281e-01, // 1.400000
  6.57432169485154149768391107500065118074417114257812e-01, // 1.500000
  6.80823787674826763627322634420124813914299011230469e-01, // 1.600000
  7.02161792699145248519698725431226193904876708984375e-01, // 1.700000
  7.21662252065084608432243840070441365242004394531250e-01, // 1.800000
  7.39515971034705121844865516322897747159004211425781e-01, // 1.900000
  7.55891404214417250706503637047717347741127014160156e-01, // 2.000000
  8.41344746068627968860198507172754034399986267089844e-01, // 2.718282
  8.73839466286817856399693482671864330768585205078125e-01, // 3.141593
  };

  public static final double[] GNUR_LOGNORM_PDF_0_1 = { //
  2.96247992534875823036207602430441206383136443181234e-106, // 0.000000
  6.58561599261676327741206387988416669540449194645082e-25, // 0.000010
  2.81590189015268332894947889144532382488250732421875e-01, // 0.100000
  3.62394132775150590308754772195243276655673980712891e-01, // 0.123457
  5.46267870758180063006648197188042104244232177734375e-01, // 0.200000
  6.28312639712100828859320245101116597652435302734375e-01, // 0.271828
  6.44203257359199588094611499400343745946884155273438e-01, // 0.300000
  6.49600535087866637340425768343266099691390991210938e-01, // 0.314159
  6.55444168060311471712964248581556603312492370605469e-01, // 0.400000
  6.27496077115924366651711352460552006959915161132812e-01, // 0.500000
  5.83573822594503965888179664034396409988403320312500e-01, // 0.600000
  5.34794832076919912644541454938007518649101257324219e-01, // 0.700000
  4.86415781111553424675975065838429145514965057373047e-01, // 0.800000
  4.40815685912026689052822803205344825983047485351562e-01, // 0.900000
  3.98942280401432702863218082711682654917240142822266e-01, // 1.000000
  3.61031261229040001392576186844962649047374725341797e-01, // 1.100000
  3.26972024074255740444527873478364199399948120117188e-01, // 1.200000
  2.96496370636105011087124694313388317823410034179688e-01, // 1.300000
  2.69276228949932694423097245817189104855060577392578e-01, // 1.400000
  2.44973651710509915391256186012469697743654251098633e-01, // 1.500000
  2.23265447430298996644992826077213976532220840454102e-01, // 1.600000
  2.03854259497871476147778935228416230529546737670898e-01, // 1.700000
  1.86472448538908125437174589933420065790414810180664e-01, // 1.800000
  1.70882238241215372864800770003057550638914108276367e-01, // 1.900000
  1.56874019278981091662927838115138001739978790283203e-01, // 2.000000
  8.90160549158889452536769226753676775842905044555664e-02, // 2.718282
  6.59491245186515084863998481523594819009304046630859e-02, // 3.141593
  };

  public static final double[] SCIPY_LOGNORM_CDF_1_3 = { //
  5.80027942989772828348821244410095933142208312997823e-16, // 0.000000
  1.51649466283750378246646117741391890376689843833447e-05, // 0.000010
  1.35478427197421924521592018209048546850681304931641e-01, // 0.100000
  1.51359152068233360211024773889221251010894775390625e-01, // 0.123457
  1.92201402041897895234257021002122201025485992431641e-01, // 0.200000
  2.21383717792045020544833278108853846788406372070312e-01, // 0.271828
  2.31274023997502686800231685992912389338016510009766e-01, // 0.300000
  2.35982646811182916746219007109175436198711395263672e-01, // 0.314159
  2.61488373016912578616910423079389147460460662841797e-01, // 0.400000
  2.86246959246035603818825165944872424006462097167969e-01, // 0.500000
  3.07268244645081289867505347501719370484352111816406e-01, // 0.600000
  3.25553456641869298771041485451860353350639343261719e-01, // 0.700000
  3.41741639131648788918482750887051224708557128906250e-01, // 0.800000
  3.56267553868274411410510538189555518329143524169922e-01, // 0.900000
  3.69441340181763666628000919445184990763664245605469e-01, // 1.000000
  3.81492504412193622620463884231867268681526184082031e-01, // 1.100000
  3.92595931185516167083449090569047257304191589355469e-01, // 1.200000
  4.02888113368478295583940962387714534997940063476562e-01, // 1.300000
  4.12477728576550517480825419625034555792808532714844e-01, // 1.400000
  4.21452783486494730880167480790987610816955566406250e-01, // 1.500000
  4.29885587708867089595798915979685261845588684082031e-01, // 1.600000
  4.37836307018417136838195347081637009978294372558594e-01, // 1.700000
  4.45355558873167967082196128103532828390598297119141e-01, // 1.800000
  4.52486345581900428758359566927538253366947174072266e-01, // 1.900000
  4.59265519021804791144347746012499555945396423339844e-01, // 2.000000
  5.00000000000000000000000000000000000000000000000000e-01, // 2.718282
  5.19238827154144821918180241482332348823547363281250e-01, // 3.141593
  };

  public static final double[] SCIPY_LOGNORM_PDF_1_3 = { //
  1.57184900048158173423151762193583635962568223476410e-05, // 0.000000
  2.21855874552272958055709750624373555183410644531250e+00, // 0.000010
  7.25485663598960939246751422615488991141319274902344e-01, // 0.100000
  6.33321891615805876440958854800555855035781860351562e-01, // 0.123457
  4.55481616703077685226475068702711723744869232177734e-01, // 0.200000
  3.64395441271923081760775176007882691919803619384766e-01, // 0.271828
  3.38429691850187064350308219218277372419834136962891e-01, // 0.300000
  3.26808430238364144848617343086516484618186950683594e-01, // 0.314159
  2.71099452010358343834184324805391952395439147949219e-01, // 0.400000
  2.26804382443861807239215977460844442248344421386719e-01, // 0.500000
  1.95237994591654751008746870866161771118640899658203e-01, // 0.600000
  1.71507274769319928431343669217312708497047424316406e-01, // 0.700000
  1.52968565294889946581946560399956069886684417724609e-01, // 0.800000
  1.38059669586597599799659974451060406863689422607422e-01, // 0.900000
  1.25794409230997716875322112173307687044143676757812e-01, // 1.000000
  1.15517735625617068428994116402463987469673156738281e-01, // 1.100000
  1.06776574753620304147894159996212692931294441223145e-01, // 1.200000
  9.92470359816487873727552937452855985611677169799805e-02, // 1.300000
  9.26911340280602569219325914673390798270702362060547e-02, // 1.400000
  8.69298985658228523520918429312587250024080276489258e-02, // 1.500000
  8.18260363727473238881771067099180072546005249023438e-02, // 1.600000
  7.72723945276378165747388493400649167597293853759766e-02, // 1.700000
  7.31840719845132658960551452764775604009628295898438e-02, // 1.800000
  6.94928962728491861344792823729221709072589874267578e-02, // 1.900000
  6.61434746059730849232494165335083380341529846191406e-02, // 2.000000
  4.89208877245799755861810353962937369942665100097656e-02, // 2.718282
  4.22798605814237726807647277382784523069858551025391e-02, // 3.141593
  };

  public static final double[] GNUR_LOGNORM_CDF_1_3 = { //
  5.80027942989774504678244839060182433065409538918561e-16, // 0.000000
  1.51649466283750666237848184203507173606340074911714e-05, // 0.000010
  1.35478427197421869010440786951221525669097900390625e-01, // 0.100000
  1.51359152068233387966600389518134761601686477661133e-01, // 0.123457
  1.92201402041897867478681405373208690434694290161133e-01, // 0.200000
  2.21383717792079853792230892395309638231992721557617e-01, // 0.271828
  2.31274023997502603533504839106171857565641403198242e-01, // 0.300000
  2.35982646811189716862244836192985530942678451538086e-01, // 0.314159
  2.61488373016912634128061654337216168642044067382812e-01, // 0.400000
  2.86246959246035659329976397202699445188045501708984e-01, // 0.500000
  3.07268244645081289867505347501719370484352111816406e-01, // 0.600000
  3.25553456641869243259890254194033332169055938720703e-01, // 0.700000
  3.41741639131648788918482750887051224708557128906250e-01, // 0.800000
  3.56267553868274355899359306931728497147560119628906e-01, // 0.900000
  3.69441340181763666628000919445184990763664245605469e-01, // 1.000000
  3.81492504412193622620463884231867268681526184082031e-01, // 1.100000
  3.92595931185516167083449090569047257304191589355469e-01, // 1.200000
  4.02888113368478295583940962387714534997940063476562e-01, // 1.300000
  4.12477728576550517480825419625034555792808532714844e-01, // 1.400000
  4.21452783486494730880167480790987610816955566406250e-01, // 1.500000
  4.29885587708867089595798915979685261845588684082031e-01, // 1.600000
  4.37836307018417136838195347081637009978294372558594e-01, // 1.700000
  4.45355558873167967082196128103532828390598297119141e-01, // 1.800000
  4.52486345581900428758359566927538253366947174072266e-01, // 1.900000
  4.59265519021804791144347746012499555945396423339844e-01, // 2.000000
  5.00000000000046740389336719090351834893226623535156e-01, // 2.718282
  5.19238827154153592680074780219001695513725280761719e-01, // 3.141593
  };

  public static final double[] GNUR_LOGNORM_PDF_1_3 = { //
  1.57184900048159257625324247698017643415369093418121e-05, // 0.000000
  2.21855874552272913646788765618111938238143920898438e+00, // 0.000010
  7.25485663598960717202146497584180906414985656738281e-01, // 0.100000
  6.33321891615805654396353929769247770309448242187500e-01, // 0.123457
  4.55481616703077685226475068702711723744869232177734e-01, // 0.200000
  3.64395441271827880136413568834541365504264831542969e-01, // 0.271828
  3.38429691850187008839156987960450351238250732421875e-01, // 0.300000
  3.26808430238347880081306584543199278414249420166016e-01, // 0.314159
  2.71099452010358343834184324805391952395439147949219e-01, // 0.400000
  2.26804382443861779483640361831930931657552719116211e-01, // 0.500000
  1.95237994591654723253171255237248260527849197387695e-01, // 0.600000
  1.71507274769319928431343669217312708497047424316406e-01, // 0.700000
  1.52968565294889946581946560399956069886684417724609e-01, // 0.800000
  1.38059669586597572044084358822146896272897720336914e-01, // 0.900000
  1.25794409230997716875322112173307687044143676757812e-01, // 1.000000
  1.15517735625617040673418500773550476878881454467773e-01, // 1.100000
  1.06776574753620290270106352181755937635898590087891e-01, // 1.200000
  9.92470359816487596171796781163720879703760147094727e-02, // 1.300000
  9.26911340280602569219325914673390798270702362060547e-02, // 1.400000
  8.69298985658228384743040351168019697070121765136719e-02, // 1.500000
  8.18260363727473238881771067099180072546005249023438e-02, // 1.600000
  7.72723945276378165747388493400649167597293853759766e-02, // 1.700000
  7.31840719845132658960551452764775604009628295898438e-02, // 1.800000
  6.94928962728491861344792823729221709072589874267578e-02, // 1.900000
  6.61434746059730849232494165335083380341529846191406e-02, // 2.000000
  4.89208877245627948848749610988306812942028045654297e-02, // 2.718282
  4.22798605814209346731580296818719943985342979431152e-02, // 3.141593
  };

  public static final double[] SCIPY_LOGNORM_CDF_01_01 = { //
  0.00000000000000000000000000000000000000000000000000e+00, // 0.000000
  0.00000000000000000000000000000000000000000000000000e+00, // 0.000010
  7.46586729512055156851629424736742001201359883385653e-128, // 0.100000
  8.62525588462026443966250245228507150042446838603379e-107, // 0.123457
  8.17136334615500295339422451815302932260019361648268e-66, // 0.200000
  5.41526143458555017502898041505360963321909770679929e-45, // 0.271828
  3.63579783587386242094605540612077634804949224374312e-39, // 0.300000
  1.38530749241422417511125868885806575792955358046047e-36, // 0.314159
  1.45144331736744006446530515598948213843059576725939e-24, // 0.400000
  1.08281884688449672381822010295852147196911971320510e-15, // 0.500000
  5.03627752858171494558555774501791210684409350051283e-10, // 0.600000
  2.47672755812896728362052650718183599565236363559961e-06, // 0.700000
  6.15850565794841134717452213465094246203079819679260e-04, // 0.800000
  2.00069613308373993321165329462019144557416439056396e-02, // 0.900000
  1.58655253931456963201185317302588373422622680664062e-01, // 1.000000
  4.81297180574880578696195243537658825516700744628906e-01, // 1.100000
  7.94807291306340868430879709194414317607879638671875e-01, // 1.200000
  9.47773950742736737140603509033098816871643066406250e-01, // 1.300000
  9.90978204147032748494439147179946303367614746093750e-01, // 1.400000
  9.98873387527683287423485580802662298083305358886719e-01, // 1.500000
  9.99892215681807416949311573262093588709831237792969e-01, // 1.600000
  9.99991698949889484815400919615058228373527526855469e-01, // 1.700000
  9.99999463803087018654025541763985529541969299316406e-01, // 1.800000
  9.99999969955960299472508268081583082675933837890625e-01, // 1.900000
  9.99999998498844355765413638437166810035705566406250e-01, // 2.000000
  1.00000000000000000000000000000000000000000000000000e+00, // 2.718282
  1.00000000000000000000000000000000000000000000000000e+00, // 3.141593
  };

  public static final double[] SCIPY_LOGNORM_PDF_01_01 = { //
  0.00000000000000000000000000000000000000000000000000e+00, // 0.000000
  0.00000000000000000000000000000000000000000000000000e+00, // 0.000010
  1.79683490350738681626883878695903692927320056006734e-124, // 0.100000
  1.53451240609964396752170892833443226944498342251986e-103, // 0.123457
  7.00795903038145442928847865311558208490483375898988e-63, // 0.200000
  2.80824126786686071155513246881328889371784643139373e-42, // 0.271828
  1.58951508355228437334280261322875927189162769346195e-36, // 0.300000
  5.58122798783872171035008357202465427911692811792789e-34, // 0.314159
  3.72276530194518232247109146801702251947511489941735e-22, // 0.400000
  1.74416764530574268824702660517399569335391176783112e-13, // 0.500000
  5.25802665152593504051015431063897942465246160281822e-08, // 0.600000
  1.68718532893121360123969698641133163619088008999825e-04, // 0.700000
  2.69358375879647123085636195582992513664066791534424e-02, // 0.800000
  5.38138135622016955394997239636722952127456665039062e-01, // 0.900000
  2.41970724519143187691838647879194468259811401367188e+00, // 1.000000
  3.62276178615723765119582822080701589584350585937500e+00, // 1.100000
  2.36903587207174082251981417357455939054489135742188e+00, // 1.200000
  8.21334006428800944554780016915174201130867004394531e-01, // 1.300000
  1.73985796664595016025600671127904206514358520507812e-01, // 1.400000
  2.50403898972403581024703811408471665345132350921631e-02, // 1.500000
  2.65452021073224507866084032059461605967953801155090e-03, // 1.600000
  2.20631598630764908571672000547891911992337554693222e-04, // 1.700000
  1.50982513130951377124371742821651309895969461649656e-05, // 1.800000
  8.84282704123672517376785005144279594446743431035429e-07, // 1.900000
  4.57223083211032647909217801161785299512985147885047e-08, // 2.000000
  3.78171735691457591044937145833685229912833032037698e-18, // 2.718282
  2.52927370006418196229588945092783163653520317423423e-24, // 3.141593
  };

  public static final double[] GNUR_LOGNORM_CDF_01_01 = { //
  0.00000000000000000000000000000000000000000000000000e+00, // 0.000000
  0.00000000000000000000000000000000000000000000000000e+00, // 0.000010
  7.46586729512139622334915485545022576086084233165236e-128, // 0.100000
  8.62525588462074367291196505542133273171533757207493e-107, // 0.123457
  8.17136334615526229146362798902434829752185485895027e-66, // 0.200000
  5.41526143485395691388299267636695240314190437945746e-45, // 0.271828
  3.63579783587375997366590782928570699643197691594649e-39, // 0.300000
  1.38530749242582464214423634101122722133110779968500e-36, // 0.314159
  1.45144331736743767674240504821798157693450453457286e-24, // 0.400000
  1.08281884688450796508611950237674858910117499290887e-15, // 0.500000
  5.03627752858173148919780880557140953501793489976990e-10, // 0.600000
  2.47672755812898803592773423754014316955363028682768e-06, // 0.700000
  6.15850565794842218919624698969528253655880689620972e-04, // 0.800000
  2.00069613308374652516086200648715021088719367980957e-02, // 0.900000
  1.58655253931457046467912164189328905194997787475586e-01, // 1.000000
  4.81297180574880967274253862342447973787784576416016e-01, // 1.100000
  7.94807291306340868430879709194414317607879638671875e-01, // 1.200000
  9.47773950742736737140603509033098816871643066406250e-01, // 1.300000
  9.90978204147032748494439147179946303367614746093750e-01, // 1.400000
  9.98873387527683287423485580802662298083305358886719e-01, // 1.500000
  9.99892215681807416949311573262093588709831237792969e-01, // 1.600000
  9.99991698949889484815400919615058228373527526855469e-01, // 1.700000
  9.99999463803087018654025541763985529541969299316406e-01, // 1.800000
  9.99999969955960299472508268081583082675933837890625e-01, // 1.900000
  9.99999998498844355765413638437166810035705566406250e-01, // 2.000000
  1.00000000000000000000000000000000000000000000000000e+00, // 2.718282
  1.00000000000000000000000000000000000000000000000000e+00, // 3.141593
  };

  public static final double[] GNUR_LOGNORM_PDF_01_01 = { //
  0.00000000000000000000000000000000000000000000000000e+00, // 0.000000
  0.00000000000000000000000000000000000000000000000000e+00, // 0.000010
  1.79683490350738660633521042548751343580552590537274e-124, // 0.100000
  1.53451240609960034664638006561340724714793008505265e-103, // 0.123457
  7.00795903038125471789064209408362002551800649974979e-63, // 0.200000
  2.80824126800431268164764045127534069443724460598013e-42, // 0.271828
  1.58951508355223927043831980742723065033874451220525e-36, // 0.300000
  5.58122798788467359181540305857831591200353537133688e-34, // 0.314159
  3.72276530194518138207561081018701614448588512162081e-22, // 0.400000
  1.74416764530575505758602047063910454925781540191920e-13, // 0.500000
  5.25802665152591651166443312281906230509775923565030e-08, // 0.600000
  1.68718532893122579851413744833621422003488987684250e-04, // 0.700000
  2.69358375879647331252453312799843843095004558563232e-02, // 0.800000
  5.38138135622017954595719402277609333395957946777344e-01, // 0.900000
  2.41970724519143365327522587904240936040878295898438e+00, // 1.000000
  3.62276178615723720710661837074439972639083862304688e+00, // 1.100000
  2.36903587207174037843060432351194322109222412109375e+00, // 1.200000
  8.21334006428800944554780016915174201130867004394531e-01, // 1.300000
  1.73985796664594405402937127291806973516941070556641e-01, // 1.400000
  2.50403898972401985079105912745944806374609470367432e-02, // 1.500000
  2.65452021073223987449041239017333282390609383583069e-03, // 1.600000
  2.20631598630763743054336578630625353980576619505882e-04, // 1.700000
  1.50982513130949767761771959651007080083218170329928e-05, // 1.800000
  8.84282704123670929190008903331143841342054656706750e-07, // 1.900000
  4.57223083211027817174440491480164050486223459301982e-08, // 2.000000
  3.78171735679367296188854058553106900541344188905750e-18, // 2.718282
  2.52927370004658260781217201514446220919197886472664e-24, // 3.141593
  };

  public static final double[] P_PROBIT = { //
  0.0001, 0.001, 0.01, 0.1, 0.25, 0.5, 0.75, 0.9, 0.99, 0.999, 0.9999 //
  };

  public static final double[] SCIPY_LOGNORM_QUANT_0_1 = { //
  2.42578140301463428263328836465007043443620204925537e-02, // 0.000100
  4.54913852476535249258482451750751351937651634216309e-02, // 0.001000
  9.76517330703359914823025178520765621215105056762695e-02, // 0.010000
  2.77606241852009827386638107782346196472644805908203e-01, // 0.100000
  5.09416283863277530308266705105779692530632019042969e-01, // 0.250000
  1.00000000000000000000000000000000000000000000000000e+00, // 0.500000
  1.96303108415825700028278788522584363818168640136719e+00, // 0.750000
  3.60222447927915734311454798444174230098724365234375e+00, // 0.900000
  1.02404736563121314674162931623868644237518310546875e+01, // 0.990000
  2.19821839795828282149159349501132965087890625000000e+01, // 0.999000
  4.12238299278443065531973843462765216827392578125000e+01, // 0.999900
  };

  public static final double[] GNUR_LOGNORM_QUANT_0_1 = { //
  2.42578140301463567041206914609574596397578716278076e-02, // 0.000100
  4.54913852476535249258482451750751351937651634216309e-02, // 0.001000
  9.76517330703359914823025178520765621215105056762695e-02, // 0.010000
  2.77606241852009716364335645266692154109477996826172e-01, // 0.100000
  5.09416283863277530308266705105779692530632019042969e-01, // 0.250000
  1.00000000000000000000000000000000000000000000000000e+00, // 0.500000
  1.96303108415825700028278788522584363818168640136719e+00, // 0.750000
  3.60222447927915911947138738469220697879791259765625e+00, // 0.900000
  1.02404736563121314674162931623868644237518310546875e+01, // 0.990000
  2.19821839795828282149159349501132965087890625000000e+01, // 0.999000
  4.12238299278442923423426691442728042602539062500000e+01, // 0.999900
  };

  public static final double[] SCIPY_LOGNORM_QUANT_1_3 = { //
  3.88015852051401416715654957290126958469045348465443e-05, // 0.000100
  2.55906882854475099201702059659169208316598087549210e-04, // 0.001000
  2.53124596262631687548783077090774895623326301574707e-03, // 0.010000
  5.81543441640198222053470544778974726796150207519531e-02, // 0.100000
  3.59346113463704264212594807759160175919532775878906e-01, // 0.250000
  2.71828182845904509079559829842764884233474731445312e+00, // 0.500000
  2.05625045661638417016092716949060559272766113281250e+01, // 0.750000
  1.27059400379280148740690492559224367141723632812500e+02, // 0.900000
  2.91913792971113252860959619283676147460937500000000e+03, // 0.990000
  2.88740029830793428118340671062469482421875000000000e+04, // 0.999000
  1.90431809934209391940385103225708007812500000000000e+05, // 0.999900
  };

  public static final double[] GNUR_LOGNORM_QUANT_1_3 = { //
  3.88015852051401484478290737634154083934845402836800e-05, // 0.000100
  2.55906882854475099201702059659169208316598087549210e-04, // 0.001000
  2.53124596262631687548783077090774895623326301574707e-03, // 0.010000
  5.81543441640197528164080154056136962026357650756836e-02, // 0.100000
  3.59346113463704319723746039016987197101116180419922e-01, // 0.250000
  2.71828182845904509079559829842764884233474731445312e+00, // 0.500000
  2.05625045661638417016092716949060559272766113281250e+01, // 0.750000
  1.27059400379280333481801790185272693634033203125000e+02, // 0.900000
  2.91913792971113298335694707930088043212890625000000e+03, // 0.990000
  2.88740029830793428118340671062469482421875000000000e+04, // 0.999000
  1.90431809934209421044215559959411621093750000000000e+05, // 0.999900
  };

  public static final double[] SCIPY_LOGNORM_QUANT_01_01 = { //
  7.61929194249313157527581097383517771959304809570312e-01, // 0.000100
  8.11376386650571768122119920008117333054542541503906e-01, // 0.001000
  8.75784881379674473578234028536826372146606445312500e-01, // 0.010000
  9.72237506076474744531878968700766563415527343750000e-01, // 0.100000
  1.03308660502291438376687437994405627250671386718750e+00, // 0.250000
  1.10517091807564771244187795673497021198272705078125e+00, // 0.500000
  1.18228496257879456265982298646122217178344726562500e+00, // 0.750000
  1.25628023042354852734092673927079886198043823242188e+00, // 0.900000
  1.39463786613445961393153993412852287292480468750000e+00, // 0.990000
  1.50534669021145739442602007329696789383888244628906e+00, // 0.999000
  1.60303971468576467174216304556466639041900634765625e+00, // 0.999900
  };

  public static final double[] GNUR_LOGNORM_QUANT_01_01 = { //
  7.61929194249313157527581097383517771959304809570312e-01, // 0.000100
  8.11376386650571768122119920008117333054542541503906e-01, // 0.001000
  8.75784881379674362555931566021172329783439636230469e-01, // 0.010000
  9.72237506076474744531878968700766563415527343750000e-01, // 0.100000
  1.03308660502291416172226945491274818778038024902344e+00, // 0.250000
  1.10517091807564771244187795673497021198272705078125e+00, // 0.500000
  1.18228496257879434061521806142991408705711364746094e+00, // 0.750000
  1.25628023042354852734092673927079886198043823242188e+00, // 0.900000
  1.39463786613445961393153993412852287292480468750000e+00, // 0.990000
  1.50534669021145739442602007329696789383888244628906e+00, // 0.999000
  1.60303971468576444969755812053335830569267272949219e+00, // 0.999900
  };

  @Test
  public void testPDF() {
    checkPDF(new LogNormalDistribution(0., 1., 0), P_CDFPDF, GNUR_LOGNORM_PDF_0_1, 1e-15);
    checkPDF(new LogNormalDistribution(1., 3., 0), P_CDFPDF, GNUR_LOGNORM_PDF_1_3, 1e-15);
    checkPDF(new LogNormalDistribution(.1, .1, 0), P_CDFPDF, GNUR_LOGNORM_PDF_01_01, 1e-15);
    checkPDF(new LogNormalDistribution(0., 1., 0), P_CDFPDF, SCIPY_LOGNORM_PDF_0_1, 1e-12);
    checkPDF(new LogNormalDistribution(1., 3., 0), P_CDFPDF, SCIPY_LOGNORM_PDF_1_3, 1e-12);
    checkPDF(new LogNormalDistribution(.1, .1, 0), P_CDFPDF, SCIPY_LOGNORM_PDF_01_01, 1e-14);
  }

  @Test
  public void testCDF() {
    checkCDF(new LogNormalDistribution(0., 1., 0), P_CDFPDF, GNUR_LOGNORM_CDF_0_1, 1e-15);
    checkCDF(new LogNormalDistribution(1., 3., 0), P_CDFPDF, GNUR_LOGNORM_CDF_1_3, 1e-15);
    checkCDF(new LogNormalDistribution(.1, .1, 0), P_CDFPDF, GNUR_LOGNORM_CDF_01_01, 1e-15);
    checkCDF(new LogNormalDistribution(0., 1., 0), P_CDFPDF, SCIPY_LOGNORM_CDF_0_1, 1e-12);
    checkCDF(new LogNormalDistribution(1., 3., 0), P_CDFPDF, SCIPY_LOGNORM_CDF_1_3, 1e-12);
    checkCDF(new LogNormalDistribution(.1, .1, 0), P_CDFPDF, SCIPY_LOGNORM_CDF_01_01, 1e-15);
  }

  @Test
  public void testProbit() {
    checkQuantile(new LogNormalDistribution(0., 1., 0), P_PROBIT, GNUR_LOGNORM_QUANT_0_1, 1e-8);
    checkQuantile(new LogNormalDistribution(1., 3., 0), P_PROBIT, GNUR_LOGNORM_QUANT_1_3, 1e-8);
    checkQuantile(new LogNormalDistribution(.1, .1, 0), P_PROBIT, GNUR_LOGNORM_QUANT_01_01, 1e-9);
    checkQuantile(new LogNormalDistribution(0., 1., 0), P_PROBIT, SCIPY_LOGNORM_QUANT_0_1, 1e-8);
    checkQuantile(new LogNormalDistribution(1., 3., 0), P_PROBIT, SCIPY_LOGNORM_QUANT_1_3, 1e-8);
    checkQuantile(new LogNormalDistribution(.1, .1, 0), P_PROBIT, SCIPY_LOGNORM_QUANT_01_01, 1e-9);
  }
}
