/* This file is part of EdiTeX, an editor of mathematical
 * expressions based on TeX syntax.
 * 
 * Copyright (C) 2002-2003 Luca Padovani <lpadovan@cs.unibo.it>,
 *                    2003 Paolo Marinelli <pmarinel@cs.unibo.it>.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For more information, please visit the project's home page
 * http://helm.cs.unibo.it/editex/
 * or send an email to <lpadovan@cs.unibo.it>
 */

#include <getopt.h>
#include <fstream>
#include <cassert>

#include "dom.hh"
#include "TPushParser.hh"
#include "TPushLexer.hh"
#include "TDictionary.hh"
#include "CLoggerConsole.hh"
#include "CMathMLFactoryXSLT.hh"
#include "CMathMLFactoryXSLTDiff.hh"
#include "AMathMLConsumer.hh"

#include "config.dirs"

enum CommandLineOptionId {
  OPTION_VERSION = 256,
  OPTION_HELP,
  OPTION_VERBOSE,
  OPTION_DICTIONARY,
  OPTION_TML_XSLT,
  OPTION_XSLT
};

static std::string tml_xslt = PKGDATADIR"/tml-mmlp.xsl";
static std::string dictionary = PKGDATADIR"/dictionary-tex.xml";
static bool xslt = true;

static bool
parseBoolean(const char* s, bool& res)
{
  assert(s != NULL);
  if (!strcmp(s, "yes")) {
    res = true;
    return true;
  } else if (!strcmp(s, "no")) {
    res = false;
    return true;
  }

  return false;
}

static void
printVersion()
{
  std::cout << "TeX to MathML converter " << VERSION << " - Luca Padovani (C) 2003" << std::endl
	    << "This program is covered by the GNU Lesser General Public Licence" << std::endl;
}

static void
printHelp()
{
  static char* helpMsg = "\
Usage: textomml [options] file\n\n\
  -V, --version                   Output version information\n\
  -h, --help                      This small usage guide\n\
  -v, --verbose[=0-3]             Display messages\n\
      --dictionary=<path>         Full path of the dictionary\n\
      --tml-xslt=<path>           Full path of the XSLT stylesheet\n\
      --xslt[=yes|no]             Enable/disable XSLT transformation (default='yes')\n\
";

  std::cout << helpMsg << std::endl;
  exit(0);
}

static void
parseError(const char* option)
{
  assert(option != NULL);
  std::cerr << "error while parsing option `" << option << "'" << std::endl << std::endl;
  printHelp();
}

int
main(int argc, char* argv[])
{
  CLoggerConsole logger;

  while (TRUE) {
    int option_index = 0;
    static struct option long_options[] =
    {
      { "version", 	 no_argument,       NULL, OPTION_VERSION },
      { "help",    	 no_argument,       NULL, OPTION_HELP },
      { "verbose",       optional_argument, NULL, OPTION_VERBOSE },
      { "dictionary",    required_argument, NULL, OPTION_DICTIONARY },
      { "tml-xslt",      required_argument, NULL, OPTION_TML_XSLT },
      { "xslt",          optional_argument, NULL, OPTION_XSLT },

      { NULL,            no_argument, NULL, 0 }
    };

    int c = getopt_long(argc, argv, "Vhv::", long_options, &option_index);

    if (c == -1) break;

    switch (c) {
    case OPTION_VERSION:
    case 'V':
      printVersion();
      break;

    case OPTION_HELP:
    case 'h':
      printHelp();
      break;

    case OPTION_VERBOSE:
    case 'v':
      if (optarg == NULL) logger.verbosity(ALogger::Warning);
      else logger.verbosity(ALogger::Level(*optarg - '0'));
      break;

    case OPTION_DICTIONARY:
      dictionary = optarg;
      break;

    case OPTION_TML_XSLT:
      tml_xslt = optarg;
      break;

    case OPTION_XSLT:
      if (optarg == NULL) printHelp();
      else if (!parseBoolean(optarg, xslt)) parseError("xslt");
      break;

    case '?':
      break;

    default:
      std::cerr << "*** getopt returned `" << c << "' value" << std::endl;
      break;
    }
  }

  TDictionary dict(logger);
  logger.info("loading dictionary: `" + dictionary + "'");
  dict.load("dictionary-test.xml");

  logger.info("loading stylesheet: `" + tml_xslt + "'");
  DOM::DOMImplementation di;
  DOM::Document docStyle = di.createDocumentFromURI("./xsl/tml-mmlp.xsl");
  DOMX::XSLTStylesheet style(docStyle);

  CMathMLFactoryXSLT factory(logger, style);
  TPushParser parser(logger, factory, dict);
  TPushLexer lexer(logger, parser);

  if (optind < argc)
    {
      std::ifstream file(argv[optind]);
      if (!file)
	{
	  std::cerr << "can't open input file `" << argv[optind] << "'" << std::endl;
	  exit(1);
	}
      
      parser.freeze();
      char ch;
      while (file.get(ch)) lexer.push(ch);
      parser.thaw();
    }
  else
    printHelp();
}
