/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_suite.t.cpp
 *
 * @brief [LEVEL: beta] testing @ref
 * diagnostics::unittest::Test_Suite
 *
 * $Id: test_suite.t.cpp,v 1.12 2005/06/23 09:54:25 esdentem Exp $
 *
 * @author Christian Schallhart
 */
#include <diagnostics/unittest.hpp>

#include <diagnostics/unittest/test_suite.hpp>

// used components
#include <diagnostics/unittest/test_system_exception.hpp>
#include <diagnostics/unittest/name_separators.hpp>
#include <diagnostics/util/to_string.hpp>

// test support
#include <diagnostics/util/dummy_test_case.ts.hpp>

#define TEST_COMPONENT_NAMESPACE diagnostics::unittest
#define TEST_COMPONENT_NAME Test_Suite

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;
TEST_COMPONENT_TEST_NAMESPACE_BEGIN;

/**
 * @brief [PRO] fct/normal: normal usage
 */
void std_usage(Test_Data & test_data)
{
    Test_Suite s("suite");
    TEST_ASSERT(s.name()=="suite");
    TEST_ASSERT(s.test_cases().size()==0);
    TEST_ASSERT(s.test_suites().size()==0);

    Stupid_Test_Case *t(new Stupid_Test_Case("case"));
    s.add(t);
    TEST_ASSERT(s.test_cases().size()==1);

    Stupid_Test_Case *t2(new Stupid_Test_Case("case2"));
    s.add(t2);
    TEST_ASSERT(s.test_cases().size()==2);

    Test_Suite *s2(new Test_Suite("suite2"));
    s.add(s2);
    TEST_ASSERT(s.test_suites().size()==1);

    Test_Suite *s3(new Test_Suite("suite3"));
    s.add(s3);
    TEST_ASSERT(s.test_suites().size()==2);
}

/**
 * @brief [PRO] fct/abnormal: invalid attempts to add a @ref Test_Case
 */
void invalid_case_adds(Test_Data & test_data)
{
    // valid beginning
    Test_Suite s("suite");
    Stupid_Test_Case *t(new Stupid_Test_Case("case"));
    s.add(t);
    TEST_ASSERT(s.test_cases().size()==1);

    // same name
    TEST_THROWING_BLOCK_ENTER;
    Stupid_Test_Case *t2(new Stupid_Test_Case("case")); // never mind the leak
    s.add(t2);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    TEST_ASSERT(s.test_cases().size()==1);

    // NULL pointer
    TEST_THROWING_BLOCK_ENTER;
    s.add(static_cast<Test_Case*>(NULL));
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    TEST_ASSERT(s.test_cases().size()==1);


    // correct usage to check
    Stupid_Test_Case *t2(new Stupid_Test_Case("case2"));
    s.add(t2);
    TEST_ASSERT(s.test_cases().size()==2);
}

/**
 * @brief [PRO] fct/abnormal: invalid attempts to add another Test_Suite
 */
void invalid_suite_adds(Test_Data & test_data)
{
    // valid beginning
    Test_Suite s("suite");
    Test_Suite *s2=new Test_Suite("suite2");
    s.add(s2);
    TEST_ASSERT(s.test_suites().size()==1);

    // same name
    TEST_THROWING_BLOCK_ENTER;
    Test_Suite *s3(new Test_Suite("suite2")); // never mind the leak
    s.add(s3);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    TEST_ASSERT(s.test_suites().size()==1);

    // NULL pointer
    TEST_THROWING_BLOCK_ENTER;
    s.add(static_cast<Test_Suite*>(NULL));
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    TEST_ASSERT(s.test_suites().size()==1);

    // empty name
    TEST_THROWING_BLOCK_ENTER;
    Test_Suite *s3(new Test_Suite("")); // never mind the leak
    s.add(s3);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    TEST_ASSERT(s.test_suites().size()==1);

    // correct usage to check
    Test_Suite *s3(new Test_Suite("suite3"));
    s.add(s3);
    TEST_ASSERT(s.test_suites().size()==2);
}

/**
 * @brief [PRO] fct/abnormal: invalid construction attempts (case separator)
 */
void case_seperator(Test_Data & test_data)
{
    // at the end
    TEST_THROWING_BLOCK_ENTER;
    Test_Suite t("name1" DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    
    // at the beginning
    TEST_THROWING_BLOCK_ENTER;
    Test_Suite t(DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR "name1");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    
    // in the middle
    TEST_THROWING_BLOCK_ENTER;
    Test_Suite t("name" DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR "name1");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    
    // solely the separator
    TEST_THROWING_BLOCK_ENTER;
    Test_Suite t(DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR "name1");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
}

/**
 * @brief [PRO] fct/abnormal: invalid construction attempts (data separator)
 */
void data_seperator(Test_Data & test_data)
{
    // at the end
    TEST_THROWING_BLOCK_ENTER;
    Test_Suite t("name1" DIAGNOSTICS_UNITTEST_TEST_DATA_NAME_SEPARATOR);
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    
    // at the beginning
    TEST_THROWING_BLOCK_ENTER;
    Test_Suite t(DIAGNOSTICS_UNITTEST_TEST_DATA_NAME_SEPARATOR "name1");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    
    // in the middle
    TEST_THROWING_BLOCK_ENTER;
    Test_Suite t("name" DIAGNOSTICS_UNITTEST_TEST_DATA_NAME_SEPARATOR "name1");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
    
    // solely the separator
    TEST_THROWING_BLOCK_ENTER;
    Test_Suite t(DIAGNOSTICS_UNITTEST_TEST_DATA_NAME_SEPARATOR "name1");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);
}

TEST_COMPONENT_TEST_NAMESPACE_END;
TEST_NAMESPACE_END;
UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

TEST_SUITE_BEGIN;
TEST_NORMAL_CASE(&std_usage,LEVEL_PROD);
TEST_ABNORMAL_CASE(&invalid_case_adds,LEVEL_PROD);
TEST_ABNORMAL_CASE(&invalid_suite_adds,LEVEL_PROD);
TEST_ABNORMAL_CASE(&case_seperator,LEVEL_PROD);
TEST_ABNORMAL_CASE(&data_seperator,LEVEL_PROD);
TEST_SUITE_END;

STREAM_TEST_SYSTEM_MAIN;
// vim:ts=4:sw=4
