/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/macros/block_annotation.t.cpp 
 *
 * @brief [LEVEL: beta] testing @ref DIAGNOSTICS_BASE_BLOCK_ENTER,
 * @ref DIAGNOSTICS_BASE_BLOCK_EXIT, and @ref
 * DIAGNOSTICS_BASE_BLOCK_GUARD
 *
 * $Id: block_annotation.t.cpp,v 1.12 2005/06/23 09:54:21 esdentem Exp $
 *
 * @author Christian Schallhart
 */

#include <diagnostics/unittest.hpp>

// component
#include <diagnostics/macros/block_annotation.hpp>

// used component
#include <diagnostics/frame/logging_config.hpp>

// backdoor
#include <diagnostics/unittest/test_system/current_test_logger.hpp>

// test support
#include <diagnostics/util/dummy_logger.ts.hpp>
#include <diagnostics/util/assert_record.ts.hpp>

#define TEST_COMPONENT_NAME block_annotation
#define TEST_COMPONENT_NAMESPACE diagnostics


DIAGNOSTICS_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;
TEST_COMPONENT_TEST_NAMESPACE_BEGIN;
using namespace unittest;
using ::diagnostics::unittest::internal::Current_Test_Logger;


#define AC_THROW(LEVEL,NAME,WHAT) throw NAME(WHAT)
#define AC_NOOP(LEVEL,NAME,WHAT) ((void)0)

void block_guard(Test_Data & data)
{
    // never mind the leak 
    Dummy_Logger * logger=new Dummy_Logger;
    Dummy_Logger::Records_t & records(logger->records());
    typedef Dummy_Logger::Records_t::iterator Iter_t;

    Logging_Config::register_logger(logger);


    TEST_NO_IMPLICIT_LOGGING_ENTER;
    using ::diagnostics::internal::Block_Guard;
    {
	Block_Guard<TYPE_TRACE,TYPE_TRACE_BINARY,LEVEL_PROD,0,1> guard("what","base","file");
    }
    {
	Block_Guard<TYPE_BLOCK_ENTER,TYPE_BLOCK_EXIT,LEVEL_DEBUG,1,2> guard("what1","base1","file1");
    }

    {
	Block_Guard<TYPE_TRACE,TYPE_TRACE_BINARY,LEVEL_PROD,2,1> guard("what","base","file");
	Block_Guard<TYPE_BLOCK_ENTER,TYPE_BLOCK_EXIT,LEVEL_DEBUG,3,2> guard1("what1","base1","file1");
    }
    TEST_NO_IMPLICIT_LOGGING_EXIT;

    Logging_Config::unregister_logger(logger);

    // checking
    Iter_t current(records.begin()); 
    TEST_ASSERT(current->type()==TYPE_LOG_OPEN);
    ++current;

    TEST_ASSERT_RECORD(LEVEL_PROD,TYPE_TRACE,0,"what","base","file",1);
    ++current;
    TEST_ASSERT_RECORD(LEVEL_PROD,TYPE_TRACE_BINARY,0,"what","base","file",1);
    ++current;

    TEST_ASSERT_RECORD(LEVEL_DEBUG,TYPE_BLOCK_ENTER,1,"what1","base1","file1",2);
    ++current;
    TEST_ASSERT_RECORD(LEVEL_DEBUG,TYPE_BLOCK_EXIT,1,"what1","base1","file1",2);
    ++current;

    TEST_ASSERT_RECORD(LEVEL_PROD,TYPE_TRACE,2,"what","base","file",1);
    ++current;
    TEST_ASSERT_RECORD(LEVEL_DEBUG,TYPE_BLOCK_ENTER,3,"what1","base1","file1",2);
    ++current;
    TEST_ASSERT_RECORD(LEVEL_DEBUG,TYPE_BLOCK_EXIT,3,"what1","base1","file1",2);
    ++current;
    TEST_ASSERT_RECORD(LEVEL_PROD,TYPE_TRACE_BINARY,2,"what","base","file",1);
    ++current;

    TEST_ASSERT(current->type()==TYPE_LOG_CLOSE);
    TEST_ASSERT(current+1==records.end()); 
    delete logger;

}


void block_guard_macro(Test_Data & data)
{
    // never mind the leak 
    Dummy_Logger * logger=new Dummy_Logger;
    Dummy_Logger::Records_t & records(logger->records());
    typedef Dummy_Logger::Records_t::iterator Iter_t;

    Logging_Config::register_logger(logger);


    TEST_NO_IMPLICIT_LOGGING_ENTER;
    using ::diagnostics::internal::Block_Guard;
    {
	DIAGNOSTICS_BASE_BLOCK_GUARD(LEVEL_PROD,TYPE_TRACE,TYPE_TRACE_BINARY,10,"what");
    }
    {
	DIAGNOSTICS_BASE_BLOCK_GUARD(LEVEL_DEBUG,TYPE_BLOCK_ENTER,TYPE_BLOCK_EXIT,1000,"what1");
    }

    {
	DIAGNOSTICS_BASE_BLOCK_GUARD(LEVEL_PROD,TYPE_TRACE,TYPE_TRACE_BINARY,10,"what"); 
	{
	    DIAGNOSTICS_BASE_BLOCK_GUARD(LEVEL_DEBUG,TYPE_BLOCK_ENTER,TYPE_BLOCK_EXIT,1000,"what1");
	}
    }
    TEST_NO_IMPLICIT_LOGGING_EXIT;

    Logging_Config::unregister_logger(logger);

    // checking
    Iter_t current(records.begin()); 
    TEST_ASSERT(current->type()==TYPE_LOG_OPEN);
    ++current;

    TEST_ASSERT_RECORD1(LEVEL_PROD,TYPE_TRACE,10,"what");
    ++current;
    TEST_ASSERT_RECORD1(LEVEL_PROD,TYPE_TRACE_BINARY,10,"what");
    ++current;

    TEST_ASSERT_RECORD1(LEVEL_DEBUG,TYPE_BLOCK_ENTER,1000,"what1");
    ++current;
    TEST_ASSERT_RECORD1(LEVEL_DEBUG,TYPE_BLOCK_EXIT,1000,"what1");
    ++current;

    TEST_ASSERT_RECORD1(LEVEL_PROD,TYPE_TRACE,10,"what");
    ++current;
    TEST_ASSERT_RECORD1(LEVEL_DEBUG,TYPE_BLOCK_ENTER,1000,"what1");
    ++current;
    TEST_ASSERT_RECORD1(LEVEL_DEBUG,TYPE_BLOCK_EXIT,1000,"what1");
    ++current;
    TEST_ASSERT_RECORD1(LEVEL_PROD,TYPE_TRACE_BINARY,10,"what");
    ++current;

    TEST_ASSERT(current->type()==TYPE_LOG_CLOSE);
    TEST_ASSERT(current+1==records.end()); 
    delete logger;

}


void block_enter_exit(Test_Data & data)
{
    // never mind the leak 
    Dummy_Logger * logger=new Dummy_Logger;
    Dummy_Logger::Records_t & records(logger->records());
    typedef Dummy_Logger::Records_t::iterator Iter_t;

    Logging_Config::register_logger(logger);

    TEST_NO_IMPLICIT_LOGGING_ENTER;
    DIAGNOSTICS_BASE_BLOCK_ENTER(LEVEL_AUDIT,TYPE_BLOCK_ENTER,TYPE_BLOCK_EXIT,10,"what");
    DIAGNOSTICS_BASE_BLOCK_ENTER(LEVEL_DEBUG,TYPE_METHOD_ENTER,TYPE_METHOD_EXIT,1000,"what1");
    DIAGNOSTICS_BASE_BLOCK_EXIT;
    DIAGNOSTICS_BASE_BLOCK_EXIT;
    TEST_NO_IMPLICIT_LOGGING_EXIT;

    Logging_Config::unregister_logger(logger);

    // checking
    Iter_t current(records.begin()); 
    TEST_ASSERT(current->type()==TYPE_LOG_OPEN);
    ++current;

    TEST_ASSERT_RECORD1(LEVEL_AUDIT,TYPE_BLOCK_ENTER,10,"what");
    ++current;
    TEST_ASSERT_RECORD1(LEVEL_DEBUG,TYPE_METHOD_ENTER,1000,"what1");
    ++current;
    TEST_ASSERT_RECORD1(LEVEL_DEBUG,TYPE_METHOD_EXIT,1000,"what1");
    ++current;
    TEST_ASSERT_RECORD1(LEVEL_AUDIT,TYPE_BLOCK_EXIT,10,"what");
    ++current;


    TEST_ASSERT(current->type()==TYPE_LOG_CLOSE);
    TEST_ASSERT(current+1==records.end()); 
    	    
    delete logger;

}

TEST_COMPONENT_TEST_NAMESPACE_END;
TEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

TEST_SUITE_BEGIN;
TEST_NORMAL_CASE(&block_guard,LEVEL_PROD);
TEST_NORMAL_CASE(&block_guard_macro,LEVEL_PROD);
TEST_NORMAL_CASE(&block_enter_exit,LEVEL_PROD);
TEST_SUITE_END;

STREAM_TEST_SYSTEM_MAIN;

// vim:ts=4:sw=4
